wandy.analyzeRun <- function( run.info.filename, sample.info.filename, sample.res.DIR,
                              if.summarize.binMatrix = TRUE, run.res.DIR = NULL, 
                              static.ref.file = NULL) 
#=== when run.res.DIR is NULL, run.res.DIR = paste(sample.res.DIR,"_runSummary.DIR/")
#=== for debugging purpose if explicitily specified    
{
  #===============================
  #=== step 0. Initialization 
  #===============================
  
  options(stringsAsFactors = FALSE)
  
  #=== loading run information
  source(run.info.filename)
  
  #=== reading sample information 
  rSample.info <- read.delim(sample.info.filename)
  rSample.info <- rSample.info[which(rSample.info$sample.ID != ""), ] # remove empty lines
  
  NAN.type.idx <- which(is.na(rSample.info$sample.type))
  if( length(NAN.type.idx) > 0 ){
    rSample.info$sample.type[NAN.type.idx] <- "N/A"  
  }
  
  
  unique.SampleType <- sort(unique(rSample.info$sample.type))
  sample.color.vec <- seq( 1, length(unique.SampleType))
  sample.pch.vec <- seq( 1, length(unique.SampleType))
  names(sample.color.vec) <- names(sample.pch.vec) <- unique.SampleType
   
  run.ID <- unique(rSample.info$run.ID)
  if( length(run.ID) != 1  ){
    cat("\n run.ID:", run.ID, "\n")
    stop("only one unique run.ID is allowed, check sample information file \n")
  }
  
  cat( paste("*** Initialization: loading run-information of", run.ID, "\n") )
  
  if(is.null(run.res.DIR)){
    run.outputDIR <- paste(sample.res.DIR, "/", run.ID, "_runSummary.DIR/", sep = "")  }  
  
  if( !is.null(static.ref.file) ){ # using static reference
    wandy.create.DIR(run.outputDIR)
    run.outputDIR <- paste(run.outputDIR, "staticRef.Summary/", sep = "")
  }
  
  cat("*** creating run-summary DIR:", run.outputDIR, "\n")
  wandy.create.DIR(run.outputDIR)
  setwd(run.outputDIR)
  
  minimum.million.cvg <- 1
  sample.run.splitStr <- "_within_"
  RunLevel_Output_Prefix <- paste(run.outputDIR, run.ID, sep = "")
  
  if( !is.null(static.ref.file) ){ # using static reference
    RunLevel_Output_Prefix <- paste(RunLevel_Output_Prefix, "_statRef", sep = "")
  }
  
  
  #=== 0a define text output filenames
  
  
  sample.Stat.TXT.filename <- paste( RunLevel_Output_Prefix,
                                     "_RunLevel_sample_stat.txt", sep = "")
  
  
  chrPerctg.TXT.filename <- paste( RunLevel_Output_Prefix,
                                   "_Chr_Perctg.txt", sep = "")
  
  chrZscore.TXT.filename <- paste( RunLevel_Output_Prefix,
                                   "_Chr_Zscore.txt", sep = "")
  
  chrAltZscore.TXT.filename <- paste( RunLevel_Output_Prefix,
                                   "_Chr_AltZscore.txt", sep = "")
  
  chrLscore.TXT.filename <- paste( RunLevel_Output_Prefix,
                                   "_Chr_Lscore.txt", sep = "")
  
  chrPerctStat.TXT.filename <- paste( RunLevel_Output_Prefix,
                                      "_Chr_PerctStat.txt", sep = "")
  
  
  
  
  binNmlz.Covg.TXT.filename <- paste( RunLevel_Output_Prefix,
                                      "_bin_normalized_Mtx.txt", sep = "")
  
  
  #=== 0b define output plot filenames
  
  QCsummary.PDF.filename <- paste( RunLevel_Output_Prefix,
                                   "_QC_summary.pdf", sep = "")
  
  runReport.PDF.filename <- paste( RunLevel_Output_Prefix,
                                   "_run_Report.pdf", sep = "")
  
  
  
  
  
  #============================================================
  #=== step 1. summarizing sample-level stat
  #============================================================
  
  cat( paste("*** 1st step: summarizing sample-level statistics of", run.ID, "\n") )
  
  #=== merging sample-level stat
  sampleStat.filepath.vec <- dir(path = sample.res.DIR, pattern = "_SampleLevel_stat.txt", recursive = TRUE, full.names = TRUE)
  sampleStat.filename.Str <- unlist(lapply(strsplit(sampleStat.filepath.vec,split = "/"),tail,1))
  sampleStat.sampleID.vec <- unlist(lapply(strsplit(sampleStat.filename.Str,split = sample.run.splitStr),head,1))
  
  r1 <- read.delim(sampleStat.filepath.vec[1])
  N.sampleStat.sample <- length(sampleStat.sampleID.vec)
  N.sampleStat.col <- ncol(r1)
  sampleStat.mtx <- NULL
  
  
  #== check # of samples between sample.info and actually found sample stat files
  if( nrow(rSample.info) != N.sampleStat.sample){
    cat("Error \n")
    cat(nrow(rSample.info),"samples found in sample information file,")
    cat(" unmatched with", N.sampleStat.sample, "sample summary files\n")
    stop("please double check \n")
  }
  
  for(s.idx in 1 : N.sampleStat.sample){
    cat( s.idx, "-th sample:", sampleStat.sampleID.vec[s.idx], "\n")
    r1 <- read.delim(sampleStat.filepath.vec[s.idx])
    sampleStat.mtx <- rbind(sampleStat.mtx, r1)
  }
  
  
  
  
  #============================================================
  #=== step 2. summarizing chr percentages and z-scores
  #============================================================
  
  cat( paste("*** 2nd step: summarizing chr percentages and z-scores of", run.ID, "\n") )
  
  #=== chr% z-score 
  
  Chr.filepath.vec <- dir(path = sample.res.DIR, pattern = "_Chr_summary.txt", recursive = TRUE, full.names = TRUE)
  Chr.filename.Str <- unlist(lapply(strsplit(Chr.filepath.vec,split = "/"),tail,1))
  Chr.sampleID.vec <- unlist(lapply(strsplit(Chr.filename.Str,split = sample.run.splitStr),head,1))
  
  r1 <- read.delim(Chr.filepath.vec[1])
  N.Chr.sample <- length(Chr.sampleID.vec)
  N.Chr.row <- nrow(r1)
  run.cvg.perc.mtx <- run.crct.cvg.mtx <- mat.or.vec(N.Chr.row, N.Chr.sample)
  rownames(run.cvg.perc.mtx) <- rownames(run.crct.cvg.mtx) <- r1$Chr
  colnames(run.cvg.perc.mtx) <- colnames(run.crct.cvg.mtx) <- Chr.sampleID.vec
  
  for(s.idx in 1 : N.Chr.sample){
    cat( s.idx, "-th sample:", Chr.sampleID.vec[s.idx], "\n")
    rtmp <- read.delim(Chr.filepath.vec[s.idx])
    run.crct.cvg.mtx[ rtmp$Chr, Chr.sampleID.vec[s.idx] ] <- rtmp$total.crct.cvg
    run.cvg.perc.mtx[ rtmp$Chr, Chr.sampleID.vec[s.idx] ] <- rtmp$crct.cvg.perc.vs.autosome
  }
  
  
  
  
  
  run.sample.crct.total.million.cvg <- apply(run.crct.cvg.mtx,2,sum)/1e6
  outlier.sample.idx <- which( run.sample.crct.total.million.cvg < minimum.million.cvg ) # if less than 1-million
  
  sel.sample.idx <- setdiff( seq(1, N.Chr.sample), outlier.sample.idx)
  
  run.Chr.perc.mean <- apply(run.cvg.perc.mtx[, sel.sample.idx], 1, median)
  run.Chr.perc.SD <- apply(run.cvg.perc.mtx[, sel.sample.idx],1,mad) * 1.4826
  
  
  if( !is.null(static.ref.file) ){ # using static reference
    cat("reading static reference baseline: \n", static.ref.file, "\n")
    rStaticRef <- read.delim(static.ref.file)
    
    
    run.Chr.perc.mean <- rStaticRef$run.Chr.perc.mean 
    run.Chr.perc.SD <- rStaticRef$run.Chr.perc.SD
  }
  
  run.Chr.zscore.mtx <- run.Chr.zscore2.mtx <- run.Chr.Lscore.mtx <- run.cvg.perc.mtx*0
  # run.Chr.zscore.mtx: null
  # run.Chr.zscore2.mtx: alternative
  
  mch.fetal.frac <- 0.01 * rSample.info$case.fraction[match( colnames(run.Chr.zscore.mtx), rSample.info$sample.ID )]
  
  for(chr.idx in 1 : N.Chr.row){
    run.Chr.zscore.mtx[ chr.idx, ] <- (run.cvg.perc.mtx[ chr.idx, ] - run.Chr.perc.mean[chr.idx])/
      run.Chr.perc.SD[ chr.idx ]
    run.Chr.zscore2.mtx[ chr.idx, ] <- (run.cvg.perc.mtx[ chr.idx, ] - run.Chr.perc.mean[chr.idx] -
                                          0.5*run.Chr.perc.mean[chr.idx] *  mch.fetal.frac )/
                                          run.Chr.perc.SD[ chr.idx ]
    
    
    # run.Chr.Lscore.mtx[ chr.idx, ] <- log(pnorm(abs(run.Chr.zscore2.mtx[chr.idx,]), lower.tail=FALSE)) - 
    #   log(pnorm(abs(run.Chr.zscore.mtx[chr.idx,]),  lower.tail=FALSE))

     run.Chr.Lscore.mtx[ chr.idx, ] <- log2(dnorm( run.Chr.zscore2.mtx[chr.idx,] )) - 
       log2(dnorm( run.Chr.zscore.mtx[chr.idx,] ))
    run.Chr.Lscore.mtx[chr.idx, ] <- pmin( 100,  run.Chr.Lscore.mtx[chr.idx, ])
    run.Chr.Lscore.mtx[chr.idx, ] <- pmax( -100,  run.Chr.Lscore.mtx[chr.idx, ])
  }

#   L-score codes from Na, Jie: 
#   Zscore1[j] <- (covgi[j]-m)/inflated.mad
#   Zscore2[j] <- (covgi[j]-m-m*ffi[j]/2)/inflated.mad
#   
#   Lscore[j] <- -log((pnorm(abs(Zscore1[j]), lower.tail=FALSE))/(pnorm(abs(Zscore2[j]),  lower.tail=FALSE)))
#   Lscore[j] <- ifelse(Lscore[j] > 600, 600, Lscore[j])
#   Lscore[j] <- ifelse(Lscore[j] < -600, -600, Lscore[j])
  
  run.Chr.perc.SNR <- run.Chr.perc.mean/run.Chr.perc.SD
  run.chr.stat.mtx <- cbind(run.Chr.perc.mean, run.Chr.perc.SD, run.Chr.perc.SNR)
  

  if( is.null(static.ref.file) ){ # not using static reference
    write.table(x = run.chr.stat.mtx, file = chrPerctStat.TXT.filename, quote = FALSE, sep = "\t")
    write.table(x = run.cvg.perc.mtx, file = chrPerctg.TXT.filename, quote = FALSE, sep = "\t")
  }

  write.table(x = run.Chr.zscore.mtx, file = chrZscore.TXT.filename, quote = FALSE, sep = "\t")
  write.table(x = run.Chr.zscore2.mtx, file = chrAltZscore.TXT.filename, quote = FALSE, sep = "\t")

  write.table(x = run.Chr.Lscore.mtx, file = chrLscore.TXT.filename, quote = FALSE, sep = "\t")


  
  
  
  zscore.mch.sample.idx <- 
    match( sampleStat.mtx[,"sample.ID"], colnames(run.Chr.zscore.mtx))
  
  T21.zscore <- run.Chr.zscore.mtx["chr21", zscore.mch.sample.idx]
  T21.Lscore <- run.Chr.Lscore.mtx["chr21", zscore.mch.sample.idx]
  T21.call   <- rep("Neg", length(T21.zscore))
  T21.call[which(T21.zscore >= T21.grey.zcut)] <- "Bdline"
  T21.call[which(T21.zscore >= T21.Pos.zcut)] <- "Pos"
  T21.call[which(T21.zscore >= T21.Pos.zcut & T21.Lscore <= 0)] <- "Bdline"
  T21.call[which(T21.zscore < T21.Pos.zcut & T21.Lscore > 0)] <- "Bdline"
  
  T18.zscore <- run.Chr.zscore.mtx["chr18", zscore.mch.sample.idx]
  T18.Lscore <- run.Chr.Lscore.mtx["chr18", zscore.mch.sample.idx]
  T18.call   <- rep("Neg", length(T18.zscore))
  T18.call[which(T18.zscore >= T18.grey.zcut)] <- "Bdline"
  T18.call[which(T18.zscore >= T18.Pos.zcut)] <- "Pos"
  T18.call[which(T18.zscore >= T18.Pos.zcut & T18.Lscore <= 0)] <- "Bdline"
  T18.call[which(T18.zscore < T18.Pos.zcut & T18.Lscore > 0)] <- "Bdline"

  T13.zscore <- run.Chr.zscore.mtx["chr13", zscore.mch.sample.idx]
  T13.Lscore <- run.Chr.Lscore.mtx["chr13", zscore.mch.sample.idx]
  T13.call   <- rep("Neg", length(T13.zscore))
  T13.call[which(T13.zscore >= T13.grey.zcut)] <- "Bdline"
  T13.call[which(T13.zscore >= T13.Pos.zcut)] <- "Pos"
  T13.call[which(T13.zscore >= T13.Pos.zcut & T13.Lscore <= 0)] <- "Bdline"
  T13.call[which(T13.zscore < T13.Pos.zcut & T13.Lscore > 0)] <- "Bdline"
  
  sampleReport.mtx <- cbind( sampleStat.mtx, 
                             T13.zscore, T13.Lscore, T13.call,
                             T18.zscore, T18.Lscore, T18.call,
                             T21.zscore, T21.Lscore, T21.call )
  
  
  write.table(x = sampleReport.mtx, file = sample.Stat.TXT.filename, row.names = FALSE,
              quote = FALSE, sep = "\t")  
  
  #============================================================
  #=== step 3. summarizing bin-level matrix
  #============================================================
  
  cat( paste("*** 3rd step: summarizing bin-level matrix of", run.ID, "\n") )
  
  
    #=== merging bin-level coverage information 
    
    binCovg.filepath.vec <- dir(path = sample.res.DIR, pattern = "_bin_normalized_covg.txt", recursive = TRUE, full.names = TRUE)
    #cat(binCovg.filepath.vec)
  
    binCovg.filename.Str <- unlist(lapply(strsplit(binCovg.filepath.vec, split = "/"),tail,1))
    binCovg.sampleID.vec <- unlist(lapply(strsplit(binCovg.filename.Str, split = sample.run.splitStr),head,1))
    
    r1 <- read.delim(binCovg.filepath.vec[1])
    sel.bin.idx <- which( r1$is.reliable.bin == 1 )
    bin.info <- cbind( r1$Chr,  r1$Start.Pos, r1$gc.content, r1$is.reliable.bin )
    colnames(bin.info) <- c("Chr","Start.Pos","gc.content","is.reliable.bin")  
    
    N.binCovg.sample <- length(binCovg.sampleID.vec)
    N.bin <- nrow(r1)
    
    run.bin.cvg.mtx <- mat.or.vec(N.bin, N.binCovg.sample)
    colnames(run.bin.cvg.mtx) <- binCovg.sampleID.vec 
    run.GC.res <- list()
    
    plot.GCcurve.ymax <- quantile(r1$original.count[sel.bin.idx], 0.98, na.rm = TRUE)
    
    
        for(s.idx in 1 : N.binCovg.sample){
          cat( s.idx, "-th sample:", binCovg.sampleID.vec[s.idx], "\n")
          
          r1 <- read.delim(binCovg.filepath.vec[s.idx])
          run.bin.cvg.mtx[, binCovg.sampleID.vec[s.idx]] <- r1$nmlz.cvg
          
          sspline.res <- smooth.spline( r1$gc.content[sel.bin.idx], r1$original.count[sel.bin.idx], 
                                        df = spline.DegreeOfFreedom )
          run.GC.res[[ s.idx ]] <- sspline.res
          
          persample.GCcurve.ymax <- quantile(r1$original.count[sel.bin.idx], 0.98, na.rm = TRUE)
          plot.GCcurve.ymax <- max( plot.GCcurve.ymax, persample.GCcurve.ymax)
        }
        
  if( if.summarize.binMatrix ){      
        bin.sum.mtx <- cbind(bin.info, run.bin.cvg.mtx)
        
        if( is.null(static.ref.file) ){ # not using static reference
        write.table(x = bin.sum.mtx, file = binNmlz.Covg.TXT.filename, 
                    row.names = FALSE, quote = FALSE, sep = "\t")    }
  }
  
  #============================================================
  #=== step 4. generating QC-summary and report plots
  #============================================================
  
  
  #   par(xpd = T, mar = c(5, 4, 4, 2) + c(0,0,0,5))
  #   plot(rnorm(length(rSample.info$case.fraction)), 
  #        pch = sample.pch.vec[rSample.info$sample.type], 
  #        col = sample.color.vec[rSample.info$sample.type])
  #   legend(23, 0.015,
  #          unique.SampleType,
  #          cex = 0.7,  col = sample.color.vec,
  #          pch = sample.pch.vec)
  #   par(mar=c(5, 4, 4, 2) + 0.1)
  #
  
  #=== plotting initialization
 
  pch.vec <- sample.pch.vec[rSample.info$sample.type]
  color.vec <- sample.color.vec[rSample.info$sample.type]
  
  cat( paste("*** 4th step: generating QC-summary and report plots of", run.ID, "\n") )
  #=== QC summary plots
  
  pdf(QCsummary.PDF.filename)
  
  default.par.config <- par(no.readonly = TRUE) # record par configurations

  side.plot(run.sample.crct.total.million.cvg, 
            pch = pch.vec, col = color.vec,
            main = "effective coverage",
            ylab = "per sample effective coverage (million reads)", xlab = "sample index",
            hpos = minimum.million.cvg, hcol = "blue", 
            legend.text = unique.SampleType, 
            legend.col = sample.color.vec,
            legend.pch = sample.pch.vec)
    
     
    
  side.plot(sampleStat.mtx[, "autosome.crct.cvg.Million"],
          pch = pch.vec, col = color.vec,
          main = "effective coverage in autosome",
          ylab = "per sample raw coverage (million reads)", xlab = "sample index",
          legend.text = unique.SampleType, 
          legend.col = sample.color.vec,
          legend.pch = sample.pch.vec )
     
  side.plot( sampleStat.mtx[, "total.raw.cvg.Million"],
          pch = pch.vec, col = color.vec,
          main = "raw coverage",
          ylab = "per sample raw coverage (million reads)", xlab = "sample index",
          legend.text = unique.SampleType, 
          legend.col = sample.color.vec,
          legend.pch = sample.pch.vec )
     
    
    
    side.plot(sampleStat.mtx[, "max.min.GCCovg.ratio"], 
         pch = pch.vec, col = color.vec,
         ylim = c(1, max(sampleStat.mtx[, "max.min.GCCovg.ratio"])), 
         ylab = "max/min ratio of GC-covg bias",
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec, 
         hpos = 1, hcol = "blue")
     
    side.plot(sampleStat.mtx[, "max.min.GCCovg.slope"], 
         pch = pch.vec, col = color.vec,
         ylab = "max/min slope of GC-covg bias",
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec)
     
    side.plot(sampleStat.mtx[, "GC.area.distortion.perc"], 
         pch = pch.vec, col = color.vec,
         ylab = "nonlinear area distortion % of GC-covg bias",
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec)
     
    
  par(mar=c(5, 4, 4, 2) + 0.1)
  par(default.par.config)  # recover default par configurations
  
  plot(c(-1,1), xlim = GCcurv.x.range, ylim = c(0, plot.GCcurve.ymax),
       ylab = "covg", xlab = "GC content")
  for(s.idx in 1 : N.binCovg.sample) {
    lines( run.GC.res[[ s.idx ]] , col = s.idx, lwd = 2)
  }
  
  plot(c(-1,1), xlim = GCcurv.x.range, ylim = c(0, 3),
       ylab = "scaled covg (by median)", xlab = "GC content")
  for(s.idx in 1 : N.binCovg.sample) {
    lines( run.GC.res[[ s.idx ]]$x,
           run.GC.res[[ s.idx ]]$y / median(run.GC.res[[ s.idx ]]$y, na.rm = TRUE),
           col = s.idx, lwd = 2)
  }
  
  par(mfrow = c(3,3))
  for(s.idx in 1:N.binCovg.sample){
    
    tmp.sampleID <- binCovg.sampleID.vec[s.idx]
    tmp.sampleidx <- match( tmp.sampleID, sampleStat.mtx[, "sample.ID"])
    
    plot(run.GC.res[[ s.idx ]],
         col = s.idx, lwd=2, type="l", xlab="", ylab="", cex.main=0.92,
         ylim = c(0, max(run.GC.res[[s.idx]]$y, na.rm = TRUE)),
         main = paste(tmp.sampleID, "\n",
                      "max/min covg ratio =", 
                      round(sampleStat.mtx[tmp.sampleidx, "max.min.GCCovg.ratio"], 2),
                      "\n",
                      "max/min slope =", 
                      round(sampleStat.mtx[tmp.sampleidx, "max.min.GCCovg.slope"], 2)
         ))
    
  }
  par(mfrow = c(1,1))
  
  dev.off()
  
  pdf(runReport.PDF.filename)
  
  default.par.config <- par(no.readonly = TRUE) # record par configurations
  
    side.plot(run.Chr.zscore.mtx["chr13",], ylab="chr13 z.score",
         pch = pch.vec, col = color.vec,
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec,
         hpos = c(T13.Pos.zcut, T13.grey.zcut), hcol = c("red", "grey") )
     
    if( !all(is.na(run.Chr.Lscore.mtx["chr13",])) ) { # at least some non-NA L-scores
      side.plot(run.Chr.Lscore.mtx["chr13",], run.Chr.zscore.mtx["chr13",], 
                xlab = "chr13 L.score", ylab="chr13 z.score",
                pch = pch.vec, col = color.vec,
                legend.text = unique.SampleType, 
                legend.col = sample.color.vec,
                legend.pch = sample.pch.vec,
                hpos = c(T13.Pos.zcut, T13.grey.zcut), hcol = c("red", "grey"), 
                vpos = 0, vcol = "cyan")  
    }
    
      
  
    side.plot(run.Chr.zscore.mtx["chr18",], ylab="chr18 z.score",
         pch = pch.vec, col = color.vec,
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec,
         hpos = c(T18.Pos.zcut, T18.grey.zcut), hcol = c("red", "grey"))
  
  if( !all(is.na(run.Chr.Lscore.mtx["chr18",])) ) { # at least some non-NA L-scores
   side.plot(run.Chr.Lscore.mtx["chr18",], run.Chr.zscore.mtx["chr18",], 
         xlab = "chr18 L.score", ylab="chr18 z.score",
         pch = pch.vec, col = color.vec,
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec,
         hpos = c(T18.Pos.zcut, T18.grey.zcut), hcol = c("red", "grey"), 
         vpos = 0, vcol = "cyan")
  }

    side.plot(run.Chr.zscore.mtx["chr21",], ylab="chr21 z.score",
         pch = pch.vec, col = color.vec,
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec,
         hpos = c(T21.Pos.zcut, T21.grey.zcut), hcol = c("red", "grey"))
     
  if( !all(is.na(run.Chr.Lscore.mtx["chr21",])) ) { # at least some non-NA L-scores
    side.plot(run.Chr.Lscore.mtx["chr21",], run.Chr.zscore.mtx["chr21",], 
         xlab = "chr21 L.score", ylab="chr21 z.score",
         pch = pch.vec, col = color.vec,
         legend.text = unique.SampleType, 
         legend.col = sample.color.vec,
         legend.pch = sample.pch.vec,
         hpos = c(T21.Pos.zcut, T21.grey.zcut), hcol = c("red", "grey"), 
         vpos = 0, vcol = "cyan")
  }
  
  par(default.par.config)  # recover default par configurations

  barplot(  run.Chr.perc.mean, horiz = TRUE, 
            las = 1 , xlab = "mean % of reads (autosome as denominator)") 
  barplot(  run.Chr.perc.SD, horiz = TRUE, 
            las = 1 , xlab = "% SD of reads (autosome as denominator)", xlim = c(0, 0.1)) 
  
  dev.off()     
  
  
  
  
  
  #============================================================
  #=== wandy per-run summary done
  #============================================================
  
  cat( paste("*** Wandy is done with per-run summary of", run.ID, ":-) \n") )
  
}
