wandy.analyzeSample <- function( run.info.filename, sample.info.filename, sel.sampleID, output.DIR, 
                                debug.rndSeed = NULL)
{
  rSample.info <- read.delim(sample.info.filename)
  
  sel.sample.idx <- match( sel.sampleID, rSample.info$sample.ID )
  
  if(is.na(sel.sample.idx)){
    stop(paste("\n input sample.ID", sel.sampleID, 
               "can't be located from sample information file.\n please check \n"))
  }
  
  ##===============================
  ##=== step 0. Initialization 
  ##===============================
  
  cat( paste("*** Initialization: loading run- and sample-information of", sel.sampleID, "\n") )
  
  ##=== loading run information
  source(run.info.filename)
  
  ##=== reading sample information of selected sample
  Sample.info.variables <- colnames(rSample.info)
  N.sample.info.var <- length(Sample.info.variables)
  for(j in 1 : N.sample.info.var){
    assign(x = Sample.info.variables[j],value = rSample.info[ sel.sample.idx, j])
  }
  
  ##=== generating random seed if Null
  if(is.null(debug.rndSeed)){
    debug.rndSeed <- 
      as.integer( (as.double(Sys.time())*1000+Sys.getpid()) %% 2^31 + 
                 sum(utf8ToInt(sample.ID)) )
  }
  cat("****** random seed =", debug.rndSeed,"\n")
  set.seed(seed = debug.rndSeed)
  
  
  ##=== create working, per-run and per-sample directories, if not created
  
  #wandy.create.DIR(output.DIR)  # created by wandy.Java module, instead of by wandy.R
  
  setwd(output.DIR)
  
  run.outputDIR <- paste( run.ID, wandyCNV.version, "/", sep = "")
  sample.outputDIR <- paste(run.outputDIR, sample.ID, "_within_", 
                            run.ID, "/", sep = "")
  wandy.create.DIR(run.outputDIR)
  wandy.create.DIR(sample.outputDIR)
  
  common.sample.prefix <- paste(sample.ID, run.ID, sep = "_within_")
  common.run.prefix <- paste(run.ID, wandyCNV.version, sep = "")
  
  SampleLevel_Output_Prefix <- paste(sample.outputDIR, common.sample.prefix, sep = "")
  
  ##=== 0a define text output filenames
  binNmlz.Covg.TXT.filename <- paste( SampleLevel_Output_Prefix,
                                     "_bin_normalized_covg.txt", sep = "")
  
  chrSum.TXT.filename <- paste( SampleLevel_Output_Prefix,
                               "_Chr_summary.txt", sep = "")
  
  zmoutBinCovg.TXT.filename <- paste( SampleLevel_Output_Prefix,
                                     "_zoomout_BinCovg.txt", sep = "")
  
  CNVseg.TXT.filename <- paste( SampleLevel_Output_Prefix,
                               "_CNV_segmentation.txt", sep = "")
  
  
  sample.Stat.TXT.filename <- paste( SampleLevel_Output_Prefix,
                                    "_SampleLevel_stat.txt", sep = "")
  
  
  ##=== 0b define output plot filenames
  rawChrCovg.PDF.filename <- paste( SampleLevel_Output_Prefix,
                                   "_rawChr_million_read.pdf", sep = "")
  
  
  insertSize.PDF.filename <- paste( SampleLevel_Output_Prefix,
                                   "_insertSize.pdf", sep = "")
  
  classfGender.PDF.filename <- paste( SampleLevel_Output_Prefix,
                                     "_gender_classification.pdf", sep = "")
  
  
  GC.Covg.perChr.PDF.filename <- paste( SampleLevel_Output_Prefix,
                                       "_GC_covg_perChr.pdf", sep = "")
  
  GCbias.QC.PDF.filename <- paste( SampleLevel_Output_Prefix,
                                  "_GC_bias_QC.pdf", sep = "")
  
  WGS.Covg.PNG.filename <- paste( SampleLevel_Output_Prefix,
                                 "_WGS_covg.png", sep = "")
  
  CNV.seg.PDF.filename <- paste( SampleLevel_Output_Prefix,
                                "_CNV_seg.pdf", sep = "")
  
  ##===================================
  ##=== step 1. filtering genomic bins  
  ##===================================
  
  cat("*** 1st step: reading bin annotation and performing filtering ... \n")
  
  rBinAnnot <- read.delim( bin.annotation.file )
  
  N.total.bin <- nrow(rBinAnnot)
  
  
  sel.GC.NAN <- ( !is.na(rBinAnnot$GC_Content) )
  sel.usable.bin <- ( rBinAnnot$usable_bin == 1)
  
  if( all(is.na( rBinAnnot$median )) ) {
    
    reliable.bin.idx <- which( sel.GC.NAN & sel.usable.bin )
    
  } else {
    sel.median.cvg <- ( rBinAnnot$median >= filtering.autosome.median.range[1] &
                       rBinAnnot$median <= filtering.autosome.median.range[2] )
    sel.SNR <- (rBinAnnot$SNR >= filtering.min.SNR)  
    reliable.bin.idx <- which( sel.median.cvg & sel.SNR &
                              sel.GC.NAN & sel.usable.bin )
  }
  
  ##=== single.ref.file 
  if( !is.na(single.ref.file) ) {  
    single.ref <- read.delim(single.ref.file)
    reliable.bin.idx <- which( single.ref$is.reliable.bin == 1  )
  }
  
  reliable.autosome.bin.idx <- intersect( reliable.bin.idx,
                                         which(!is.element(rBinAnnot$chr, c("chrX", "chrY", "chrM"))) )
  
  
  if( all(is.na( rBinAnnot$median )) ) {
    reliable.chrX.bin.idx <- which( is.element(rBinAnnot$chr, c("chrX")) & 
                                   sel.GC.NAN & sel.usable.bin)
    
    reliable.chrY.bin.idx <- which( is.element(rBinAnnot$chr, c("chrY")) &
                                   sel.GC.NAN & sel.usable.bin) 
  }  else {
    
    reliable.chrX.bin.idx <- which( is.element(rBinAnnot$chr, c("chrX")) &
                                   rBinAnnot$median >= filtering.chrX.median.range[1] &
                                   rBinAnnot$median <= filtering.chrX.median.range[2] & 
                                   sel.GC.NAN & sel.usable.bin)
    
    reliable.chrY.bin.idx <- which( is.element(rBinAnnot$chr, c("chrY")) &
                                   rBinAnnot$median >= filtering.chrY.median.range[1] &
                                   rBinAnnot$median <= filtering.chrY.median.range[2] & 
                                   sel.GC.NAN & sel.usable.bin) 
    
    
  }
  
  ##====================================================================================
  ##=== step 2. loading sample coverage information and checking chr-level raw coverage
  ##====================================================================================
  
  cat("*** 2nd step: loading sample and checking raw coverage ... \n")
  
  
  rBinRecord <- read.delim(bin.record.file)
    
  ChrName.Str <- paste("chr", c(seq(1,22),"X","Y","M"), sep = "")
  N.Chr <- length(ChrName.Str)
  Chr.raw.million.reads <- mat.or.vec( N.Chr, 1)
  names(Chr.raw.million.reads) <- paste(ChrName.Str, "raw.MR", sep = ".")
  for(j in 1 : N.Chr){
    Chr.raw.million.reads[j] <- sum(rBinRecord$High_Quality_Read[
      which(rBinRecord$Chromosome==ChrName.Str[j])])/1e6
  }
  
  pdf(rawChrCovg.PDF.filename)
  barplot(Chr.raw.million.reads, horiz = FALSE,las=2, names.arg = ChrName.Str,
          cex.names=0.55, ylab="Million reads", 
          main=paste("total raw reads in each Chr\n", common.sample.prefix))
  dev.off() # end of pdf(rawChrCovg.PDF.filename)
  
  total.raw.cvg.Million <- sum(Chr.raw.million.reads)
  
  ##==========================================
  ##=== step 3. checking insert size (optional)
  ##==========================================
  
  cat("*** 3rd step: checking insert size ... \n")
  
  if(if.check.InsertSize){
    insert.start.idx <- which(colnames(rBinRecord)=="X0.49")
    insert.stop.idx <-  ncol(rBinRecord)
    colnames(rBinRecord)[ insert.start.idx : insert.stop.idx ]
    insert.list <- strsplit(gsub("X","",colnames(rBinRecord)[ 
      insert.start.idx : insert.stop.idx ]),split = "\\.")
    N.InsertInterval <- length(insert.list)
    insert.vec <- mat.or.vec(N.InsertInterval,1)
    for(k in 1:N.InsertInterval){
      insert.vec[k] <- mean(as.numeric(insert.list[[k]]),na.rm = TRUE)
    }
    
    insert.readcount <- apply(rBinRecord[,insert.start.idx:insert.stop.idx],2,sum)
    insert.mod.idx <- which.max(insert.readcount)
    insertSize.mode <- round(insert.vec[insert.mod.idx])
    
    pdf(insertSize.PDF.filename)
    
    par(mfrow = c(1,2))
    plot( insert.vec, insert.readcount/1e6, lwd=2,
          main = paste("insert.size mode around\n", insert.size.mode , "bp"),
          type = "b", ylab="Million reads", xlab="insert.size")
    abline( v = insert.size.mode, lty=2, col="blue", lwd=2)
    plot( insert.vec, (1+insert.readcount)/1e6, lwd=2,
          type="b", log="y", ylab="Million reads in log-scale", xlab="insert.size")
    abline( v = insert.size.mode, lty=2, col="blue", lwd=2)
    par(mfrow = c(1,1))
    
    
    
    
    AvgInsertSize <- apply(rBinRecord[, insert.start.idx:insert.stop.idx],1,
                           myWeighted.Avg, insert.vec)
    hist(AvgInsertSize,500,xlim=c(150,300))
    print(summary(AvgInsertSize))
    insertSize.mean <- mean(AvgInsertSize)
    
    dev.off() # end of pdf(insertSize.PDF.filename)
    
  }   else {  # not checking insert-size for single-end reads
    cat("\t \t 3rd step skipped \n")
    insertSize.mean <- NA 
    insertSize.mode <- NA
  }
  
  ##=============================
  ##=== 4. checking gender
  ##=============================
  
  cat("*** 4th step: checking gender ... \n")
  
  
  selY.idx <- reliable.chrY.bin.idx
  chrY.cvg <- rBinRecord$High_Quality_Read [ selY.idx ]
  chrY.pos <- rBinRecord$Start[ selY.idx ]
  
  selX.idx <- reliable.chrX.bin.idx
  chrX.cvg <- rBinRecord$High_Quality_Read [ selX.idx ]
  chrX.pos <- rBinRecord$Start[ selX.idx ]
  
  YXratio <- max( mean(chrY.cvg), 1e-10) / max( mean(chrX.cvg), 1e-10)
  print(paste("YXratio = ",YXratio,sep=""))
  
  if(if.check.gender){	
    
    pdf(classfGender.PDF.filename)
    
    N.sexchr.hist.break <- 20
    X.hres <- hist( chrX.cvg, N.sexchr.hist.break, plot = FALSE)
    Y.hres <- hist( chrY.cvg, N.sexchr.hist.break, plot = FALSE)
    XY.plotrange <- c(0, quantile(c(chrX.cvg,chrY.cvg),0.999))
    plot( X.hres$mids, X.hres$counts, type="l", col="black", lwd=2,
         xlab="coverage count",  main=paste("Y/X coverage ratio", round(YXratio,2)),
         ylab="freq count", xlim=XY.plotrange)
    lines( Y.hres$mids, Y.hres$counts, type="l", col="red", lwd=2)
    legend("topright", c("chr-X", "chr-Y"), col=c("black","red"), lwd=2)
    
    par(mfrow = c(1,2))
    plot(chrX.pos/1e6, chrX.cvg, ylab="cvg", ylim=XY.plotrange, xlab="Mb")
    plot(chrY.pos/1e6, chrY.cvg, ylab="cvg", ylim=XY.plotrange, xlab="Mb")
    par(mfrow = c(1,1))
    
    dev.off()
    
    
    classified.gender <- "male"
    if( is.na(YXratio) ){  # checking BAM files, indicating 0 reads on chr-X, very unlikely
      classified.gender <- "NA"
    } else {
      if(YXratio < gender.cut){
        classified.gender <- "female"
      }  
    }
    
    if ( !is.na(reported.gender) ){
      if(reported.gender != classified.gender) {
        warning(paste("reported gender:", reported.gender, 
                      " inconsistent with classified gender:", classified.gender))
      }    
    } # gender warning if inconsistent
    
  } else {  # not checking gender for some known gender cases
    cat("\t \t 4th step skipped \n")
    classified.gender <- NA 
  }
  
  ##====================================
  ##=== 5. GC-coverage bias correction
  ##====================================
  
  cat("*** 5th step: correcting GC-coverage bias ... \n")
  
  
  ##   table(rBinRecord$Chromosome == rBinAnnot$chr)
  ##   table(rBinRecord$Start == rBinAnnot$start)
  ##   table(rBinRecord$End == rBinAnnot$end)
  ##   
  gc.res <- correctGCbias( chr=rBinRecord$Chromosome, loc=rBinRecord$Start,
                          counts=rBinRecord$High_Quality_Read, gc=rBinAnnot$GC_Content, 
                          train.autosome.idx = reliable.autosome.bin.idx,
                          sspline.df = spline.DegreeOfFreedom,
                          sample.name = paste(sample.ID,run.ID,sep = "_within_"),
                          plot_filename=GC.Covg.perChr.PDF.filename)
  
  ##================================================
  ##=== 6. quantitatively assess GC-covg bias curve
  ##================================================
  cat("*** 6th step, quantifing GC-coverage bias\n")
  
  X.gc.vec <- seq(GCcurv.x.range[1], GCcurv.x.range[2], by = 0.02)
  Y.covg.vec <- predict(gc.res$sspline.res,X.gc.vec)$y
  
  maxCovg.idx <- which.max(Y.covg.vec)
  minCovg.idx <- which.min(Y.covg.vec)
  ##=== 1e-6 as a small constant to avoid dividing zero
  max.min.GCCovg.ratio <- (1e-6 + Y.covg.vec[maxCovg.idx])/( 1e-6 + Y.covg.vec[minCovg.idx] )
  max.min.GCCovg.slope <- ((Y.covg.vec[maxCovg.idx]) - (Y.covg.vec[minCovg.idx]))/ 
    ((X.gc.vec[maxCovg.idx]) - (X.gc.vec[minCovg.idx]))
  
  linear.area <- (tail(Y.covg.vec,1) + Y.covg.vec[1]) * diff(range(X.gc.vec)) /2
  nonlinear.area <- sum(diff(X.gc.vec)*Y.covg.vec[1:length(X.gc.vec)-1])
  GC.area.distortion.perc <- 100*(nonlinear.area - linear.area)/linear.area
  
  ##===============================================================
  ##=== 7. scan whole-genome and output plot and bin.summary.txt
  ##===============================================================
  
  cat("*** 7th step: producing whole-genome bin summary\n")
  
  all.reliable.idx <- c(reliable.autosome.bin.idx, 
                        reliable.chrX.bin.idx, reliable.chrY.bin.idx)
  
  binsum.res <- 
    processWholeGenomeCovg(res_list = gc.res,
                           statfile = single.ref.file,
                           reliable.idx = all.reliable.idx,
                           png_filename = WGS.Covg.PNG.filename,
                           txt_filename = binNmlz.Covg.TXT.filename)
  
  ##=== QC plot for GC-coverage bias curve and normalized coverage
  reliable.nmlz.cvg <- as.numeric(binsum.res$nmlz.cvg[ binsum.res$is.reliable.bin==1 ])
  hist.reliabile.break <- 100
  mad.diff.nmlz.cvg <- mad( diff(reliable.nmlz.cvg), na.rm = TRUE)
  
  pdf(GCbias.QC.PDF.filename)
  ##=== GC bias curve
  main.str <- paste(
                    "max.min.GCCovg.ratio=", round(max.min.GCCovg.ratio,digits = 3), ";\n",
                    "max.min.slope=",round(max.min.GCCovg.slope,digits = 3),";",
                    "nonlinear.distortion% =",round(GC.area.distortion.perc,digits = 3),"% \n") 
  
  plot(X.gc.vec, Y.covg.vec, type="l", lwd=2, col="red", ylim=c(0, max(Y.covg.vec)),
       xlab="GC content", ylab=" Covg ", main=main.str)
  points(X.gc.vec[maxCovg.idx], Y.covg.vec[maxCovg.idx], col="blue", cex=2,lwd = 2)
  points(X.gc.vec[minCovg.idx], Y.covg.vec[minCovg.idx], col="green", cex=2,lwd = 2)
  lines( x = c(tail(X.gc.vec,1) , X.gc.vec[1]),
        y = c(tail(Y.covg.vec,1) , Y.covg.vec[1]), col="black", lwd=2)
  abline(v=c(tail(X.gc.vec,1) , X.gc.vec[1]), col="grey", lty=2, lwd=2)
  
  legend("bottom", 
         c("obs.GC-bias curve","linear curve",
           paste('max.covg ~=',round(Y.covg.vec[maxCovg.idx])),
           paste('min.covg ~=',round(Y.covg.vec[minCovg.idx]))),
         lwd=2, col=c("red","black",'blue','green'), cex=1.25,
         lty=c(1,1,0,0), pch=c(-1,-1,1,1))
  
  hist(reliable.nmlz.cvg, hist.reliabile.break, xlim = c(0,3.5), 
       main = "Histogram of normalized coverage", xlab = "normalized coverage")
  abline( v=c(1-3*mad.diff.nmlz.cvg, 1+3*mad.diff.nmlz.cvg), lwd = 2, lty = 2, col = "red")
  
  hist(diff(reliable.nmlz.cvg), hist.reliabile.break, xlim = c(-1.5, 1.5),
       main = "Histogram of difference of normalized coverage", 
       xlab = "normalized coverage difference of 2 consecutive bins")
  abline( v=c(-3*mad.diff.nmlz.cvg, 3*mad.diff.nmlz.cvg), lwd = 2, lty = 2, col = "red")
  
  dev.off()
  
  
  ##==============================
  ##=== 8. chr-level summary
  ##==============================
  
  cat("*** 8th step: chr-level summary \n")
  
  sel.autosome.chr.idx <- which( is.element(binsum.res$Chr, paste("chr",seq(1,22),sep = "")) &
                                binsum.res$is.reliable.bin==1) 
  total.autosome.crct.cvg <- sum(as.numeric(binsum.res$corrected.count[sel.autosome.chr.idx]),
                                 na.rm = TRUE)
  autosome.crct.cvg.Million <- total.autosome.crct.cvg/1e6
  
  chrsum.res <- list()
  chrsum.res$Chr <- ChrName.Str
  for(k in 1 : N.Chr){
    tmp.chr <- ChrName.Str[k]
    sel.chr.idx <- which(binsum.res$Chr == tmp.chr & binsum.res$is.reliable.bin==1)
    if(tmp.chr == "chrM") { sel.chr.idx <- which(binsum.res$Chr == tmp.chr) } # all bins in chrM
    tmp.crct.cvg <- sum(as.numeric(binsum.res$corrected.count[sel.chr.idx]),na.rm = TRUE)
    chrsum.res$total.crct.cvg[k] <- tmp.crct.cvg
    chrsum.res$crct.cvg.perc.vs.autosome[k] <- 100*tmp.crct.cvg/total.autosome.crct.cvg
  }
  
  chrsum.res <- data.frame(chrsum.res)
  
  write.table(x = chrsum.res,file = chrSum.TXT.filename, 
              sep = "\t", quote = FALSE, row.names = FALSE)
  
  
  ##======================================
  ##=== 9. computing CNV segmentation
  ##======================================
  
  cat("*** 9th step: segmenting copy number variations (CNVs)\n")
  
  min.median.bin.cvg <- CNV.min.median.BinCvg
  
  median( as.numeric(binsum.res$nmlz.cvg[reliable.autosome.bin.idx]) )
  median.crt.cvg <- median( as.numeric(binsum.res$corrected.count[reliable.autosome.bin.idx]) )
  
  if(median.crt.cvg == 0) { # low-pass 
    median.crt.cvg <- mean( as.numeric(binsum.res$corrected.count[reliable.autosome.bin.idx]) )
  }
  
  ori.CNV.cvg <- as.numeric(binsum.res$corrected.count[all.reliable.idx])
  ori.CNV.chrom <- binsum.res$Chr[all.reliable.idx]
  ori.CNV.pos <- as.numeric(binsum.res$Start.Pos[all.reliable.idx])
  
  ##=== single.ref.file 
  if( !is.na(single.ref.file) ) { ori.CNV.cvg <- as.numeric(binsum.res$nmlz.cvg[all.reliable.idx]) }
  
  ##=== zoom-out view
  zmout.list <- NULL
  
  zmout.n <- ceiling( min.median.bin.cvg/median.crt.cvg )
  
  cat("median.crt.cvg", median.crt.cvg,"\n",
      "min.median.bin.cvg", min.median.bin.cvg, "\n",
      "zmout.n", zmout.n,"\n")
  
  chr.Nbin.vec <- mat.or.vec(24,1)
  for(k in 1:24){ tmp.chr <- ChrName.Str[k]
                  cat(k,tmp.chr,"\n")
                  sel.chr.idx <- which(ori.CNV.chrom == tmp.chr) 
                  chr.Nbin.vec[k] <- length(sel.chr.idx)}
  
  if(is.na(CNV.default.zmout.N)) {
    zmout.n <- floor( min( c(zmout.n, chr.Nbin.vec/10) ) ) # making sure zoom.out ratio is not too large
  } else {
    zmout.n <- CNV.default.zmout.N
  }
  
  ##=== single.ref.file 
  if( !is.na(single.ref.file) ) { zmout.n <- 1 }
  
  cat("single.ref.file", single.ref.file, "\n")
  cat("final zmout.n", zmout.n,"\n")
  
  for(k in 1:24){
    
    tmp.chr <- ChrName.Str[k]
    cat(k,tmp.chr,"\n")
    sel.chr.idx <- which(ori.CNV.chrom == tmp.chr)
    
    
    end.vec <- seq(zmout.n, length(sel.chr.idx),by = zmout.n)
    start.vec <-  end.vec - zmout.n + 1
    
    for(j in 1:length(start.vec)){
      tmp.cvg <- sum(ori.CNV.cvg[sel.chr.idx[start.vec[j]:end.vec[j]]],na.rm = TRUE)
      tmp.pos <- mean(ori.CNV.pos[sel.chr.idx[start.vec[j]:end.vec[j]]],na.rm = TRUE)
      zmout.list <- rbind(zmout.list, c(tmp.chr, tmp.pos, tmp.cvg))
    }
  }
  
  write.table(x = zmout.list, file = zmoutBinCovg.TXT.filename, 
              quote = FALSE, sep = "\t", row.names = FALSE)
  
  
  CNV.chrom <- zmout.list[,1]
  CNV.pos <- as.numeric(zmout.list[,2])
  CNV.cvg <- as.numeric(zmout.list[,3])
  
  CNV.vec <- log2( CNV.cvg + LR.small.delta  ) - 
    log2(median(CNV.cvg,na.rm = TRUE) + LR.small.delta)
  
  ##=== single.ref.file 
  if( !is.na(single.ref.file) ) { CNV.vec <- log2( CNV.cvg + LR.small.delta ) - 
                                    log2( single.ref$median[all.reliable.idx] + LR.small.delta )	}
  
  CNV.diff.MAD <- mad(diff(CNV.vec), na.rm = TRUE) 
  
  cna.obj <- CNA(genomdat = CNV.vec, chrom = CNV.chrom, maploc = CNV.pos, 
                 data.type=c("logratio"), sampleid=common.sample.prefix)
  seg.res <- segment(cna.obj, verbose=0, min.width=5)
  
  seg.output.mtx <- seg.res$output
  seg.output.mtx[,"num.mark"] <- seg.output.mtx[,"num.mark"] * zmout.n
  seg.output.mtx[,"loc.start"]  <- round( seg.output.mtx[,"loc.start"] )
  seg.output.mtx[,"loc.end"]  <- round( seg.output.mtx[,"loc.end"] )
  
  write.table(x = seg.output.mtx, file = CNVseg.TXT.filename, quote = FALSE, sep = "\t", row.names = FALSE)
  
  pdf(CNV.seg.PDF.filename)
  for(k in 1:24){
    par(mfrow = c(2,1))
    plotSample(seg.res, chromlist= ChrName.Str[k] , ylim=CNV.plot.ylim, 
               xmaploc=TRUE, 
               main=paste(ChrName.Str[k], "of", common.sample.prefix,"(fixed-scale)"))
    plotSample(seg.res, chromlist= ChrName.Str[k] , 
               xmaploc=TRUE, main=paste("(auto-scale)"))  
    par(mfrow = c(1,1))
  }
  dev.off()
  
  
  ori.binsize <- (rBinAnnot$stop[1] - rBinAnnot$start[1] + 1)
  ori.binsize.Mb <- ori.binsize/1e6
  
  CNV.binsize <- (rBinAnnot$stop[1] - rBinAnnot$start[1] + 1) * zmout.n
  CNV.binsize.Mb <- CNV.binsize/1e6
  
  
  cat("heree\n")
  ##sample.output.attr[1]
  ##get(sample.output.attr[1])
  
  ##===========================================
  ##=== final step: output sample-level output
  ##===========================================
  cat("****** 10th step: generating final per-sample report table\n")
  
  sampleLevel.output.txt <- c(unlist(rSample.info[sel.sample.idx,]),
                              unlist(mget(sample.output.attr,ifnotfound = NA)))
  
  write.table( x = t(sampleLevel.output.txt), file = sample.Stat.TXT.filename, 
              quote = FALSE, row.names = FALSE, sep = "\t")
  
  cat("*** Wandy is done :=) \n")
  
} # end of wandy.analyzeSample(.) funciton
## debug use only remove later.
## wandy.analyzeSample(run.info.filename="/data2/bsi/RandD/s115463.Aneuploidy/SinlgeSampleWandy/Wandy_test_sample/results/run_info_Feb2_2015.ini",
##                      sample.info.filename="/data2/bsi/RandD/s115463.Aneuploidy/SinlgeSampleWandy/Wandy_test_sample/results/sample.info",
##                      sel.sampleID= "EX06.sorted",
##                      output.DIR= "/data5/bsi/bioinf_ext1/s200798.sequenome_analysis/Ascore/results/paper/testSingleSample/")
  
 
