
/*------------------------------------------------------------------*
 | MACRO NAME: time_to_ae
 | SHORT DESC: Perform the time-to-event and competing risk analysis for
 |             individual/overall adverse events or individual symptoms/QOL items
 |             by treatment groups over time
 *------------------------------------------------------------------*
 | CREATED BY: ToxT Team                                (07/20/2016)
 |
 | 1. Version 2 (6/1/2017): Added competing risk analysis and included
 |                          cumulative incidence plot after the KM-curve
 |                          The CRSE dataset is now required two more
 |                          variables: fu_stat and fu_date
 |                          completed - 1/7/2019
 *------------------------------------------------------------------*
 | PURPOSE
 |
 | This program perform the time-to-event analysis for individual
 | or overall adverse events per NCI Common Terminology Criteria
 | for Adverse Events (CTCAE) or individual symptoms reported
 | per Patient Report Outcomes (PRO) for each treatment groups
 | across data collection time points
 |
 | The program will produce multiple documents in the directory where SAS is run:
 |
 | For CTCAE:
 | 1. time_to_ae_figures_ae.doc: Produce a compilation of graphs and
 |    tables for individual or overall adverse events
 |    a) Kaplan-Meier Curves for time to >= user specified grade of individual or
 |       overall adverse event by treatment group
 |    b) Summary table of events and censored cases
 |    c) Cumulative Incidence Plots with death as competing risk for user
 |       specified grade of individual or overall adverse event by treatment group
 |    d) Summary table of events, censored and competing risk events
 |    f) Bar Charts of median time to first occurrence of AE (grade>=1) or
 |       worst grade for each treatment group
 |    g) Box Plots of time to first occurrence of AE (grade>=1) or
 |       worst grade for each treatment group
 |    h) Table of Summary Statistics for time to first occurrence of AE (grade>=1)
 |       or worst grade for each treatment group
 |
 | 2. time_to_ae_tbl_ov_ae.doc: Table Summary of Cox proportional
 |                              hazard model analysis for time to >= user specified
 |                              grade of overall AE with user specified factors,
 |                              variable specified in BY parameter and
 |                              Age (continuous variable) as predictors
 |    time_to_ae_tbl_x_ae.doc:  Table Summary of Cox proportional
 |                              hazard model analysis for time to >= user specified
 |                              grade of individual AE with user specified factors,
 |                              variable specified in BY parameter and
 |                              Age (continuous variable) as predictors
 |
 | NOTES: Overall AE is summarized across all AEs that are included
 |        in the CYTOX dataset. If user would like to see the Overall AE
 |        across a subset of AEs that are specified in TOXICITY_LIST
 |        parameter, the CYTOX dataset need to be modified to include
 |        only the subset AEs that are of user interest
 |
 | For PRO Symptoms/QOL items:
 | 1. time_to_ae_figures_qol.doc: Produce a compilation of graphs and
 |                                tables for individual PRO Symptoms/QOL items
 |    a) Kaplan-Meier Curves for time to <=user specified scores of individual
 |       PRO Symptoms/QOL items by treatment group
 |    b) Summary table of events and censored cases
 |    c) Cumulative Incidence Plots with death as competing risk for user
 |       specified grade of individual or overall adverse event by treatment group
 |    d) Summary table of events, censored and competing risk events
 |    e) Bar Charts of median time to first occurrence of <= user specified scores
 |       or worst scores for each treatment group
 |    f) Box Plots of time to first occurrence of <= user specified scores or
 |       worst scores for each treatment group
 |    g) Table of Summary Statistics for time to first occurrence of <= user specified scores
 |       or worst scores for each treatment group
 |
 | 2. time_to_ae_tbl_x_qol.doc: Table Summary of Cox proportional
 |                              hazard model analysis for time to <=user specified scores of
 |                              individual PRO syptoms/QOL items
 |                              with user specified factors, variable specified
 |                              in BY parameter and Age (continuous variable) as predictors
 |
 | ~ IMPORTANT NOTES: ~
 | 1. The program will produce the graphs in black/white version ONLY
 | 2. Datasets that are used to produce the bar chars and summary tables
 |    can be found in WORK directory with prefix of f_ in dataset name
 |    where user can further modify or tweak the graphing per user preference
 | 3. If user would like to apply data cut-off to the time-to-event analysis
 |    the data modification need to be done in cytox data set prior to running the macro.
 |    For example, update the values in eval_dt or endat_dt variables and remove any observations
 |    that are beyond the date cut-off
 |
 *------------------------------------------------------------------*
 | REQUIRED DATASETS and DATA FIELDS
 |
 | 1. crse dataset with one observation per patient with the following data fields:
 |    dcntr_id          : patient identifier
 |    evalae            : evaluable for toxicity (numeric)
 |                        (1= at least one AE form is entered other than
 |                            the baseline timepoint
 |                         0= otherwise)
 |    date_on           : date of initial randomization/registration
 |                        (SAS date with mmddyy10. format)
 |    [comparison group]: group to be used for data comparison
 |                        which will be specified in BY macro parameter
 |                        (i.e. arm)
 |    age               : age of patient at the time of study entry (numeric)
 |    fu_date           : date patient last known to be alive or dead
 |                        (SAS date with mmddyy10. format)
 |    fu_stat           : patiet status at last follow up (numeric with fu_stat. format)
 |                        (1= non-event
 |                         2= competing event)
 |                        (e.g. 1=alive, 2=death or 1=alive, 2=death/progression)
 |                         Notes: user need to format the variable accordingly prior
 |                         to macro run in order to display the correct value of competing event
 |                         on the title of the figure)
 |    [factors]         : include the data for each descriptive/stratification factors
 |                        of user interest (i.e. data for gender, race, disease_type)
 |
 | 2. cytox dataset with one observation per maximum grade per adverse event per
 |    timepoint per patient with the following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    eval_dt           : date that the patient was seen for toxicity evaluation
 |                        (SAS date with mmddyy10. format)
 |    endat_dt          : date decision was made to end active treatment/intervention or not to
 |                        initiate protocol treatment/intervention (SAS date with mmddyy10. format)
 |    toxicity          : Adverse event as in medra codes (numeric and should be formatted
 |                        to give adverse event description) (i.e. 900182)
 |    grade             : Severity of the adverse event according to CTC guidelines.
 |                        (numeric) (i.e. 0, 1, 2, 3, 4, 5)
 |    Notes: The dataset should include all adverse event reported up to the last time point
 |           indicated in protdata dataset of user interest for the analysis
 |
 | 3. qol dataset with one observation per symptom per timepoint per patient
 |    with the following data fields (REQUIRED for PRO Symptoms analysis ONLY):
 |    dcntr_id          : patient identifier
 |    comp_dt           : date of completion for questionnaire (SAS date with mmddyy10. format)
 |    qol               : SAS variable name for each symptom (character)
 |                        (i.e. p_q10, poms04)
 |    _label_           : Description/Label for each symptom (character)
 |                        (i.e. Overall QOL, Sleep Interference, Nervousness)
 |    scores            : Symptoms/QOL items scores in a scale of 0 to 100 (numeric)
 |                        with 0=Low Quality of Life and 100=Best Quality of Life
 |                        Notes: for symptoms with 0=no symptom and 100=worst symptom
 |                               the scores need to be reversed before using this program
 |    Notes: The dataset should include all symptoms/qol items reported up to the
 |           last time point of user interest for the analysis
 |
 | OPTIONAL DATASET and DATA FIELDS
 |
 | 1. factors dataset with one observation per factor (up to 8 factors) with the following data field:
 |    sasname          : Variable name of descriptive/stratification factors (character)
 |    Notes: This data set is required only if user would like to perform Cox Proportional
 |           Hazard Model analysis in addition to treatment arm assignment and Age variable.
 |           The dataset should include all descriptive/stratification factors
 |           of user interest which will be used as predictor in both Univariate
 |           and Multivariate Cox Proportional Hazard Model Analysis. The data field of the
 |           listed factors must be included in the CRSE dataset
 |            (The variable specified in BY parameter should not be included)
 |            (If Age group will be one of the factor, please do not name it as Age, as
 |             it has been used in the CRSE dataset to represent the age as continuous variable)
 |
 *------------------------------------------------------------------*
 | REQUIRED PARAMETERS
 |
 | datasource   : The location where analysis datasets are saved
 |                1 = Internal CCS database
 |                2 = User provided datasets in WORK directory (DEFAULT)
 |
 | datatype     : Type of adverse events/symptoms reported
 |                1 = Adverse Events per CTCAE (DEFAULT)
 |                2 = Symptoms/QOL items per PRO
 |
 | by           : Group Comparison (i.e. arm)
 |
 | toxicity_list: List of toxicity codes separated by space for CTCAE
 |                (i.e. 800001 832222 1002222)
 |                OR List of SAS variable names separated by space for PRO
 |                symptoms/QOL items (i.e. lasa01 poms03 lasa09 lccs4)
 |                Notes: This allows up to 8 AEs/Symptoms/QOL items
 |
 | grade        : CTCAE Grade cut-off to perform time-to-event analysis
 |                of >=user specified grade (i.e. 3) (numeric)
 |                OR PRO Symptoms/QOL items Scores to determine the first occurrence of
 |                <=user specified scores and time-to-event analysis of <=user specified
 |                scores (i.e. 50)
 |                (Analysis will be inclusive of the grade/scores provided (i.e. >=3 or <=50)
 |
 | OPTIONAL PARAMETERS
 |
 | studynum     : study reference number (REQUIRED if datasource=1) (i.e. MCXXXX)
 |
 | *------------------------------------------------------------------*
 | SAMPLE CODES that can be used to prepare the QOL dataset
 |
 | For example:
 | QOL_tmp dataset contains one observation per timepoint with data fields include:
 | dcntr_id, visit, comp_dt, p_q01, p_q07, p_q09, p_q11
 |
 | proc sort data=qol_tmp;
 |    by dcntr_id visit comp_dt;
 | run;
 |
 | proc transpose=qol_tmp out=qol (rename=(col1=scores));
 |    by dcntr_id visit comp_dt;
 |    var p_q01 p_q07 p_q09 p_q11;
 | run;
 |
 | data qol;
 |    set qol;
 |    rename _name_=qol;
 |    label _name_='QOL';
 | run;
 |
 *------------------------------------------------------------------*
 | EXAMPLE:
 |
 | For CTCAE:
 |    %time_to_ae (by=arm, timepoint=cycle, grade=3,
 |                 toxicity_list=8000001 900182 900146 21162 21197 21521);
 | For PRO Symptoms/QOL items:
 |    %time_to_ae (by=arm, datatype=2, timepoint=visit, grade=60,
 |                 toxicity_list=p_q08 p_q09 p_q10 p_q11 p_q01 p_q06);
 *------------------------------------------------------------------*
 */

%macro time_to_ae (datasource=2, studynum=, datatype=1, by=, toxicity_list=, grade=);

   ****============================================================****;
   ********************************************************************;
   **** Customized template for K-M curves using Lifetest Procedure ***;
   ********************************************************************;
   ****============================================================****;

   %macro ltesttmpl (mytit=);
      proc template;
         source Stat.Lifetest.Graphics.ProductLimitSurvival;
         define statgraph Stat.Lifetest.Graphics.ProductLimitSurvival;
         dynamic NStrata xName plotAtRisk plotCensored plotCL plotHW plotEP labelCL labelHW labelEP maxTime xtickVals xtickValFitPol
            method StratumID classAtRisk plotBand plotTest GroupName yMin Transparency SecondTitle TestName pValue _byline_ _bytitle_
            _byfootnote_;

         BeginGraph;

         if (NSTRATA=1)
            if (EXISTS(STRATUMID))
                 entrytitle "&mytit.";
            else
                 entrytitle "&mytit.";
            endif;
            if (PLOTATRISK=1)
               entrytitle "With Number of Subjects at Risk" / textattrs=GRAPHVALUETEXT;
            endif;
            layout overlay / xaxisopts=(shortlabel=XNAME offsetmin=.05 linearopts=(viewmax=MAXTIME tickvaluelist=XTICKVALS
               tickvaluefitpolicy=XTICKVALFITPOL)) yaxisopts=(label="Adverse Event Free Probability"
               shortlabel="Event Free Probability"
               linearopts=(viewmin=0 viewmax=1 tickvaluelist=(0 0.2 0.4 0.6 0.8 1.0)));
               if (PLOTHW=1 AND PLOTEP=0)
                  bandplot LimitUpper=HW_UCL LimitLower=HW_LCL x=TIME / displayTail=false modelname="Survival" fillattrs=
                     GRAPHCONFIDENCE name="HW" legendlabel=LABELHW;
               endif;
               if (PLOTHW=0 AND PLOTEP=1)
                  bandplot LimitUpper=EP_UCL LimitLower=EP_LCL x=TIME / displayTail=false modelname="Survival" fillattrs=
                     GRAPHCONFIDENCE name="EP" legendlabel=LABELEP;
               endif;
               if (PLOTHW=1 AND PLOTEP=1)
                  bandplot LimitUpper=HW_UCL LimitLower=HW_LCL x=TIME / displayTail=false modelname="Survival" fillattrs=GRAPHDATA1
                     datatransparency=.55 name="HW" legendlabel=LABELHW;
                  bandplot LimitUpper=EP_UCL LimitLower=EP_LCL x=TIME / displayTail=false modelname="Survival" fillattrs=GRAPHDATA2
                     datatransparency=.55 name="EP" legendlabel=LABELEP;
               endif;
               if (PLOTCL=1)
                  if (PLOTHW=1 OR PLOTEP=1)
                     bandplot LimitUpper=SDF_UCL LimitLower=SDF_LCL x=TIME / displayTail=false modelname="Survival" display=(outline)
                        outlineattrs=GRAPHPREDICTIONLIMITS name="CL" legendlabel=LABELCL;
                  else
                     bandplot LimitUpper=SDF_UCL LimitLower=SDF_LCL x=TIME / displayTail=false modelname="Survival" fillattrs=
                        GRAPHCONFIDENCE name="CL" legendlabel=LABELCL;
                  endif;
               endif;
               stepplot y=SURVIVAL x=TIME / name="Survival" rolename=(_tip1=ATRISK _tip2=EVENT) tiplabel=(_tip1="Number at Risk"
                  _tip2
                  ="Observed Events") tip=(x y _tip1 _tip2) legendlabel="Survival";
               if (PLOTCENSORED=1)
                  scatterplot y=CENSORED x=TIME / markerattrs=(symbol=plus) tiplabel=(y="Adverse Event Free Probability")
                     name="Censored"
                     legendlabel="Censored";
               endif;
               if (PLOTCL=1 OR PLOTHW=1 OR PLOTEP=1)
                  discretelegend "Censored" "CL" "HW" "EP" / location=outside halign=center;
               else
                  if (PLOTCENSORED=1)
                     discretelegend "Censored" / location=inside autoalign=(topright bottomleft);
                  endif;
               endif;
               if (PLOTATRISK=1)
                  innermargin / align=bottom;
                     axistable x=TATRISK value=ATRISK / display=(label) valueattrs=(size=7pt);
                  endinnermargin;
               endif;
            endlayout;
         else
            entrytitle "&mytit.";

            if (EXISTS(SECONDTITLE))
               entrytitle SECONDTITLE / textattrs=GRAPHVALUETEXT;
            endif;
            layout overlay / xaxisopts=(shortlabel=XNAME offsetmin=.05 linearopts=(viewmax=MAXTIME tickvaluelist=XTICKVALS
               tickvaluefitpolicy=XTICKVALFITPOL)) yaxisopts=(label="Adverse Event Free Probability"
               shortlabel="Event Free Probability" linearopts=(viewmin=0
               viewmax=1 tickvaluelist=(0 .2 .4 .6 .8 1.0)));
               if (PLOTHW=1)
                  bandplot LimitUpper=HW_UCL LimitLower=HW_LCL x=TIME / displayTail=false group=STRATUM index=STRATUMNUM modelname=
                     "Survival" datatransparency=Transparency;
               endif;
               if (PLOTEP=1)
                  bandplot LimitUpper=EP_UCL LimitLower=EP_LCL x=TIME / displayTail=false group=STRATUM index=STRATUMNUM modelname=
                     "Survival" datatransparency=Transparency;
               endif;
               if (PLOTCL=1)
                  if (PLOTHW=1 OR PLOTEP=1)
                     bandplot LimitUpper=SDF_UCL LimitLower=SDF_LCL x=TIME / displayTail=false group=STRATUM index=STRATUMNUM
                        modelname="Survival" display=(outline) outlineattrs=(pattern=ShortDash);
                  else
                     bandplot LimitUpper=SDF_UCL LimitLower=SDF_LCL x=TIME / displayTail=false group=STRATUM index=STRATUMNUM
                        modelname="Survival" datatransparency=Transparency;
                  endif;
               endif;
               stepplot y=SURVIVAL x=TIME / group=STRATUM index=STRATUMNUM name="Survival" rolename=(_tip1=ATRISK
                  _tip2=EVENT)
                  tiplabel=(_tip1="Number at Risk" _tip2="Observed Events") tip=(x y _tip1 _tip2);
               if (PLOTCENSORED=1)
                  scatterplot y=CENSORED x=TIME / group=STRATUM index=STRATUMNUM
                     tiplabel=(y="Adverse Event Free Probability") markerattrs=(
                     symbol=plus);
               endif;
               if (PLOTATRISK=1)
                  innermargin / align=bottom;
                     axistable x=TATRISK value=ATRISK / display=(label) valueattrs=(size=7pt) class=CLASSATRISK colorgroup=CLASSATRISK;
                  endinnermargin;
               endif;
               DiscreteLegend "Survival" / title=GROUPNAME location=outside;
               if (PLOTCENSORED=1)
                  if (PLOTTEST=1)
                     layout gridded / rows=2 autoalign=(TOPRIGHT BOTTOMLEFT TOP BOTTOM) border=true BackgroundColor=GraphWalls:Color
                        Opaque=true;
                        entry "+ Censored";
                        if (PVALUE < .0001)
                           entry TESTNAME " p " eval (PUT(PVALUE, PVALUE6.4));
                        else
                           entry TESTNAME " p=" eval (PUT(PVALUE, PVALUE6.4));
                        endif;
                     endlayout;
                  else
                     layout gridded / rows=1 autoalign=(TOPRIGHT BOTTOMLEFT TOP BOTTOM) border=true BackgroundColor=GraphWalls:Color
                        Opaque=true;
                        entry "+ Censored";
                     endlayout;
                  endif;
               else
                  if (PLOTTEST=1)
                     layout gridded / rows=1 autoalign=(TOPRIGHT BOTTOMLEFT TOP BOTTOM) border=true BackgroundColor=GraphWalls:Color
                        Opaque=true;
                        if (PVALUE < .0001)
                           entry TESTNAME " p " eval (PUT(PVALUE, PVALUE6.4));
                        else
                           entry TESTNAME " p=" eval (PUT(PVALUE, PVALUE6.4));
                        endif;
                     endlayout;
                  endif;
               endif;
            endlayout;
         endif;
         if (_BYTITLE_)
            entrytitle _BYLINE_ / textattrs=GRAPHVALUETEXT;
         else
            if (_BYFOOTNOTE_)
               entryfootnote halign=left _BYLINE_;
            endif;
         endif;
         EndGraph;
         end;
      run;
   %mend ltesttmpl;

   ****============================================================================****;
   ************************************************************************************;
   **** Customized template for Cumulative Incidence Plots using Lifetest Procedure ***;
   ************************************************************************************;
   ****============================================================================****;

   %macro ltstcomptmpl (myctit=);
      proc template;
         source Stat.Lifetest.Graphics.cifPlot;
         define statgraph Stat.Lifetest.Graphics.cifPlot;
         notes "define cifPlot template";
         dynamic nCurves groupVar groupNum groupName plotTest pValue plotCL Confidence Transparency maxTime xName _byline_ _bytitle_
            _byfootnote;
         BeginGraph;
            if (NCURVES=1)
               entrytitle "&myctit.";
               entrytitle "with &compevent. as competing risk";
            else
               entrytitle "&myctit.";
               entrytitle "with &compevent. as competing risk";
            endif;
            if (PLOTCL)
               entrytitle "With " CONFIDENCE " Confidence Limits" / textattrs=GRAPHVALUETEXT;
            endif;
            layout overlay / xaxisopts=(shortlabel=XNAME offsetmin=0.05 linearopts=(viewmax=MAXTIME)) yaxisopts=(label=
               "Cumulative Incidence" shortlabel="CIF" linearopts=(viewmin=0 viewmax=1 tickvaluelist=(0 .2 .4 .6 .8 1.0)));
               if (PLOTCL=1)
                 Bandplot LimitUpper=CIF_UCL LimitLower=CIF_LCL x=TIME / displayTail=false group=GROUPVAR index=GROUPNUM
                     dataTransparency=Transparency modelname="CIF" name="CL";
               endif;
               stepplot y=CIF x=TIME / group=GROUPVAR index=GROUPNUM name="CIF" legendlabel="CIF";
               if (NCURVES > 1)
                  discretelegend "CIF" / title=GROUPNAME location=outside;
               endif;
               if (PLOTTEST=1)
                  layout gridded / rows=1 autoalign=(TOPLEFT BOTTOMRIGHT TOP BOTTOM) border=true BackgroundColor=GraphWalls:Color Opaque=
                     true;
                     if (PVALUE < .0001)
                        entry "Gray's Test p " eval (PUT(PVALUE, PVALUE6.4));
                     else
                        entry "Gray's Test p=" eval (PUT(PVALUE, PVALUE6.4));
                     endif;
                  endlayout;
               endif;
            endlayout;
            if (_BYTITLE_)
               entrytitle _BYLINE_ / textattrs=GRAPHVALUETEXT;
            else
               if (_BYFOOTNOTE_)
                  entryfootnote halign=left _BYLINE_;
               endif;
            endif;
         EndGraph;
         end;
      run;
   %mend ltstcomptmpl;


**********************************;
**** Variables Initializations ***;
**********************************;

%let studynum = %upcase(&studynum.);

%local myErrMsg chkdayson chkdayseval dtype dname scrgrd ddate1 ddate2 aevar survtlt figtbtlt dworst
       fig2tbtlt dylbl factors_list compevent;

%let chkdayson=0;
%let chkdayseval=0;
%let factors_list=;
%let compevent=Death;

%if &datatype.=1 %then %do;
   %let dtype=ae;
   %let dname=cytox;
   %let scrgrd=Grade;
   %let ddate1=eval_dt;
   %let ddate2=endat_dt;
   %let aevar=toxicity;
   %let survtlt=%str(Grade &grade.+);
   %let figtbtlt=%str (First Occurrence);
   %let fig2tbtlt=%str (AE);
   %let dylbl=%str (Toxicity);
   %let dworst=last;
%end;

%if &datatype.=2 %then %do;
   %let dtype=qol;
   %let dname=qol;
   %let scrgrd=Scores;
   %let ddate1=comp_dt;
   %let ddate2=;
   %let aevar=qol;
   %let survtlt=%str(Scores of <=&grade.);
   %let figtbtlt=%str(First Occurrence of <=&grade. Scores);
   %let fig2tbtlt=%str (Patient Reported QOL/Symptom);
   %let dylbl=%str (QOL/Symptom);
   %let dworst=first;
%end;

**================**;
**  Data Format   **;
**================**;

proc format;
   value &dtype.typef 1='First Occurrence'
                      2="Worst &scrgrd.";
run;

**================**;
** Style Template **;
**================**;

proc template;
   define Style lineplotstyle;
      parent = styles.journal;

      style fonts /
      'docFont' = ("<sans-serif>, Helvetica, Helv",2)
      'headingFont' = ("<sans-serif>, Helvetica, Helv",2,roman)
      'headingEmphasisFont' = ("<sans-serif>, Helvetica, Helv",2,bold roman)
      'FixedFont' = ("<monospace>, Courier",2)
      'BatchFixedFont' = ("SAS Monospace, <monospace>, Courier, monospace",2)
      'FixedHeadingFont' = ("<monospace>, Courier",2,roman)
      'FixedStrongFont' = ("<monospace>, Courier",2,bold)
      'FixedEmphasisFont' = ("<sans-serif>, Helvetica, Helv",2,bold roman)
      'EmphasisFont' = ("<sans-serif>, Helvetica, Helv",2,roman)
      'StrongFont' = ("<sans-serif>, Helvetica, Helv",2,bold roman)
      'TitleFont' = ("<sans-serif>, Helvetica, Helv",2,bold roman)
      'TitleFont2' = ("<sans-serif>, Helvetica, Helv",2,bold roman)
      'SASTitleFont' = ("<sans-serif>, Helvetica, Helv",2,bold roman);

      style GraphFonts from GraphFonts
      "Fonts used in graph styles" /
      'GraphTitleFont' = ("Arial",12.5pt, Bold)
      'GraphLabelFont' = ("Arial",12.5pt, Bold)
      'GraphValueFont' = ("Arial",12.5pt, Bold)
      'GraphDataFont' = ("Arial",12.5pt, Bold)
      'GraphFootnoteFont' = ("Arial",10pt);

      class GraphDataDefault / linethickness = 3px;
      style GraphAxisLines   / linethickness = 2px;
      style GraphWalls       / linethickness = 2px FrameBorder=on;

      class GraphData1 / linestyle=1;
      class GraphData2 / linestyle=2;
      class GraphData3 / linestyle=4;
      class GraphData4 / linestyle=5;
      class GraphData5 / linestyle=8;
      class GraphData6 / linestyle=15;
      class GraphData7 / linestyle=34;
      class GraphData8 / linestyle=35;
      class GraphData9 / linestyle=41;
      class GraphData10 / linestyle=42;

   end;
run;

**================================================**;
** Attribute Maps - Style to be used in box plots **;
**================================================**;

proc sql;
   create table attrmap_boxbar
      (id char (8),
       value char (16),
       fillcolor char (5),
       linecolor char (5),
       markercolor char (5));

   insert into attrmap_boxbar
          (id, value, fillcolor, linecolor, markercolor)
   values ("&dtype.type",  "First Occurrence", "white", "black", "black")
   values ("&dtype.type",  "Worst &scrgrd.",  "gray",  "black", "black")
   values ("&dtype.typec", "First Occurrence", "white", "black", "black")
   values ("&dtype.typec", "Worst &scrgrd.",  "gray",  "black", "black");

quit;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 PROMPT ERROR MESSAGE WHEN NECESSARY                *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

***************************************************************;
**** Display error when data source is not found or invalid ***;
***************************************************************;

%if &datasource. = %then %do;
   %let myErrMsg = "Data source is not found.";
   %goto exit;
%end;

%else %if &datasource. ^=1 and &datasource. ^=2 %then %do;
   %let myErrMsg = "Data source is invalid.";
   %goto exit;
%end;

********************************************************************;
**** Display error when data source is 1 and studynum is missing ***;
********************************************************************;

%if &datasource. = 1 and &studynum. = %then %do;
   %let myErrMsg = "Data Source of Cancer Center database was indicated, yet, study number is not provided.";
   %goto exit;
%end;

***********************************************************;
**** Display error when data type is missing or invalid ***;
***********************************************************;

%if &datatype. = %then %do;
   %let myErrMsg = "Datatype is not found.";
   %goto exit;
%end;

%else %if &datatype. ^=1 and &datatype. ^=2 %then %do;
   %let myErrMsg = "Datatype is invalid.";
   %goto exit;
%end;

*******************************************;
**** Display error when by is not found ***;
*******************************************;

%if &by. = %then %do;
   %let myErrMsg = "BY group is not found.";
   %goto exit;
%end;

*********************************************************;
**** Display error when toxicity_list is missing     ****;
*********************************************************;
%if &toxicity_list. = %then %do;
   %let myErrMsg = "No variable found in parameter TOXICITY_LIST.";
   %goto exit;
%end;

%let cntvar=1;

**** Get the number of toxicities specify by user;
%do %while (%scan(&toxicity_list., &cntvar.) ^= );
   %let cntvar   = %eval(&cntvar. + 1);
%end;

%let cntvar = %eval(&cntvar. - 1);

**********************************************;
**** Display error when grade is not found ***;
**********************************************;

%if &grade. = %then %do;
   %let myErrMsg = "Grade/Scores Cut-off is not found.";
   %goto exit;
%end;

**********************************************;
**** Display error when grade is invalid   ***;
**********************************************;

%if &datatype.=1 and (&grade.^=0 and &grade.^=1 and &grade.^=2 and
                      &grade.^=3 and &grade.^=4 and &grade.^=5) %then %do;
   %let myErrMsg = "Grade Cut-Off is invalid";
   %goto exit;
%end;

%if &datatype.=2 and (%eval (&grade. < 0) or %eval (&grade. > 100)) %then %do;
   %let myErrMsg = "Grade (Scores) Cut-Off is invalid";
   %goto exit;
%end;


**=======================**;
**  Required Datasets    **;
**=======================**;

**** if using cancer center database;
%if &datasource. = 1 %then %do;
   %crtlibn (d=&studynum.);

   proc copy in=pdata out=work;
   run;

%end;

*******************************************************;
**** Display error when CRSE dataset does not exist ***;
*******************************************************;

%if %sysfunc(exist(crse)) = 0 %then %do;
   %let myErrMsg = "crse dataset does not exist.";
   %goto exit;
%end;

************************************************************;
**** Display error when CYTOX/QOL dataset does not exist ***;
************************************************************;

%if %sysfunc(exist(&dname)) = 0 %then %do;
   %let myErrMsg = "&dname. dataset does not exist.";
   %goto exit;
%end;

**=========================**;
** Other Required Datasets **;
**=========================**;

%if %sysfunc(exist(factors)) = 1 %then %do;
   ****** Bring in the needed datasets *********;
   ****** Factors dataset to determine the descriptive and stractification factors for the study;
   data factors_dsn;
      length factors_list $500.;
      set work.factors end=eof;
      retain factors_list;

      if upcase (sasname) in (%upcase ("&by."), "AGE") then delete;

      factors_list = compbl (factors_list || sasname);

      if eof then do;
         call symput ('factors_list', factors_list);
         call symput ('factors_tot', trim (left(_n_)));
      end;
   run;

   %if %eval (&factors_tot. > 8) %then %do;
      %let myErrMsg = "Maximum number of factors allow is 8";
      %goto exit;
   %end;

%end;

*** edit 5/26/2017 ***;
**** CRSE dataset;
data crse0;
   set work.crse;
   keep dcntr_id &by. date_on age fu_date fu_stat &factors_list.;
run;

proc sort data=crse0;
   by dcntr_id;
run;

**** Determine the variable type, format and label for CRSE dataset;
proc sql;
  create table _w1 as
  select *
  from sashelp.vcolumn
  where upcase(libname)="WORK"
  and upcase(memname)="CRSE0";
quit;

data _null_;
   set _w1 end=eof;
   where upcase (name) not in ('DATE_ON', 'DCNTR_ID', 'EXCLUDED', 'MEMBER');

   if (label=' ') then label=name;

   if upcase(name)='AGE' then call symput('fctype' || trim (left(_n_)), 1);
   else call symput('fctype' || trim (left(_n_)), 2);

   call symput('fcf' || trim (left(_n_)), trim(left(format)));
   call symput('fclbl' || trim (left (_n_)), trim(left(label)));

   if (upcase(name)=%upcase("&by")) then do;
      if (format=' ') and (type='char') then format='$char'
         || trim(left(length)) || '.';
      if (format=' ') and (type='num') then format='8.';

      call symput('byf', trim(left(format)));
      call symput('bylbl', trim(left(label)));
      call symput('bytype', trim(left(type)));
   end;

   if eof then call symput ('numfc', trim(left(_n_)));

run;

**** Determine the number of level used in the By parameter;
proc sql;
   create table _wchkbylvl as
   select distinct &by.
   from crse0;
quit;

data _null_;
   set _wchkbylvl end=eof;
   call symput ('bylevel' || trim (left(_n_)), &by.);
   call symput ('bylvllbl' || trim (left(_n_)), put (&by., &byf.));

   if eof then call symput ('numlevel', trim(left(_n_)));
run;

**** Retrieve the formatted values of competing event;
**** so we can display the correct values of the competing event in the title of the figure;
data _null_;
   set crse0 end=eof;
   where fu_stat=2;

   if eof then call symput ('compevent', put (fu_stat, fustat.));
run;


*********************************************************************************;
**** Verify the toxicity list provided by users to make sure they are unique ****;
*********************************************************************************;

%if &datatype.=1 %then %do;
   **** CYTOX dataset - Overall AE analysis;
   proc sql;
      create table cytox0 as
      select dcntr_id, eval_dt, toxicity, grade, endat_dt
      from work.cytox
      order by dcntr_id;
   quit;
%end;

%if &datatype.=2 %then %do;
   proc sql;
      create table qol_items as
      select distinct qol, _label_
      from work.qol
   quit;

   data _null_;
      set qol_items end=eof;

      call symput ('qlist_'  || trim (left(_n_)), qol);
      call symput ('qlabel_' || trim (left(_n_)), _label_);

      if eof then call symput ('n_allqol', trim (left(_n_)));
   run;

   **** Create the format for the QOL items used;
   proc format;
      value $ qolf
         %do i=1 %to &n_allqol.;
            "&&&qlist_&i." = "&&&qlabel_&i."
         %end;
      ;
   run;

   **** If there is no entries of QOL whatsoever, that QOL will be set to missing;
   **** QOL dataset - Overall QOL analysis;
   proc sql;
      create table qol0 as
      select dcntr_id, comp_dt, qol format=$qolf., _label_, scores
      from work.qol
      order by dcntr_id;
   quit;
%end;

proc sql;
   create table _w2 as
   select *
   from sashelp.vcolumn
   where libname="WORK"
   and memname=%upcase("&dname.0");
quit;

data _null_;
   set _w2;
   if (label=' ') then label=name;
   if (upcase(name)=%upcase("&aevar.")) then do;
      call symput('txf',trim(left(format)));
      call symput('txlbl',trim(left(label)));
    end;
run;

**** Declare each toxicity as macro variable;
data _wtxlist0;
   %do i=1 %to &cntvar.;
      %if &datatype.=1 %then %do;
         toxvar&i.=%scan (&toxicity_list., &i.);
         toxlbl&i.=put (toxvar&i., &txf.);
      %end;

      %if &datatype.=2 %then %do;
         length toxvar&i toxlbl&i. $ 255.;
         format toxlbl&i. $qolf.;

         toxvar&i.="%scan (&toxicity_list., &i.)";
         toxlbl&i.= toxvar&i.;
      %end;
   %end;
run;


**** Remove any duplicates toxicities codes;
proc transpose data=_wtxlist0 out=_wf_txlist0;
   var %do i=1 %to &cntvar.;
       toxvar&i.
      %end;
      ;
run;

proc sort data=_wf_txlist0 nodupkeys;
   by col1;
run;

**** To retain the order of AE as what was specified by user;
proc sort data=_wf_txlist0;
   by _name_;
run;

**** Change the obs number if we want to accept more toxicity variables for analysis;
data _null_;
   set _wf_txlist0 (firstobs=1 obs=8) end=eof;
   call symput ('txvar' || trim(left(_n_)),  trim(left(col1)));
   call symput ('utxlbl' || trim (left(_n_)), put (col1, &txf.));
   if eof then call symput ('numvar', trim(left( _n_)));
run;

**--------------------------------------------------------------------------**;
******************************************************************************;
***** Data Manipulations for CTCAE                                       *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

%if &datatype.=1 %then %do;

   **** Check to see if the time to event values is negative;
   proc sql;
      create table _wchktime as
      select l.*, r.date_on, l.endat_dt - r.date_on as dayson, l.eval_dt - r.date_on as dayseval
      from cytox0 l left join crse0 r
      on l.dcntr_id = r.dcntr_id;
   quit;

   proc sql noprint;
      select min (dayson), min (dayseval)
      into: chkdayson, : chkdayseval
      from _wchktime;
   quit;

   proc sql;
      create table cytox1 as
      select dcntr_id, eval_dt, toxicity, grade, endat_dt
      from work.cytox where toxicity in
      (%do i=1 %to &numvar.;
         &&&txvar&i.
      %end;)
      ;
   quit;

   **** Overall AE Dataset to be used to determine 1st occurence of grade 1 and/above AE;
   proc sort data=cytox0 out=_wov_tmp1 (where= (eval_dt ^=. and grade not in (., 0)));
      by dcntr_id eval_dt grade;
   run;

   **** Overall AE Dataset to be used to determine the worst grade of AE;
   **** subset twice to make sure we only pick the first occurence of the worst grade;
   proc sort data=cytox0 out=_wov_tmp2 (where= (eval_dt ^=. and grade not in (.,0)));
      by dcntr_id grade eval_dt;
   run;

   **** Overall AE Dataset to determine the first occurrence of grade >= per user specify (time-to-event);
   proc sort data=cytox0 out=_wov_tmp3 (where=(eval_dt ^=. and grade >=&grade.));
      by dcntr_id eval_dt;
   run;

   **** Overall AE Dataset, determine the last AE evaluation date, which will be used in the case where no such AE reported per patient;
   proc sort data=cytox0 out=_wov_tmp4 (where=(eval_dt ^=.));
      by dcntr_id eval_dt;
   run;

   **** Specific AE Dataset, 1st occurence of grade 1 and/above AE;
   proc sort data=cytox1 out=_wtmp1 (where= (eval_dt ^=. and grade not in (., 0)));
      by dcntr_id toxicity eval_dt;
   run;

   **** Specific AE Dataset to be used to determine the worst grade of AE;
   proc sort data=cytox1 out=_wtmp2 (where= (eval_dt ^=. and grade not in (.,0)));
      by dcntr_id toxicity grade eval_dt;
   run;

   **** Specific AE Dataset to determine the first occurence of grade >= per user specify (time-to-event);
   proc sort data=cytox1 out=_wtmp3 (where=(eval_dt ^=. and grade >=&grade.));
      by dcntr_id toxicity eval_dt;
   run;

   **** Specific AE Dataset to determine the last AE evaluation date, which will be used in the case where no such AE reported per patient;
   proc sort data=cytox1 out=_wtmp4 (where=(eval_dt ^=.));
      by dcntr_id eval_dt;
   run;

%end;

**--------------------------------------------------------------------------**;
******************************************************************************;
***** Data Manipulations for PRO QOL                                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

%if &datatype.=2 %then %do;

   **** Check to see if the time to event values is negative;
   proc sql;
      create table _wchktime as
      select l.*, r.date_on, l.comp_dt - r.date_on as dayseval
      from qol0 l left join crse0 r
      on l.dcntr_id = r.dcntr_id;
   quit;

   proc sql noprint;
      select min (dayseval)
      into: chkdayseval
      from _wchktime;
   quit;

   **** QOL dataset - Specific AE analysis;
   proc sql;
      create table qol1 as
      select dcntr_id, comp_dt, qol format=$qolf., _label_, scores
      from work.qol where qol in
      (%do i=1 %to &numvar.;
         "&&&txvar&i."
      %end;)
      ;
   quit;

   /**** The overall figures are not applicable for QOL AE - so will skip that part ****/
   **** Specific QOL Dataset, 1st occurence of scores <= per user specify;
   proc sort data=qol1 out=_wtmp1 (where= (comp_dt ^=. and scores ^=. and scores <=&grade.));
      by dcntr_id qol comp_dt;
   run;

   **** Specific QOL Dataset to be used to determine the worst QOL Scores;
   **** subset twice to make sure we only pick the first occurrence of the worst scores;
   proc sort data=qol1 out=_wtmp2 (where= (comp_dt ^=. and scores ^=.));
      by dcntr_id qol scores comp_dt;
   run;

   **** Specific QOL Dataset to determine the first occurence of scores <= per user specify (time-to-event);
   **** For QOL AE, tmp1 and tmp3 would be the same, as we are looking at the scores cut-off specify by user;
   proc sort data=qol1 out=_wtmp3 (where=(comp_dt ^=. and scores ^=. and scores <=&grade.));
      by dcntr_id qol comp_dt;
   run;

   **** Specific QOL Dataset to determine the last AE evaluation date, which will be used in the case where no such AE reported per patient;
   proc sort data=qol1 out=_wtmp4 (where=(comp_dt ^=.));
      by dcntr_id comp_dt;
   run;

%end;

*********************************************************;
**** Display error when time is negative             ****;
*********************************************************;

%if %eval (&chkdayson. < 0) or %eval (&chkdayseval. < 0) %then %do;
   %let myErrMsg = "One or more negative time was found, data must be corrected.";
   %goto exit;
%end;


***** Txlist will be used later;
proc sql;
   create table _wtxlist as
   select distinct &aevar.
   from &dname.1;
quit;


%if &datatype.=1 %then %do;

   **=============================================================**;
   **                   DATA MANIPULATIONS                        **;
   **                      (Overall AE)                           **;
   ** FOR time to AE Bar Chart (First occurrence vs. Worst Grade) **;
   **=============================================================**;

   data _wov_ae1stocc_a;
      set _wov_tmp1;
      by dcntr_id eval_dt grade;
      if first.dcntr_id;
      keep dcntr_id grade eval_dt aetype;
      format aetype aetypef.;

      aetype=1;
   run;

   **** worst grade ae;
   **** subset twice to make sure we only pick the first occurence of the worst grade;
   data _wov_ae1stocc_b;
      set _wov_tmp2;
      by dcntr_id grade eval_dt;
      if first.grade;
      keep dcntr_id grade eval_dt aetype;
      format aetype aetypef.;

      aetype=2;
   run;

   proc sort data=_wov_ae1stocc_b out=_wov_tmp2b;
      by dcntr_id grade;
   run;

   data _wov_ae1stocc_c;
      set _wov_tmp2b;
      by dcntr_id grade;
      if last.dcntr_id;
   run;

   data _wov_aebar0;
      set _wov_ae1stocc_a _wov_ae1stocc_c;

      label aetype = "AE Type";
   run;

   proc sql;
      create table f_ov_aebar1 as
      select l.*, r.date_on, r.&by., l.eval_dt - r.date_on as aedays label='Days'
      from _wov_aebar0 l left join crse0 r
      on l.dcntr_id = r.dcntr_id
      order by r.&by., l.aetype;
   quit;

   proc univariate data=f_ov_aebar1;
      var aedays;
      class &by. aetype;
      output out=_wov_outuniv median=median;
   run;

   **** dataset to create bar chart - Overall AE;
   data f_ov_aebar;
      length aetypec $25. mygroup $25.;
      set _wov_outuniv;

      aetypec = put (aetype, aetypef.);
      mygroup = put (&by., &byf);

      label aetypec ="AE Type"
            mygroup ="&bylbl."
            median  ='Days';
   run;

   proc sort data=f_ov_aebar;
      by mygroup aetypec;
   run;

   **==================================================================**;
   **                   DATA MANIPULATIONS                             **;
   **                     (Overall AE)                                 **;
   ** FOR time to AE KM curve for each specific AE for specified grade **;
   **==================================================================**;

   **** Determine the first occurrence of grade >= (or <=) per user specify;
   data _wov_aeworst_a;
      set _wov_tmp3;
      by dcntr_id eval_dt;
      keep dcntr_id eval_dt grade ae_occ;

      if first.dcntr_id;

      ae_occ=1;
   run;

   proc sort data=_wov_aeworst_a;
      by dcntr_id;
   run;

   **** Determine last AE evaldt per patient;
   data _wov_aeworst_b;
      set _wov_tmp4;
      by dcntr_id eval_dt;
      keep dcntr_id lst_evaldt endat_dt;
      format lst_evaldt mmddyy10.;

      if last.dcntr_id;

      lst_evaldt = eval_dt;
   run;

   proc sql;
      create table _wov_aeworst_c as
      select l.*, r.lst_evaldt, r.endat_dt
      from crse0 l left join _wov_aeworst_b r
      on l.dcntr_id = r.dcntr_id
      order by l.dcntr_id;
   quit;

   **** data set to run time to event analysis;
   data f_ov_aeworst;
      merge _wov_aeworst_c (in=in1) _wov_aeworst_a;
      by dcntr_id;
      if in1;

      if ae_occ=. then do;
         if lst_evaldt ^=. or endat_dt ^=. then do;
            eval_dt = max (of lst_evaldt, endat_dt);
            ae_occ = 0;
         end;

         if lst_evaldt = . and endat_dt = . then do;
            eval_dt = date_on;
            ae_occ = 0;
         end;
      end;

      aedays = eval_dt - date_on;

      **** edit 5/26/2017 *****;
      **** competing risk analysis ****;
      crk_ae_occ = ae_occ;
      crk_aedays = aedays;

      if ae_occ=0 and fu_stat=2 then do;
         crk_ae_occ = 2;
         crk_aedays = fu_date - date_on;
      end;

      label aedays = 'Days'
            crk_aedays = 'Days';
   run;

   **** Workaround to remove the default failed event title from the header of the document prior to cif plot;
   title;
   ods noproctitle;
   ods graphics on;
   ods document name=_wciplot (write);
   ods select cifplot;
   ods output FailureSummary=_wfsum;
   proc lifetest data=f_ov_aeworst plots=cif (test);
      time crk_aedays*crk_ae_occ(0) / eventcode=1;
      strata &by. ;
   run;
   ods output close;
   ods document close;
   ods graphics off;

   title;
   ods noproctitle;
   proc document name=_wciplot;
      list / levels=all;
   run;
   quit;
%end;

**=============================================================**;
**                   DATA MANIPULATIONS                        **;
**                      (Specific AE)                          **;
** FOR time to AE Bar Chart (First occurrence vs. Worst Grade) **;
**=============================================================**;

data _wae1stocc_a;
   set _wtmp1;
   by dcntr_id &aevar. &ddate1.;
   if first.&aevar.;
   keep dcntr_id &aevar. &scrgrd. &ddate1. &dtype.type;
   format &dtype.type &dtype.typef.;

   &dtype.type=1;

   label &aevar.="&dylbl.";
run;

**** worst grade ae;
**** subset twice to make sure we only pick the first occurence of the worst grade;
data _wae1stocc_b;
   set _wtmp2;
   by dcntr_id &aevar. &scrgrd. &ddate1.;
   if first.&scrgrd.;
   keep dcntr_id &aevar. &scrgrd. &ddate1. &dtype.type;
   format &dtype.type &dtype.typef.;

   &dtype.type=2;

   label &aevar.="&dylbl.";
run;

proc sort data=_wae1stocc_b out=_wtmp2b;
   by dcntr_id &aevar. &scrgrd.;
run;

**** For CTCAE - highest grade is worst grade;
**** For QOLAE - lowest scores is worst scores;
data _wae1stocc_c;
   set _wtmp2b;
   by dcntr_id &aevar. &scrgrd.;
   if &dworst..&aevar.;
run;

data _waebar_tmp;
   set _wae1stocc_a _wae1stocc_c;

   label &dtype.type="%upcase(&dtype.) Type";
run;

proc sql;
   create table f_aebar1 as
   select l.*, r.date_on, r.&by., l.&ddate1. - r.date_on as &dtype.days label='Days'
   from _waebar_tmp l left join crse0 r
   on l.dcntr_id = r.dcntr_id
   order l.&aevar., l.&dtype.type;
quit;


**** retrieve the median days per toxicity per AE type;
%do i=1 %to &numlevel.;

   proc univariate data=f_aebar1;
      where
      %if &bytype.=num %then %do;
         &by. = &&&bylevel&i.
      %end;
      %else %if &bytype.=char %then %do;
         &by. = "&&&bylevel&i."
      %end;
      ;

      var &dtype.days;
      class &aevar. &dtype.type;
      output out=_woutuniv_&i. median=median;
   run;

   **** dataset to create bar chart;
   data f_aebar_&i.;
      length &dtype.typec $25. tox $25.;
      set _woutuniv_&i.;

      &dtype.typec = put (&dtype.type, &dtype.typef.);
      tox     = put (&aevar., &txf);

      label &dtype.typec ="%upcase(&dtype.) Type"
            tox          ="&dylbl."
            median       ='Days';
   run;

   proc sort data=f_aebar_&i.;
      by tox &dtype.typec;
   run;

%end;


**==================================================================**;
**                   DATA MANIPULATIONS                             **;
**                     (Specific AE)                                **;
** FOR time to AE KM curve for each specific AE for specified grade **;
**==================================================================**;

**** Determine the first occurrence of user specify AE grade;
data _waeworst_a;
   set _wtmp3;
   by dcntr_id &aevar. &ddate1.;
   keep dcntr_id &ddate1. &aevar. &scrgrd. &dtype._occ;

   if first.&aevar.;

   &dtype._occ=1;

run;

proc sort data=_waeworst_a;
   by dcntr_id &aevar.;
run;

**** Last AE follow up date will be determined using the last AE evaldt or off treatment date;
**** whichever come later;
**** Merge in the toxicity list for each patient;
proc sql;
   create table crse1 as
   select l.*, r.&aevar.
   from crse0 l left join _wtxlist r
   on l.dcntr_id
   order by l.dcntr_id;
quit;

**** Determine last AE evaldt per cytox;
data _waeworst_b;
   set _wtmp4;
   by dcntr_id &ddate1.;
   keep dcntr_id lst_evaldt &ddate2.;
   format lst_evaldt mmddyy10.;

   if last.dcntr_id;

   lst_evaldt = &ddate1.;

run;

proc sql;
   create table _waeworst_c as
   select l.*, r.lst_evaldt
      %if &datatype.=1 %then %do;
         , r.endat_dt
      %end;
   from crse1 l left join _waeworst_b r
   on l.dcntr_id = r.dcntr_id
   order by l.dcntr_id, l.&aevar.;
quit;

**** data set to run time to event analysis;
data _waeworst;
   merge _waeworst_c (in=in1) _waeworst_a;
   by dcntr_id &aevar.;
   if in1;

   if &dtype._occ=. then do;
      %if &datatype.=1 %then %do;
         if lst_evaldt ^=. or endat_dt ^=. then do;
            eval_dt = max (of lst_evaldt, endat_dt);
            ae_occ = 0;
         end;

         if lst_evaldt = . and endat_dt = . then do;
             eval_dt = date_on;
             ae_occ = 0;
         end;
      %end;

      %else %if &datatype.=2 %then %do;
         if lst_evaldt ^=. then do;
            comp_dt = lst_evaldt;
            qol_occ = 0;
         end;
      %end;

   end;

   &dtype.days = &ddate1. - date_on;

   **** edit 5/26/2017 *****;
   **** competing risk analysis ****;
   crk_&dtype._occ = &dtype._occ;
   crk_&dtype.days = &dtype.days;

   if &dtype._occ=0 and fu_stat=2 then do;
      crk_&dtype._occ = 2;
      crk_&dtype.days = fu_date - date_on;
   end;

   label &dtype.days = "Days"
         crk_&dtype.days = 'Days';

run;

%do i=1 %to &numvar.;
   proc sql;
      create table f_aeworst_&i. as
      select *
      from _waeworst

      %if &datatype.=1 %then %do;
         where toxicity=&&&txvar&i.
      %end;

      %else %if &datatype.=2 %then %do;
         where qol="&&&txvar&i."
      %end;
      ;
   quit;

   **** Workaround to remove the default failed event title from the header of the document prior to cif plot;
   title;
   ods noproctitle;
   ods graphics on;
   ods document name=_wciplot_&i. (write);
   ods select cifplot;
   ods output FailureSummary =_wfsum_&i.;
   proc lifetest data=f_aeworst_&i. plots=cif (test);
       time crk_&dtype.days*crk_&dtype._occ(0) / eventcode=1;
       strata &by. ;
   run;
   ods output close;
   ods document close;
   ods graphics off;

   title;
   ods noproctitle;
   proc document name=_wciplot_&i.;
      list / levels=all;
   run;
   quit;

%end;

ods graphics / outputfmt=png;

options nodate orientation=portrait;
ods rtf file="time_to_ae_figures_&dtype..doc" style=lineplotstyle bodytitle startpage=No;

%do i=1 %to &numvar.;

   ods path WORK.TEMPLAT(UPDATE) SASHELP.TMPLMST (READ);

   /**** Modified versionn of lifetest template to plot specific AE/QOL items*/
   %ltesttmpl (mytit=Figure 1_&i.a: Time to &survtlt. %trim(&&&utxlbl&i.) by %trim(&bylbl.));

   title;
   ods noproctitle;
   ods select survivalplot;
   ods output CensoredSummary=_www_&i.;
   proc lifetest data=f_aeworst_&i. plots=survival (test atrisk);
      time &dtype.days*&dtype._occ(0);
      strata &by. / test=logrank;
   run;
   ods output close;

   title;
   proc print data=_www_&i. noobs label;
      var &by. total failed censored pctcens;
   run;

   **** edit 5/26/2017 ***;
   ods path WORK.TEMPLAT(UPDATE) SASHELP.TMPLMST (READ);
   %ltstcomptmpl (myctit=Figure 1_&i.b: Incidence of &survtlt. %trim(&&&utxlbl&i.) by %trim(&bylbl.));

   title;
   ods noproctitle;
   proc document name=_wciplot_&i.;
      obstitle  \Lifetest#1\Failcode#1\cifPlot#1;
      replay    \Lifetest#1\Failcode#1\cifPlot#1;
   run;
   quit;

   title;
   proc print data=_wfsum_&i. noobs label;
      var &by. censored event compete total;
   run;

%end;

%do i=1 %to &numlevel.;
   title "Figure 2_&i.a: %trim(&bylbl.) %trim (&&&bylvllbl&i.): Median Time to &figtbtlt. and Worst %trim(&scrgrd.) for Patients Experiencin
g the %trim(&fig2tbtlt.)";
   proc sgplot data=f_aebar_&i. dattrmap=attrmap_boxbar;
      hbar tox / response=median stat=mean group=&dtype.typec groupdisplay=cluster attrid=&dtype.typec;
   run;

   title "Figure 2_&i.b: Box Plots of %trim(&bylbl.) %trim (&&&bylvllbl&i.): Time to &figtbtlt. and Worst %trim (&scrgrd.) for Patients Expe
riencing the %trim (&fig2tbtlt.)";
   proc sgplot data=f_aebar1 dattrmap=attrmap_boxbar;
      where
      %if &bytype.=num %then %do;
         &by. = &&&bylevel&i.
      %end;
      %else %if &bytype.=char %then %do;
         &by. = "&&&bylevel&i."
      %end;
      ;

      hbox &dtype.days / category=&aevar. group=&dtype.type attrid=&dtype.type;
      xaxis label='Days';
      keylegend / title="%upcase(&dtype.) Type";
   run;

   title "Table 2_&i.c: Summary Statistics of %trim(&bylbl.) %trim (&&&bylvllbl&i.): Time to &figtbtlt. and Worst %trim (&scrgrd.) (Days) f
or Patients Experiencing the %trim(&fig2tbtlt.)";

   proc sql;
      create table f_aebox_&i. as
      select &aevar., &dtype.type,  count (&dtype.days) as n_group label='N',
           mean (&dtype.days) as mn_&dtype.days format=8.2 label='Mean',
           std (&dtype.days) as std_&dtype.days format=8.2 label='Standard Deviation',
           median (&dtype.days) as md_&dtype.days format=8.2 label='Median',
           trim (left(put(min (&dtype.days), 8.2))) || '-' || trim(left(put (max (&dtype.days), 8.2))) as range label='Range'
      from f_aebar1
      where
      %if &bytype.=num %then %do;
         &by. = &&&bylevel&i.
      %end;
      %else %if &bytype.=char %then %do;
         &by. = "&&&bylevel&i."
      %end;
      group by &aevar., &dtype.type;
   quit;

   proc sql;
      select l.*
      from f_aebox_&i. l left join _wf_txlist0 r
      on l.&aevar. = r.col1
      order by r._name_, l.&dtype.type;
   quit;
   title;

%end;


%if &datatype.=1 %then %do;
   ods path WORK.TEMPLAT(UPDATE) SASHELP.TMPLMST (READ);

   /**** Modified versionn of lifetest template to plot specific AE/QOL items*/
   %ltesttmpl (mytit=Figure 3a: Time to &survtlt. Overall AE by %trim(&bylbl.));

   title;
   ods noproctitle;
   ods select survivalplot;
   ods output CensoredSummary=_wov_www;
   proc lifetest data=f_ov_aeworst plots=survival (test atrisk);
      time aedays*ae_occ(0);
      strata &by. / test=logrank;
   run;
   ods output close;

   title;
   proc print data=_wov_www noobs label;
      var &by. total failed censored pctcens;
   run;


   **** edit 5/26/2017 ***;
   ods path WORK.TEMPLAT(UPDATE) SASHELP.TMPLMST (READ);

   %ltstcomptmpl (myctit=Figure 3b: Incidence of &survtlt. Overall AE by %trim(&bylbl.));

   title;
   ods noproctitle;
   proc document name=_wciplot;
      obstitle  \Lifetest#1\Failcode#1\cifPlot#1;
      replay    \Lifetest#1\Failcode#1\cifPlot#1;
   run;
   quit;

   title;
   proc print data=_wfsum noobs label;
      var &by. censored event compete total;
   run;

title "Figure 4a: Median Time to &figtbtlt. and Worst %trim(&scrgrd.) on Overall %trim(%upcase(&dtype.)) by %trim(&bylbl.) for Patients Expe
riencing the %trim(%upcase(&dtype.))";
proc sgplot data=f_ov_aebar dattrmap=attrmap_boxbar;
   hbar &by. / response=median stat=mean group=aetypec groupdisplay=cluster attrid=aetypec;
run;

/* add the extreme option if we would like to reflect the actual min and max on box plot */
title "Figure 4b: Box Plots of Time to &figtbtlt. and Worst %trim(&scrgrd.) on Overall %trim(%upcase(&dtype.)) by %trim(&bylbl.) for Patient
s Experiencing the %trim(%upcase(&dtype.))";
proc sgplot data=f_ov_aebar1 dattrmap=attrmap_boxbar;
   hbox aedays / category=&by. group=aetype attrid=aetype;
   xaxis label='Days';
   keylegend / title="TOXICITY Type";
run;

title "Table 4c: Summary Statistics of Time to &figtbtlt. and Worst %trim (&scrgrd.) (Days) on Overall %trim (%upcase(&dtype.)) by %trim(&b
ylbl.) for Patients Experiencing the %trim (%upcase(&dtype.))";

   proc sql;
      select &by., aetype,  count (aedays) as n_group label='N',
             mean (aedays) as mn_aedays format=8.2 label='Mean',
             std (aedays) as std_aedays format=8.2 label='Standard Deviation',
             median (aedays) as md_aedays format=8.2 label='Median',
             trim (left(put (min (aedays), 8.2))) || '-' || trim (left(put (max (aedays), 8.2))) as range label='Range'
      from f_ov_aebar1
      where
         %if &bytype.=num %then %do;
            &by. ^ = .
         %end;
         %else %if &bytype.=char %then %do;
            &by. ^= ' '
         %end;
       group by &by., aetype;
   quit;
%end;

ods rtf close;
ods graphics off;


**** Delete the modified version of the template for KM curve;
proc template;
   delete Stat.Lifetest.Graphics.ProductLimitSurvival;
run;

**** Delete the modified version of the template for CIF;
proc template;
   delete Stat.Lifetest.Graphics.cifPlot;
run;

%do z=1 %to &numvar.;
   %stable (dsn     =f_aeworst_&z.,
            time    =&dtype.days,
            status  =&dtype._occ,
            censor  =0,
            percent =N,
            pctci   =N,
            var     =&by. &factors_list.,
            cvar    =age,
            outdoc  =time_to_ae_tbl_&z._&dtype..doc,
            title1  =Table 1_&z.: Time to &survtlt. &&&utxlbl&z.,
            title2  =on patients experiencing the &fig2tbtlt.);
%end;

%if &datatype.=1 %then %do;
   %stable (dsn     =f_ov_aeworst,
            time    =aedays,
            status  =ae_occ,
            censor  =0,
            percent =N,
            pctci   =N,
            var     =&by. &factors_list.,
            cvar    =age,
            outdoc  =time_to_ae_tbl_ov_ae.doc,
            title1  =Table 2 : Time to &survtlt. Overall AE,
            title2  =on patients experiencing the AE);
%end;

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                        REMOVE WORKING DATA SET                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

proc datasets library=work nolist;
   delete _w:;
quit;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 EXIT THE PROGRAM WHEN ERROR OCCUR                  *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
%exit:

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                       DISPLAY THE MACRO STATUS                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
data _null_;
   put "---------------------------";
   %if &myErrMsg. ^= %then %do;
      put "ERROR: " &myErrMsg.;
   %end;
   put "---------------------------";
run;


%mend time_to_ae;

