
/*------------------------------------------------------------------*
 | MACRO NAME: heatmap
 | SHORT DESC: Generate the heat maps to show overall patterns in
 |             adverse event for each treatment group over time
 *------------------------------------------------------------------*
 | CREATED BY: ToxT Team                               (07/20/2016)
 |
 | Version 2 (4/27/2018): Sort the _wcytox_tmp dataset for merging purpose
 |                        Completed - 1/7/2019
 *------------------------------------------------------------------*
 | PURPOSE
 |
 | This program generate the heat maps to show overall patterns in
 | individual or overall adverse events per NCI Common Terminology Criteria
 | for Adverse Events (CTCAE) or individual symptoms/QOL items
 | reported per Patient Report Outcomes (PRO) for each treatment groups
 | across data collection time points
 |
 | The program will produce one document in the directory where SAS is run:
 |
 | For CTCAE:
 | heatmap_ae.doc: Heat maps for individual or overall
 |                 adverse event for each treatment group
 |                 (white area indicates missing,
 |                  pinkish grey indicates grade 0 and
 |                  black indicates grade 5)
 |
 | NOTES: Overall AE is summarized across all AEs that are included
 |        in the CYTOX dataset. If user would like to see the Overall AE
 |        across a subset of AEs that are specified in TOXICITY_LIST
 |        parameter, the CYTOX dataset need to be modified to include
 |        only the subset AEs that are of user interest
 |
 | For PRO Symptoms/QOL items:
 | heatmap_qol.doc: Heat maps for individual symptom/
 |                  QOL items for each treatment group
 |                  over time (Categories: 0-50 (black), >50-75
 |                  and >75-100 (light gray) with 0=low QOL
 |                  and 100=Best QOL)
 |
 | ~ IMPORTANT NOTES: ~
 | 1. The heat maps will be produced in black/white version ONLY
 | 2. Datasets that are used to produce the bar chars and summary tables
 |    can be found in WORK directory with prefix of f_ in dataset name
 |    where user can further modify or tweak the graphing per user preference
 |
 *------------------------------------------------------------------*
 | REQUIRED DATASETS and DATA FIELDS
 |
 | 1. crse dataset with one observation per patient with the following data fields:
 |    dcntr_id          : patient identifier
 |    evalae            : evaluable for toxicity (numeric)
 |                        (1= at least one AE form is entered other than
 |                            the baseline timepoint
 |                         0= otherwise)
 |    [comparison group]: group to be used for data comparison
 |                        which will be specified in BY macro parameter
 |                        (i.e. arm)
 |
 | 2. prodata dataset with one observation per time point per patient with the
 |    following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for treatment (numeric) which will be specified
 |                        in TIMEPOINT macro parameter (i.e. cycle)
 |    Notes: The dataset should include all treatment time point that are
 |            of user interest for the analysis
 |
 | 3. cytox dataset with one observation per maximum grade per adverse event per
 |    timepoint per patient with the following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for adverse events reporting (numeric) which
 |                        will be specified in TIMEPOINT macro parameter (i.e. cycle)
 |    toxicity          : Adverse event as in medra codes (numeric and should be formatted
 |                        to give adverse event description) (i.e. 900182)
 |    grade             : Severity of the adverse event according to CTC guidelines.
 |                        (numeric) (i.e. 0, 1, 2, 3, 4, 5)
 |    Notes: The dataset should include all adverse event reported up to the last time point
 |           indicated in protdata dataset of user interest for the analysis
 |
 | 4. qol dataset with one observation per symptom per timepoint per patient
 |    with the following data fields (REQUIRED for PRO Symptoms analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for symptoms reporting (numeric) which
 |                        will be specified in TIMEPOINT macro parameter (i.e. cycle)
 |    qol               : SAS variable name for each symptom (character)
 |                        (i.e. p_q10, poms04)
 |    _label_           : Description/Label for each symptom (character)
 |                        (i.e. Overall QOL, Sleep Interference, Nervousness)
 |    scores            : Symptoms/QOL items scores in a scale of 0 to 100 (numeric)
 |                        with 0=Low Quality of Life and 100=Best Quality of Life
 |                        Notes: for symptoms with 0=no symptom and 100=worst symptom
 |                               the scores need to be reversed before using this program
 |    Notes: The dataset should include all symptoms/qol items reported up to the
 |           last time point of user interest for the analysis
 *------------------------------------------------------------------*
 | REQUIRED PARAMETERS
 |
 | datasource   : The location where analysis datasets are saved
 |                1 = Internal CCS database
 |                2 = User provided datasets in WORK directory (DEFAULT)
 |
 | datatype     : Type of adverse events/symptoms reported
 |                1 = Adverse Events per CTCAE (DEFAULT)
 |                2 = Symptoms/QOL items per PRO
 |
 | by           : Group Comparison (i.e. arm)
 |
 | timepoint    : Time variable (numeric)
 |
 | toxicity_list: List of toxicity codes separated by space for CTCAE
 |                (i.e. 800001 832222 1002222)
 |                OR List of SAS variable names separated by space for PRO
 |                symptoms/QOL items (i.e. lasa01 poms03 lasa09 lccs4)
 |                Notes: This allows up to 8 AEs/Symptoms/QOL items

 | OPTIONAL PARAMETERS
 |
 | studynum     : study reference number (REQUIRED if datasource=1) (i.e. MCXXXX)
 |
 | incmiss      : AE grade assumption when specified AE entry is not available (for CTCAE ONLY)
 |                y = assumed grade 0 if there is no entry for user specified AE in the data set,
 |                    with condition that data entry is available for other AE at specific
 |                    timepoint (DEFAULT)
 |                n = assumed missing or do nothing if there is no entry for user specified
 |                    AE in the dataset
 *------------------------------------------------------------------*
 | SAMPLE CODES that can be used to prepare the QOL dataset
 |
 | For example:
 | QOL_tmp dataset contains one observation per timepoint with data fields include:
 | dcntr_id, visit, p_q01, p_q07, p_q09, p_q11
 |
 | proc sort data=qol_tmp;
 |    by dcntr_id visit;
 | run;
 |
 | proc transpose=qol_tmp out=qol (rename=(col1=scores));
 |    by dcntr_id visit;
 |    var p_q01 p_q07 p_q09 p_q11;
 | run;
 |
 | data qol;
 |    set qol;
 |    rename _name_=qol;
 |    label _name_='QOL';
 | run;
 |
 *------------------------------------------------------------------*
 | EXAMPLE:
 |
 | For CTCAE:
 |    %heatmap (by=arm, timepoint=cycle,
 |              toxicity_list=8000001 900182 900146 21162 21197 21521);
 | For PRO Symptoms/QOL items:
 |    %heatmap (by=arm, datatype=2, timepoint=visit,
 |              toxicity_list=p_q08 p_q09 p_q10 p_q11 p_q01 p_q06);
 *------------------------------------------------------------------*
 */

%macro heatmap (datasource=2, studynum=, datatype=1, by=, timepoint=, toxicity_list=, incmiss=y);

**********************************;
**** Variables Initializations ***;
**********************************;
%let studynum = %upcase(&studynum.);

%local myErrMsg dtype dname scrgrd aevar dfigtbtlt dsort dfnote char_tmpt;

%if &datatype.=1 %then %do;
   %let dtype     =ae;
   %let dname     =cytox;
   %let scrgrd    =Grade;
   %let aevar     =toxicity;
   %let dfigtbtlt =%str(CTCAE Grade);
   %let dsort     =last;
   %let dfnote    =toxicity per CTCAE e.g. black indicates grade 5;
%end;

%if &datatype.=2 %then %do;
   %let dtype     =qol;
   %let dname     =qol;
   %let scrgrd    =Scores;
   %let aevar     =qol;
   %let dfigtbtlt =%str(Patient Reported QOL/Symptom with 0:Low and 100:Best QOL);
   %let dsort     =first;
   %let dfnote    =QOL measured symptom;
%end;

**================**;
**  Data Format   **;
**================**;

proc format;
   value yyesnof 1='Yes'
                 2='No';

run;

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 PROMPT ERROR MESSAGE WHEN NECESSARY                *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

***************************************************************;
**** Display error when data source is not found or invalid ***;
***************************************************************;

%if &datasource. = %then %do;
   %let myErrMsg = "Data source is not found.";
   %goto exit;
%end;

%else %if &datasource. ^=1 and &datasource. ^=2 %then %do;
   %let myErrMsg = "Data source is invalid.";
   %goto exit;
%end;


********************************************************************;
**** Display error when data source is 1 and studynum is missing ***;
********************************************************************;

%if &datasource. = 1 and &studynum. = %then %do;
   %let myErrMsg = "Data Source of Cancer Center database was indicated, yet, study number is not provided.";
   %goto exit;
%end;

***********************************************************;
**** Display error when data type is missing or invalid ***;
***********************************************************;

%if &datatype. = %then %do;
   %let myErrMsg = "Datatype is not found.";
   %goto exit;
%end;

%else %if &datatype. ^=1 and &datatype. ^=2 %then %do;
   %let myErrMsg = "Datatype is invalid.";
   %goto exit;
%end;

*****************************************;
**** Display error when by is missing ***;
*****************************************;

%if &by. = %then %do;
   %let myErrMsg = "By group is not found.";
   %goto exit;
%end;

************************************************;
**** Display error when timepoint is missing ***;
************************************************;

%if &timepoint. = %then %do;
   %let myErrMsg = "Timepoint is not found.";
   %goto exit;
%end;

*********************************************************;
**** Display error when toxicity_list is missing     ****;
*********************************************************;
%if &toxicity_list. = %then %do;
   %let myErrMsg = "No variable found in parameter TOXICITY_LIST.";
   %goto exit;
%end;

%let cntvar=1;

**** Get the number of toxicities specify by user;
%do %while (%scan(&toxicity_list., &cntvar.) ^= );
   %let cntvar   = %eval(&cntvar. + 1);
%end;

%let cntvar = %eval(&cntvar. - 1);

**=======================**;
**  Required Datasets    **;
**=======================**;

**** if using cancer center database;
%if &datasource. = 1 %then %do;
   %crtlibn (d=&studynum.);

   proc copy in=pdata out=work;
   run;

%end;

*******************************************************;
**** Display error when CRSE dataset does not exist ***;
*******************************************************;

%if %sysfunc(exist(crse)) = 0 %then %do;
   %let myErrMsg = "crse dataset does not exist.";
   %goto exit;
%end;

************************************************************;
**** Display error when CYTOX/QOL dataset does not exist ***;
************************************************************;

%if %sysfunc(exist(&dname)) = 0 %then %do;
   %let myErrMsg = "&dname. dataset does not exist.";
   %goto exit;
%end;

************************************************************;
**** Display error when PROTDATA dataset does not exist ***;
************************************************************;

%if %upcase (&dname.)=CYTOX %then %do;
   %if %sysfunc(exist(protdata)) = 0 %then %do;
      %let myErrMsg = "protdata dataset does not exist.";
      %goto exit;
   %end;
%end;

**==========================**;
**  Other Required Datasets **;
**==========================**;

**** CRSE dataset;
proc sql;
   create table crse0 as
   select dcntr_id, evalae, &by.
   from work.crse
   order by dcntr_id;
quit;

**** Determine the variable type, format and label for CRSE dataset;
proc sql;
  create table _w1 as
  select *
  from sashelp.vcolumn
  where upcase(libname)="WORK"
  and upcase(memname)="CRSE0";
quit;

**** Retrieve the type, format and label for By variable;
data _null_;
   set _w1 end=eof;

   if (label=' ') then label=name;

   if (upcase(name)=%upcase("&by")) then do;
      if (format=' ') and (type='char') then format=compress('$char' || trim(left(length)) || '.');
      if (format=' ') and (type='num') then format='8.';

      call symput('byf', trim(left(format)));
      call symput('bylbl', trim(left(label)));
      call symput('bytype', trim(left(type)));
   end;

run;

**** Determine the number of level used in the By parameter;
proc sql;
   create table _wchkbylvl as
   select distinct &by.
   from crse0;
quit;

data _null_;
   set _wchkbylvl end=eof;
   call symput ('bylevel' || trim (left(_n_)), &by.);
   call symput ('bylvllbl' || trim (left(_n_)), put (&by., &byf.));

   if eof then call symput ('numlevel', trim(left(_n_)));
run;

*********************************************************************************************;
**** Determine the number of evaluation patients per protdata - applicable to CTCAE data ONLY;
**** Verify the toxicity list provided by users to make sure they are unique             ****;
*********************************************************************************************;

%if &datatype.=1 %then %do;
   proc sort data=work.cytox;
      by dcntr_id;
   run;

   proc sort data=work.protdata;
      by dcntr_id;
   run;

   **** Make sure the cytox dataset provided are for the intended population;
   data _wcytox_tmp;
      merge crse0 (in=in1 keep=dcntr_id)
            cytox (in=in2 keep=dcntr_id &timepoint. toxicity grade);
      by dcntr_id;
      if in1 and in2;
   run;

   **** Make sure the protdata dataset provided are for the intended population;
   data _wprotdata_tmp;
      merge crse0 (in=in1 keep=dcntr_id)
            protdata (in=in2 keep=dcntr_id &timepoint.);
      by dcntr_id;
      if in1 and in2;
   run;

   **** All cycles needed to exist in order to run AUC macro, data can be missing at that cycle;
   **** Even though this does not apply to this macro, will still keep it the same as;
   **** what was written in the other macro for consistency;
   proc sql;
      create table _wchktimelvl0 as
      select distinct &timepoint.
      from _wcytox_tmp
      order by &timepoint.;
   quit;

   **** Determine the unique AE evaluation cycles and if patient is evaluable for AE;
   proc sql;
      create table _wpttime as
      select l.*, r.&timepoint.
      from crse0 l left join _wchktimelvl0 r
      on l.dcntr_id
      where l.evalae=1
      order by l.dcntr_id, r.&timepoint.;
   quit;

   proc sql;
      create table _wpttime_tmp as
      select distinct dcntr_id, max (&timepoint.) as mxcyc_prot
      from _wprotdata_tmp
      group by dcntr_id
      order by dcntr_id;
   quit;

   proc sql;
      create table protdata0 as
      select l.*, r.mxcyc_prot
      from _wpttime l left join _wpttime_tmp r
      on l.dcntr_id = r.dcntr_id
      having l.&timepoint. <= r.mxcyc_prot
      order by l.dcntr_id, l.&timepoint.;
   quit;

   proc sort data=_wcytox_tmp;
      by dcntr_id &timepoint.;
   run;

   **** Assume Grade 0 when certain toxicity is not reported, but, AE entries is available at that cycle;
   proc sql;
      create table checkAEgrade as
      select distinct l.dcntr_id, l.&timepoint., 1 as assume_grd0, 0 as grade, r.&by., r.mxcyc_prot
      from _wcytox_tmp l left join protdata0 r
      on l.dcntr_id = r.dcntr_id and l.&timepoint. = r.&timepoint.
      having l.&timepoint. <= r.mxcyc_prot
      order l.dcntr_id, l.&timepoint.;
   quit;

   *** Include only patients who indicated as evaluable for AE and with treatment cycles available;
   data _wcytox0;
      merge protdata0    (in=in1)
            _wcytox_tmp  (in=in2);
      by dcntr_id &timepoint.;
      if in1;
      evalpt=1;
   run;

   **** CYTOX dataset - Overall AE analysis (include evaluable patients only);
   **** Grade 0 will be pre-filled at each cycle if at least one AE was reported at that cycle;
   **** and was indicated by user to do so;
   data cytox0;
      set _wcytox0

      %if %upcase (&incmiss.)=Y %then %do;
         checkAEgrade (drop=assume_grd0 mxcyc_prot)
      %end;
      ;
   run;
%end;

%if &datatype.=2 %then %do;
   proc sql;
      create table qol_items as
      select distinct qol, _label_
      from work.qol
   quit;

   data _null_;
      set qol_items end=eof;

      call symput ('qlist_'  || trim (left(_n_)), qol);
      call symput ('qlabel_' || trim (left(_n_)), _label_);

      if eof then call symput ('n_allqol', trim (left(_n_)));
   run;

   **** Create the format for the QOL items used;
   proc format;
      value $ qolf
         %do i=1 %to &n_allqol.;
            "&&&qlist_&i." = "&&&qlabel_&i."
         %end;
      ;
   run;

   **** If there is no entries of QOL whatsoever, that QOL will be set to missing;
   **** QOL dataset - Base dataset;
   proc sql;
      create table qol0 as
      select dcntr_id, &timepoint., qol format=$qolf., _label_, scores
      from work.qol
      order by dcntr_id;
   quit;
%end;

proc sql;
   create table _c2 as
   select *
   from sashelp.vcolumn
   where libname="WORK"
   and memname=%upcase("&dname.0");
quit;

**** Retrieve the format and label for Toxicity/QOL AE and timepoints;
data _null_;
   set _c2;
   if (label=' ') then label=name;
   if (upcase(name)=%upcase("&aevar.")) then do;
      call symput('txf',trim(left(format)));
      call symput('txlbl',trim(left(label)));
   end;

   if (upcase(name)=%upcase ("&timepoint.")) then do;
      if type='char' then do;
         call symput ('char_tmpt', 'y');
      end;

      if type='num'  and format=' ' then format=compress(trim(left(length)) || '.');

       call symput('tmptf',trim(left(format)));
       call symput('tmptlbl',trim(left(label)));
   end;
run;

************************************************************;
**** Display error when Timepoint is Characters field    ***;
************************************************************;

%if %upcase (&char_tmpt.)=Y %then %do;
   %let myErrMsg = "TIMEPOINT required a numeric field";
   %goto exit;
%end;

**** Declare each toxicity as macro variable;
data _wtxlist0;
   %do i=1 %to &cntvar.;
      %if &datatype.=1 %then %do;
         toxvar&i.=%scan (&toxicity_list., &i.);
         toxlbl&i.=put (toxvar&i., &txf.);
      %end;

      %if &datatype.=2 %then %do;
         length toxvar&i toxlbl&i. $ 255.;
         format toxlbl&i. $qolf.;

         toxvar&i.="%scan (&toxicity_list., &i.)";
         toxlbl&i.= toxvar&i.;
      %end;
   %end;
run;


**** Remove any duplicates toxicities codes;
proc transpose data=_wtxlist0 out=_wf_txlist0;
   var %do i=1 %to &cntvar.;
       toxvar&i.
      %end;
      ;
run;

proc sort data=_wf_txlist0 nodupkeys;
   by col1;
run;

**** To retain the order of AE as it was specified by user;
proc sort data=_wf_txlist0;
   by _name_;
run;


**** Change the obs number if we want to accept more toxicity variables for analysis;
data _null_;
   set _wf_txlist0 (firstobs=1 obs=8) end=eof;
   call symput ('txvar' || trim(left(_n_)),  trim(left(col1)));
   call symput ('utxlbl' || trim (left(_n_)), put (col1, &txf.));
   if eof then call symput ('numvar', trim(left( _n_)));
run;

**** Determine the number of timepoints used in the timepoint parameter;
proc sql;
   create table _wchktimelvl as
   select distinct &timepoint.
   from &dname.0
   order by &timepoint.;
quit;

data _null_;
   length alltmpt1 alltmpt2 $255.;
   set _wchktimelvl end=eof;
   retain alltmpt1 alltmpt2;

   alltmpt1 = compbl (alltmpt1 || &timepoint.);
   alltmpt2 = compbl (alltmpt2 || compress ('_' || &timepoint.));

   if _n_=1 then call symput ('tmpt1level', &timepoint.);

   call symput ('tmptlevel' || trim (left(_n_)), &timepoint.);
   call symput ('dtmptlevel' || trim (left(_n_)), compress('_' || &timepoint.));
   call symput ('tmptlvllbl' || trim (left(_n_)), put (&timepoint., &tmptf.));

   if eof then do;
      call symput ('numtmptlevel', trim(left(_n_)));
      call symput ('alltmptlevel1', alltmpt1);
      call symput ('alltmptlevel2', alltmpt2);
   end;

run;

**=============================================================**;
**                   DATA MANIPULATIONS                        **;
** *********This applicable for CTCAE datasets ONLY *************;
** Maximum grade across ALL Adverse Events that are included   **;
** in cytox dataset at each timepoints                         **;
**=============================================================**;

%if &datatype.=1 %then %do;

   proc sort data=cytox0 out=_wov_mxtox;
      by dcntr_id &timepoint. grade;
   run;

   data _wmxtox;
      set _wov_mxtox;
      by dcntr_id &timepoint. grade;
      if last.&timepoint.;

      mytimepoint=&timepoint.;
   run;

   **** OVERALL AE: Dataset for each By-group;
   %do y=1 %to &numlevel.;
      data _wsmxallae_&y.;
         set _wmxtox;

         %if &bytype.=num %then %do;
            where &by. = &&&bylevel&y.;
         %end;

         %else %if &bytype.=char %then %do;
            where &by. = "&&&bylevel&y.";
         %end;
      run;

      proc sort data=_wsmxallae_&y.;
         by dcntr_id &by.;
      run;

      **** Transpose the dataset for further sorting purpose;
      proc transpose data=_wsmxallae_&y. out=_wf_smxallae_&y.;
         by dcntr_id &by.;
         id mytimepoint;
         var grade;
      run;

      **** Sort the dataset by grade reported at each timepoint;
      **** To group together the patients with similiar trend of AE grade/QOL Scores across timepoints;
      proc sort data=_wf_smxallae_&y.;
         by %do i=1 %to &numtmptlevel.;
               _%trim(&&tmptlevel&i.)
            %end;
         ;
      run;

      **** Assign patient id which indicated the order to display the grade/scores;
      data _wf_smxallae_&y.;
         set _wf_smxallae_&y.;
         myptid=_n_;

         label myptid='Patient';
      run;

      **** FINAL dataset to generate the Overall AE event chart;
      proc sql;
         create table f_mxallae_&y. as
         select l.*, r.myptid
         from _wsmxallae_&y. l left join _wf_smxallae_&y. r
         on l.dcntr_id = r.dcntr_id
         order by r.myptid, l.grade;
      quit;

      **** Determine the total patients to plot the max y-axis;
      **** and to include the information in the title of the table;
      data _null_;
         set f_mxallae_&y. end=eof;
         by myptid;

         if eof then call symput ('totpt_' || trim(left(&y.)), myptid);
      run;

   %end;
%end;

**==================================================================**;
**                   DATA MANIPULATIONS                             **;
**                (For Both CTCAE and QOLAE)                        **;
** Grade per adverse events or Scores per QOL AE at each timepoints **;
** If multiple grade was reported per AE at specific timepoints     **;
** proc transpose will fail and users need to fix their data        **;
** before re-run the program                                        **;
**==================================================================**;

%do y=1 %to &numvar.;

   **** To make sure all patients have an observation for each timepoint;
   %if &datatype.=1 %then %do;
      proc sql;
         create table _wcytox1 as
         select distinct l.dcntr_id, l.&timepoint., l.&by., l.evalae, l.evalpt, l.mxcyc_prot, r.assume_grd0
         from cytox0 (where=(grade^=0 and toxicity^=. and evalae^=.)) l left join checkAEgrade r
         on l.dcntr_id = r.dcntr_id;
      quit;

      proc sql;
         create table _ws_mxtox_&y. as
         select l.*, r.toxicity, r.grade
         from _wcytox1 l left join cytox0 (where=(toxicity=&&&txvar&y.)) r
         on l.dcntr_id    = r.dcntr_id and
            l.&timepoint. = r.&timepoint.;
      quit;

      **** If AE is not reported at the time point, assumed patient do not have that AE, will code as 0;
      data _wmxtox_&y.;
         set _ws_mxtox_&y.;

         %if %upcase (&incmiss.)=Y %then %do;
            if grade=. and assume_grd0=1 then grade=0;
         %end;

         mytimepoint=&timepoint.;

      run;
   %end;

   %if &datatype.=2 %then %do;
      proc sql;
         create table _wmxtox_&y.  as
         select l.dcntr_id, l.&timepoint., l.qol, l.scores, r.&by.
         from qol0 (where=(upcase(qol)=%upcase("&&&txvar&y."))) l left join crse0 r
         on l.dcntr_id = r.dcntr_id
         having l.scores ^=.;
      quit;

      **** Need this extra step in order to create the mytimepoint variable without carried over the data format;
      data _wmxtox_&y.;
         set _wmxtox_&y.;
         mytimepoint=&timepoint.;
      run;
   %end;

   **** Create analysis for each By-group;
   %do z=1 %to &numlevel.;
      data _wsmx1ae_&y._&z.;
         set _wmxtox_&y.;

         %if &bytype.=num %then %do;
            where &by. = &&&bylevel&z.;
         %end;

         %else %if &bytype.=char %then %do;
            where &by. = "&&&bylevel&z.";
         %end;

      run;

      proc sort data=_wsmx1ae_&y._&z.;
         by dcntr_id &by.;
      run;

      **** Transpose the dataset for further sorting purpose;
      proc transpose data=_wsmx1ae_&y._&z. out=_wf_smx1ae_&y._&z.;
         by dcntr_id &by.;
         id mytimepoint;
         var &scrgrd.;
      run;

      **** Sort the dataset by grade reported at each timepoint;
      **** To group together the patients with similiar trend of AE grade/QOL Scores across timepoints;
      proc sort data=_wf_smx1ae_&y._&z.;
         by %do i=1 %to &numtmptlevel.;
             _%trim(&&tmptlevel&i.)
            %end;
         ;
      run;

      **** Assign patient id which indicated the order to display the grade/scores;
      data _wf_smx1ae_&y._&z.;
         set _wf_smx1ae_&y._&z.;
         myptid=_n_;

         label myptid='Patient';
      run;

      **** FINAL dataset to generate the INDIVIDUAL AE/QOLAE event chart;
      proc sql;
         create table f_mx1ae_&y._&z. as
         select l.*, r.myptid
         from _wsmx1ae_&y._&z. l left join _wf_smx1ae_&y._&z. r
         on l.dcntr_id = r.dcntr_id
         order by r.myptid;
      quit;

      data _null_;
         set f_mx1ae_&y._&z. end=eof;
         by myptid;

         if eof then call symput ('totpt_' || trim(left(&y.)) || '_' || trim(left(&z.)) , myptid);
      run;

      /**** new codes for QOL AE change from Baseline Scores event chart ****/
      %if &datatype.=2 %then %do;
         proc sql;
            create table _wschgmx1ae_&y._&z. as
            select l.*, l.scores - r.scores as chgscores label='Change from Baseline'
            from _wsmx1ae_&y._&z.  l left join _wsmx1ae_&y._&z. (where=(&timepoint.=&tmpt1level.)) r
            on l.dcntr_id = r.dcntr_id;
         quit;

         **** Need this extra step in order to create the mytimepoint variable without carried over the data format;
         **** remove the baseline data so, we will plot the first timepoint beyond baseline;
         data _wschgmx1ae_&y._&z.;
            set _wschgmx1ae_&y._&z.;
            mytimepoint=&timepoint.;
            if &timepoint.=&tmpt1level. then delete;
         run;

         proc sort data=_wschgmx1ae_&y._&z.;
            by dcntr_id &by.;
         run;

         **** Transpose the dataset for further sorting purpose;
         proc transpose data=_wschgmx1ae_&y._&z. out=_wf_schgmx1ae_&y._&z.;
            by dcntr_id &by.;
            id mytimepoint;
            var chgscores;
         run;

         **** Sort the dataset by grade reported at each timepoint;
         **** To group together the patients with similiar trend of QOL Scores across timepoints;
         proc sort data=_wf_schgmx1ae_&y._&z.;
            by %do i=2 %to &numtmptlevel.;
                _%trim(&&tmptlevel&i.)
               %end;
            ;
         run;

         **** Assign patient id which indicated the order to display the grade/scores;
         data _wf_schgmx1ae_&y._&z.;
            set _wf_schgmx1ae_&y._&z.;
            myptid=_n_;

            label myptid='Patient';
         run;

         **** FINAL dataset to generate the INDIVIDUAL QOLAE Change from Baseline Scores Event Chart;
         proc sql;
            create table f_mx1aechg_&y._&z. as
            select l.*, r.myptid
            from _wschgmx1ae_&y._&z. l left join _wf_schgmx1ae_&y._&z. r
            on l.dcntr_id = r.dcntr_id
            order by r.myptid;
         quit;

         data _null_;
            set f_mx1aechg_&y._&z. end=eof;
            by myptid;

            if eof then call symput ('chgtotpt_' || trim(left(&y.)) || '_' || trim(left(&z.)) , myptid);
         run;
      %end;
   /*** Change from Baseline Scores Event Chart End ****/

   %end;

%end;

**=============================================================**;
**=============================================================**;
**=============================================================**;
**                  CREATE HEAT MAP                            **;
*==============================================================**;
**=============================================================**;
**=============================================================**;
/*
** The templates codes can be moved outside the %do-loop and only keep the proc sgrender codes;
** within %do-loop if we decide to let the procedure itself to decide what would be the best;
** appropriate y-axis ranges to plot the graph;
** We will round up the max range of y-axis - per stat;
*/

option nodate;
ods graphics / outputfmt=png;
ods rtf file="heatmap_&dtype..doc" bodytitle;

/***** Create heat map of each arm for each specify AE ***/
%do y=1 %to &numvar.;
   %do z=1 %to &numlevel.;

      **** Use this color gwh, if we want to differentiate missing and grade 0 for CTCAE;
      proc template;
         define statgraph heatmap;
         begingraph;

            rangeattrmap name="rmap";
               range missing / rangecolor=white;
               %if &datatype.=1 %then %do;
                  range 0 - < 1  / rangecolor=pkgr;
                  range 1 - < 2  / rangecolor=ligr;
                  range 2 - < 3  / rangecolor=megr;
                  range 3 - < 4  / rangecolor=dagr;
                  range 4 - < 5  / rangecolor=dagray;
                  range 5 - 5    / rangecolor=black;
               %end;
               %else %do;
                  range 0  -  50    / rangecolor=black;
                  range 50 < - 75   / rangecolor=dagr;
                  range 75 < - 100  / rangecolor=ligr;
               %end;
            endrangeattrmap;

            rangeattrvar attrmap="rmap" var=&scrgrd. attrvar=pColor;

            layout overlay  / yaxisopts=(label="Individual Patient"  offsetmin=0.1 offsetmax=0.1 type=linear
                              linearopts=(viewmin=0 viewmax=&&&&&totpt_&y._&z. thresholdmin=0 thresholdmax=1 minorticks=true)
                              display=(label ticks tickvalues line) tickvalueattrs=(size=8))

                              xaxisopts=(label="&tmptlbl." offsetmin=0.1 offsetmax=0.1
                              display=(label ticks tickvalues line) tickvalueattrs=(size=8)
                              discreteopts=(tickvaluefitpolicy=stagger));

               heatmapparm x=&timepoint. y=myptid colorresponse=pColor /  name="mheatmap";
               continuouslegend "mheatmap" / valueattrs=(size=8) ;

            endlayout;
         endgraph;
         end;
      run;

      title1 "Figure 1a_&y._&z.: %trim(&&&bylvllbl&z.) %trim (&&&utxlbl&y.) Heat Map";
      title2 "for %trim(&dfigtbtlt.) (N=%trim(&&&&&totpt_&y._&z.))";
      proc sgrender data=f_mx1ae_&y._&z. template=heatmap;
      run;
      title2;

      ods rtf text="Each horizontal line indicates an individual patient%bquote(')s experience.";
      ods rtf text="No color (i.e. white) indicates missing data.";
      ods rtf text="A darker saturation of color indicates worse %trim(&dfnote.).";

      %if &datatype.=2 %then %do;
         **** Use this color gwh, if we want to differentiate missing and grade 0 for CTCAE;
         proc template;
            define statgraph heatmap;
            begingraph;

               rangeattrmap name="rmap";
                  range missing / rangecolor=white;
                  range -100 - < -20 / rangecolor=black;
                  range -20 -  < 0   / rangecolor=dagray;
                  range 0 - 0        / rangecolor=megr;
                  range 0 -    < 20  / rangecolor=pkgr;
                  range 20 -   < 100 / rangecolor=ligr;
               endrangeattrmap;

               rangeattrvar attrmap="rmap" var=chgscores attrvar=pColor;

               layout overlay  / yaxisopts=(label="Individual Patient"  offsetmin=0.1 offsetmax=0.1 type=linear
                                 linearopts=(viewmin=0 viewmax=&&&&&totpt_&y._&z. thresholdmin=0 thresholdmax=1 minorticks=true)
                                 display=(label ticks tickvalues line) tickvalueattrs=(size=8))

                                 xaxisopts=(label="&tmptlbl." offsetmin=0.1 offsetmax=0.1
                                 display=(label ticks tickvalues line) tickvalueattrs=(size=8)
                                 discreteopts=(tickvaluefitpolicy=stagger));

               heatmapparm x=&timepoint. y=myptid colorresponse=pColor /  name="mheatmap";
               continuouslegend "mheatmap" / valueattrs=(size=8) ;

               endlayout;
            endgraph;
            end;
         run;

         title1 "Figure 1b_&y._&z.: %trim(&&&bylvllbl&z.) %trim (&&&utxlbl&y.) Heat Map of Change from Baseline Scores";
         title2 "for %trim(&dfigtbtlt.) (N=%trim(&&&&&chgtotpt_&y._&z.))";
         proc sgrender data=f_mx1aechg_&y._&z. template=heatmap;
         run;
         title2;

         ods rtf text="A 20% (or more) increase or decrease in score from baseline is clinically significant (black or lightest gray).";
         ods rtf text="No change from baseline is indicated in mid-level gray.";
         ods rtf text="No color (i.e. white) indicates missing data.";
      %end;
   %end;
%end;

%if &datatype.=1 %then %do;
   /**** Create heat map of each arm on overall AE ***/
   %do y=1 %to &numlevel.;
      **** Use this color gwh, if we want to differentiate missing and grade 0 for CTCAE;
      proc template;
         define statgraph heatmap;
         begingraph;

            rangeattrmap name="rmap";
               range missing / rangecolor=white;
               range 0 - < 1  / rangecolor=pkgr;
               range 1 - < 2  / rangecolor=ligr;
               range 2 - < 3  / rangecolor=megr;
               range 3 - < 4  / rangecolor=dagr;
               range 4 - < 5  / rangecolor=dagray;
               range 5 - 5    / rangecolor=black;
            endrangeattrmap;

            rangeattrvar attrmap="rmap" var=&scrgrd. attrvar=pColor;

            layout overlay  / yaxisopts=(label="Individual Patient"  offsetmin=0.1 offsetmax=0.1 type=linear
                              linearopts=(viewmin=0 viewmax=&&&totpt_&y. thresholdmin=0 thresholdmax=1 minorticks=true)
                              display=(label ticks tickvalues line) tickvalueattrs=(size=8))

                              xaxisopts=(label="&tmptlbl." offsetmin=0.1 offsetmax=0.1
                              display=(label ticks tickvalues line) tickvalueattrs=(size=8)
                              discreteopts=(tickvaluefitpolicy=stagger));

               heatmapparm x=&timepoint. y=myptid colorresponse=pColor /  name="mheatmap";
               continuouslegend "mheatmap" / valueattrs=(size=8) ;

            endlayout;
         endgraph;
         end;
      run;

      title1 "Figure 2_&y.: %trim(&&&bylvllbl&y.) Overall Heat Map";
      title2 "for CTCAE Grade (N=%trim(&&&totpt_&y.))";
      proc sgrender data=f_mxallae_&y. template=heatmap;
      run;
      title2;

      ods rtf text="Each horizontal line indicates an individual patient%bquote(')s experience.";
      ods rtf text="No color (i.e. white) indicates missing data.";
      ods rtf text="A darker saturation of color indicates worse %trim(&dfnote.).";

   %end;
%end;


ods graphics off;
ods rtf close;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                        REMOVE WORKING DATA SET                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
proc datasets library=work nolist;
   delete _w:;
quit;

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 EXIT THE PROGRAM WHEN ERROR OCCUR                  *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
%exit:

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                       DISPLAY THE MACRO STATUS                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
data _null_;
   put "---------------------------";
   %if &myErrMsg. ^= %then %do;
      put "ERROR: " &myErrMsg.;
   %end;
   put "---------------------------";
run;

%mend heatmap;
