
/*------------------------------------------------------------------*
 | MACRO NAME: ae_over_time
 | SHORT DESC: Perform longitudinal analysis that involved repeated
 |             measures and profile analysis on individual/overall
 |             adverse events between treatment groups over time
 *------------------------------------------------------------------*
 | CREATED BY: ToxT Team                                (07/20/2016)
 |
 | Version 2 (5/9/2017): Added new macro parameter to allow user
 |                       to select the type of convariance structure
 |                       for random effect to run the proc mixed model
 |                       Completed - 1/7/2019
 *------------------------------------------------------------------*
 | PURPOSE
 |
 | This program performs longitudinal analysis that involved
 | repeated measures and profile analysis on individual or overall adverse
 | events per NCI Common Terminology Criteria for Adverse Events (CTCAE) OR
 | individual symptoms/QOL items reported per Patient Report Outcomes (PRO)
 | for each treatment groups across data collection time points. This macro
 | also utilized the profile macro written by Paul Novotny to perform
 | the profile analysis.
 |
 | The program will produce two documents in the directory where SAS is run:
 |
 | For CTCAE:
 | 1. ae_over_time_figures_ae.doc: Line graph of Mean Grade with 95% CI for individual
 |                                 or overall adverse event by treatment group
 |                                 over time
 | 2. ae_over_time_tables_ae.doc : 1) Table Summary of Repeated Measures Analysis on
 |                                    individual/overall adverse event by treatment
 |                                    group over time
 |                                 2) Table Summary of Profile Analysis on
 |                                    individual/overall adverse event by
 |                                    treatment group over time
 | NOTES: Overall AE is summarized across all AEs that are included
 |        in the CYTOX dataset. If user would like to see the Overall AE
 |        across a subset of AEs that are specified in TOXICITY_LIST
 |        parameter, the CYTOX dataset need to be modified to include
 |        only the subset AEs that are of user interest
 |
 | For PRO Symptoms/QOL items:
 | 1. ae_over_time_figures_qol.doc: Line graph of Mean Grade with 95% CI for individual
 |                                  symptom/QOL items by treatment group over time
 | 2. ae_over_time_tables_qol.doc : 1) Table Summary of Repeated Measures Analysis on
 |                                     individual symptom/QOL items by treatment
 |                                     group over time
 |                                  2) Table Summary of Profile Analysis on
 |                                     individual symptom/QOL items by
 |                                     treatment group over time
 |
 | ~ IMPORTANT NOTES: ~
 | 1. The line graph will be produced in COLOR version ONLY
 | 2. Datasets that are used to produce the line graphs
 |    can be found in WORK directory with prefix of f_ in dataset name
 |    where user can further modify or tweak the graphing per user preference
 | 3. If by-group consist of 4 levels and the N count below the graph only shows
 |    the information for two levels means the label for by-group is too long,
 |    and it needs to be shortned to have the N counts displayed correctly for
 |    all groups
 |
 *------------------------------------------------------------------*
 | REQUIRED DATASETS and DATA FIELDS
 |
 | 1. crse dataset with one observation per patient with the following data fields:
 |    dcntr_id          : patient identifier
 |    evalae            : evaluable for toxicity (numeric)
 |                        (1= at least one AE form is entered other than
 |                            the baseline timepoint
 |                         0= otherwise)
 |    [comparison group]: group to be used for data comparison
 |                        which will be specified in BY macro parameter
 |                        (i.e. arm)
 |
 | 2. prodata dataset with one observation per time point per patient with the
 |    following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for treatment (numeric) which will be specified
 |                        in TIMEPOINT macro parameter (i.e. cycle)
 |    Notes: The dataset should include all treatment time point that are
 |            of user interest for the analysis
 |
 | 3. cytox dataset with one observation per maximum grade per adverse event per
 |    timepoint per patient with the following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for adverse events reporting (numeric) which
 |                        will be specified in TIMEPOINT macro parameter (i.e. cycle)
 |    toxicity          : Adverse event as in medra codes (numeric and should be formatted
 |                        to give adverse event description) (i.e. 900182)
 |    grade             : Severity of the adverse event according to CTC guidelines.
 |                        (numeric) (i.e. 0, 1, 2, 3, 4, 5)
 |    Notes: The dataset should include all adverse event reported up to the last time point
 |           indicated in protdata dataset of user interest for the analysis
 |
 | 4. qol dataset with one observation per symptom per timepoint per patient
 |    with the following data fields (REQUIRED for PRO Symptoms analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for symptoms reporting (numeric) which
 |                        will be specified in TIMEPOINT macro parameter (i.e. cycle)
 |    qol               : SAS variable name for each symptom (character)
 |                        (i.e. p_q10, poms04)
 |    _label_           : Description/Label for each symptom (character)
 |                        (i.e. Overall QOL, Sleep Interference, Nervousness)
 |    scores            : Symptoms/QOL items scores in a scale of 0 to 100 (numeric)
 |                        with 0=Low Quality of Life and 100=Best Quality of Life
 |                        Notes: for symptoms with 0=no symptom and 100=worst symptom
 |                               the scores need to be reversed before using this program
 |    Notes: The dataset should include all symptoms/qol items reported up to the
 |           last time point of user interest for the analysis
 *------------------------------------------------------------------*
 | REQUIRED PARAMETERS
 |
 | datasource   : The location where analysis datasets are saved
 |                1 = Internal CCS database
 |                2 = User provided datasets in WORK directory (DEFAULT)
 |
 | datatype     : Type of adverse events/symptoms reported
 |                1 = Adverse Events per CTCAE (DEFAULT)
 |                2 = Symptoms/QOL items per PRO
 |
 | by           : Group Comparison (i.e. arm)
 |
 | timepoint    : Time variable (numeric)
 |
 | toxicity_list: List of toxicity codes separated by space for CTCAE
 |                (i.e. 800001 832222 1002222)
 |                OR List of SAS variable names separated by space for PRO
 |                symptoms/QOL items (i.e. lasa01 poms03 lasa09 lccs4)
 |                Notes: This allows up to 8 AEs/Symptoms/QOL items

 | OPTIONAL PARAMETERS
 |
 | studynum     : study reference number (REQUIRED if datasource=1) (i.e. MCXXXX)
 |
 | incmiss      : AE grade assumption when specified AE entry is not available (for CTCAE ONLY)
 |                y = assumed grade 0 if there is no entry for user specified AE in the data set,
 |                    with condition that data entry is available for other AE at specific
 |                    timepoint (DEFAULT)
 |                n = assumed missing or do nothing if there is no entry for user specified
 |                    AE in the dataset
 | covrstr      : Specify the covariance structure for the random effect
 |                un    = Unstructured (DEFAULT)
 |                cs    = Compound symmetry
 |                ar(1) = Autoregressive(1)
 |
 *------------------------------------------------------------------*
 | SAMPLE CODES that can be used to prepare the QOL dataset
 |
 | For example:
 | QOL_tmp dataset contains one observation per timepoint with data fields include:
 | dcntr_id, visit, p_q01, p_q07, p_q09, p_q11
 |
 | proc sort data=qol_tmp;
 |    by dcntr_id visit;
 | run;
 |
 | proc transpose=qol_tmp out=qol (rename=(col1=scores));
 |    by dcntr_id visit;
 |    var p_q01 p_q07 p_q09 p_q11;
 | run;
 |
 | data qol;
 |    set qol;
 |    rename _name_=qol;
 |    label _name_='QOL';
 | run;
 |
 *------------------------------------------------------------------*
 | EXAMPLE:
 |
 | For CTCAE:
 |    %ae_over_time (by=arm, timepoint=cycle,
 |                    toxicity_list=8000001 900182 900146 21162 21197 21521);
 | For PRO Symptoms/QOL items:
 |    %ae_over_time (by=arm, datatype=2, timepoint=visit,
 |                    toxicity_list=p_q08 p_q09 p_q10 p_q11 p_q01 p_q06);
 *------------------------------------------------------------------*
 */

%macro ae_over_time (datasource=2, studynum=, datatype=1, by=, timepoint=, toxicity_list=, incmiss=y, covrstr=un);

*** Per project MS stat request, will include Profile analysis macro within this code;
*** So, user do not need to explicitly save out this macro somewhere else;
/*%include "profile_ed.macro";*/

%macro profile(dsn,class,var,min,max,short,debug,myout=);
/****************************************************************************************/
/* This macro does a profile analysis of two curves                                     */
/*                                                                                      */
/* Written by: Paul Novotny    5/2000                                                   */
/* Updated to ODS: Paul Novotny 2/2016                                                  */
/*                                                                                      */
/* Input Parameters:                                                                    */
/* -----------------                                                                    */
/*      dsn = dataset name (default=_last_)                                             */
/*      class = the class variable for analysis (required, can be numeric or character) */
/*      var = the array name for the analysis (required, must be numeric)               */
/*            (e.g. if the variable are in x1,x2,x3,x4,x5  then enter x)                */
/*      min = the lowest array index (default=1)                                        */
/*      max = the highest array index (required)                                        */
/*      short = 'y' to print only a summary of the results,                             */
/*              'n' to print the detailed GLM output (default=y)                        */
/*      debug = 'y' to print notes, source, and macro output to help                    */
/*              in debugging problems (default=none)                                    */
/* Examples of execution:                                                               */
/*     %profile(master,arm,ps,0,5);                                                     */
/****************************************************************************************/

/*******************************************/
/* sets the default parameters and options */
/*******************************************/
%local dsn class var min max debug;

%if (&dsn=)    %then %let dsn=_last_;
%if (&debug=Y) %then %let debug=y;
%if (&min=)    %then %let min=1;
%if (short=Y)  %then %let short=y;
%if (short=)   %then %let short=y;

%if (&debug=y) %then %do; options mprint mtrace macrogen mlogic; %end;
               %else %do; options nonotes nomprint nosource;     %end;


/******************************************************/
/* creates the analysis file for the profile analysis */
/******************************************************/
data _tmp_;
  set &dsn;
  array _score  (&min:&max)  %do i = &min %to &max;  _%trim(&&&var&i)  %end; ;
  array _diff   (&min:&max)  %do i = &min %to &max;  _diff&i  %end; ;

  b=1; /* dummy variable */

  do i = &min to (&max-1);
     _diff[i]=_score[i+1]-_score[i];  /* successive differences */
  end;
  _z=mean(  %do i = &min %to %eval(&max-1);  _%trim(&&&var&i), %end;
   _%trim(&&&var&max));      /* overall mean */
run;

proc sort data=_tmp_;
  by &class;
run;


/**************************/
/* distributions by class */
/**************************/
proc means data=&dsn;
  class &class;
  var %do i = &min %to &max;  _%trim(&&&var&i)  %end; ;
  title2 "Means by &class";
  run;


/********************************/
/* testing of levels hypothesis */
/********************************/
ods output OverallANOVA=overall;
ods output ModelANOVA=model;
ods output LSMeans=mult1;

proc glm data=_tmp_;
  class &class;
  model _z=&class;
  lsmeans &class / stderr /* pdiff */;
  title2 'PROFILE ANALYSIS - test of levels hypothesis';
  run;
  quit;

ods select all;
ods output close;

data mult1;
  set mult1;
  mrg=1;

data model (keep=mrg ProbtDiff);
  set model;
  if (HypothesisType='3');
  mrg=1;
  ProbtDiff=ProbF;

data mult1 (keep=&class LSMean Probt ProbtDiff top);
  merge mult1 model;
  by mrg;
  if (_n_=1) then top=1;


/*****************************************************/
/* MANOVA test for equal mean vectors between groups */
/* via Hotelling's T-squared statistic               */
/*****************************************************/

ods select MultStat(persist);
ods output MultStat(match_all persist=proc)=mult2;

proc glm data=_tmp_;
  class &class;
  model  %do i = &min %to &max;  _%trim(&&&var&i) %end; = &class;
  manova h=&class / printh printe;
  means &class;
  title2 'PROFILE ANALYSIS - MANOVA test for equal mean vectors';
  run;
  quit;

ods select all;
ods output close;


data mult2 (keep=Statistic ProbF);
  set mult2;
  if (Statistic='Hotelling-Lawley Trace');


/*******************************/
/* test of flatness hypothesis */
/*******************************/
ods select MultStat(persist);
ods output MultStat(match_all persist=proc)=mult3;

proc glm data=_tmp_;
  class b;
  model %do i = &min %to %eval(&max-1); _diff&i %end; = b / noint;
  manova h=b;
  title2 'PROFILE ANALYSIS - test of flatness hypothesis';
  run;
  quit;

ods select all;
ods output close;

data mult3 (keep=Statistic ProbF);
  set mult3;
  if (Statistic='Hotelling-Lawley Trace');


/************************************************************/
/* test parallelism hypothesis (group*variable interaction) */
/************************************************************/
ods select MultStat(persist);
ods output MultStat(match_all persist=proc)=mult4;

proc glm data=_tmp_;
  class &class;
  model %do i = &min %to %eval(&max-1);  _diff&i %end; = &class;
  manova h=&class;
  title2 'PROFILE ANALYSIS - MANOVA test of parallelism hypothesis';
  run;
  quit;

ods select all;
ods output close;

data mult4 (keep=Statistic ProbF);
  set mult4;
  if (Statistic='Hotelling-Lawley Trace');


/******************************************/
/* combines the output to print a summary */
/******************************************/
data &myout.;
  set mult1 (in=a) mult2 (in=b) mult3 (in=c) mult4 (in=d);
  length group $ 50;
  if a then group='1. Equal Levels';
  if b then group='2. Equal Means';
  if c then group='3. Flat Lines';
  if d then group='4. Parallel Lines';

proc sort data=&myout.;
  by group &class;
run;

%mend profile;

*---------------------------------------------------------------------------**;
******************************************************************************;
*****             CODES PERTINENT TO AE_OVER_TIME MACRO                  *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

**********************************;
**** Variables Initializations ***;
**********************************;
%let studynum = %upcase(&studynum.);
%if (covrstr=) %then %let covrstr=un;

%local myErrMsg dtype dname scrgrd aevar char_tmpt;

%if &datatype.=1 %then %do;
   %let dtype=ae;
   %let dname=cytox;
   %let scrgrd=Grade;
   %let aevar=toxicity;
%end;

%if &datatype.=2 %then %do;
   %let dtype=qol;
   %let dname=qol;
   %let scrgrd=Scores;
   %let aevar=qol;
%end;

**================**;
**  Data Format   **;
**================**;

proc format;
   value yyesnof 1='Yes'
                 2='No';
run;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 PROMPT ERROR MESSAGE WHEN NECESSARY                *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

***************************************************************;
**** Display error when data source is not found or invalid ***;
***************************************************************;

%if &datasource. = %then %do;
   %let myErrMsg = "Data source is not found.";
   %goto exit;
%end;

%else %if &datasource. ^=1 and &datasource. ^=2 %then %do;
   %let myErrMsg = "Data source is invalid.";
   %goto exit;
%end;


********************************************************************;
**** Display error when data source is 1 and studynum is missing ***;
********************************************************************;

%if &datasource. = 1 and &studynum. = %then %do;
   %let myErrMsg = "Data Source of Cancer Center database was indicated, yet, study number is not provided.";
   %goto exit;
%end;

***********************************************************;
**** Display error when data type is missing or invalid ***;
***********************************************************;

%if &datatype. = %then %do;
   %let myErrMsg = "Datatype is not found.";
   %goto exit;
%end;

%else %if &datatype. ^=1 and &datatype. ^=2 %then %do;
   %let myErrMsg = "Datatype is invalid.";
   %goto exit;
%end;

*****************************************;
**** Display error when by is missing ***;
*****************************************;

%if &by. = %then %do;
   %let myErrMsg = "By group is not found.";
   %goto exit;
%end;

************************************************;
**** Display error when timepoint is missing ***;
************************************************;

%if &timepoint. = %then %do;
   %let myErrMsg = "Timepoint is not found.";
   %goto exit;
%end;

*********************************************************;
**** Display error when toxicity_list is missing     ****;
*********************************************************;
%if &toxicity_list. = %then %do;
   %let myErrMsg = "No variable found in parameter TOXICITY_LIST.";
   %goto exit;
%end;

%let cntvar=1;

**** Get the number of toxicities specify by user;
%do %while (%scan(&toxicity_list., &cntvar.) ^= );
   %let cntvar   = %eval(&cntvar. + 1);
%end;

%let cntvar = %eval(&cntvar. - 1);

**=======================**;
**  Required Datasets    **;
**=======================**;

**** if using cancer center database;
%if &datasource. = 1 %then %do;
   %crtlibn (d=&studynum.);

   proc copy in=pdata out=work;
   run;

%end;

*******************************************************;
**** Display error when CRSE dataset does not exist ***;
*******************************************************;

%if %sysfunc(exist(crse)) = 0 %then %do;
   %let myErrMsg = "crse dataset does not exist.";
   %goto exit;
%end;

************************************************************;
**** Display error when CYTOX/QOL dataset does not exist ***;
************************************************************;

%if %sysfunc(exist(&dname)) = 0 %then %do;
   %let myErrMsg = "&dname. dataset does not exist.";
   %goto exit;
%end;


************************************************************;
**** Display error when PROTDATA dataset does not exist ***;
************************************************************;

%if %upcase (&dname.)=CYTOX %then %do;
   %if %sysfunc(exist(protdata)) = 0 %then %do;
      %let myErrMsg = "protdata dataset does not exist.";
      %goto exit;
   %end;
%end;

**==========================**;
**  Other Required Datasets **;
**==========================**;

**** CRSE dataset;
proc sql;
   create table crse0 as
   select dcntr_id, &by., evalae
   from work.crse
   order by dcntr_id;
quit;

**** Determine the variable type, format and label for CRSE dataset;
proc sql;
  create table _w1 as
  select *
  from sashelp.vcolumn
  where upcase(libname)="WORK"
  and upcase(memname)="CRSE0";
quit;

**** Retrieve the type, format and label for By variable;
data _null_;
   set _w1 end=eof;

   if (label=' ') then label=name;

   if (upcase(name)=%upcase("&by")) then do;
      if (format=' ') and (type='char') then format=compress('$char' || trim(left(length)) || '.');
      if (format=' ') and (type='num') then format='8.';

      call symput('byf', trim(left(format)));
      call symput('bylbl', trim(left(label)));
      call symput('bytype', trim(left(type)));
   end;

run;

**** Determine the number of level used in the By parameter;
proc sql;
   create table _wchkbylvl as
   select distinct &by.
   from crse0;
quit;

data _null_;
   set _wchkbylvl end=eof;
   call symput ('bylevel' || trim (left(_n_)), &by.);
   call symput ('bylvllbl' || trim (left(_n_)), put (&by., &byf.));

   if eof then call symput ('numlevel', trim(left(_n_)));
run;

*********************************************************************************;
**** Verify the toxicity list provided by users to make sure they are unique ****;
*********************************************************************************;

%if &datatype.=1 %then %do;
   proc sort data=work.cytox;
      by dcntr_id;
   run;

   proc sort data=work.protdata;
      by dcntr_id;
   run;

   **** Make sure the cytox dataset provided are for the intended population;
   data _wcytox_tmp;
      merge crse0 (in=in1 keep=dcntr_id)
            cytox (in=in2 keep=dcntr_id &timepoint. toxicity grade);
      by dcntr_id;
      if in1 and in2;
   run;

   **** Make sure the protdata dataset provided are for the intended population;
   data _wprotdata_tmp;
      merge crse0 (in=in1 keep=dcntr_id)
            protdata (in=in2 keep=dcntr_id &timepoint.);
      by dcntr_id;
      if in1 and in2;
   run;

   **** All cycles needed to exist in order to run AUC macro, data can be missing at that cycle;
   **** Even though this does not apply to this macro, will still keep it the same as;
   **** what was written in the other macros for consistency;
   proc sql;
      create table _wchktimelvl0 as
      select distinct &timepoint.
      from _wcytox_tmp
      order by &timepoint.;
   quit;

   **** Determine the unique treatment cycles and if patient is evaluable for AE;
   proc sql;
      create table _wpttime as
      select l.*, r.&timepoint.
      from crse0 l left join _wchktimelvl0 r
      on l.dcntr_id
      where l.evalae=1
      order by l.dcntr_id, r.&timepoint.;
   quit;

   proc sql;
      create table _wpttime_tmp as
      select distinct dcntr_id, max (&timepoint.) as mxcyc_prot
      from _wprotdata_tmp
      group by dcntr_id
      order by dcntr_id;
   quit;

   proc sql;
      create table protdata0 as
      select l.*, r.mxcyc_prot
      from _wpttime l left join _wpttime_tmp r
      on l.dcntr_id = r.dcntr_id
      having l.&timepoint. <= r.mxcyc_prot
      order by l.dcntr_id, l.&timepoint.;
   quit;

   proc sort data= _wcytox_tmp;
      by dcntr_id &timepoint.;
   run;

   **** Assume Grade 0 when certain toxicity is not reported, but, AE entries is available at that cycle;
   proc sql;
      create table checkAEgrade as
      select distinct l.dcntr_id, l.&timepoint., 1 as assume_grd0, 0 as grade, r.&by., r.mxcyc_prot
      from _wcytox_tmp l left join protdata0 r
      on l.dcntr_id = r.dcntr_id and l.&timepoint. = r.&timepoint.
      having l.&timepoint. <= r.mxcyc_prot
      order l.dcntr_id, l.&timepoint.;
   quit;

   *** Include only patients who indicated as evaluable for AE and with treatment cycles available;
   data _wcytox0;
      merge protdata0    (in=in1)
            _wcytox_tmp  (in=in2);
      by dcntr_id &timepoint.;
      if in1;
      evalpt=1;
   run;

   **** CYTOX dataset - Overall AE analysis;
   **** Grade 0 will be pre-filled at each cycle if at least one AE was reported at that cycle;
   **** and was indicated by user to do so;
   data cytox0;
      set _wcytox0

      %if %upcase (&incmiss.)=Y %then %do;
         checkAEgrade (drop=assume_grd0 mxcyc_prot)
      %end;
      ;
   run;

%end;

%if &datatype.=2 %then %do;
   proc sql;
      create table qol_items as
      select distinct qol, _label_
      from work.qol
   quit;

   data _null_;
      set qol_items end=eof;

      call symput ('qlist_'  || trim (left(_n_)), qol);
      call symput ('qlabel_' || trim (left(_n_)), _label_);

      if eof then call symput ('n_allqol', trim (left(_n_)));
   run;

   **** Create the format for the QOL items used;
   proc format;
      value $ qolf
         %do i=1 %to &n_allqol.;
            "&&&qlist_&i." = "&&&qlabel_&i."
         %end;
      ;
   run;

   **** If there is no entries of QOL whatsoever, that QOL will be set to missing;
   **** QOL dataset - Base dataset;
   proc sql;
      create table qol0 as
      select dcntr_id, &timepoint., qol format=$qolf., _label_, scores
      from work.qol
      order by dcntr_id;
   quit;
%end;

proc sql;
   create table _c2 as
   select *
   from sashelp.vcolumn
   where libname="WORK"
   and memname=%upcase("&dname.0");
quit;

**** Retrieve the format and label for Toxicity/QOL AE and timepoints;
data _null_;
   set _c2;
   if (label=' ') then label=name;
   if (upcase(name)=%upcase("&aevar.")) then do;
      call symput('txf',trim(left(format)));
      call symput('txlbl',trim(left(label)));
   end;

   if (upcase(name)=%upcase ("&timepoint.")) then do;
      if type='char' then do;
         call symput ('char_tmpt', 'y');
      end;

      if type='num'  and format=' ' then format=compress(trim(left(length)) || '.');

      call symput('tmptf',trim(left(format)));
      call symput('tmptlbl',trim(left(label)));
   end;
run;


************************************************************;
**** Display error when Timepoint is Characters field    ***;
************************************************************;

%if %upcase (&char_tmpt.)=Y %then %do;
   %let myErrMsg = "TIMEPOINT required a numeric field";
   %goto exit;
%end;

**** Declare each toxicity as macro variable;
data _wtxlist0;
   %do i=1 %to &cntvar.;
      %if &datatype.=1 %then %do;
         toxvar&i.=%scan (&toxicity_list., &i.);
         toxlbl&i.=put (toxvar&i., &txf.);
      %end;

      %if &datatype.=2 %then %do;
         length toxvar&i toxlbl&i. $ 255.;
         format toxlbl&i. $qolf.;

         toxvar&i.="%scan (&toxicity_list., &i.)";
         toxlbl&i.= toxvar&i.;
      %end;
   %end;
run;


**** Remove any duplicates toxicities codes;
proc transpose data=_wtxlist0 out=_wf_txlist0;
   var %do i=1 %to &cntvar.;
       toxvar&i.
      %end;
      ;
run;

proc sort data=_wf_txlist0 nodupkeys;
   by col1;
run;

**** To retain the order of AE as it was specified by user;
proc sort data=_wf_txlist0;
   by _name_;
run;


**** Change the obs number if we want to accept more toxicity variables for analysis;
data _null_;
   set _wf_txlist0 (firstobs=1 obs=8) end=eof;
   call symput ('txvar' || trim(left(_n_)),  trim(left(col1)));
   call symput ('utxlbl' || trim (left(_n_)), put (col1, &txf.));
   if eof then call symput ('numvar', trim(left( _n_)));
run;

**** Determine the number of timepoints used in the timepoint parameter;
proc sql;
   create table _wchktimelvl as
   select distinct &timepoint.
   from &dname.0
   order by &timepoint.;
quit;

data _null_;
   length alltmpt1 alltmpt2 $255.;
   set _wchktimelvl end=eof;
   retain alltmpt1 alltmpt2;

   alltmpt1 = compbl (alltmpt1 || &timepoint.);
   alltmpt2 = compbl (alltmpt2 || compress ('_' || &timepoint.));

   if _n_=1 then call symput ('tmpt1level', trim(left(&timepoint.)));

   call symput ('tmptlevel' || trim (left(_n_)), &timepoint.);
   call symput ('tmptlvllbl' || trim (left(_n_)), put (&timepoint., &tmptf.));

   if eof then do;
      call symput ('numtmptlevel', trim(left(_n_)));
      call symput ('tmptlstlevel', trim(left(&timepoint.)));
      call symput ('alltmptlevel1', alltmpt1);
      call symput ('alltmptlevel2', alltmpt2);
   end;

run;

**** All timepoints are needed in order to run AUC macro;
proc sql;
   create table crse1 as
   select l.*, r.&timepoint.
   from crse0 l left join _wchktimelvl  r
   on l.dcntr_id;
quit;

**=============================================================**;
**                   DATA MANIPULATIONS                        **;
** *********This applicable for CTCAE datasets ONLY *************;
** Maximum grade across ALL Adverse Events at each timepoints  **;
**=============================================================**;

%if &datatype.=1 %then %do;

   proc sort data=cytox0 out=_wov_mxtox;
      by dcntr_id &timepoint. grade;
   run;

   data _wmxtox;
      set _wov_mxtox;
      by dcntr_id &timepoint. grade;
      if last.&timepoint.;

      mytimepoint=&timepoint.;
   run;

   **** one obs per pt will be used to run profile analysis;
   proc sort data=_wmxtox out=_ws_mxtox;
      by dcntr_id &by.;
   run;

   proc transpose data=_ws_mxtox out=_wf_mxtox;
      by dcntr_id &by.;
      id mytimepoint;
      var grade;
   run;

   **** Get the summary statistics of grade;
   proc means data=_wmxtox;
      var grade;
      class &timepoint. &by.;
      output out=_wmxtox_stat n=n median=median mean=mean stderr=stderr
             lclm(grade)=lcl
             uclm(grade)=ucl;
   run;

   **** Dataset to create line graphs;
   data f_aetime_ovgraph;
      length mycount $8.;
      set _wmxtox_stat;
      format mean 8.2;
      keep &timepoint. &by. n mean lcl ucl mycount;

      mycount = put (n, 8.);

      where &timepoint. ^=. and missing (&by.) =0;

      label mean='Mean CTCAE Grade'
            lcl='lower 95% CI'
            ucl='upper 95% CI';
   run;

   ***** Performed Repeated Measures Analysis - Overall AE Grade;
   ods output tests3    =_wtest3out   (keep=effect probf)
              solutionf =_wsfout      (keep=effect &by. estimate stderr probt)
              LSMeans   =_wlsmeansout (keep=effect &by. estimate stderr probt)
              ConvergenceStatus=_wconvout;
   proc mixed data=_wmxtox;
      class &by. dcntr_id;
      model grade = &by. &timepoint. &by.*&timepoint. / solution ddfm=KR;
      random &timepoint. / type=&covrstr. subject=dcntr_id (&by.);
      lsmeans &by.;
   run;
   ods output close;

   ***** To determine the convergence status on random effect model;
   data _wconvout;
      length mreason $256.;
      set _wconvout;

      mreason="The Newton-Raphson algorithm for random effect model fail to converge, please consider a different covariance structure.";

      label mreason="Repeated Measures Analysis Results Not Available";
   run;

   data _null_;
      set _wconvout;
      call symput ('ov_conv', trim (left (status)));
      call symput ('ov_conv_rea', trim(reason));
   run;

   %profile(dsn=_wf_mxtox,class=&by.,var=tmptlevel,min=1,max=&numtmptlevel.,
            myout=_wf_mxtox_out);
   run;
%end;

**=============================================================**;
**                   DATA MANIPULATIONS                        **;
**                (For Both CTCAE and QOLAE)                   **;
** Grade per adverse events or Scores per QOL AE at each timepoints **;
** If multiple grade was reported per AE at specific timepoints **;
** proc transpose will fail and users need to fix their data before re-run the program;
**=============================================================**;

%do y=1 %to &numvar.;

   **** To make sure all patients have an observation for each timepoint;
   **** and also an observation for each toxicity within each timepoint;
   %if &datatype.=1 %then %do;

      proc sql;
         create table _wcytox1 as
         select distinct l.dcntr_id, l.&timepoint., l.&by., l.evalae, l.evalpt, l.mxcyc_prot, r.assume_grd0
         from cytox0 (where=(grade^=0 and toxicity^=. and evalae^=.)) l left join checkAEgrade r
         on l.dcntr_id = r.dcntr_id;
      quit;

      proc sql;
         create table _ws_mxtox_&y. as
         select l.*, r.toxicity, r.grade
         from _wcytox1 l left join cytox0 (where=(toxicity=&&&txvar&y.)) r
         on l.dcntr_id    = r.dcntr_id and
            l.&timepoint. = r.&timepoint.;
      quit;

      **** If AE is not reported at the time point, assumed patient do not have that AE, will code as 0;
      data _wmxtox_&y.;
         set _ws_mxtox_&y.;

         %if %upcase (&incmiss.)=Y %then %do;
            if grade=. and assume_grd0=1 then grade=0;
         %end;

         mytimepoint=&timepoint.;

      run;

   %end;

   %if &datatype.=2 %then %do;

      **** To make sure all patients have an observation for each timepoint;
      proc sql;
         create table _wmxtox_&y.  as
         select l.*, r.qol, r.scores
         from crse1 l left join qol0 (where=(upcase(qol)=%upcase("&&&txvar&y."))) r
         on l.dcntr_id = r.dcntr_id and
            l.&timepoint. = r.&timepoint. ;
      quit;

      **** Need this extra step in order to create the mytimepoint variable without carried over the data format;
      data _wmxtox_&y.;
         set _wmxtox_&y.;
         mytimepoint=&timepoint.;
      run;
   %end;

   **** one obs per patient data will be used to run profile analysis;
   proc sort data=_wmxtox_&y. out=_wss_mxtox_&y.;
      by dcntr_id &by.;
   run;

   proc transpose data=_wss_mxtox_&y. out=_wf_mxtox_&y.;
      by dcntr_id &by.;
      id mytimepoint;
      var &scrgrd.;
   run;

   **** Get the summary statistics of grade;
   proc means data=_wmxtox_&y.;
      var &scrgrd.;
      class &timepoint. &by.;
      output out=_wmxtox_stat_&y. n=n median=median mean=mean stderr=stderr
             lclm(&scrgrd.)=lcl
             uclm(&scrgrd.)=ucl;
   run;

   **** Dataset to create line graphs;
   data f_aetime_graph_&y.;
      length mycount $8.;
      set _wmxtox_stat_&y.;
      format mean 8.2;
      keep &timepoint. &by. n mean lcl ucl mycount;

      mycount = put (n, 8.);

      where &timepoint. ^=. and missing (&by.) =0;

      label mean="Mean &scrgrd."
            lcl='lower 95% CI'
            ucl='upper 95% CI';
   run;

   **** Perform Repeated Measures Analysis on Individual AE;
   ods graphics off;
   ods output tests3    =_wtest3out_&y.   (keep=effect probf)
              solutionf =_wsfout_&y.      (keep=effect &by. estimate stderr probt)
              LSMeans   =_wlsmeansout_&y. (keep=effect &by. estimate stderr probt)
              ConvergenceStatus=_wconvout_&y.;
   proc mixed data=_wmxtox_&y.;
      class &by. dcntr_id;
      model &scrgrd. = &by. &timepoint. &by.*&timepoint. / solution ddfm=KR;
      random &timepoint. / type=&covrstr. subject=dcntr_id (&by.);
      lsmeans &by.;
   run;
   ods output close;

   ***** To determine the convergence status on random effect model;
   data _wconvout_&y.;
      length mreason $256.;
      set _wconvout_&y.;

      mreason="The Newton-Raphson algorithm for random effect model fail to converge, please consider a different covariance structure.";

      label mreason="Repeated Measures Analysis Results Not Available";
   run;

   data _null_;
      set _wconvout_&y.;
      call symput (compress('conv_' || &y.) , trim (left (status)));
      call symput (compress('conv_rea_' || &y.), trim(reason));
   run;

   **** Perform Profile Summary Analysis on Individual AE;
   %profile(dsn=_wf_mxtox_&y.,class=&by.,var=tmptlevel,min=1,max=&numtmptlevel.,
            myout=_wf_mxtox_out_&y.);

   run;

%end;

**=============================================================**;
**=============================================================**;
**=============================================================**;
**              CREATE AE OVER TIME GRAPH                      **;
**=============================================================**;
**=============================================================**;
**=============================================================**;

proc template;
   define Style lineplotstyle;
      parent = styles.statistical;
      style GraphFonts from GraphFonts
      "Fonts used in graph styles" /
      'GraphTitleFont' = ("Arial",12pt, Bold)
      'GraphLabelFont' = ("Arial",12pt, Bold)
      'GraphValueFont' = ("Arial",12pt, Bold)
      'GraphDataFont' = ("Arial",12pt, Bold)
      'GraphFootnoteFont' = ("Arial",12pt);

      style GraphDataDefault / linethickness = 2px;
      style GraphAxisLines   / linethickness = 2px;
      style GraphWalls       / linethickness = 2px FrameBorder=on;
   end;
run;

options nodate;
ods graphics / outputfmt=png;
ods rtf file="ae_over_time_figures_&dtype..doc" style=lineplotstyle;
%do y=1 %to &numvar.;
   title1 "Figure 1_&y.: %trim(&&&utxlbl&y.) %trim (&scrgrd.) Over Time By %trim(&bylbl.) with 95% CI";
   proc sgplot data=f_aetime_graph_&y.;
      format &timepoint. &tmptf. &by. &byf.;
      scatter x=&timepoint. y=mean / group=&by. yerrorlower=lcl
                                     yerrorupper=ucl;
      series x=&timepoint. y=mean / group=&by. name='s' markers markerattrs=(size=5 symbol=circlefilled)
                                    lineattrs=(thickness=3 pattern=solid);
      xaxistable n / x=&timepoint. class=&by. location=outside colorgroup=&by.;
      keylegend 's';
      yaxis label="Mean &scrgrd.";
      xaxis values=(&tmptlevel1. to &&&tmptlevel&numtmptlevel. by %eval (&tmptlevel2. - &tmptlevel1.));
   run;

%end;

%if &datatype.=1 %then %do;
   title1 "Figure 2: Maximum Adverse Events Grade Over Time By %trim(&bylbl.) with 95% CI";
   proc sgplot data=f_aetime_ovgraph;
      format &timepoint. &tmptf. &by. &byf.;
      scatter x=&timepoint. y=mean / group=&by. yerrorlower=lcl
                                     yerrorupper=ucl;
      series x=&timepoint. y=mean / group=&by. name='s' markers markerattrs=(size=5 symbol=circlefilled)
                                    lineattrs=(thickness=3 pattern=solid);
      xaxistable n / x=&timepoint. class=&by. location=outside colorgroup=&by.;
      keylegend 's';
      yaxis  label="Mean &scrgrd.";
      xaxis values=(&tmptlevel1. to &&&tmptlevel&numtmptlevel. by %eval (&tmptlevel2. - &tmptlevel1.));
   run;
%end;


ods rtf close;

ods graphics off;

**=============================================================**;
**=============================================================**;
**=============================================================**;
**                 GENERATE SUMMARY TABLES                     **;
**=============================================================**;
**=============================================================**;
**=============================================================**;

ods escapechar='^';
options nodate;
ods rtf file="ae_over_time_tables_&dtype..doc" style=journal2 bodytitle startpage=no;

%do y=1 %to &numvar.;

   title1 "Table 1_&y.: Repeated Measures for %trim(&&&utxlbl&y.) By %trim(&bylbl.)";
   %if %eval(&&conv_&y. =1) and "&&conv_rea_&y." ="WARNING: Did not converge." %then %do;

      proc print data=_wconvout_&y. noobs label;
         var mreason;
      run;

   %end;
   %else %do;
     title2 "Solution for Fixed Effects";
     proc print data=_wsfout_&y. label noobs;
     run;
     title2;

     ods rtf text="^S={font_size=9pt}The Solution for Fixed Effects Table indicates where there are differences in the model parameters.";
     ods rtf text="^R'\line'";

     ods rtf text="^S={font_size=9pt}The p-value for the Intercept tests whether the reference group (group where estimate=0) has an effect";
     ods rtf text="^S={font_size=9pt}that is different from zero. Significance means the reference group has a positive or negative effect";
     ods rtf text="^S={font_size=9pt}on the mean grade/score.";

     ods rtf text="^S={font_size=9pt}The p-values for subsequent groups indicate whether they have the same effect on mean grade/score as";
     ods rtf text="^S={font_size=9pt}the reference group. Significance means the group has an effect different than the reference group after";
     ods rtf text="^S={font_size=9pt}adjusting for all of the other effects in the model.";
     ods rtf text="^R'\line'";

     ods rtf text="^S={font_size=9pt}The p-value for the time point tests whether there is a change over all time points";
     ods rtf text="^S={font_size=9pt}(example, mean of all cycle 1 grades vs mean of cycle 2 grades... etc).";
     ods rtf text="^S={font_size=9pt}Significance means the slope of a graphed line of the means is not 0 (i.e. the line is not flat).";
     ods rtf text="^S={font_size=9pt}The slope of the line is the time point estimate.";
     ods rtf text="^R'\line'";

     ods rtf text="^S={font_size=9pt}The p-value for the time point and group interaction tests whether the first group listed";
     ods rtf text="^S={font_size=9pt}changes over time differently than the reference group. Significance means yes.";
     ods rtf text="^S={font_size=9pt}The p-values for subsequent interactions indicate whether the change over time is different than";
     ods rtf text="^S={font_size=9pt}the reference group given the previous group (s) may or may not contribute to the model.";

     title1 "Type 3 Tests of Fixed Effects";
     proc print data=_wtest3out_&y. label noobs;
     run;

     ods rtf text="^S={font_size=9pt}The Tests of Fixed Effects table indicates tests for the entire model";
     ods rtf text="^R'\line'";

     ods rtf text="^S={font_size=9pt}The group p-value indicates a difference between groups after adjusting for the slopes";
     ods rtf text="^S={font_size=9pt}and interactions. Significance means there is a difference between groups.";
     ods rtf text="^R'\line'";

     ods rtf text="^S={font_size=9pt}The time point p-value tests whether the overall growth curve increases or decreases";
     ods rtf text="^S={font_size=9pt}rather than staying flat over time. Significance means it is not flat.";

     ods rtf text="^S={font_size=9pt}The interaction p-value is a statistical combination of the interaction p-values";
     ods rtf text="^S={font_size=9pt}from the previous table and tests whether there is a difference between groups over time.";
     ods rtf text="^S={font_size=9pt}Significance means values are changing differently over time by groups.";

     ods startpage=now;

     title1 " Least Squares Means";
     title2 "Test whether overall estimate=0";
     proc print data=_wlsmeansout_&y. label noobs;
     run;
     title2;

     ods rtf text="^S={font_size=9pt}The Least Squares Means table provides estimates of the overall means of grade/score";
     ods rtf text="^S={font_size=9pt}in each group after adjusting for group, time point, and group * time point interactions.";
     ods rtf text="^S={font_size=9pt}A significant p-value indicates that the mean grade/score is different than 0.";

   %end;

   ods startpage=now;

  /*****************************/
  /* prints the summary report */
  /*****************************/
  title1 "Table 2_&y.: PROFILE ANALYSIS SUMMARY for %trim(&&&utxlbl&y.) By %trim(&bylbl.)";

  data _null_;
     set _wf_mxtox_out_&y.;
     file print;
     if (_n_=1) then do;
        put _page_;
        put @1 'PROFILE ANALYSIS SUMMARY';
        put @1 '------------------------';
        put @1 "     Comparing %trim(&&&utxlbl&y.) between %trim(&bylbl.)";
     end;

     if (group='1. Equal Levels') & (top=1) then do;
        put //  @1 'Test of Levels: Are the overall means equal for each group';
        put     @1 '---------------';
        put     @1 '     p-value= ' ProbtDiff pvalue6.4;
        if (ProbtDiff<0.05) then put @1 '     The overall mean levels averaged over all time periods are different between groups';
                            else put @1 '     There is no evidence that the overall mean levels averaged over all time periods are different between groups';
     end;
     if (group='1. Equal Levels') then put @1 "          %trim(&bylbl.) " &by. " mean level = " LSMean;

     if (group='2. Equal Means') then do;
       put //  @1 'Test of Equal Mean Vectors: Are the mean values equal at each time point';
       put     @1 '---------------------------';
       put     @1 '     Hotelling-Lawley Trace p-value= ' ProbF pvalue6.4;
       if (ProbF<0.05) then put @1 '     The mean values are signficantly different between groups for at least one time point';
                      else put @1 '     There is no evidence that the mean values are different between groups at any of the time points';
     end;

     if (group='3. Flat Lines') then do;
       put //  @1 'Test of Flatness: Are the lines flat (slope=0) at each time point for all groups';
       put     @1 '-----------------';
       put     @1 '     Hotelling-Lawley Trace p-value= ' ProbF pvalue6.4;
       if (ProbF<0.05) then put @1 '     The lines are not flat across all of the time points';
                      else put @1 '     There is no evidence that the lines are different from being flat across all of the time points';
     end;

     if (group='4. Parallel Lines') then do;
       put //  @1 'Test of Parallel Vectors: Are the lines for the groups parallel between each pair of time points';
       put     @1 '------------------------';
       put     @1 '     Hotelling-Lawley Trace p-value= ' ProbF pvalue6.4;
       if (ProbF<0.05) then do; put @1 '     The lines for the groups are not parallel between each time point';
                                put @1 '     This suggests there is a group by time interaction';
                            end;
                       else do; put @1 '     There is no evidence that the lines for the groups are different from parallel between each time point';
                                put @1 '     This suggests there is no evidence of a group by time interaction';
                            end;
      end;
   run;

   ods startpage=now;

%end;

%if &datatype.=1 %then %do;

   title1 "Table 3: Repeated Measures for Maximum Adverse Event Grade By %trim(&bylbl.)";

   %if %eval(&ov_conv.=1) and "&ov_conv_rea." ="WARNING: Did not converge." %then %do;
      proc print data=_wconvout noobs label;
         var mreason;
      run;
   %end;
   %else %do;
      title2 "Solution for Fixed Effects";
      proc print data=_wsfout label noobs;
      run;
      title2;

      ods rtf text="^S={font_size=9pt}The Solution for Fixed Effects Table indicates where there are differences in the model parameters.";
      ods rtf text="^R'\line'";

      ods rtf text="^S={font_size=9pt}The p-value for the Intercept tests whether the reference group (group where estimate=0) has an effect";
      ods rtf text="^S={font_size=9pt}that is different from zero. Significance means the reference group has a positive or negative effect";
      ods rtf text="^S={font_size=9pt}on the mean grade/score.";

      ods rtf text="^S={font_size=9pt}The p-values for subsequent groups indicate whether they have the same effect on mean grade/score as";
      ods rtf text="^S={font_size=9pt}the reference group. Significance means the group has an effect different than the reference group after";
      ods rtf text="^S={font_size=9pt}adjusting for all of the other effects in the model.";
      ods rtf text="^R'\line'";

      ods rtf text="^S={font_size=9pt}The p-value for the time point tests whether there is a change over all time points";
      ods rtf text="^S={font_size=9pt}(example, mean of all cycle 1 grades vs mean of cycle 2 grades... etc).";
      ods rtf text="^S={font_size=9pt}Significance means the slope of a graphed line of the means is not 0 (i.e. the line is not flat).";
      ods rtf text="^S={font_size=9pt}The slope of the line is the time point estimate.";
      ods rtf text="^R'\line'";

      ods rtf text="^S={font_size=9pt}The p-value for the time point and group interaction tests whether the first group listed";
      ods rtf text="^S={font_size=9pt}changes over time differently than the reference group. Significance means yes.";
      ods rtf text="^S={font_size=9pt}The p-values for subsequent interactions indicate whether the change over time is different than";
      ods rtf text="^S={font_size=9pt}the reference group given the previous group (s) may or may not contribute to the model.";


      title1 "Type 3 Tests of Fixed Effects";
      proc print data=_wtest3out label noobs;
      run;

      ods rtf text="^S={font_size=9pt}The Tests of Fixed Effects table indicates tests for the entire model";
      ods rtf text="^R'\line'";

      ods rtf text="^S={font_size=9pt}The group p-value indicates a difference between groups after adjusting for the slopes";
      ods rtf text="^S={font_size=9pt}and interactions. Significance means there is a difference between groups.";
      ods rtf text="^R'\line'";

      ods rtf text="^S={font_size=9pt}The time point p-value tests whether the overall growth curve increases or decreases";
      ods rtf text="^S={font_size=9pt}rather than staying flat over time. Significance means it is not flat.";

      ods rtf text="^S={font_size=9pt}The interaction p-value is a statistical combination of the interaction p-values";
      ods rtf text="^S={font_size=9pt}from the previous table and tests whether there is a difference between groups over time.";
      ods rtf text="^S={font_size=9pt}Significance means values are changing differently over time by groups.";

      ods startpage=now;

      title1 " Least Squares Means";
      title2 "Test whether overall estimate=0";
      proc print data=_wlsmeansout label noobs;
      run;
      title2;

      ods rtf text="^S={font_size=9pt}The Least Squares Means table provides estimates of the overall means of grade/score";
      ods rtf text="^S={font_size=9pt}in each group after adjusting for group, time point, and group * time point interactions.";
      ods rtf text="^S={font_size=9pt}A significant p-value indicates that the mean grade/score is different than 0.";

   %end;

   ods startpage=now;

  /*****************************/
  /* prints the summary report */
  /*****************************/
  title1 "Table 4: PROFILE ANALYSIS SUMMARY for Maximum Adverse Event Grade By %trim(&bylbl.)";

  data _null_;
     set _wf_mxtox_out;
     file print;
     if (_n_=1) then do;
        put _page_;
        put @1 'PROFILE ANALYSIS SUMMARY';
        put @1 '------------------------';
        put @1 "     Comparing Maximum Adverse Event Grade between &by.";
     end;

     if (group='1. Equal Levels') & (top=1) then do;
        put //  @1 'Test of Levels: Are the overall means equal for each group';
        put     @1 '---------------';
        put     @1 '     p-value= ' ProbtDiff pvalue6.4;
        if (ProbtDiff<0.05) then put @1 '     The overall mean levels averaged over all time periods are different between groups';
                            else put @1 '     There is no evidence that the overall mean levels averaged over all time periods are different between groups';
     end;
     if (group='1. Equal Levels') then put @1 "          &by. " &by. " mean level = " LSMean;

     if (group='2. Equal Means') then do;
       put //  @1 'Test of Equal Mean Vectors: Are the mean values equal at each time point';
       put     @1 '---------------------------';
       put     @1 '     Hotelling-Lawley Trace p-value= ' ProbF pvalue6.4;
       if (ProbF<0.05) then put @1 '     The mean values are signficantly different between groups for at least one time point';
                      else put @1 '     There is no evidence that the mean values are different between groups at any of the time points';
     end;

     if (group='3. Flat Lines') then do;
       put //  @1 'Test of Flatness: Are the lines flat (slope=0) at each time point for all groups';
       put     @1 '-----------------';
       put     @1 '     Hotelling-Lawley Trace p-value= ' ProbF pvalue6.4;
       if (ProbF<0.05) then put @1 '     The lines are not flat across all of the time points';
                      else put @1 '     There is no evidence that the lines are different from being flat across all of the time points';
     end;

     if (group='4. Parallel Lines') then do;
       put //  @1 'Test of Parallel Vectors: Are the lines for the groups parallel between each pair of time points';
       put     @1 '------------------------';
       put     @1 '     Hotelling-Lawley Trace p-value= ' ProbF pvalue6.4;
       if (ProbF<0.05) then do; put @1 '     The lines for the groups are not parallel between each time point';
                                put @1 '     This suggests there is a group by time interaction';
                            end;
                       else do; put @1 '     There is no evidence that the lines for the groups are different from parallel between each time point';
                                put @1 '     This suggests there is no evidence of a group by time interaction';
                            end;
      end;
   run;

%end;

ods rtf close;

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                        REMOVE WORKING DATA SET                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

proc datasets library=work nolist;
   delete _w:;
quit;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 EXIT THE PROGRAM WHEN ERROR OCCUR                  *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
%exit:

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                       DISPLAY THE MACRO STATUS                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

data _null_;
   put "---------------------------";
   %if &myErrMsg. ^= %then %do;
      put "ERROR: " &myErrMsg.;
   %end;
   put "---------------------------";
run;


%mend ae_over_time;
