
/*------------------------------------------------------------------*
 | MACRO NAME: bytimept
 | SHORT DESC: Compares individual/overall adverse events between
 |             treatment groups over time
 *------------------------------------------------------------------*
 | CREATED BY: ToxT Team                               (07/20/2016)
 |
 | Version 2 (6/1/2017): Re-create the layout for summary statistics tables
 |                       for means and freq from one table per cycle data
 |                       into one table for all cycles data
 |                       Completed - 1/7/2019
 *------------------------------------------------------------------*
 | PURPOSE
 |
 | This program compares individual or overall adverse events per
 | NCI Common Terminology Criteria for Adverse Events (CTCAE) or
 | individual symptoms/QOL items reported per Patient Report Outcomes (PRO)
 | for each treatment groups across data collection time points
 |
 | The program will produce two documents in the directory where SAS is run:
 |
 | For CTCAE:
 | 1. bytimept_figures_ae.doc: Bar Charts for Incidence of individual
 |                             or overall adverse event by treatment group,
 |                             timepoint and adverse event grade
 | 2. bytimept_tbl_ae.doc: 1) Table Summary of Mean grade of individual or
 |                            overall adverse event by treatment group
 |                            at each timepoint
 |                         2) Table Summary of Frequecy per grade of
 |                            individual or overall adverse event by
 |                            treatment group at each timepoint
 | NOTES: Overall AE is summarized across all AEs that are included
 |        in the CYTOX dataset. If user would like to see the Overall AE
 |        across a subset of AEs that are specified in TOXICITY_LIST
 |        parameter, the CYTOX dataset need to be modified to include
 |        only the subset AEs that are of user interest
 |
 | For PRO Symptoms/QOL items:
 | 1. bytimept_figures_qol.doc: Bar Charts for Incidence of individual
 |                              symptom/QOL items by treatment group,
 |                              timepoint and symptom grade categorized
 |                              per symptom scores
 | 2. bytimept_tbl_qol.doc: 1) Table Summary of Mean grade of individual
 |                             symptom/QOL items by treatment group
 |                             at each timepoint
 |                          2) Table Summary of Frequecy per grade of
 |                             individual symptoms/QOL items by
 |                             treatment group at each timepoint
 |
 | ~ IMPORTANT NOTES: ~
 | 1. The program will produce the bar charts with readable axis label
 |     when the levels of cross product of BY * TIMEPOINT is <=32
 | 2. The bar charts will be produced in black/white version ONLY
 | 3. Datasets that are used to produce the bar chars and summary tables
 |    can be found in WORK directory with prefix of f_ in dataset name
 |    where user can further modify or tweak the graphing per user preference
 | 4. DO NOT use # in the label of a variable, it will get interpreted as
 |    displaying text in new line
 |
 *------------------------------------------------------------------*
 | REQUIRED DATASETS and DATA FIELDS
 |
 | 1. crse dataset with one observation per patient with the following data fields:
 |    dcntr_id          : patient identifier
 |    evalae            : evaluable for toxicity (numeric)
 |                        (1= at least one AE form is entered other than
 |                            the baseline timepoint
 |                         0= otherwise)
 |    [comparison group]: group to be used for data comparison
 |                        which will be specified in BY macro parameter
 |                        (i.e. arm)
 |
 | 2. prodata dataset with one observation per time point per patient with the
 |    following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for treatment (numeric) which will be specified
 |                        in TIMEPOINT macro parameter (i.e. cycle)
 |    Notes: The dataset should include all treatment time point that are
 |            of user interest for the analysis
 |
 | 3. cytox dataset with one observation per maximum grade per adverse event per
 |    timepoint per patient with the following data fields (REQUIRED for CTCAE analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for adverse events reporting (numeric) which
 |                        will be specified in TIMEPOINT macro parameter (i.e. cycle)
 |    toxicity          : Adverse event as in medra codes (numeric and should be formatted
 |                        to give adverse event description) (i.e. 900182)
 |    grade             : Severity of the adverse event according to CTC guidelines.
 |                        (numeric) (i.e. 0, 1, 2, 3, 4, 5)
 |    Notes: The dataset should include all adverse event reported up to the last time point
 |           indicated in protdata dataset of user interest for the analysis
 |
 | 4. qol dataset with one observation per symptom per timepoint per patient
 |    with the following data fields (REQUIRED for PRO Symptoms analysis ONLY):
 |    dcntr_id          : patient identifier
 |    [timepoint]       : timepoint for symptoms reporting (numeric) which
 |                        will be specified in TIMEPOINT macro parameter (i.e. cycle)
 |    qol               : SAS variable name for each symptom (character)
 |                        (i.e. p_q10, poms04)
 |    _label_           : Description/Label for each symptom (character)
 |                        (i.e. Overall QOL, Sleep Interference, Nervousness)
 |    scores            : Symptoms/QOL items scores in a scale of 0 to 100 (numeric)
 |                        with 0=Low Quality of Life and 100=Best Quality of Life
 |                        Notes: for symptoms with 0=no symptom and 100=worst symptom
 |                               the scores need to be reversed before using this program
 |    Notes: The dataset should include all symptoms/qol items reported up to the
 |           last time point of user interest for the analysis
 *------------------------------------------------------------------*
 | REQUIRED PARAMETERS
 |
 | datasource   : The location where analysis datasets are saved
 |                1 = Internal CCS database
 |                2 = User provided datasets in WORK directory (DEFAULT)
 |
 | datatype     : Type of adverse events/symptoms reported
 |                1 = Adverse Events per CTCAE (DEFAULT)
 |                2 = Symptoms/QOL items per PRO
 |
 | by           : Group Comparison (i.e. arm)
 |
 | timepoint    : Time variable (numeric)
 |
 | toxicity_list: List of toxicity codes separated by space for CTCAE
 |                (i.e. 800001 832222 1002222)
 |                OR List of SAS variable names separated by space for PRO
 |                symptoms/QOL items (i.e. lasa01 poms03 lasa09 lccs4)
 |                Notes: This allows up to 8 AEs/Symptoms/QOL items

 | OPTIONAL PARAMETERS
 |
 | studynum     : study reference number (REQUIRED if datasource=1) (i.e. MCXXXX)
 |
 | incmiss      : AE grade assumption when specified AE entry is not available (for CTCAE ONLY)
 |                y = assumed grade 0 if there is no entry for user specified AE in the data set,
 |                    with condition that data entry is available for other AE at specific
 |                    timepoint (DEFAULT)
 |                n = assumed missing or do nothing if there is no entry for user specified
 |                    AE in the dataset
 *------------------------------------------------------------------*
 | SAMPLE CODES USED to prepare the QOL dataset
 |
 | For example:
 | QOL_tmp dataset contains one observation per timepoint with data fields include:
 | dcntr_id, visit, p_q01, p_q07, p_q09, p_q11
 |
 | proc sort data=qol_tmp;
 |    by dcntr_id visit;
 | run;
 |
 | proc transpose=qol_tmp out=qol (rename=(col1=scores));
 |    by dcntr_id visit;
 |    var p_q01 p_q07 p_q09 p_q11;
 | run;
 |
 | data qol;
 |    set qol;
 |    rename _name_=qol;
 |    label _name_='QOL';
 | run;
 |
 *------------------------------------------------------------------*
 | EXAMPLE:
 |
 | For CTCAE:
 |    %bytimept (by=arm, timepoint=cycle,
 |               toxicity_list=8000001 900182 900146 21162 21197 21521);
 | For PRO Symptoms/QOL items:
 |    %bytimept (by=arm, datatype=2, timepoint=visit,
 |               toxicity_list=p_q08 p_q09 p_q10 p_q11 p_q01 p_q06);
 *------------------------------------------------------------------*
 */

%macro bytimept (datasource=2, studynum=, datatype=1, by=, timepoint=, toxicity_list=, incmiss=y);

%macro outstat (dsn=, by=, timepoint=, numlevel=, myy=, myz=);

  *************************************************************************************;
  **** Data Setup to retrieve n (%) and p-value of grade/scores for each timepoint ****;
  *************************************************************************************;

   ods output crosstabfreqs=_wfrout&myy.&myz.;
   proc freq data=&dsn. ;
      %if &datatype.=2 %then %do;
         format &scrgrd2 8.;
      %end;

      tables &timepoint*&by*&scrgrd2 / sparse nowarn missprint outpct chisq;
   run;
   ods output close;

   *** Retrieve the timepoint;
   proc sql;
      create table _woutd&myy.&myz. as
      select distinct &timepoint.
      from &dsn.;
   quit;


   **** Determine the level of grade/scores for each AE/QOL;
   proc sql noprint;
      select max (&scrgrd2) into: mxsc&myy.&myz.
      from _wfrout&myy.&myz.;
   quit;

   **** retrieve the p-values (both ordinal/continous data analysis share same p-values);
   **** since we have ordered data, we should use wilcoxon-rank sum for 2-by level data;
   **** and kruskal wallis for >2-by level data;
   proc npar1way data=&dsn. wilcoxon anova noprint;
      class &by.;
      var &scrgrd.;
      output out=_w2outd&myy.&myz. wilcoxon anova;
   run;

   data _woutd2&myy.&myz.;
      set _w2outd&myy.&myz.;
      format pval pvalue6.4;
      keep &timepoint. pval;

      if _n_=1 then set _woutd&myy.&myz. (keep=&timepoint.);

      **** Wilcoxon p-value will be missing, if by-group>2 levels;
      %if %eval(&numlevel.>2) %then %do;
         pval=p_kw;
         label pval='Kruskal Wallis P-value';
      %end;

      %else %do;
         pval=p2_wil;
         label pval='Wilcoxon Rank-Sum P-value';
      %end;
   run;

   data _wfrout2&myy.&myz. (keep=&timepoint. &by. &scrgrd2. freqstat)
        _wofrout2&myy.&myz. (keep=&by. frequency);

      length freqstat $50.;
      set _wfrout&myy.&myz.;

      freqstat = compbl (trim(left(frequency)) || ' (' || put (rowpercent, 8.1) || '%)');

      if _type_='110' then output _wofrout2&myy.&myz.;
      else if _type_='111' and &scrgrd2. ^=. then output _wfrout2&myy.&myz.;

   run;

   proc sort data=_wfrout2&myy.&myz.;
      by &timepoint. &by. &scrgrd2.;
   run;

   proc transpose data=_wfrout2&myy.&myz. out=_wfr_out2&myy.&myz. (drop=_name_);
      by &timepoint. &by.;
      id &scrgrd2.;
      var freqstat;
   run;

   proc sql;
      create table _wfqtab&myy.&myz. as
      select a.*, b.frequency label='Total', c.pval
      from _wfr_out2&myy.&myz. a, _wofrout2&myy.&myz. b, _woutd2&myy.&myz. c
      where a.&by.=b.&by. and a.&timepoint. = c.&timepoint.;
   quit;

   data _wfqtab&myy.&myz.;
      set _wfqtab&myy.&myz.;

      if _n_ > 1 then do;
         &timepoint.=.;
         pval=' ';
      end;

   run;

   ***************************************************************************************;
   **** Data Setup to retrieve n, mean and p-value of grade/scores for each timepoint ****;
   ***************************************************************************************;

   **** 4/26/2018 updated the var statement from &scrgrd2. to &scrgrd.;
   **** retrieve n and means;
   proc univariate data=&dsn. noprint;
      class &by.;
      var &scrgrd.;
      output out=_wmout&myy.&myz. n=n mean=mean;
   run;

   data
      %do g=1 %to &numlevel.;
         _wa&g.&myy.&myz. (rename=(n=n&g. mean=mn&g.))
      %end;
      ;

      drop &by.;

      set _wmout&myy.&myz.;

      dummvar=1;

      %do g=1 %to &numlevel.;
         if &by.=trim(left("&&&bylevel&g.")) then output _wa&g.&myy.&myz.;
      %end;

      label n='N'
            mean='Mean';

   run;

    **** 4/26/2018 updated the var statement from &scrgrd2. to &scrgrd.;
   **** retrieve total n and means;
   proc univariate data=&dsn. noprint;
      var &scrgrd.;
      output out=_wmtout&myy.&myz. n=n mean=mn;
   run;

   data _wmtout&myy.&myz.;
      set _wmtout&myy.&myz.;
      format mn 8.1;

      dummvar=1;

      label n='N'
            mn='Mean';
   run;

   **** Merge in N, means and the total N and total Means;
   data  _wtttmout&myy.&myz.;
      merge %do g=1 %to &numlevel.;
           _wa&g.&myy.&myz.
           %end;
           _wmtout&myy.&myz.;
      by dummvar;

      format %do g=1 %to &numlevel.;
             mn&g.
           %end;
           8.1;
   run;

   data _wtmout2&myy.&myz.;
      set _wtttmout&myy.&myz.;

      if _n_=1 then set _woutd&myy.&myz. (keep=&timepoint.);
   run;

   **** Merge in all stat;
   data _wctst&myy.&myz.;
      merge _wtmout2&myy.&myz. _woutd2&myy.&myz.;
      by &timepoint.;
   run;

%mend outstat;

**********************************;
**** Variables Initializations ***;
**********************************;
%let studynum = %upcase(&studynum.);

%local myErrMsg dtype dname scrgrd aevar dfigtbtlt dxlbl scrgrd2 annoymax char_tmpt;

%if &datatype.=1 %then %do;
   %let dtype     =ae;
   %let dname     =cytox;
   %let scrgrd    =Grade;
   %let scrgrd2   =grade;
   %let aevar     =toxicity;
   %let dfigtbtlt =%str(AE);
   %let dxlbl     =%str(CTCAE Grade);
%end;

%if &datatype.=2 %then %do;
   %let dtype     =qol;
   %let dname     =qol;
   %let scrgrd    =Scores;
   %let scrgrd2   =scrcat;
   %let aevar     =qol;
   %let dfigtbtlt =%str(Patient Reported QOL/Symptom with 0:Low and 100:Best QOL);
   %let dxlbl     =%str(QOL/Symptom Scores);
%end;

**================**;
**  Data Format   **;
**================**;

proc format;
   value yyesnof 1='Yes'
                 2='No';

   value scrcatf 1='0-50'
                 2='51-74'
                 3='75-100';

run;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 PROMPT ERROR MESSAGE WHEN NECESSARY                *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

***************************************************************;
**** Display error when data source is not found or invalid ***;
***************************************************************;

%if &datasource. = %then %do;
   %let myErrMsg = "Data source is not found.";
   %goto exit;
%end;

%else %if &datasource. ^=1 and &datasource. ^=2 %then %do;
   %let myErrMsg = "Data source is invalid.";
   %goto exit;
%end;

********************************************************************;
**** Display error when data source is 1 and studynum is missing ***;
********************************************************************;

%if &datasource. = 1 and &studynum. = %then %do;
   %let myErrMsg = "Data Source of Cancer Center database was indicated, yet, study number is not provided.";
   %goto exit;
%end;

***********************************************************;
**** Display error when data type is missing or invalid ***;
***********************************************************;

%if &datatype. = %then %do;
   %let myErrMsg = "Datatype is not found.";
   %goto exit;
%end;

%else %if &datatype. ^=1 and &datatype. ^=2 %then %do;
   %let myErrMsg = "Datatype is invalid.";
   %goto exit;
%end;

*****************************************;
**** Display error when by is missing ***;
*****************************************;

%if &by. = %then %do;
   %let myErrMsg = "By group is not found.";
   %goto exit;
%end;

************************************************;
**** Display error when timepoint is missing ***;
************************************************;

%if &timepoint. = %then %do;
   %let myErrMsg = "Timepoint is not found.";
   %goto exit;
%end;

*********************************************************;
**** Display error when toxicity_list is missing     ****;
*********************************************************;
%if &toxicity_list. = %then %do;
   %let myErrMsg = "No variable found in parameter TOXICITY_LIST.";
   %goto exit;
%end;

%let cntvar=1;

**** Get the number of toxicities specify by user;
%do %while (%scan(&toxicity_list., &cntvar.) ^= );
   %let cntvar   = %eval(&cntvar. + 1);
%end;

%let cntvar = %eval(&cntvar. - 1);

**=======================**;
**  Required Datasets    **;
**=======================**;

**** if using cancer center database;
%if &datasource. = 1 %then %do;
   %crtlibn (d=%upcase(&studynum.));

   proc copy in=pdata out=work;
   run;

%end;

*******************************************************;
**** Display error when CRSE dataset does not exist ***;
*******************************************************;

%if %sysfunc(exist(crse)) = 0 %then %do;
   %let myErrMsg = "crse dataset does not exist.";
   %goto exit;
%end;

************************************************************;
**** Display error when CYTOX/QOL dataset does not exist ***;
************************************************************;

%if %sysfunc(exist(&dname)) = 0 %then %do;
   %let myErrMsg = "&dname. dataset does not exist.";
   %goto exit;
%end;

************************************************************;
**** Display error when PRODATA dataset does not exist   ***;
**** data type is CTCAE                                  ***;
************************************************************;

%if %upcase (&dname.)=CYTOX %then %do;
   %if %sysfunc(exist(protdata)) = 0 %then %do;
      %let myErrMsg = "protdata dataset does not exist.";
      %goto exit;
   %end;
%end;

**==========================**;
**  Other Required Datasets **;
**==========================**;

**** CRSE dataset;
proc sql;
   create table crse0 as
   select dcntr_id, evalae, &by.
   from work.crse
   order by dcntr_id;
quit;

**** Determine the variable type, format and label for CRSE dataset;
proc sql;
  create table _w1 as
  select *
  from sashelp.vcolumn
  where upcase(libname)="WORK"
  and upcase(memname)="CRSE0";
quit;

**** Retrieve the type, format and label for By variable;
data _null_;
   set _w1 end=eof;

   if (label=' ') then label=name;

   if (upcase(name)=%upcase("&by")) then do;
      if (format=' ') and (type='char') then format=compress('$char' || trim(left(length)) || '.');
      if (format=' ') and (type='num') then format='8.';

      call symput('byf', trim(left(format)));
      call symput('bylbl', trim(left(label)));
      call symput('bytype', trim(left(type)));
   end;

run;

**** Determine the number of level used in the BY parameter;
proc sql;
   create table _wchkbylvl as
   select distinct &by.
   from crse0;
quit;

data _null_;
   set _wchkbylvl end=eof;
   call symput ('bylevel' || trim (left(_n_)), &by.);
   call symput ('bylvllbl' || trim (left(_n_)), put (&by., &byf.));

   if eof then call symput ('numlevel', trim(left(_n_)));
run;

**** Check to see if TIMEPOINT is character field ****;
proc sql;
   create table &dname.tmp as
   select *
   from work.&dname.;
quit;

**** Check to see if TIMEPOINT is character field ****;
proc sql;
   create table _ww1 as
   select *
   from sashelp.vcolumn
   where libname="WORK"
   and memname=%upcase("&dname.tmp");
quit;

data _null_;
   set _ww1;

   if upcase(name)=%upcase ("&timepoint.") and type='char' then
      call symput ('char_tmpt', 'y');
run;

************************************************************;
**** Display error when Timepoint is Characters field    ***;
************************************************************;

%if %upcase (&char_tmpt.)=Y %then %do;
   %let myErrMsg = "TIMEPOINT required a numeric field";
   %goto exit;
%end;

*********************************************************************************************;
**** Determine the number of evaluation patients per protdata - applicable to CTCAE data ONLY;
**** Verify the toxicity list provided by users to make sure they are unique             ****;
*********************************************************************************************;

%if &datatype.=1 %then %do;
   proc sort data=work.cytox;
      by dcntr_id;
   run;

   proc sort data=work.protdata;
      by dcntr_id;
   run;

   **** Make sure the cytox dataset provided are for the intended population;
   data _wcytox_tmp;
      merge crse0 (in=in1 keep=dcntr_id)
            cytox (in=in2 keep=dcntr_id &timepoint. toxicity grade);
      by dcntr_id;
      if in1 and in2;
   run;

   **** Make sure the protdata dataset provided are for the intended population;
   data _wprotdata_tmp;
      merge crse0 (in=in1 keep=dcntr_id)
            protdata (in=in2 keep=dcntr_id &timepoint.);
      by dcntr_id;
      if in1 and in2;
   run;

   **** All cycles needed to exist in order to run AUC macro, data can be missing at that cycle;
   **** Even though this does not apply to this macro, will still keep it the same as;
   **** what was written in the other macro for consistency;
   proc sql;
      create table _wchktimelvl0 as
      select distinct &timepoint.
      from _wcytox_tmp
      order by &timepoint.;
   quit;

   **** Determine the unique treatment cycles and if patient is evaluable for AE;
   proc sql;
      create table _wpttime as
      select l.*, r.&timepoint.
      from crse0 l left join _wchktimelvl0 r
      on l.dcntr_id
      where l.evalae=1
      order by l.dcntr_id, r.&timepoint.;
   quit;

   proc sql;
      create table _wpttime_tmp as
      select distinct dcntr_id, max (&timepoint.) as mxcyc_prot
      from _wprotdata_tmp
      group by dcntr_id
      order by dcntr_id;
   quit;

   proc sql;
      create table protdata0 as
      select l.*, r.mxcyc_prot
      from _wpttime l left join _wpttime_tmp r
      on l.dcntr_id = r.dcntr_id
      having l.&timepoint. <= r.mxcyc_prot
      order by l.dcntr_id, l.&timepoint.;
   quit;

   proc sort data=_wcytox_tmp;
      by dcntr_id &timepoint.;
   run;

   **** 4/20/2018 - added evalae indicator so we could indicate evalpt in the OVERALL AE calculation in the case;
   *****where cytox datasets include more AE data that are beyond the specific variables of interest;
   **** Assume Grade 0 when certain toxicity is not reported, but, AE entries is available at that cycle;
   proc sql;
      create table checkAEgrade as
      select distinct l.dcntr_id, l.&timepoint., 1 as assume_grd0, 0 as grade, r.&by., r.mxcyc_prot, r.evalae
      from _wcytox_tmp l left join protdata0 r
      on l.dcntr_id = r.dcntr_id and l.&timepoint. = r.&timepoint.
      having l.&timepoint. <= r.mxcyc_prot
      order l.dcntr_id, l.&timepoint.;
   quit;

   *** Include only patients who indicated as evaluable for AE and with treatment cycles available;
   data _wcytox0;
      merge protdata0    (in=in1)
            _wcytox_tmp  (in=in2);
      by dcntr_id &timepoint.;
      if in1;
      evalpt=1;
   run;

   **** 4/20/2018 added the condition of evalae to indicate evalpt;
   **** CYTOX dataset - Overall AE analysis (include evaluable patients only);
   **** Grade 0 will be pre-filled at each cycle if at least one AE was reported at that cycle;
   **** and was indicated by user to do so;
   data cytox0;
      set _wcytox0

      %if %upcase (&incmiss.)=Y %then %do;
         checkAEgrade (drop=assume_grd0 mxcyc_prot)
      %end;
      ;

      if evalae=1 then evalpt=1;

   run;

   **** 4/20/2018 added the grade^=. condition to account for the case if pt is treated at a cycle (i.e. cycle 1);
   **** and ae data was not reported, then pt is considered not evaluable for ae at that cycle;
   **** Determine the number of evaluation patients;
   proc sql;
      create table _wchk_evalpt as
      select distinct dcntr_id, &timepoint., &by., evalpt
      from cytox0 (where=(evalpt=1 and grade^=.));
   quit;

   ods output crosstabfreqs=_wchk_evalpt2;
      proc freq data=_wchk_evalpt;
         table &by. * &timepoint.;
      run;
   ods output close;

   data getevlpt;
      set _wchk_evalpt2;
      keep &by. &timepoint. frequency;
      rename frequency=tot_evalpt;
      where _type_='11';
   run;
%end;

%if &datatype.=2 %then %do;
   proc sql;
      create table qol_items as
      select distinct qol, _label_
      from work.qol
   quit;

   data _null_;
      set qol_items end=eof;

      call symput ('qlist_'  || trim (left(_n_)), qol);
      call symput ('qlabel_' || trim (left(_n_)), _label_);

      if eof then call symput ('n_allqol', trim (left(_n_)));
   run;

   **** Create the format for the QOL items used;
   proc format;
      value $ qolf
         %do i=1 %to &n_allqol.;
            "&&&qlist_&i." = "&&&qlabel_&i."
         %end;
      ;
   run;

   **** If there is no entries of QOL whatsoever, that QOL will be set to missing;
   **** QOL dataset - Base dataset;
   proc sql;
      create table qol0 as
      select dcntr_id, &timepoint., qol format=$qolf., _label_, scores,
             case when  0<= scores <=50  then 1
                  when 50 < scores <75  then 2
                  when 75<= scores <=100 then 3
            else .
            end as scrcat label='Scores Category' format=scrcatf.
      from work.qol
      order by dcntr_id;
   quit;
%end;

proc sql;
   create table _c2 as
   select *
   from sashelp.vcolumn
   where libname="WORK"
   and memname=%upcase("&dname.0");
quit;

**** Retrieve the format and label for Toxicity/QOL AE and timepoints;
data _null_;
   set _c2;
   if (label=' ') then label=name;
   if (upcase(name)=%upcase("&aevar.")) then do;
      call symput('txf',trim(left(format)));
      call symput('txlbl',trim(left(label)));
   end;

   if (upcase(name)=%upcase ("&timepoint.")) then do;
      if type='num'  and format=' ' then format=compress(trim(left(length)) || '.');

       call symput('tmptf',trim(left(format)));
       call symput('tmptlbl',trim(left(label)));
   end;
run;

**** Declare each toxicity as macro variable;
data _wtxlist0;
   %do i=1 %to &cntvar.;
      %if &datatype.=1 %then %do;
         toxvar&i.=%scan (&toxicity_list., &i.);
         toxlbl&i.=put (toxvar&i., &txf.);
      %end;

      %if &datatype.=2 %then %do;
         length toxvar&i toxlbl&i. $ 255.;
         format toxlbl&i. $qolf.;

         toxvar&i.="%scan (&toxicity_list., &i.)";
         toxlbl&i.= toxvar&i.;
      %end;
   %end;
run;


**** Remove any duplicates toxicities codes;
proc transpose data=_wtxlist0 out=_wf_txlist0;
   var %do i=1 %to &cntvar.;
       toxvar&i.
      %end;
      ;
run;

proc sort data=_wf_txlist0 nodupkeys;
   by col1;
run;

**** To retain the order of AE as it was specified by user;
proc sort data=_wf_txlist0;
   by _name_;
run;

**** Change the obs number if we want to accept more toxicity variables for analysis;
data _null_;
   set _wf_txlist0 (firstobs=1 obs=8) end=eof;
   call symput ('txvar' || trim(left(_n_)),  trim(left(col1)));
   call symput ('utxlbl' || trim (left(_n_)), put (col1, &txf.));
   if eof then call symput ('numvar', trim(left( _n_)));
run;

**** Determine the number of timepoints used in the timepoint parameter;
proc sql;
   create table _wchktimelvl as
   select distinct &timepoint.
   from &dname.0
   order by &timepoint.;
quit;

data _null_;
   length alltmpt1 alltmpt2 $255.;
   set _wchktimelvl end=eof;
   retain alltmpt1 alltmpt2;

   alltmpt1 = compbl (alltmpt1 || &timepoint.);
   alltmpt2 = compbl (alltmpt2 || compress ('_' || &timepoint.));

   if _n_=1 then call symput ('tmpt1level', &timepoint.);

   call symput ('tmptlevel' || trim (left(_n_)), &timepoint.);
   call symput ('dtmptlevel' || trim (left(_n_)), compress('_' || &timepoint.));
   call symput ('tmptlvllbl' || trim (left(_n_)), put (&timepoint., &tmptf.));

   if eof then do;
      call symput ('numtmptlevel', trim(left(_n_)));
      call symput ('alltmptlevel1', alltmpt1);
      call symput ('alltmptlevel2', alltmpt2);
   end;

run;

**** Assign the y-level to plot the Total N wording on the graph, max=32;
%if %eval(&numtmptlevel. * &numlevel.) >= 10 %then %do;
   %let annoymax=92;
%end;
%else %if %eval(&numtmptlevel. * &numlevel.) >= 7 %then %do;
   %let annoymax=82;
%end;
%else %do;
   %let annoymax=78;
%end;


**=============================================================**;
**                   DATA MANIPULATIONS                        **;
** *********This applicable for CTCAE datasets ONLY *************;
** Maximum grade across ALL Adverse Events at each timepoints  **;
**=============================================================**;

%if &datatype.=1 %then %do;

   proc sort data=cytox0 out=_wov_mxtox;
      by dcntr_id &timepoint. grade;
   run;

   data _wmxtox;
      set _wov_mxtox;
      by dcntr_id &timepoint. grade;
      if last.&timepoint.;
   run;

   **** OVERALL AE: Dataset to use in %table to generate the summary statistics per timepoint;
   %do i=1 %to &numtmptlevel.;
      proc sql;
         create table f_mxtox_&i. as
         select dcntr_id, &by., &timepoint.,
                grade label= "Overall AE: %trim(&tmptlbl.) %trim(&&&tmptlvllbl&i.)"
         from _wmxtox
         where &timepoint. = &&&tmptlevel&i.;
      quit;

      %outstat (dsn=f_mxtox_&i.,
                by=&by.,
                numlevel=&numlevel.,
                timepoint=&timepoint.,
                myy=_&i.);
   %end;

   **** Determine the Frequency Count at each timepoint, grade and bygroup;
   ods output crosstabfreqs=_wmxtoxf1 (keep=&by. grade &timepoint. frequency _type_);
   proc freq data=_wmxtox;
      table &timepoint.*grade*&by.;
   run;
   ods output close;

   **** OVERALL AE: Dataset for generating the Overall AE Bar chart;
   proc sql;
      create table f_ovhbar as
      select l.&by., l.&timepoint., l.grade, l.frequency, r.tot_evalpt,
             (l.frequency / r.tot_evalpt)*100 as pctpt_ae
      from _wmxtoxf1 (where=(_type_='111')) l left join getevlpt r
      on l.&by. = r.&by. and l.&timepoint. = r.&timepoint.
      having l.grade not in (., 0)
      order l.&by., l.&timepoint., l.grade;
   quit;

   **** annotate dataset;
   data f_ovhbar_anno;
      set _wchk_evalpt2 (where=(_type_='11'));
      xsys='3';
      ysys='2';
      when="a";
      midpoint=&by.;
      group=&timepoint.;
      x=97;
      text=cat(frequency);
      output;

      xsys='3';
      ysys='3';
      y=&annoymax.;
      x=96;
      text="Total N";
      output;
   run;

%end;

**=============================================================**;
**                   DATA MANIPULATIONS                        **;
**                (For Both CTCAE and QOLAE)                   **;
** Grade per adverse events or Scores per QOL AE at each timepoints **;
** If multiple grade was reported per AE at specific timepoints **;
** proc transpose will fail and users need to fix their data before re-run the program;
**=============================================================**;

%do y=1 %to &numvar.;

   **** To make sure all patients have an observation for each timepoint;
   %if &datatype.=1 %then %do;
      proc sql;
         create table _wcytox1 as
         select distinct l.dcntr_id, l.&timepoint., l.&by., l.evalae, l.evalpt, l.mxcyc_prot, r.assume_grd0
         from cytox0 (where=(grade^=0 and toxicity^=. and evalae^=.)) l left join checkAEgrade r
         on l.dcntr_id = r.dcntr_id;
      quit;

      proc sql;
         create table _ws_mxtox_&y. as
         select l.*, r.toxicity, r.grade
         from _wcytox1 l left join cytox0 (where=(toxicity=&&&txvar&y.)) r
         on l.dcntr_id    = r.dcntr_id and
            l.&timepoint. = r.&timepoint.;
      quit;

      **** If AE is not reported at the time point, assumed patient do not have that AE, will code as 0;
      data _wmxtox_&y.;
         set _ws_mxtox_&y.;

         %if %upcase (&incmiss.)=Y %then %do;
            if grade=. and assume_grd0=1 then grade=0;
         %end;
      run;

      **** Determine the number of evaluation patients;
      **** This would work fine for when incmiss=Y;
      proc sql;
         create table _wchk_evalpt_&y. as
         select distinct dcntr_id, &timepoint., &by., evalpt
         from _wmxtox_&y. (where=(evalpt=1));
      quit;

      ods output crosstabfreqs=_w2chk_evalpt_&y.;
         proc freq data=_wchk_evalpt_&y.;
            table &by. * &timepoint.;
         run;
      ods output close;

     **** Need to replace this dataset when incmiss=N;
      data getevlpt_&y.;
         set _w2chk_evalpt_&y.;
         keep &by. &timepoint. frequency;
         rename frequency=tot_evalpt;
         where _type_='11';
      run;

      **** SPECIFIC AE: Dataset to use in %table to generate the summary statistics per timepoint;
      %do i=1 %to &numtmptlevel.;
         proc sql;
            create table f_mxtox_&y._&i.  as
            select dcntr_id, &by., &timepoint., evalpt,
                   grade label= "%trim(&&&utxlbl&y.): %trim(&tmptlbl.) %trim(&&&tmptlvllbl&i.)"
            from _wmxtox_&y.
            where evalpt=1 and &timepoint. = &&&tmptlevel&i.;
         quit;

         %outstat (dsn=f_mxtox_&y._&i.,
                   by=&by.,
                   timepoint=&timepoint.,
                   numlevel=&numlevel.,
                   myy=_&y.,
                   myz=_&i.);
      %end;

      ods output crosstabfreqs=_wmxtoxfa_&y.  (keep=&by. grade &timepoint. frequency _type_);
         proc freq data=_wmxtox_&y. ;
            table &timepoint.*grade*&by.;
         run;
      ods output close;

      /*** new added on 12/10/2018 ****/
      %if %upcase(&incmiss.)=N %then %do;
         ods output crosstabfreqs=_w2chk_evalpt_&y.;
         proc freq data=_wmxtox_&y. ;
            where grade ^=.;
            table &by. * &timepoint.;
         run;
         ods output close;

         data getevlpt_&y.;
            set _w2chk_evalpt_&y.;
            keep &by. &timepoint. frequency;
            rename frequency=tot_evalpt;
            where _type_='11';
         run;
      %end;
      /*** end added on 12/10/2018 ****/


      proc sql;
         create table f_hbar_&y. as
         select l.&by., l.&timepoint., l.grade, l.frequency, r.tot_evalpt,
                (l.frequency / r.tot_evalpt)*100 as pctpt_ae
         from _wmxtoxfa_&y. (where=(_type_='111')) l left join getevlpt_&y. r
         on l.&by. = r.&by. and l.&timepoint. = r.&timepoint.
         having l.grade not in (0, .)
         order l.&by., l.&timepoint., l.grade;
      quit;

      **** annotate dataset;
      data f_hbar_&y._anno;
         set _w2chk_evalpt_&y. (where=(_type_='11'));
         xsys='3';
         ysys='2';
         when="a";
         midpoint=&by.;
         group=&timepoint.;
         x=97;
         text=cat(frequency);
         output;

         xsys='3';
         ysys='3';
         y=&annoymax.;
         x=96;
         text="Total N";
         output;
      run;

   %end;

   %if &datatype.=2 %then %do;

      proc sql;
         create table _wmxtox_&y.  as
         select l.dcntr_id, l.&timepoint., l.qol, l.scores, l.scrcat, r.&by.
         from qol0 (where=(upcase(qol)=%upcase("&&&txvar&y."))) l left join crse0 r
         on l.dcntr_id = r.dcntr_id
         having l.scores ^=.;
      quit;

      **** SPECIFIC QOLAE: Dataset to use in %table to generate the summary statistics per timepoint;
      %do i=1 %to &numtmptlevel.;
         proc sql;
            create table f_mxtox_&y._&i.  as
            select dcntr_id, &by., &timepoint.,
                   scores label= "%trim(&&&utxlbl&y.): %trim(&tmptlbl.) %trim(&&&tmptlvllbl&i.)",
                   scrcat label= "%trim(&&&utxlbl&y.): %trim(&tmptlbl.) %trim(&&&tmptlvllbl&i.)"
            from _wmxtox_&y.
            where &timepoint. = &&&tmptlevel&i.;
         quit;

         %outstat (dsn=f_mxtox_&y._&i.,
                   by=&by.,
                   timepoint=&timepoint.,
                   numlevel=&numlevel.,
                   myy=_&y.,
                   myz=_&i.);
      %end;

      **** Determine the evaluable patients based on those completed QOL question;
      ods output crosstabfreqs=_womxtox_&y.;
      proc freq data=_wmxtox_&y.;
         table &by.*&timepoint.;
      run;
      ods output close;

      data getevlpt_&y.;
         set _womxtox_&y.;
         keep &by. &timepoint. frequency;
         rename frequency=tot_evalpt;
         where _type_='11';
      run;

      ods output crosstabfreqs=_wmxtoxfa_&y.  (keep=&by. scrcat &timepoint. frequency _type_);
      proc freq data=_wmxtox_&y. ;
         table &timepoint.*scrcat*&by.;
      run;
      ods output close;

      proc sql;
         create table f_hbar_&y. as
         select l.&by., l.&timepoint., l.scrcat, l.frequency, r.tot_evalpt,
                (l.frequency / r.tot_evalpt)*100 as pctpt_qol
         from _wmxtoxfa_&y. (where=(_type_='111')) l left join getevlpt_&y. r
         on l.&by. = r.&by. and l.&timepoint. = r.&timepoint.
         order l.&by., l.&timepoint., l.scrcat;
      quit;

      **** annotate dataset;
      data f_hbar_&y._anno;
         set _womxtox_&y. (where=(_type_='11'));
         xsys='3';
         ysys='2';
         when="a";
         midpoint=&by.;
         group=&timepoint.;
         x=98;
         text=cat(frequency);
         output;

         xsys='3';
         ysys='3';
         y=&annoymax.;
         x=96;
         text="Total N";
         output;
      run;

   %end;

%end;


**=============================================================**;
**=============================================================**;
**=============================================================**;
**                  CREATE BAR CHARTS                          **;
**=============================================================**;
**=============================================================**;
**=============================================================**;


%if &datatype.=1 %then %do;
   pattern1 color=black     value=empty;
   pattern2 color=black     value=r1;
   pattern3 color=lightgray value=solid;
   pattern4 color=darkgray  value=solid;
   pattern5 color=black     value=solid;
%end;

%if &datatype.=2 %then %do;
   pattern1 color=black     value=solid;
   pattern2 color=darkgray  value=solid;
   pattern3 color=black     value=empty;
%end;


options nodate orientation=landscape;
ods graphics / outputfmt=png;
ods rtf file="bytimept_figures_&dtype..doc" style=journal;

goptions device=png htext=1.5 target=png xmax=9 in ymax=7 in;

axis1 value=(height=1.5) order=(0 to 100 by 10)
      label=("Percent of Patients with &dxlbl.") offset=(,12);
legend1 label=("&scrgrd.");

%do y=1 %to &numvar.;
   title1 "Figure 1_&y.: %trim(&&&utxlbl&y.) Incidence By %trim(&bylbl.)";
   %if &datatype.=1 %then %do;
      title2 "% Based on # eval pts at each &tmptlbl.";
   %end;

   %else %if &datatype.=2 %then %do;
      title2 "For Patient Experiencing the &dfigtbtlt.";
      title3 "% Based on # eval pts at each &tmptlbl.";
   %end;
   proc gchart data=f_hbar_&y. anno=f_hbar_&y._anno ;
      format pctpt_&dtype. 8.;
      hbar &by. / nostats
           sumvar=pctpt_&dtype.
           group=&timepoint.
           subgroup=&scrgrd2.
           patternid=subgroup
           raxis=axis1
           legend=legend1
           frame
           annotate=f_hbar_&y._anno
           space=0.2

      %if &bytype.=num %then %do;
         discrete
      %end;
      ;
   run;
   quit;

%end;

%if &datatype.=1 %then %do;

   axis1 value=(height=1.5) order=(0 to 100 by 10)
         label=("Percent of Patients with Maximum CTCAE Grade") offset=(,12);
   legend1 label=('Grade');

   title1 "Figure 2: Overall Adverse Events Indicidence By %trim(&bylbl.)";
   title2 "% Based on # eval pts at each &tmptlbl.";
   proc gchart data=f_ovhbar anno=f_ovhbar_anno ;
      format pctpt_ae 8.;
      hbar &by. / nostats
           sumvar=pctpt_ae
           group=&timepoint.
           subgroup=grade
           patternid=subgroup
           raxis=axis1
           legend=legend1
           frame
           annotate=f_ovhbar_anno
           space=0.2


      %if &bytype.=num %then %do;
         discrete
      %end;
      ;
   run;
   quit;

%end;

ods rtf close;
ods graphics off;


**=============================================================**;
**=============================================================**;
**=============================================================**;
**                 GENERATE SUMMARY TABLES                     **;
**=============================================================**;
**=============================================================**;
**=============================================================**;

%do y=1 %to &numvar.;
   **** Determine the level of grade/scores for each AE/QOL;
   **** The previously defined grade/scores macro variable is not accessible here;
   data _wmxsc_&y.;
      set %do z=1 %to &numtmptlevel.;
                    _wfrout_&y._&z.
                    %end;
                ;
   run;

   proc sql noprint;
       select max (&scrgrd2) into: rmxsc&y.
          from _wmxsc_&y.;
   quit;

   %if &datatype.=1 %then %do;
      proc sql;
         create table _wfqtabw_&y.
         (%do g=0 %to &&&rmxsc&y.;
             _&g. char,
          %end;
          dummv num);

      insert into _wfqtabw_&y.
          (%do g=0 %to &&&rmxsc&y.;
             _&g.,
          %end;
          dummv)

      values
          (%do g=0 %to &&&rmxsc&y.;
             "0 (0%)",
          %end;
          1)
          ;

      quit;
   %end;

   %if &datatype.=2 %then %do;
      proc sql;
         create table _wfqtabw_&y.
          (%do g=1 %to &&&rmxsc&y.;
             _&g. char,
           %end;
          dummv num);

      insert into _wfqtabw_&y.
          (%do g=1 %to &&&rmxsc&y.;
             _&g.,
           %end;
           dummv)

      values
          (%do g=1 %to &&&rmxsc&y.;
             "0 (0%)",
          %end;
          1)
          ;

      quit;
   %end;

   data f_indtb_&y.;
      set %do z=1 %to &numtmptlevel.;
          _wfqtab_&y._&z.
       %end;
       _wfqtabw_&y.
      ;


      %if &datatype.=1 %then %do;
         %do g=0 %to &&&rmxsc&y.;
            if _&g.=' ' then _&g.="0 (0%)";
            label _&g.="Grade &g.";
         %end;
      %end;
      %else %if &datatype.=2 %then %do;
         %if %eval(&&&rmxsc&y.>0) %then %do;
            if _1=' ' then _1="0 (0%)";
            label _1="0-50";
         %end;
         %if %eval(&&&rmxsc&y.>1) %then %do;
            if _2=' ' then _2="0 (0%)";
            label _2="51-74";
         %end;
         %if %eval(&&&rmxsc&y.>2) %then %do;
            if _3=' ' then _3="0 (0%)";
            label _3="75-100";
         %end;
         ;

      %end;
   run;

   data f_indtbc_&y.;
      set %do z=1 %to &numtmptlevel.;
          _wctst_&y._&z.
          %end;
      ;
   run;
%end;

%if &datatype.=1 %then %do;
    proc sql noprint;
       select max (&scrgrd2) into: ovmxsc
       from _wmxtox;
    quit;

    proc sql;
         create table _wfqtabwov
         (%do g=0 %to &ovmxsc.;
             _&g. char,
          %end;
          dummv num);

      insert into _wfqtabwov
         (%do g=0 %to &ovmxsc.;
             _&g.,
          %end;
          dummv)

      values
         (%do g=0 %to &ovmxsc.;
             "0 (0%)",
          %end;
          1)
       ;

      quit;

   data f_ovtb;
      set %do z=1 %to &numtmptlevel.;
          _wfqtab_&z.
          %end;
          _wfqtabwov
          ;

      %do g=0 %to &ovmxsc.;
         if _&g.=' ' then _&g.="0 (0%)";
         label _&g.="Grade &g.";
      %end;
   run;

   data f_ovtbc;
      set %do z=1 %to &numtmptlevel.;
             _wctst_&z.
          %end;
      ;
   run;
%end;

options nodate orientation=landscape;
ods rtf file="bytimept_tbl_&dtype..doc" style=journal bodytitle;

   %do y=1 %to &numvar.;
      ods startpage=no;
      title1 "Table A_&y.: Mean &scrgrd. of %trim(&&&utxlbl&y.) Summary by &bylbl.";
      proc report data=f_indtbc_&y. nowd headline headskip split='*';
         column &timepoint.
                   %do g=1 %to &numlevel.;
                      ("%trim(&&&bylvllbl&g.)" n&g. mn&g.)
                   %end;
                   ("Total" n mn) pval;
         define &timepoint. / display;

         %do g=1 %to &numlevel.;
            define n&g. / display;
            define mn&g. / display;
         %end;

         define n / display;
         define mn / display;
         define pval / display;
      run;

      title1 "Table B_&y.: Frequency per &scrgrd. of %trim(&&&utxlbl&y.) Summary by &bylbl.";
      proc print data=f_indtb_&y. label noobs;
         where dummv=.;
         var &timepoint. &by.
            %if &datatype.=1 %then %do;
               %do i=0 %to &&&rmxsc&y.;
                  _&i.
               %end;
            %end;
            %else %if &datatype.=2 %then %do;
               %do i=1 %to &&&rmxsc&y.;
                  _&i.
               %end;
            %end;
            frequency pval;
      run;

      ods startpage=now;
   %end;

   %if &datatype.=1 %then %do;
      title1 "Table C: Overall Adverse Events Mean Summary by &bylbl.";
      proc report data=f_ovtbc nowd headline headskip split='*';
         column &timepoint.
                   %do g=1 %to &numlevel.;
                      ("%trim(&&&bylvllbl&g.)" n&g. mn&g.)
                   %end;
                   ("Total" n mn) pval;
         define &timepoint. / display;

         %do g=1 %to &numlevel.;
            define n&g. / display;
            define mn&g. / display;
         %end;

         define n / display;
         define mn / display;
         define pval / display;
      run;

      title1 "Table D: Overall Adverse Events Frequency per Grade Summary by &bylbl.";
      proc print data=f_ovtb label noobs;
         where dummv=.;
         var &timepoint. &by.
             %do i=0 %to &ovmxsc.;
                _&i.
             %end;
            frequency pval;
      run;
   %end;

ods rtf close;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                        REMOVE WORKING DATA SET                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;

proc datasets library=work nolist;
   delete _w: tb1: ;
quit;


**--------------------------------------------------------------------------**;
******************************************************************************;
*****                 EXIT THE PROGRAM WHEN ERROR OCCUR                  *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
%exit:

**--------------------------------------------------------------------------**;
******************************************************************************;
*****                       DISPLAY THE MACRO STATUS                     *****;
******************************************************************************;
**--------------------------------------------------------------------------**;
data _null_;
   put "---------------------------";
   %if &myErrMsg. ^= %then %do;
      put "ERROR: " &myErrMsg.;
   %end;
   put "---------------------------";
run;


%mend bytimept;
