package edu.mayo.genotype;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.Iterator;
import java.util.Collections;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.HashSet;
import java.util.TreeSet;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.Map.Entry;


/**
 * This is not the most elegant object oriented code, but I am trying to use native data
 * types to speed-up the dynamic search.
 * 
 * @author Hugues Sicotte
 *
 */
public class Cluster {


	private int clusterid;
	private static int nclusters=0;
	private int length; // numbers of bins
	private Bin[] bins=null;
	private int[] binId2BinIndex =null;
	private BinEdge[] edges=null;
	private Snp[] snps = null; // contains all the tag SNPs in a cluster.
							   // XXX- Current Cluster constructor only stores SNPs in Edges.
	private HashMap<String,ArrayList<BinEdge>> snp2Edges=null; // Map by string of snpid to list of bin-edges
	private int type=0; //0 - empty, 1- single, 2-pair, 3- chain, 4- fishbone, 5- complex) [any with >=2 can be processed by same alg.]
	private int status=0; // 0- in process - 1 tag-snps picked
	
	private static HashMap<String,Cluster> clusters = new HashMap<String,Cluster>();
	private static HashMap<String,Cluster> snpId2Cluster= new HashMap<String,Cluster>();
	private Comparator<Snp> snpOrderComp= new SnpOrderComparator();
	private TreeSet<Snp> orderedSnps = new TreeSet<Snp>(snpOrderComp);
	private HashMap<Snp,LinkedList<Snp>> snpToRangeSnps = new HashMap<Snp,LinkedList<Snp>>();;

	
	private static boolean debug = true;
	private static int maxtime = SNPPicker.maxCPU;// Maximum time to Spend on one cluster(10 minutes is default) .. returns with best solution so far.

	private long starttime=0L;
	private long lasttime=0L;
	
	private int[] temp_edgeWeights=null; // set to 2 for edge that is the only link to two other ones.

	private int nsolns=0; // Number of Solutions Considered for Exhaustive search.
	private boolean isSemiOptimal=false; // To track clusters with semi-optimal solution.
	private static int nSemiOptimal=0; // used by compareSemiOptimalWithRandom to figure out a filename to store random solutions
	
	/**
	 * @return the isSemiOptimal
	 */
	private boolean isSemiOptimal() {
		return isSemiOptimal;
	}

	private SolutionTracker st = null;
	
	public static boolean isDebug() {
		return debug;
	}
	
	private static void buildSnp2ClusterMap() {
		for(int i=1;i<=nclusters;i++) {
			String cid = Integer.toString(i);
			Cluster c = clusters.get(cid);
			if(c !=null && c.snps!=null) {
				for(int j=0;j<c.snps.length;j++) {
					Snp s = c.snps[j];
					if(s!=null) {
						snpId2Cluster.put(Integer.toString(s.getId()), c);
					}
				}
			}
		}
	}
	public static void setDebug(boolean debug) {
		Cluster.debug = debug;
	}

	public static int getMaxtime() {
		return maxtime;
	}

	public static void setMaxtime(int maxtime) {
		Cluster.maxtime = maxtime;
	}
	public Cluster(Bin[] bs) throws Exception {
		this.bins=bs;
		this.length=bs.length;
		Cluster.nclusters++;
		this.clusterid=nclusters;
		if(this.length==1) {
			this.type=1;
			//			System.out.println("Singleton Cluster");
		} else if (this.length==2) { // Cluster with 2 bins.
			this.type=2;
			//			System.out.println("Pair Cluster");
		} else {
			//			System.out.println("Complex Cluster");
			this.type=5;
		}
		String cid = Integer.toString(clusterid);
		Cluster.clusters.put(cid,this);

		if(this.length>1) {// Cannot have length ==1 and an edge.
			// Count Unique BinEdges
			int nedges = BinEdge.getNbinedges();
			if(nedges==0) {return;}


			int[] beids=new int[nedges+1];

			for(int i=0;i<this.length;i++) {
				Bin b = bs[i];
				if(b!=null) {
					int a=1;
					if(b.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:38") ||
							b.getName().equals("CEU-GPC6|10082;SNPApp_EUR_0.90_0.05_1228_1231Hapmap10082ldselect.out/13:34") ||
							b.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:536") ||
							b.getName().equals("CEU-STAT1|6772;SNPApp_EUR_0.90_0.05_1228_1231Hapmap6772ldselect.out/2:18") ||
							b.getName().equals("AFR-STAT1|6772;SNPApp_AFR_0.90_0.05_1228_1232Hapmap6772ldselect.out/2:35")
					) {

						a++; // debugger breakpoint.
					}
					BinEdge[] bes = b.getBinedges();
					if(bes!=null) {
						for(int j=0;j<bes.length;j++) {
							BinEdge be = bes[j];
							if(be!=null) {
								beids[be.getBinedgeid()]=1;
							}
						}
					}
				}
			}

			// Finish Setting Status
			int nbeis=0;
			for(int i=0;i<=nedges;i++) {
				nbeis+=beids[i];
				beids[i]=0;
			}

			// Fill up with Unique BinEdges

			this.edges = new BinEdge[nbeis]; 
			this.temp_edgeWeights= new int[nbeis]; 

			for(int i=0;i<nedges;i++) {beids[i]=0;} // 3-30-2010. Fixed Important Bug - only relevant for greedy solution of old code.

			nbeis=0;
			int a=1;
			for(int i=0;i<this.length;i++) {
				Bin b = bs[i];
				if(b!=null) {
					if(b.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:38") ||
							b.getName().equals("CEU-GPC6|10082;SNPApp_EUR_0.90_0.05_1228_1231Hapmap10082ldselect.out/13:34") ||
							b.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:536") ||
							b.getName().equals("CEU-STAT1|6772;SNPApp_EUR_0.90_0.05_1228_1231Hapmap6772ldselect.out/2:18") ||
							b.getName().equals("AFR-STAT1|6772;SNPApp_AFR_0.90_0.05_1228_1232Hapmap6772ldselect.out/2:35")
					) {

						a++; // debugger breakpoint.
					}
					BinEdge[] bes = b.getBinedges();
					if(bes!=null) {
						for(int j=0;j<bes.length;j++) {
							BinEdge be = bes[j];
							if(be!=null) {
								if(beids[be.getBinedgeid()]==0) {
									beids[be.getBinedgeid()]=1;
									int old_ord = be.getOrder();
									if(old_ord==-1) {// this means this bin edge has not already been added.
										this.edges[nbeis++]=be;
										be.setOrder(nbeis-1); // needed for Searching
										Snp[] snps = be.getSnpids();
										be.setSnpmaxpop(0);
										be.setSnpmaxpopscore(0);
										be.setSnpidmaxpop(0);
										if(snps!=null && snps.length>0) {
											int maxpop=0;
											double old_maxScore=-1.0;
											double old_rawScore=Double.NEGATIVE_INFINITY;
											int besnps = be.getNsnps();
											for(int k=0;k<besnps;k++) {
												Snp s = snps[k];

												if(!(s.isExcluded() || s.isSelected() || s.isObligate() || s.isObligateButNotGenotyped())) {
													int npop=s.getNPop();
													double s_score = s.getSuccessProb();
													if(npop>maxpop || (npop==maxpop && s_score>old_maxScore)
															|| (npop==maxpop && s_score==old_maxScore && s.getScore()>old_rawScore)		
													) {
														be.setSnpmaxpop(npop);
														be.setSnpidmaxpop(s.getSnpid());
														be.setSnpmaxpopscore(s_score);
														old_maxScore=s_score;
														old_rawScore = s.getScore();
													} 
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
			this.populateSnpArrayInCluster(); // Populates all SNPs in cluster to this.snps and orderedSnps.

			//
			// For each SNP (in order) need the Snp Object of the first and last SNP in the block of SNps.
			// subSet(E fromElement,boolean fromInclusive, E toElement,boolean toInclusive)
			//

			int nsnps = this.orderedSnps.size();

			Iterator<Snp> it =this.orderedSnps.iterator();
			if(nsnps>0 && it!=null) {
				int lastPos = -(SNPPicker.tooClose+1);
				Snp lastSNP=null;
				Snp sfirst =  this.orderedSnps.first();
				if(sfirst!=null) {
					String lastChr =sfirst.getChromosome();
					int stackIndx=-1;
					Snp[] snpStack = new Snp[nsnps];
					while(it.hasNext()) {
						Snp s = it.next();
						int newPos = s.getPosition();
						String chr = s.getChromosome();
						if(chr!=null && chr.length()>0 && lastChr!=null && lastChr.length()>0 && !lastChr.equals(chr)){
							String msg="SNPPicker: ERROR: Two SNPs in same cluster SNP "+s.getSnpName()+"(bins " + s.getBinsString()+") has a conflicting chromosome assignment than SNP "+sfirst.getSnpName()+" (bins "+sfirst.getBinsString() +")";
							StringBuffer bf = new StringBuffer("Cluster bins");
							Iterator<Snp> it2 =this.orderedSnps.iterator();
							while(it2.hasNext()) {
								Snp s2 =  it2.next();
								if(s2!=null) {
									bf.append("\n");bf.append(s2.getBinsString());
								}
							}
							System.err.println(msg+bf.toString());
							System.err.flush();
							throw new Exception(msg+bf.toString());
						}
						if((!chr.equals(lastChr)) || newPos-lastPos>=SNPPicker.tooClose) {
							// pop stack
							if(stackIndx>=0) {
								LinkedList<Snp> l = new LinkedList<Snp>();
								l.add(snpStack[0]);
								l.add(snpStack[stackIndx]); 

								for(int i=0;i<=stackIndx;i++) {
									Snp ss = snpStack[i];
									snpToRangeSnps.put(ss,l);
								}
							}
							stackIndx=-1;
						} else {
							// push SNP in stack.
							if(newPos-lastPos<SNPPicker.tooClose) {
								if(stackIndx==-1) {
									snpStack[0]=lastSNP;
									stackIndx=0;
								}
								stackIndx++;
								snpStack[stackIndx]=s;
							}
						}
						lastChr = chr;
						lastPos = newPos;
						lastSNP=s;
					}
					// Store terminal segment
					if(stackIndx>=0) {
						LinkedList<Snp> l = new LinkedList<Snp>();
						l.add(snpStack[0]);
						l.add(snpStack[stackIndx]); 

						for(int i=0;i<=stackIndx;i++) {
							Snp ss = snpStack[i];
							snpToRangeSnps.put(ss,l);
						}
					}
					
				}
			}

		}

	/*
	 * Get the list of Too Close SNPs  in a cluster for each SNPs (includes itself).
	 */
	public TreeSet<Snp> getNeighborsSubset(Snp s) {
		LinkedList<Snp> l = snpToRangeSnps.get(s);
		TreeSet<Snp> posCluster = null;
		if(l!=null) {
			posCluster = (TreeSet<Snp>)this.orderedSnps.subSet(l.get(0), true,l.get(1),true);
		}
		return posCluster;
	}

	public static void buildClusters() throws Exception {
		Bin[] bs = Bin.getBinHeads(); // the clustering is already done 
		if(bs!=null) {
			for(int i=0;i<bs.length;i++) {
				Bin bhead=bs[i];
				int nchilds=1;
				Bin b1=bhead;
				while(b1.getNextbin()!=null) {
					b1=b1.getNextbin();
					nchilds++;
				}
				Bin[] bb = new Bin[nchilds];
				b1=bhead;
				bb[0]=bhead;
				nchilds=1;
				while(b1.getNextbin()!=null) {
					b1=b1.getNextbin();
					bb[nchilds++]=b1;
				}
				new Cluster(bb);
			}
		} else {
			System.err.println("SNPPicker:builClusters: WARNING: No clusters were built\n");
			System.err.flush();
		}
		Cluster.buildSnp2ClusterMap();
	}
	
	
	/**
	 * @return the bins
	 */
	public Bin[] getBins() {
		return this.bins;
	}


	/**
	 * @return the clusterid
	 */
	public int getClusterid() {
		return clusterid;
	}



	/**
	 * @return the clusters
	 */
	public HashMap<String,Cluster> getClusters() {
		return clusters;
	}



	/**
	 * @return the edges
	 */
	public BinEdge[] getEdges() {
		return this.edges;
	}



	/**
	 * @return the length
	 */
	public int getLength() {
		return length;
	}

	

	/**
	 * @return the status ( 1 if cluster has been picked)
	 */
	public int getStatus() {
		return status;
	}

	/**
	 * @param status the status to set
	 */
	public void setStatus(int status) {
		this.status = status;
	}

	/**
	 * @return the type
	 */
	public int getType() {
		return type;
	}

	/**
	 * @param type the type to set
	 */
	public void setType(int type) {
		this.type = type;
	}

	

	
	/*
	 * Iterate over all possible choices of up to NSB tag SNPs
	 * XXX-HS I do not think this function works.. it was never finished
	 */
	private void pickBinExhaustive(Bin b,boolean tmpToo) throws Exception {
		if(b==null) {
			return;
		}
		// If Called from Cluster PickBinEdgesInternal, bin status includes temporarely picked tag SNPs.
		ClusterState cs = new ClusterState(this);
		if(b.getStatus()==1) {
			return; // no more tagSNPs to pick or avail to pick.
		}

		// Assume that we need to pick at least 1 more.
		b.cacheMaxCoverage(tmpToo);

		if(SNPPicker.verbose) {
			System.out.println("SNPPicker:INFO: Picking bin "+b.getName());System.out.flush();
		}
		int[] ss = b.getMaxCoverageSnpids(); // sorted by Max Coverage, then prob, then func, then score
		if(SNPPicker.verbose) {
			System.out.println("SNPPicker:INFO: "+b.getName()+" started with "+(ss==null ? 0: ss.length)+" max coverage SNPids");System.out.flush();
			System.out.println("SNPPicker:INFO: "+b.getName()+" start status="+b.getStatus());System.out.flush();
		}
// Solution Tracker is needed if we are going to score each bin independently.
		
		SolutionTracker soln = new SolutionTracker();
		bins[0]=b;
		int nNeeded =b.getNSB()-b.getNSelected();
		int nAvailable = b.getNsnps()-b.getNExcluded();
		// Total tagSNPs that have to choose.
		int nRealisticallyNeeded = Math.min(nNeeded, nAvailable);

		Bin[] bins = new Bin[nRealisticallyNeeded];
		for(int i=0;i<nRealisticallyNeeded;i++) {
			bins[i]=b;
		}
		
		int nLeft = nAvailable-b.getNSelected();
		// Additional tagSNPs left to Select.
		int wantedMAXSNPs = Math.min(nNeeded,nLeft);

//		Enumerate all possible combinations of remaining tag SNPs to pick in this bin.

		if(ss!=null && ss.length>0) {
			int size = ss.length; // Limit to 25 tag SNPs per bin.
			if(size>25) {size=25;} // only consider first 25 tag SNPs... Unrealistic case.. but just code it.
			Snp[] soln_snps = new Snp[size];

			LinkedHashSet<Integer> sss = new LinkedHashSet<Integer>();
			for (int i = 0;i<size;i++ ) {
				sss.add(Integer.valueOf(ss[i]));
				soln_snps[i] = Snp.getSnpById(ss[i]);
			}
			PowerSet<Integer> pow = new PowerSet<Integer>(sss);
			Iterator<PowerSet<Integer>.BitMaskSet>  it = pow.iterator();
			while(it.hasNext()) {
				edu.mayo.genotype.PowerSet<Integer>.BitMaskSet bm = it.next();
				int nelems = bm.size();
				if(nelems<=wantedMAXSNPs) {
					// Note: The solution are in numerical order of binary representations
					// of integers. For a given nelems, the solutions are therefore considered
					// in order that respects the rank.
					
					int[] indexes = new int[nelems];
					Iterator<Integer> itp = bm.iterator();
					int i=0;
					while(itp.hasNext()) {
						int sid = itp.next().intValue();
						indexes[i++]=sid;
						soln_snps[sid].setTmpSelected(true, true);
					}

					// By the time this function is called, all the remaining
					// choices of tag SNP are independent.
					// Evaluating the score for the bin only would may lead to a different
					// solution than evaluating within the context of the cluster mainly
					// in the case of Infinium .. and because we allow less
					// than NSB tag SNPs to be selected.
					
					SolutionTracker.updateTmpScore(bins,soln);
					//this.nsolns++;
					//this.pickBestSoFar();// Move to best-so-far soln. No resetting is done.

					//To Undo Selection.
					while(i>0) {
						soln_snps[indexes[--i]].setTmpSelected(false, true);
					}
					// Since we do not touch the bin or edge status, we do not need
					//    to restore the cluster state.
				}
			}
		}
		this.restoreState(cs);
	}
	
	


	

	/**
	 * 
	 * @return The number of Complex Bins
	 */
	public static int countComplexClusters() {

		int nbins=0;
		Set<Map.Entry<String, Cluster>> s = clusters.entrySet();
		for (Map.Entry<String, Cluster> me : s) {	  
			Cluster c = me.getValue();
			if(c.getType()==5) {
				nbins++;
			    int csize = c.getLength();
			    int cid = c.getClusterid();
			    if(SNPPicker.verbose) {
			         System.out.println("Complex cluster "+cid+" of size "+csize);
			    }
			}
		}
		return nbins;
	}
	
	/**
	 * Picks SNPs in a Single Bin or where all bins are to be independently picked.
	 *
	 */
	public void pickAllBinsIndependently(boolean tmpToo)  throws Exception {
			Bin[] bs = this.getBins();
			if(bs!=null) {
				for(int i=0;i<bs.length;i++) {
					Bin b = bs[i];
					if(b!=null) {
						if(this.getType()==1) {
							this.unFixAndUnexcludeTooCloseTypeI();
						}
						if(SNPPicker.greedy) {
							b.pickBinGreedy(false,tmpToo);
						} else {
							b.pickBinByProb(false,tmpToo);
						}
					}
				}
			}
			this.setStatus(1);
	}
	/**
	 * Pick Singleton bins (and unfix TooClose)
	 * @return the number of singleton bins
	 */
	
	public static int pick1Clusters(boolean tmpToo) throws Exception {
		int optbins=0;
		Set<Map.Entry<String, Cluster>> s = clusters.entrySet();
		for (Map.Entry<String, Cluster> me : s) {	  
			Cluster c = me.getValue();
			if(c.getType()==1 && c.getStatus()==0) {
				optbins++;
				c.pickAllBinsIndependently(tmpToo);
			}
		}
		return optbins;
	}
	
	public static int pickBinByType(int t_type) throws Exception {
		int noptbins=0;
		Set<Map.Entry<String, Cluster>> s = clusters.entrySet();
		System.out.println("mesize="+s.size());System.out.flush();
		for (Map.Entry<String, Cluster> me : s) {	  
			Cluster c= me.getValue();
//			System.out.println("pbbt_me");System.out.flush();
			if(c!=null) {
				if(c.getType()==t_type && c.getStatus()==0) {
					noptbins++;
					if(SNPPicker.verbose) {
						System.out.println("Picking cluster of size "+c.length);
					}
					long cpstart = System.currentTimeMillis();
					if(t_type==1) {
						System.out.println("pbbt_unfix");System.out.flush();
						c.unFixAndUnexcludeTooCloseTypeI();
					}
					System.out.println("pbbt_cpb");System.out.flush();
					c.pickBins();
					System.out.println("pbbt");System.out.flush();
					//				if(SNPPicker.verbose) {
					System.out.println("Picked cluster of size "+c.length+" in "+(System.currentTimeMillis()-cpstart)+" ms");
					//				}
				}
			}
		}
		System.out.println("pbbt_X");System.out.flush();
		return noptbins;
	}
	
	/*
	 * Function to search for an optimal solution.
	 * 	unfix=true means to unfix fixed SNPs (unless they were excluded), prior to optimization.
	 *    these fixed SNPs are the too-close tag SNPs, and type II tag SNPs.
	 *  finalOptimize means to optimize all not-fixed tag SNPs.
	 *  if finalOptimize=false, only attempt to swap in the solution the newly unfixed tag SNPs.
	 * Choices are 
	 * 		unfix && finalOptimize, - Optimize All
	 * 		unfix && ! finalOptimize,  - Only optimize the newly unfixed SNPs.[ Good for adding type II in the solution]
	 * 		!unfix && !finalOptimize  This doesn't do anything
	 *      !unfix && finalOptimize - Only optimize not fixed SNPs
	 */
	public static void unFixAndOptimizeSolution(boolean unfix,boolean finalOptimize,boolean lastCall) {
		if(!(unfix || finalOptimize)) {
			return;
		}

		Set<Map.Entry<String, Cluster>> s = clusters.entrySet();
		for (Map.Entry<String, Cluster> me : s) {	  
			Cluster c= me.getValue();
			Snp[] notSelectedSnps = null;
			boolean noNewSnps=true;
			if(unfix && !finalOptimize) {// Only optimize SNPs that were newly unfixed by the calling function
				notSelectedSnps = c.getNotSelectedNorFixedSnps(); 
			}
			if(unfix) { // Are there any SNPs to unfix ? Unfix them.
				int nsuTC=0;
				if(SNPPicker.tooClose>0) {
					nsuTC = c.unFixAndUnexcludeTooCloseTypeI();
				}
				int nsuII= c.unFixAndUnexcludeTypeII();
				if(nsuTC>0 || (nsuII>0)) {
					noNewSnps=false;
				}
			}
//			System.out.println("au");System.out.flush();
 			if(unfix && !finalOptimize) {// refix SNPs
 				c.fixSnps(notSelectedSnps,true); // Fix SNPs (to prevent them being selected) that were not chosen for inclusion in pre-optimization.
 			}
//			System.out.println("ufof~");System.out.flush();
			if(unfix  || finalOptimize) {
//				System.out.println("ufof+");System.out.flush();
				c.findOptimal(SNPPicker.getMaxCPU(),true,lastCall);
//				System.out.println("ufof-");System.out.flush();
			}
			if(unfix && !finalOptimize) {
				c.fixSnps(notSelectedSnps,false); // UnFix SNPs that were not chosen for inclusion in pre-optimization.
			}
		}
	}

	/**
	 * 
	 * @param firstPick Set to True if this is this is the first time we pick SNPS
	 * @param unfixPick Set to True if want to attempt adding/swapping fixed SNPs back in Solution. THis function will unfix the fixed SNPs.
	 * @param optimalPick Set to True if want to perform exhaustive search of swapping all not-in-solution SNPs back in the solution (at the cost of removing some in-solution SNPs)
	 *       
	 * @return
	 * @throws Exception
	 */
	public static int pickAll(boolean firstPick, boolean unfixPick, boolean optimalPick, boolean lastCall) throws Exception {
		int n1= 0;
		int n2=0;
		int n5=0;
		
		if(firstPick) { // initial picking of tag SNPs
			n1= Cluster.pick1Clusters(false); // pick real
		
			if(SNPPicker.verbose) {System.out.println("Found "+n1+" Single Bins solutions");}

			n2 = Cluster.pickBinByType(2); // Could easily create a special function for 2-bin clusters.
			if(SNPPicker.verbose) {System.out.println("Found "+n2+" Paired Bins solutions");}

			n5 = Cluster.pickBinByType(5);
			//if(SNPPicker.verbose) {
				System.out.println("Finished "+n5+" Complex Bins");
			//}
		}
		if(unfixPick || optimalPick) {
			System.out.println("ufo+");System.out.flush();
			unFixAndOptimizeSolution(unfixPick,optimalPick,lastCall);
			System.out.println("ufo-");System.out.flush();
		}
		return (n1+n2+n5);
	}

	/**
	 * 
	 *  @return same as countAvailableBins, except that it also sets bin status based on fully selected SNPs
	 */

	private int countConsistentBinsStatus(boolean tmpToo) {
		int nremain=0;
		for(int i=0;i<bins.length;i++) {
			Bin b = bins[i];
			if(b!=null) {
				int _nExcluded = b.getNExcluded();
				int nsel = b.getNSelected();
				if(((b.getNsnps()-_nExcluded) >nsel) && nsel<b.getNSB()) {
					b.setStatus(0);
				}
				if( ((!tmpToo) && b.getNSB()>b.getNSelected()) || (tmpToo && b.getNSB()>b.getTmpNSelected() )) {
					if(b.getAvailableSnpIds(tmpToo)!=null) {
						nremain++;
					} else {
						b.setStatus(1);
					}
				} else {
					b.setStatus(1);
				}
			}
		}
		return nremain;
	}
	
	/**
	 * 
	 *  @return same as countAvailableBins, except that it also sets bin status based on fully selected SNPs
	 */
	private int[] getTmpSelectedSnps() {

		HashSet<Integer> selsnps = new HashSet<Integer>();
		for(int i=0;i<this.bins.length;i++) {
			Bin b = this.bins[i];
			if(b!=null) {
				int[] snpsids = b.getTmpSelectedSnpIds();
				if(snpsids!=null) {
					for(int j=0;j<snpsids.length;j++) {
						int sid = snpsids[j];
						if(sid>0) {
							Integer selsnp = Integer.valueOf(sid);
							selsnps.add(selsnp);
						}
					}
				}
			}
		}
		int nsel = selsnps.size();
		if(nsel==0) {
			return null;
		} else {
			int[] sels = new int[nsel];
			Iterator<Integer> it = selsnps.iterator();
			int i=0;
			while(it.hasNext()) {
				Integer selsnp = it.next();
				sels[i++]=selsnp.intValue();
			}
			return sels;
		}
	}
	
	/**
	 * 
	 * Count binedges where both bins have  have not yet completely been chosen. (if 1 bin fully chosen, don't count binedge)
	 */

	private int countAvailableEdges(boolean tmpToo) {
		int nremain=0;
		for(int i=0;i<edges.length;i++) {
			BinEdge be = edges[i];
			if(be!=null) {
				nremain +=updateAvailableEdges(be,tmpToo);
			}
		}
		return nremain;
	}
	
	private int updateAvailableEdges(BinEdge be,boolean tmpToo) {
		
			// It would be faster to simply check the Valid Flag.. but right now the code does not
			// deal with the status and TMP selected SNPs.
			if(be==null) {
				return 0;
			}
			Bin b1 = Bin.getBinById(be.getBinid1());
			Bin b2 = Bin.getBinById(be.getBinid2());
			Snp[] snps = be.getSnpids();
			boolean valid=false;
			if(snps!=null) {
				for(int j=0;j<snps.length;j++) {
					Snp s = snps[j];
					if(s!=null && !(((!tmpToo) && s.isSelected()) || s.isExcluded() || (tmpToo && s.isTmpSelected()) )) {
						valid=true;
						break;// at least 1 SNP available
					}
				}
			}
			be.setValidflag(1); // assume invalid (==don't need more snps) until proven otherwise.
			if(valid) {// at least 1 SNP available
				double pthreshold = SNPPicker.minP; // cache the minP to a local variable.
				if(SNPPicker.verbose) {
					System.out.println("tmpToo="+(tmpToo ? "true":"false")+",b1.nsb="+b1.getNSB()+",b1.nsnps="+b1.getNsnps()+",b1.ex="+b1.getNExcluded()+",b1.nsel="+b1.getNSelected()+",b1.tmpNsel="+b1.getTmpNSelected());
					System.out.println("tmpToo="+(tmpToo ? "true":"false")+",b2.nsb="+b2.getNSB()+",b2.nsnps="+b2.getNsnps()+",b2.ex="+b2.getNExcluded()+",b2.nsel="+b2.getNSelected()+",b2.tmpNsel="+b2.getTmpNSelected());
				}
				if( (((tmpToo && (b1.getNSB()>b1.getTmpNSelected())) || ((!tmpToo) && (b1.getNSB()>b1.getNSelected()))) && b1.getAvailableSnpIds(tmpToo)!=null)  
						|| (((tmpToo && (b2.getNSB()>b2.getTmpNSelected())) || ((!tmpToo) && (b2.getNSB()>b2.getNSelected()))) && b2.getAvailableSnpIds(tmpToo)!=null)) {
					if(pthreshold<1.0) {// Threshold option was activated.
						boolean b1Need=true;
						boolean b2Need=true;
						if(tmpToo) {
							if(b1.getTmpProb()>=pthreshold) {
								b1Need=false; // b1 doesn't need any more tagSNPs
							} 
							if (b2.getTmpProb()>=pthreshold) {
								b2Need=false; // b2 doesn't need any more tagSNPs.
							}
						} else {
							if(b1.getProb()>=pthreshold) {
								b1Need=false; // b1 doesn't need any more tagSNPs
							} 
							if (b2.getProb()>=pthreshold) {
								b2Need=false; // b2 doesn't need any more tagSNPs.
							}
						}
						
						if(b1Need || b2Need) {
							//At least one Bin needs more SNPS.
							if(SNPPicker.verbose) {
								System.out.println("P:Need at least one SNP for bins "+b1.getName()+","+b2.getName());
							}
							if( ((!tmpToo) && ((b1.getNsnps()-b1.getNExcluded() > b1.getNSelected()) 
									&& (b2.getNsnps()-b2.getNExcluded() > b2.getNSelected())))
									||
									((tmpToo) && ((b1.getNsnps()-b1.getNExcluded() > b1.getTmpNSelected()) 
											&& (b2.getNsnps()-b2.getNExcluded() > b2.getTmpNSelected()))) ) {
							//Both bins have more available SNPS.
								if(SNPPicker.verbose) {
									System.out.println("P:Both bins have available bins. "+b1.getName()+","+b2.getName());
								}
								be.setValidflag(0);
								return 1;
							} 
						}
					} else {
						if(SNPPicker.verbose) {
							System.out.println("Need at least one SNP for bins "+b1.getName()+","+b2.getName());
						}
						if( ((!tmpToo) && ((b1.getNsnps()-b1.getNExcluded() > b1.getNSelected()) 
								&& (b2.getNsnps()-b2.getNExcluded() > b2.getNSelected())))
								||
								((tmpToo) && ((b1.getNsnps()-b1.getNExcluded() > b1.getTmpNSelected()) 
										&& (b2.getNsnps()-b2.getNExcluded() > b2.getTmpNSelected()))) ) {
							//Both bins have more available SNPS.
							if(SNPPicker.verbose) {
								System.out.println("Both bins have available bins. "+b1.getName()+","+b2.getName());
							}
							be.setValidflag(0);
							return 1;
						} 
					}
				}
			} else {
				if(SNPPicker.verbose) {
					System.out.println("no available SNPs for bins "+b1.getName()+","+b2.getName()+", where snps is "+(snps==null ? "null" : "not null")+", and has nsnps="+be.getNsnps());
					if(snps!=null) {
						System.out.print("snps: ");
						for(int i=0;i<snps.length;i++) {
							if(snps[i]!=null) {
								System.out.print(snps[i].getSnpName());
							}
						}
						System.out.println("");
					}
				}
			}
		return 0;
	}
	
	
	
	/**
	 * 
	 * Count edges that have not yet been fully chosen.
	 * Set Status of bins and bin-edges. if (tmpToo==true) take temporarily chosen SNPs into account.
	 */

	private int countConsistentEdgesStatus(boolean tmpToo) {
		int nremain=0;
		for(int i=0;i<edges.length;i++) {
			BinEdge be = edges[i];
			if(be!=null) {
				Bin b1 = Bin.getBinById(be.getBinid1());
				Bin b2 = Bin.getBinById(be.getBinid2());
				Snp[] snps = be.getSnpids();
				int besnps = 0;
				if(snps!=null) {
					besnps=snps.length;
				} else {
					continue;
				}
				// only set Status based on permanently selected SNPs

				int nsel1 = b1.getNSelected();
				b1.computeStatus();
				int nsel2 = b2.getNSelected();
				b2.computeStatus();
				
				if(((!tmpToo) && b1.getNSB()>nsel1) || (tmpToo && b1.getNSB()>b1.getTmpNSelected())) {
					
						int[] ssids = b1.getAvailableSnpIds(tmpToo);
						if( ssids==null || ssids.length==0) {
							b1.setStatus(1); // no more snps to choose
						}
				}
				if(((!tmpToo) && b2.getNSB()>nsel2) || (tmpToo && b2.getNSB()>b2.getTmpNSelected())){
						int[] ssids = b2.getAvailableSnpIds(tmpToo);
						if ( ssids==null || ssids.length==0) {
							b2.setStatus(1); //  no more snps to choose
						}
				}
				boolean valid=false;
				if(snps!=null) {
					for(int j=0;j<besnps;j++) {
						Snp s = snps[j];
						if(s!=null && !(((!tmpToo) &&s.isSelected()) || s.isExcluded() || s.isFixed() || (tmpToo && s.isTmpSelected()) )) {
							valid=true;
							break;// at least 1 SNP available
						}
					}
				}
				// we check the tmp-dependent status of bins here..
				be.setValidflag(1);//unavailable.
				if(valid) {// at least 1 SNP available
					if(b1.getStatus()==0 ) {
						if(b2.getStatus()==0) {
							//Both bins still need more SNPs.(and have more SNPS left to pick).. with tmpToo taken into account.
							be.setValidflag(0);// available.
							nremain++;
						}
					}
				}
			}
		}
		return nremain;
	}
	
	private static int dplevel=0;
	
	/**
	 * 
	 * @return false if aborted because of cpu time limit.
	 * 
     * Recursive function to pick SNPs in an aggressive cluster breakup algorithm
	 */
	private boolean pickBinEdgesInternal(int[] subset, ClusterState cs) {
		dplevel++;
		
		// Find the maximal number of populations that any remaining SNP touches.
		if(subset==null || subset.length==0) {
			dplevel--;
			return false; // Normally this should not be recursively calle with a null subset.
		}
		int maxGroup=-1;
		for( int i=0;i<subset.length;i++) {
			int ipicked = subset[i];
            BinEdge picked = edges[ipicked];
            int group = picked.getSnpmaxpop();
            // note that maxpop does not get recomputed.
            if(group>maxGroup) {
            	maxGroup=group;
            }
		}
		int nFound=0;
		while(nFound==0 && maxGroup>0) {
			for( int i=0;i<subset.length;i++) {
				int ipicked = subset[i];
				BinEdge pickedBE = edges[ipicked];
				int group = pickedBE.getSnpmaxpop(); // Number of Bins the most promiscuous SNP touches in this Edge.
				if(group==maxGroup) {
					if(SNPPicker.verbose) {
						System.out.println("group="+group);
					}
					// Find subset of SNPs who touch the most Pops.
					// Because these snps HAVE to be selected first, only need to consider
					// edges where they are going to be picked first.

					// Temporately pick Snp.

					//XXX for looping will loop through solutions in order of best SNPs.
					int bestpickedsnp=pickedBE.pickBestTouchSnp();
					if(bestpickedsnp>=0) { // Picked SNP Temporarely touching more pops.
						nFound++;
						Snp bestPickedSnp = pickedBE.getSnpids()[bestpickedsnp];
						if(SNPPicker.verbose) {
							System.out.println("Found a SNP in a binedge :"+bestPickedSnp.getSnpName());
						}
						// 
						// Count edges that have not yet been (temporarely) chosen and set status 
						// of bins and bin edges.
						int currentEC = countConsistentEdgesStatus(true);

						//find remaining edges part of complex clusters.
						fillEdgeWeights();
						int[] remaining_subset = getGreedyOrder();

						if(remaining_subset==null || remaining_subset.length==0) {
							// found a solution
							if(SNPPicker.verbose) {
								System.out.println("Found a soln and updating best-so-far");
							}
							// if new global solution is better, save that one.
							this.nsolns++;
							this.pickBestSoFar();// Evaluate solution and move to best solution if warranted.
							if(this.nsolns % 100 == 0) { // check cpu-time usage every so-often
								long cur_time = System.currentTimeMillis()-starttime;
								if(cur_time>(long)maxtime) {
									Snp sp = pickedBE.getSnpids()[bestpickedsnp];
									this.restoreState(cs);
									sp.setTmpSelected(false, true);
									dplevel--;
									return false;
								}
							}
						} else {
							//					BinEdge check=edges[remaining_subset[0]]; // Set up Variable for Debugger
							if(dplevel>10) {
								dplevel=dplevel +1-1;// provide stop point for debugger.
							}
							// Pick remaining snps
							ClusterState newcs = new ClusterState(this);
							if(!pickBinEdgesInternal(remaining_subset, newcs)) {
								Snp sp = pickedBE.getSnpids()[bestpickedsnp];
								this.restoreState(cs);
								sp.setTmpSelected(false, true);
								dplevel--;
								return false;
							}
							// else finished DP exploration.
						}
						// reset 
						Snp sp = pickedBE.getSnpids()[bestpickedsnp];
						sp.setTmpSelected(false, true);
					} // if was able to pick a SNP
					this.restoreState(cs);
				} // loop to limit to current npop group
			} // loop over subset
			if(nFound==0) {
				maxGroup--; // Perhaps maxGroup was incorrectly computed because of too Close SNPs
			}
		}
		dplevel--;
		return true;
	}

	/**
	 * 
	 * @return false if aborted because of cpu time limit.
	 * 
     * Recursive function to pick SNPs in an aggressive cluster breakup algorithm
	 */
	private boolean pickMostLikelyInternal(int[] subset, ClusterState cs) {
		
		if(subset==null || cs==null) {
			return true;
		}
		double maxProb=0;
		
		// sort according to highest probability.
		HashMap<Integer,Snp> loopingIndexes=new HashMap<Integer,Snp>();
		for( int i=0;i<subset.length;i++) {
			int ipicked = subset[i];
			BinEdge picked = edges[ipicked];
			Snp  s= picked.getMostReliableAvailableSnp();
			double prob = s.getSuccessProb();
			try {
				if(s.canISelect(true) && prob>maxProb) {
					maxProb=prob;
					loopingIndexes.clear();
					loopingIndexes.put(Integer.valueOf(i),s);
				} else if (prob==maxProb) {
					loopingIndexes.put(Integer.valueOf(i),s);
				}
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		//System.out.println("pbmli_s "+subset.length);System.out.flush();
		Set<Map.Entry<Integer, Snp>> lies = loopingIndexes.entrySet();
		Iterator<Map.Entry<Integer, Snp>> it = lies.iterator();

		long cur_time = System.currentTimeMillis();
		if(cur_time>lasttime) {
			this.restoreState(cs);
			return false;
		}
		while(it.hasNext()) {
			Map.Entry<Integer, Snp> esm = it.next();
//			Integer I =  esm.getKey();
//			int i=I.intValue()+1-1; // for debugger
			Snp s = esm.getValue();
			

			try {
				if(s.canISelect(true)) {
					Snp.tmpSelectSnpId(s.getId(), true);
					int currentEC = countConsistentEdgesStatus(true);

					//find remaing edges part of complex clusters.
					fillEdgeWeights();
					int[] remaining_subset = getLikelyOrder();

					if(remaining_subset==null) {
//						System.out.println("pbmli_remainnull");System.out.flush();
						// found a solution
						// if new global solution is better, save that one.

						this.nsolns++;
						this.pickBestSoFar(); // incremental saving of temporarily selected SNPs. (uses selected + obligates + temp selected)
						if(this.nsolns % 10==0) { // check cpu-time usage every so-often
							cur_time =System.currentTimeMillis();
							if(cur_time>lasttime) {
								s.setTmpSelected(false, true);
								this.restoreState(cs);
								return false;
							}
						}
					} else {
						// Pick remaining snps
//						System.out.println("pbmli_remain "+Integer.toString(remaining_subset.length));System.out.flush();
						ClusterState newcs = new ClusterState(this);
						if(!pickMostLikelyInternal(remaining_subset, newcs)) {
							s.setTmpSelected(false, true);
							this.restoreState(cs);
							return false;
						}
						// else finished DP exploration.
					}
					this.restoreState(cs);
					s.setTmpSelected(false, true);
				}
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		} // loop over subset
//		System.out.println("pbmli_X "+subset.length);System.out.flush();
		return true;
	}

	/**
	 * 
	 * @return the number of bins that Still need more tag SNPs
	 */
	private int calculateMissing() {
		int missing=0;
		if(bins!=null) {
			for(int i=0;i<bins.length;i++) {
				Bin b = bins[i];
				if(b!=null) {
					int bsel = b.getTmpNSelected();// includes obligates or Fixed-Selected
					int bNeeded = Math.min(b.getNSB(),Math.max(b.getNsnps()-b.getNExcluded(), 0));
					if(bsel ==0 && bNeeded>0) {
						double P = b.getTmpProb();
						if(P<SNPPicker.minP) {
							int still_to_pick = Math.max(0,bNeeded-bsel);
							//missing +=still_to_pick;
							missing +=Math.min(1,still_to_pick);
						}
					}
				}
			}
		}
		return missing;
	}

	/**
	 * 
	 * @return the number of bins that Still need more tag SNPs
	 */
	private int calculateNecessaryButNotPicked() {
		int missing=0;
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b = this.bins[i];
				if(b!=null) {
					int bsel = b.getTmpNSelected();// includes obligates or Fixed-Selected
					int bNeeded = Math.min(b.getNSB(),Math.max(b.getNsnps()-b.getNExcluded(), 0));
					if(bsel ==0 && bNeeded>0) {
						double P = b.getTmpProb();
						missing +=1;
					}
				}
			}
		}
		return missing;
	}
	
	private void pickBestBinsSoFar() {
		if(bins!=null) {
			for(int i=0;i<bins.length;i++) {
				Bin b = bins[i];
				if(b!=null) {
					if(SNPPicker.verbose) {
						System.out.println("Found Best-so-far soln for Bin "+b.getName());
					}
					b.pickBestSoFar(true);
				}
			}
			if(SNPPicker.verbose) {
				System.out.println("--");
			}
		}
	}
	
	/**
	 * Evaluate New Solution and updates SolutionTRacker object and/or bestSoFar caches.
	 * Uses a Solution Object for whole Cluster
	 * 
	 * Look at current TMP solution and pick best solution as judged by
	 *   1) Least amount of unfilled bins.
	 *   2) Maximize Optimization Function.
	 *   
	 *   After calling this function, the user can keep temporately selecting more SNPs to see
	 *   if there is a better solution.
	 */
	
	private boolean pickBestSoFar() {
		boolean saveThisSoln=false;
		int nSelected =0;
		if(this.st==null) {
			this.st = new SolutionTracker();
			saveThisSoln=true;// Save First Solution
		}
		
		int nMissing = calculateNecessaryButNotPicked(); // Count Bins That have no tag SNPs selected, that need tag SNPs, and that have tag SNPs available for selection.
		double totalScore=0.0d;
		totalScore = this.getTmpScore(); // Equation 3 Cluster Score (SNPs over NSB quota do not count .. in order of selection)
		int[] selectedSnpids = getTmpSelectedSnps();
		if(selectedSnpids!=null) {
			nSelected = selectedSnpids.length;
		}

		if((!saveThisSoln)) {
			int nNecessaryButNotPicked=st.getNNecessaryButNotPicked(); // only choice for a bin - and selectable
			// Favor more complete solutions (more bins).
			if(nMissing<=nNecessaryButNotPicked) {
				if(nMissing<nNecessaryButNotPicked) {
					saveThisSoln=true; // better bin coverage
				}
				if(!saveThisSoln) {
					if(totalScore>st.getTotScore()) {
						saveThisSoln=true;
						// The way this is coded, additional tag SNPs per bin
						// will only be selected if the additional tag SNPs
						// pay for themselves (e.g. if the increase in bin probability*nSNPs_in_bin
						// makes up for the increase in the number of tag SNPs.)
						///
					} else {
						// By default, the search strategy searches tag SNPs with
						// better functional class first.. so we do not want to
						// override this in cases total Score is equal.	
					}
				}
			}
		}
		if(saveThisSoln) {
			st.setHasCompleteSolution();
			st.setNtags(nSelected);
			st.setNNecessaryButNotPicked(nMissing);
			st.setTotScore(totalScore);
			this.pickBestBinsSoFar();
		}
		return saveThisSoln;
	}
	
	/**
	 *  Pick Bins. Depending on the SNPPicker.isGreedy flag, find the seed solution as the greedy solution
	 *             (minimum number of SNPS) or most reliable solution.
	 *
	 */

	public void pickBins() throws Exception  {
		this.st=null; // reset solution Tracker
		this.resetTmpSolution();
		if(this.getType()==1) {
			this.unFixAndUnexcludeTooCloseTypeI();
		}
		if(SNPPicker.isGreedy()) {
//			System.out.println("pb_pbgl+");System.out.flush();
			pickBinsGreedyOrLikely(true);// favor minimum number of SNPs (bang for buck)
//			System.out.println("pb_pbgl-");System.out.flush();
		} else {
//			System.out.println("pb_pbbp+");System.out.flush();
			pickBinsByProb(); // favor most Snps most likely to yield result. but Still use bang for buck metric.
//			System.out.println("pb_pbbp-");System.out.flush();
		}

		// Now go though and pick remaining SNPs for unlinked bins.
		if(this.bins!=null) {
			for(int i=0;i<bins.length;i++) {
				Bin b = bins[i];
				if(b!=null) {
					if(b.getStatus()==0) {
//						System.out.println("pb_pbg+");System.out.flush();
						if(SNPPicker.isGreedy()) {
							b.pickBinGreedy(false,false); // finish picking the disconnected bins.
						} else {
							b.pickBinByProb(false,false);
						}
					} else {
						if(SNPPicker.verbose) {
							System.out.println("SNPPicker:INFO: Bin "+b.getName()+" already has Status==0");
							System.out.flush();
						}
					}
				}
			}
		}
		if(SNPPicker.isOptimal()) {
			if(SNPPicker.verbose) {
				System.out.println("Optimal soln search");System.out.flush();
			}
			long cpust = System.currentTimeMillis();
			try {
//				System.out.println("pb_fo+");System.out.flush();
				// Find Optimal Solution by NOT touching any fixed SNPs.
				int nbetter = findOptimal(SNPPicker.getMaxCPU(),true,false); // Refine Solution with Brute force
//				System.out.println("pb_fo-");System.out.flush();
			} catch (Exception eo) {
				eo.printStackTrace();System.out.flush();
				System.err.println(eo.getMessage());
				System.err.println("SNPPicker:Cluster: WARNING: Aborted during optimal solution search"); System.err.flush();
			}
		}
	}

	
/**
 * pick bins according to Probability, then score, then number of Snps.
 *
 */
	public void pickBinsByProb() throws Exception  {
		pickBinsGreedyOrLikely(false);
	}
	
	
	/**
	 * Greedy Solution Part II:
	 * This can be used to add SNPs to a partially selected solution.
	 * Only focus on choosing among SNPs with overlap (in bin-edges)
	 */
	
	
	public void pickBinsGreedyOrLikely(boolean greedy) throws Exception {
		if(this.length==0) {
			return;
		}
		boolean tmpToo=false;
		if(this.length==1) { // e.g. only 1 Cluster
			 this.pickAllBinsIndependently(tmpToo);
			 return;
		}
		if(SNPPicker.verbose) {
			System.out.println("processing >1 bins in cluster: "+this.getString());System.out.flush();
		}
		starttime = System.currentTimeMillis() ;
		lasttime =starttime + SNPPicker.maxCPU;
		this.fillEdgeWeights(); // Number of Bins touching an edge (multiple SNP per Edge)
		this.nsolns=0;
		int[] greedy_order =  null;
		if(greedy) {
			greedy_order = this.getGreedyOrder();
		} else {
			greedy_order = this.getLikelyOrder();
		}
//		System.out.println("pbgolbeg");System.out.flush();
		if(greedy_order==null || greedy_order.length==0) {
			if(SNPPicker.verbose) {
				System.out.println("SNPPicker:WARNING: This cluster is already in it's single bin components: "+this.getString()+"\n");
				System.out.flush();
			}
			this.pickAllBinsIndependently(tmpToo);
			return;
		}
//		int[] subset = new int[greedy_order.length];
//		for(int i=0;i<subset.length;i++) {
//			subset[i]=i;
//		}		
		ClusterState cs = new ClusterState(this);
		// aborted means that the cpu-timelimit expired and that the best
		// solution so far is available.
		boolean completed = false;
		dplevel=0;
		if(greedy) {
//			System.out.println("pbie+"+Integer.toString(dplevel));System.out.flush();
			completed = this.pickBinEdgesInternal(greedy_order,cs);
//			System.out.println("pbie-"+Integer.toString(dplevel));System.out.flush();
		} else {
//			System.out.println("pbmli+"+Integer.toString(dplevel));System.out.flush();
			completed = this.pickMostLikelyInternal(greedy_order,cs);
//			System.out.println("pbmli"+Integer.toString(dplevel));System.out.flush();

		}
		if(SNPPicker.verbose) {
			System.out.println("final picking for cluster "+this.getString());System.out.flush();
		}
//		System.out.println("pbgol_fp+");System.out.flush();
		this.finalPick();// move best solution to final solution.
//		System.out.println("pbgol_fp-");System.out.flush();
		if(SNPPicker.verbose) {
			System.out.println("picking broken up bins");
			System.out.flush();
		}
		this.pickAllBinsIndependently(tmpToo); //Finish picking Single bins.
		this.setStatus(1);
		this.resetTmpSolution(); // Set tmp == actual solution
		long endtime = System.currentTimeMillis();
		if(!completed) {
			System.err.println("WARNING: Could not explore all solutions for Phase 1 "+Integer.toString(this.clusterid)+", quick Soln: "+this.getString()+" ,size= "+this.length+" ,type= "+this.type+" ,time(ms)= "+Long.toString(endtime-starttime) );
		} else {// if(Cluster.debug) {
			if(SNPPicker.verbose) {
				System.out.println("Finished: Greedy "+(greedy ? "best coverage" : "most reliable")+" Cluster: "+this.getString()+" ,size= "+this.length+" ,type= "+this.type+" ,time(ms)= "+Long.toString(endtime-starttime) );
				System.out.flush();
			}
		}
//		System.out.println("pbgol_X");System.out.flush();
	}

	private int[] getGreedyOrder() {
		// 1. First sort bin edges according to those who contain Snps who tag the most Population
		// 2. Within each edge weight category (touches same Number of pops, not necessarely same pattern)
		//       Secondary Sort is edges according to b1.getNbinedges()+ b1.getNbinedges();
		// 
		// only need to process edges that are in complex clusters.
		// Greedy Order of Bin-Edge works OK to select solutions that minimize Total Number of SNPs 
		//   when only need 1 SNP/bin.
		if(this.edges==null) {
			return null;
		}
		BinEdgeComp c = new BinEdgeComp();
		ArrayList<BinEdge> ar = new ArrayList<BinEdge>();
		for(int i=0;i<this.edges.length;i++) {
			// Only include edge if BOTH bins still need SNPs.
			if(this.edges[i]!=null) {
				// Check if bin edge has more unselected tagSNPs (which should be common to both bins
				// and at least one bin needs more tag SNPs 
				// and both bins have tag snps that can be picked.(otherwise it's not an edge?!)
				int haveSomeBinsToPick=updateAvailableEdges(this.edges[i],true);// count assuming that TMP picked is as good as picked
				int[] ew = this.getTemp_edgeWeights();
				if(haveSomeBinsToPick==1 && ew!=null && ew[i]>=2) {
					ar.add(this.edges[i]);
				} else if(haveSomeBinsToPick==1) {
					if(SNPPicker.verbose) {
						System.out.println("edge with SNPs to pick, rejected because of weight");System.out.flush();
					}
				} else if(haveSomeBinsToPick==0) {
					// No Common SNPs to pick (after score cutoff, excluded SNPs, and other filtering)
				} 
			}
		}
		if(ar.size()==0) {
			return null;
		}
		try {
			Collections.sort(ar, c);
		} catch (Exception e) {
			e.printStackTrace();
			System.err.println("SNPPicker:ERROR: getGreedyOrder: Error with sort on array of size= "+ar.size());
			System.err.flush();
			System.out.flush();
			System.exit(-1);
		}		
		int[] order = new int[ar.size()];
		for(int i=0;i<ar.size();i++) {
			BinEdge be =  ar.get(i);
			order[i]=be.getOrder();// XXX this only works if each bin belongs to only one cluster.
		}
		return order;
	}

	private int[] getLikelyOrder() {
		BinEdgeComp2 c = new BinEdgeComp2();
		ArrayList<BinEdge> ar = new ArrayList<BinEdge>();
		for(int i=0;i<this.edges.length;i++) {
			// Only include edge if BOTH bins still need SNPs.
			if(this.edges[i]!=null) {
				int haveSomeBinsToPick=updateAvailableEdges(this.edges[i],true);// count assuming that TMP picked is as good as picked
				int[] ew = this.getTemp_edgeWeights();
				if(haveSomeBinsToPick==1 && ew!=null && ew[i]>=2) {
						ar.add(edges[i]);
				}
			}
		}	
		if(ar.size()==0) {
			return null;
		}
		Collections.sort(ar, c);
		
		int[] order = new int[ar.size()];
		for(int i=0;i<ar.size();i++) {
			BinEdge be =  ar.get(i);
				order[i]=be.getOrder();
		}
		return order;
	}

	
	


	
	public void restoreState(ClusterState cs) {// Restore the state
		int[] edgeStatus = cs.getEdgeStatus();
		int[] edgeWeight = cs.getEdgeWeights();
		int[] binStatus = cs.getBinStatus();
		if(edges!=null) {
			for(int i=0;i<edges.length;i++) {
				BinEdge e = edges[i];
				if(e!=null) {
					e.setValidflag(edgeStatus[i]);
					e.setWeight(edgeWeight[i]);
				}
			}
		}
		if(bins!=null) {
			for(int i=0;i<bins.length;i++) {
				Bin b = bins[i];
				if(b!=null) {
					b.setStatus(binStatus[i]);
				}
			}
		}

	}
	
	public void fillEdgeWeights() {
		if(temp_edgeWeights==null || temp_edgeWeights.length==0) {
			return ;
		}
		for(int i=0;i<temp_edgeWeights.length;i++) {
			BinEdge be = edges[i];
			if(be!=null && be.getValidflag()==0) {// available
				Bin b1 = Bin.getBinById(be.getBinid1());
				Bin b2 = Bin.getBinById(be.getBinid2());
				// XXX-HS Not sure which option to pick.
//				if(b1.getStatus()==0 && b2.getStatus()==0) {// Don't reward overlap if one bin already full. Let code pick best remaining SNP.
				if(b1.getStatus()==0 || b2.getStatus()==0) {// Reward overlap as long as at least one bin is not full.
					BinEdge[] be1 = b1.getBinedges();
					int nw =2;
					for(int j=0;j<be1.length;j++) {
						if(be1[j]!=null && be1[j].getValidflag()==0 && be1[j].getBinedgeid()!=be.getBinedgeid()) {
							nw++;
						}
					}
					BinEdge[] be2 = b1.getBinedges();
					for(int j=0;j<be2.length;j++) {
						if(be2[j]!=null && be2[j].getValidflag()==0 && be2[j].getBinedgeid()!=be.getBinedgeid()) {
							nw++;
						}
					}
					// 1 from be1 one from be2
					temp_edgeWeights[i]=nw;
				} else {
					be.setValidflag(1); // Happens when tooClose pre-picked bins.
					temp_edgeWeights[i]=0;
				}
		    } else {
		    	temp_edgeWeights[i]=0;
		    }
		}
	}
	
	public int[] getTemp_edgeWeights() {
		return this.temp_edgeWeights;
	}

	private TreeSet<Snp> snpsTooClose= new TreeSet<Snp>(new SnpOrderComparator());
	private ArrayList<ArrayList<Snp>>tooCloseClusters = null;
	private HashSet<String> tooClosePairs = new HashSet<String>(4*this.length);
	private HashMap<Integer,Integer> tooCloseSnpToClusterId = new HashMap<Integer,Integer>(2*this.length);
	
	/*
	 * Populate ordered snpsTooClose list 
	 * (see also Constructor that populates orderedSnps
	 */
	private void populateTooClose() { // only do this once.
		if(this.tooCloseClusters==null && SNPPicker.tooClose>0) {
			// Cluster SNPs in this Cluster and create a sorted list of tooClose SNPs.
			// That list can then be scanned for SNPs that are too Close.
			// by helper functions like canISelect.
			TreeSet<Snp> t = new TreeSet<Snp>(new SnpOrderComparator());
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null && !(s.isObligateButNotGenotyped() || (s.isExcluded() && SNPPicker.nOPA!=0))) {
					t.add(s);
				}
			}
			this.tooCloseClusters = Snp.findContigousSNP(t,SNPPicker.tooClose,false,true,SNPPicker.nOPA);
			snpsTooClose= Snp.expandListToTreeSet(tooCloseClusters);
			//
			// Create a HashSet of pairs of ids that are too close (hasmap.put(id1+id2) [ being in the same tooCloseCluster is not enough]
			Iterator<ArrayList<Snp>> itls = tooCloseClusters.iterator();
			int cidint = 0;
			while(itls.hasNext()) {
				ArrayList<Snp> ls = itls.next();
				cidint++;
				Integer cid = Integer.valueOf(cidint);
				Iterator<Snp> its = ls.iterator();
				while(its.hasNext()) {
					Snp s = its.next();
					tooCloseSnpToClusterId.put(Integer.valueOf(s.getId()),cid);
					Iterator<Snp> its2 = ls.iterator();
					int i1 = s.getId();
					int p1 = s.getPosition();
					String str1 = Integer.toString(i1)+"-";
					while(its2.hasNext()) {
						Snp s2 = its2.next();
						int i2 = s2.getId();
						if(i1!=i2) {
							int p2 = s2.getPosition();
							if(Math.abs(p1-p2)<SNPPicker.tooClose) {
								String str2 = Integer.toString(i2);
								String key = (new StringBuffer(str1)).append(str2).toString();
								tooClosePairs.add(key);
							}
						}
					}
				}
			}
			// Does not include all SNPs, just ones in too Close groups within clusters.
			if(this.snpsTooClose==null) {
				this.snpsTooClose = new TreeSet<Snp>(new SnpOrderComparator());
			}
		}
	}
	
	/*
	 * Function to check whether a SNP can be selected without violating the tooClose constraints in this Cluster.
	 * Note that this function assumes that the SNP is NOT already selected.. otherwise, results do not
	 * mean anything.
	 * 
	 */
	
	public boolean canISelect(Snp s, boolean tmpToo) {
		int nOPA = SNPPicker.nOPA;
		if(nOPA<=0) {
			return true; // no constraints on OPA.
		}
		if(SNPPicker.tooClose<=0) {
			return true; // No proximity constraints.
		}
		TreeSet<Snp> tooCloseCluster = this.getNeighborsSubset(s);
		if(tooCloseCluster==null) {
			return true;
		}
		int tooCloseClusterSize =  tooCloseCluster.size();
		if((tooCloseClusterSize+1)<=nOPA) {
			return true;
		}
		// Count Selected Snps
		int nSel=0;
		{
			Iterator<Snp> it = tooCloseCluster.iterator();
			while(it.hasNext()) {
				Snp stc = it.next();
				if(stc!=null && (stc.isObligate() || stc.isSelected() || (tmpToo && stc.isTmpSelected()))) {
					nSel++;
				} else if (stc!=null && stc.equals(s)) {
					nSel++; // count SNP wants to Select.
				}
			}
			if(nSel<=nOPA) {
				return true;
			}
		}
		// The easy cases are dealth with. Now, we have to re-check the proximity of
		// the currently selected(and/or optionally selected) tag-SNPs 
		int tooClose = SNPPicker.tooClose; // Cache it into method local variable for performance.

		Iterator<Snp> it =tooCloseCluster.iterator();
	    nSel=0;
	    boolean foundCandidateSnp=false;
	    boolean finishedCandidateSnpCluster=false;

	    int lastPos = -(tooClose+1);
	    Snp lastSelectedSnp =null;
	    int stackIndx=-1;
	    int begOffset=0; // e.g. Stack starts at position 0.
	    Snp[] snpStack = new Snp[tooCloseClusterSize];

		while(it.hasNext() && !finishedCandidateSnpCluster) {
			Snp its = it.next();
			foundCandidateSnp=false;// reset if was set in previous loop.
			if(its.equals(s)) {
				foundCandidateSnp=true;
			}
			if(its.isObligate() || its.isSelected() || (tmpToo && its.isTmpSelected()) || foundCandidateSnp) {
				int pos = its.getPosition();
				int diff = pos-lastPos;
				if(Math.abs(diff)<tooClose) {
					if(stackIndx==-1) {
						snpStack[0]=lastSelectedSnp;
						stackIndx++;
					}
					stackIndx++;
					snpStack[stackIndx]=its;

					// Trim front of stack From Current using begOffset.
					int newOffset=begOffset;
					for(int i=begOffset;i<stackIndx-1;i++) { // last 2 items in Stack are OK.
						Snp sc = snpStack[i];
						if(pos-sc.getPosition()>=tooClose) {
							newOffset=i+1;
							if(s.equals(sc)) {
								finishedCandidateSnpCluster=true;
							}
						}
					}
					if(finishedCandidateSnpCluster) {
						// if one that is trimmed is CandidateSNP, take previous Stack Size
						// and unset foundCandidateSnp flag
						nSel = Math.max(nSel, stackIndx-begOffset);
						finishedCandidateSnpCluster=true;
					}
					begOffset=newOffset;
				} else {
					// end of run.
					// If Candidate SNP was Found, then update nSel
					for(int i=begOffset;i<=stackIndx;i++) {
						Snp sc = snpStack[i];
						if(s.equals(sc)) {
							nSel = Math.max(nSel, stackIndx+1-begOffset);
							finishedCandidateSnpCluster=true;
							break;
						}
					}
					stackIndx=-1;
					begOffset=0;
				}
				lastPos=pos;
				lastSelectedSnp=its;
			}
		}
		if(!finishedCandidateSnpCluster) { // Deal with cases where cluster is at the end.
			for(int i=begOffset;i<=stackIndx;i++) {
				Snp sc = snpStack[i];
				if(s.equals(sc)) {
					nSel = Math.max(nSel, stackIndx+1-begOffset);
					break;
				}
			}
		}
		if(nSel<=nOPA) {
			return true;
		} else {
			return false;
		}

	}
	/*
	 * 
	 * Function to count the number of OPA required to select all SNPs without conflicts by searching for the biggest continuous
	 * cluster. This function is used by the optimal solution search.
	 */
	private int nOPAForTmpSolution() {
		if(SNPPicker.tooClose>0 && SNPPicker.nOPA>0) {
			if(this.snpsTooClose==null || this.snpsTooClose.size()<=1) {
				return 1;
			}
			Iterator<Snp> os = this.snpsTooClose.iterator();
			int lastPos = -(SNPPicker.tooClose+1);
			int needOPA=1;
			ArrayList<Snp> snpStack = new ArrayList<Snp>();
			int stackSize=0;
			// The maximum number of OPA's is determined by the
			// maximal number of SNPs in a 60bp window.
			// Groups of SNPs linked by 60BP constraints, can be simply
			// accomodated with nOPA=2 (if only 2 SNPs max per 60bp windows), by alternating
			int nsnpsinsoln=0;
			while(os.hasNext()) {
				Snp s = os.next();
				if(s!=null && (s.isTmpSelected() || s.isObligate() ) && !(s.isExcluded() && SNPPicker.nOPA!=0) ) {
					nsnpsinsoln++; // just need for debugging.
					int newPos = s.getPosition();
					int currentOPA=1;
					if(newPos-lastPos>=SNPPicker.tooClose) {
						stackSize=0;
						snpStack.clear();
					} else {
						if(stackSize==1) {
							currentOPA=2;// Don't need to remove last one since we know it is within 60bp
						} else {
							int nRemove=0;
							for(int i=stackSize-1;i>=0;i--) {
								Snp sstack = snpStack.get(i);
								int stackPos = sstack.getPosition();
								if(newPos-stackPos<SNPPicker.tooClose) {
									currentOPA++;
								} else{
									nRemove++;
								}
							}
							for(int i=1;i<=nRemove;i++) {
								snpStack.remove(0);
							}
							stackSize-=nRemove;
						}
						if(currentOPA>needOPA) {
							needOPA=currentOPA;
						}
					}
					snpStack.add(s);
					stackSize++;
					lastPos=newPos;
				}
			}
			if(needOPA>1) {
				needOPA += (1-1);
			}
			return needOPA;
		} else {
			return 1;
		}

	}
	
	private int noverlap=0; // overlapping non-selected SNPS (to have a chance at replacement)
	private HashMap<Bin,Integer> bins2pick =null;
	/**
	 * Find Optimal solution.
	 * 
	 * 	noNewSnps=false means that there are new SNPs "added" (or made available for choosing by unFixed)
	 *  to the cluster since last optimization. 
	 */

	public int findOptimal(long maxCPU, boolean unfixedTooClose,boolean lastCall) {
		if(this.bins==null || this.bins.length==0) {
			return 0;
		}		

		this.cacheBinId2BinIndex(); // cache map of bin id (from Bin object) to bin indexes (in this cluster)
		// Initialize bins2pick array needed by Score updating function.
		if(bins2pick==null) {
			bins2pick = new HashMap<Bin,Integer>(2*this.bins.length);
			for(int i=0;i<this.bins.length;i++) {
				Bin b = this.bins[i];
				if(b!=null) {
					Snp[] sss =b.getSnps();
					if(sss!=null && sss.length>0) {
						int nselectable =0;
						for(int is=0;is<sss.length;is++) {
							Snp ss = sss[is];
							if(ss!=null && !(ss.isExcluded() || (ss.getScore()<SNPPicker.minScore))) {
								nselectable++;
							}
						}
						bins2pick.put(b, Integer.valueOf(Math.min(b.getNSB(),nselectable)));
					}
				}
			}
		}
		this.populateTooClose(); //only gets run once.

		// Our optimization algorithm needs to start from an existing solution that is as "full" as possible.
		// We need to avoid selecting too many tooClose Snps.
		
		int ret=1;
		System.out.println("Optimizing "+bins2pick.size()+ " bins for cluster "+Integer.toString(this.clusterid));
		System.out.flush();
		int firsttime=0;
		long currcpu = System.currentTimeMillis();
		long lastcpu = currcpu+3*SNPPicker.maxCPU;
		while(ret>0 && (currcpu<=lastcpu)) {
			firsttime++;
			if(firsttime>1) {
				System.out.println("iterative optim");System.out.flush();
			}
			ret = findOptimalEx(SNPPicker.maxCPU,bins2pick,unfixedTooClose,lastCall);
			currcpu = System.currentTimeMillis();
		}
		if(ret>0 && currcpu>lastcpu) {
			System.err.println("SNPPicker: ran out of cputime in findOptimal for cluster "+Integer.toString(this.clusterid)+ " = " + this.getString());
			System.err.flush();
		}
//		System.out.println("fo");System.out.flush();
		return ret;
	}

	
	/**
	 * Find Optimal solution.
	 * 
	 * 	noNewSnps=false means that there are new SNPs "added" (or made available for choosing by unFixed) to the cluster since last optimization. 
	 *  only when lastCall is true will the exceptional solution for complex cluster be called and permuted.
	 */
	private int findOptimalEx(long maxCPU, HashMap<Bin,Integer> bins2pick, boolean unfixedTooClose,boolean lastCall) {
		if(this.snps==null) {
			System.err.print("SNPPicker:findOptimalEx: got called on a cluster with no SNPs!\n");
			return 0;
		}
		if(this.bins.length==0) {// No bins in this cluster
			System.err.print("SNPPicker:findOptimalEx: got called on a cluster with no Bins!\n");
			return 0;
		} 
		int noptimals =0; // Number of "replacements" that were better than the original solution for each subcluster
		this.noverlap=0;
		int nbetterkept=0; // Number of Kept solutions
		//int ntried=0; // number of attempted replacements
		//    For Each SNP, 
		//     Snp.bins is a vector of length nPOP which contains the bin id a SNP is in
		//     Snp.uniquelyCovers is true if that SNP is the only one covering at least one bin.

		//snps_in_solution= snps_picked_in_greedy_solution+obligate_snps+snps_fixed_by_too_close
		//snps_not_in_solution = snps_not_picked + snps_excluded_by_input+snps_excluded_by_too_close

		//selectSNPs= unique_selected_snps + obligate+snps_fixed_by_too_close
		//            + redundant_selected_snps

		// discarded_SNPs = snp.isSelected==false  && !fixed  && !excluded

		ArrayList<Snp> redundant_selected_snps = new ArrayList<Snp>();
		ArrayList<Snp> discarded_snps = new ArrayList<Snp>(); // not Selected SNPs

		
		for(int j=0;j<this.snps.length;j++) {
			Snp s = this.snps[j];
			if(s!=null) {
				if(s.isSelected()) {
					if(  (!(
						// s.isUniquelyCovers() ||   // Uniquely covers have to be free to be swapped out.
						 s.isObligate() || s.isObligateButNotGenotyped() || s.isExcluded() 
							|| s.isFixed()))
							&& (!redundant_selected_snps.contains(s))) {
						// Selected SNPs that can be (optionally) deselected because they were
						// "normally" selected (e.g. not obligates or unique) selected tag SNPs can be de-selected
						// Must allow uniquely useful SNPs to be optimized away (because of proximity constraints).
						redundant_selected_snps.add(s);
						
					}
				} else {
					if(!(s.isFixed() || s.isExcluded())) {// Cannot Select Fixed or Excluded
						if(!discarded_snps.contains(s)) {
							discarded_snps.add(s); // Discarded SNPs that can be selected.
						}
					}
				}
			}
		}//for

 

		
		HashMap<Integer,HashMap<Snp,Snp>> clusteredDiscarded = null;
		
		Set<Map.Entry<Integer, HashMap<Snp,Snp>>> cdm = null;
		HashMap<Integer,HashMap<Snp,Snp>> clusteredSelected  = new HashMap<Integer,HashMap<Snp,Snp>>(); 
		

			
		//  overlapRedundant = For Each redundant selected SNP, find non-selected SNPs who overlap in bin coverage
		//# this hash keeps track of which discarded SNPs are "singly connected"; each such
		//# SNP overlaps with only one redundant selected SNP (Easy to Test)

		HashMap<Snp,ArrayList<Snp>> overlapRedundant = new HashMap<Snp,ArrayList<Snp>>();
		for(int i=0;i<redundant_selected_snps.size();i++) {
			ArrayList<Snp> thisOverlap = new ArrayList<Snp>();
			Snp sr =  redundant_selected_snps.get(i);
			if(sr!=null) {
				int[] selBinCov = sr.getBins();// Bin Coverage indexed by Population.
				for(int j=0;j<discarded_snps.size();j++) {
					Snp sd =  discarded_snps.get(j);
					int[] relBinCov = sd.getBins();// Bin Coverage indexed by Population.
					for(int k=0;k<relBinCov.length;k++) { // Loop over populations
						if(selBinCov.length>k && relBinCov.length>k) {
							if(((selBinCov[k] == relBinCov[k]) && selBinCov[k]>0) && !thisOverlap.contains(sd)) {
								thisOverlap.add(sd); //at least one bin in same Pop.
								break;
							}
						}
					}
				}
				overlapRedundant.put(sr, thisOverlap); // For each selected SNP List of overlapping non-selected SNPS
				if(thisOverlap!=null && thisOverlap.size()>0) {
					noverlap++;
				}
			}
		}
		//   discardedSinglyConnected (hashmap of each non-selected tag SNPs that is the only coverage alternative for a selected tag SNP)
		// but this discarded tag SNP could cover more than one bin.

		HashMap<Snp,Snp> discardedSinglyConnected= new HashMap<Snp,Snp>();
		{
			HashSet<Snp> toRemove = new HashSet<Snp>();
			for(int i=0;i<redundant_selected_snps.size();i++) {
				Snp sr =  redundant_selected_snps.get(i);
				if(sr!=null) {
					ArrayList<Snp> ov = overlapRedundant.get(sr);
					if((ov!=null) && (ov.size()==1) && (ov.get(0)!=null)) {
						if(discardedSinglyConnected.containsKey(ov.get(0))) {
							toRemove.add(ov.get(0)); // not truly singly Connected
						} else {
							discardedSinglyConnected.put(ov.get(0),sr);
							noverlap--;
						}
					}
				}
			}
			Iterator<Snp> itr = toRemove.iterator();
			while(itr.hasNext()) {
				discardedSinglyConnected.remove(itr.next());
				noverlap++;
			}
		}
		//   overlapDiscarded = for each discarded, list of selected SNPs overlapping
		//   nonOverlapDiscarded = for each discarded, list of selected SNPs NOT overlapping
		HashMap<Snp,ArrayList<Snp>> overlapDiscarded = new HashMap<Snp,ArrayList<Snp>>();
//		HashMap<Snp,ArrayList<Snp>> nonOverlapDiscarded = new HashMap<Snp,ArrayList<Snp>>();
		if(noverlap>0) {
			noverlap=noverlap+1-1; // break for debugger
		}
 
		for(int j=0;j<discarded_snps.size();j++) {
			Snp sd =  discarded_snps.get(j);
			if(sd!=null) {
				int[] relBinCov = sd.getBins();// Vector of bins ids for this SNP, indexed by population.
				ArrayList<Snp> thisOverlap = new ArrayList<Snp>();
				ArrayList<Snp> thisNonOverlap = new ArrayList<Snp>();
				for(int i=0;i<redundant_selected_snps.size();i++) {				
					Snp sr =  redundant_selected_snps.get(i);
					if(sr!=null) {
						int[] selBinCov = sr.getBins();// Vector of bins ids for this SNP, indexed by population.
						boolean overlap=false;
						if(relBinCov!=null && selBinCov!=null) {
							for(int k=0;k<relBinCov.length;k++) { // Loop over populations
								if(selBinCov.length>k && relBinCov.length>k) {
									if((selBinCov[k] == relBinCov[k]) && relBinCov[k]>0 ) {
										overlap=true;
										break;
									}
								}
							}
						}
						if(overlap) {
							if(!thisOverlap.contains(sr)){
								thisOverlap.add(sr);
							}
						} else {
							if(!thisNonOverlap.contains(sr)) {
								thisNonOverlap.add(sr);
							}
						}
					}
				}
				overlapDiscarded.put(sd, thisOverlap); // For each non-selected("discarded") SNP List of overlapping selected SNPS
//				nonOverlapDiscarded.put(sd, thisNonOverlap); // For each non-selected SNP List of non-overlapping selected SNPS
			}
		}

		/*
		    # cluster SNPs that were discarded if they share overlapping redundant selected SNPs 
		    #
		    # Discarded (not selected) SNPs without overlap with selected SNPs 
		    # will be returned as a single cluster. These are from tooClose unfixing and should be optimized together.
		 */

		
		 
		



		if(unfixedTooClose) {

			// This turns off a lot of possible optimizations. because the whole cluster is clustered.
			HashMap<Snp,Snp> cd = new HashMap<Snp,Snp>(2+2*discarded_snps.size());
			Iterator<Snp> it = discarded_snps.iterator();
			while(it.hasNext()) {
				Snp s = it.next();
				cd.put(s, null);
			}
			clusteredDiscarded = new HashMap<Integer,HashMap<Snp,Snp>>(1);
			clusteredDiscarded.put(Integer.valueOf(1), cd);
			cdm = clusteredDiscarded.entrySet();

			HashMap<Snp,Snp> cs = new HashMap<Snp,Snp>(2+2*redundant_selected_snps.size());
			it = redundant_selected_snps.iterator();
			while(it.hasNext()) {
				Snp s = it.next();
				cs.put(s, null);
			}
			clusteredSelected = new HashMap<Integer,HashMap<Snp,Snp>>(2);
			clusteredSelected.put(Integer.valueOf(1), cs);

		} else {
			clusteredDiscarded = Cluster.clusterDiscarded(discarded_snps,null,overlapDiscarded);
			// Returned clusters, keyed by Integer 1..nclusters.
			// Content of Cluster are groups of discarded Snps linked by shared bins (or singleton not-selected SNPs)
			// The hope is that we can "break" the original cluster and lower the computational complexity of the problem.
			//


			// Swapping technique: Add 0-k SNPs and simultaneously remove 0-m. Ideally swap SNPs with equivalent coverage.


			// Now find groups of selected SNPs overlapping with the non-selected clusters.
			clusteredSelected = new HashMap<Integer,HashMap<Snp,Snp>>(); 
			cdm = clusteredDiscarded.entrySet();		
			Iterator<Map.Entry<Integer, HashMap<Snp,Snp>>> meit = cdm.iterator();
			while(meit.hasNext()) {
				Map.Entry<Integer, HashMap<Snp,Snp>> me = meit.next();
				Integer cid = me.getKey();
				HashMap<Snp,Snp> cdiscMap =me.getValue();
				HashMap<Snp,Snp> clusterSel = new HashMap<Snp,Snp>();

				Snp[] cdiscList = cdiscMap.keySet().toArray(new Snp[cdiscMap.keySet().size()]);
				for(int k=0;k<cdiscList.length;k++) {// Loop over not-in-solution ("discarded") Snps in Cluster
					Snp cdiscSnp = cdiscList[k];
					ArrayList<Snp> od = overlapDiscarded.get(cdiscSnp);// selected Snps overlapping with this non-selected SNP
					if(od!=null && od.size()>0) {
						for(int j=0;j<od.size();j++) {// Loop over overlapping Selected Snps for each discarded Snp
							clusterSel.put(od.get(j), null);// Accumulate Selected SNPs overlapping this cluster of discarded SNPs
						}
					}
				}
				clusteredSelected.put(cid,clusterSel); // selected SNPs overlapping with non-selected SNPs in Cluster.
			}
		}
		 
		try {

			/*
		        Loop Over SubClusters of SNPs tied to connected discarded SNPs (e.g. by Bin Coverage).
		    	Because subclusters are independently optimizable, we can still look for
		    	the global cluster-level score even though we are only changing a subset of the SNPs.

			 */
			int nchanged=0;
			if(SNPPicker.verbose) {
				System.out.println("SNPPicker: Looping over "+cdm.size()+ " clusters");System.out.flush();
			}
//			Iterator<Map.Entry<Bin, Integer>> itb2p= bins2pick.entrySet().iterator();


			int cloop=0;
			
			boolean noNotInSolutionSnps = (cdm==null || cdm.size()==0);
			
			Iterator<Map.Entry<Integer, HashMap<Snp,Snp>>> meit2 = cdm.iterator();


			// Loop Over subClusters (if cluster is separable)
			System.out.println("cluster "+this.clusterid+" for " +(cdm==null? 0: cdm.size())+" subclusters ");
			System.out.flush();
			while (meit2.hasNext()
					|| (cloop==0 && noNotInSolutionSnps) 
			) {
				
				cloop++;
				noptimals=0; // reset for each sub-Cluster.
				// XXX  - Reset CPU Time limit
				long cpustart = System.currentTimeMillis();
				if(SNPPicker.verbose) {
					System.out.println("c");System.out.flush();
				}			
				
				this.resetTmpSolution();// only currently selected SNPs will be tmp Selected.

				SolutionTracker soln = new SolutionTracker();
				this.copySolutionToTmp();
				SolutionTracker.updateTmpScore(bins2pick,soln); // Set current baseline solution	


//				int baselineNSnps=this.countSelectedSnps();
				//int bestNSnps=baselineNSnps;
//				double bestScore =soln.getScore();

				Snp[] notInSoln =null;
				Snp[] inSoln =null;
				int innerloop=0;

				/*
				 * Create lists of swappable (notInSoln=swapin/inSoln=swapout) SNPs
				 */
				
				if(cloop==1 && noNotInSolutionSnps) {// Case where there are no not-in-solutions SNPs
													 // linking the in-solution SNPs
													// so optimization will try to select a smaller number of
													// SNPs among the selectable ones.
					notInSoln = new Snp[0];
					inSoln = new Snp[redundant_selected_snps.size()];
					Iterator<Snp> iti = redundant_selected_snps.iterator();
					int ii=0;
					while(iti.hasNext()) {
						inSoln[ii++] = iti.next();
					}
				} else {
					
					Map.Entry<Integer, HashMap<Snp,Snp>> me = meit2.next();
					Integer cid = me.getKey();
					HashMap<Snp,Snp> cdiscMap =me.getValue();  // Value in this HashMap is always NULL.

					notInSoln = cdiscMap.keySet().toArray(new Snp[cdiscMap.keySet().size()]); // Snp[]
				// inSoln could be empty
					inSoln = clusteredSelected.get(cid).keySet().toArray(new Snp[clusteredSelected.get(cid).size()]); // Snp[]
				}
	
				
				
				/*
		        Create binary "instructions" (choices of subset) for all possible subsets of SNPs already picked
		        and in cluster
		        .. and all possible subsets not already picked  and in cluster
				 */


				// Only 1 set swap is going to be "best"
				int bestSubsetIn = 0;
				int bestSubsetNotIn = 0;

				if(inSoln.length>27 || notInSoln.length>27) {
					System.err.println("SNPPicker:WARNING: For Cluster cid="+Integer.toString(this.clusterid)+" : " + this.getString()+" Number of solutions to be explored too high: 2^(" +Integer.toString(inSoln.length)+"+"+Integer.toString(notInSoln.length)+"), Exceeds coded limit of permutation");
					System.err.flush();
					semiOptimalSoln(inSoln,notInSoln,soln,bins2pick,overlapDiscarded,lastCall);
					noptimals=0;// Solution no longer guaranteed to be optimal, so don't reloop.
					continue; // continue looping over subclusters (if the cluster was separable)
				} 


//				ArrayList inSolnSubsets = findSubsets(inSoln); // Includes subset were we keep all or keep none.

				LinkedHashSet<Integer> lhsin = new LinkedHashSet<Integer>();
				for (int ilhs = 0;ilhs<inSoln.length;ilhs++ ) {
					lhsin.add(Integer.valueOf(ilhs));
				}
				LinkedHashSet<Integer> lhsnotIn = new LinkedHashSet<Integer>();
				for (int ilhs = 0;ilhs<notInSoln.length;ilhs++ ) {
					lhsnotIn.add(Integer.valueOf(ilhs));
				}

				PowerSet<Integer> pow = new PowerSet<Integer>(lhsin); // First solution will have no SNPs, so no SNPs will be excluded.
				PowerSet<Integer> powN = new PowerSet<Integer>(lhsnotIn);
				
				
				
				/*
				 * Subset loop: over inSolution Subsets.
				 */
				
				Iterator<PowerSet<Integer>.BitMaskSet> it = pow.iterator();
				int inloop=0;
				boolean noInSolutionSnps = (inSoln.length==0);
				long cpunow = System.currentTimeMillis();
				long cpulast = cpunow+maxCPU;
				while(it.hasNext()       	// Loop over in-solution subsets.
						|| ((!it.hasNext()) && inloop==0 && noInSolutionSnps && !noNotInSolutionSnps)) {  // Allow Case where no in-solution SNPs overlap with the added SNPs
								// but must have some SNPs either in or out of solution that are swappable.
					inloop++; // Counter for percent of solutions Explored.
					int nelems=0;
					int[] inSolnSubset=null;
					if(inloop==1 && noInSolutionSnps && !noNotInSolutionSnps) {
						inSolnSubset = new int[0];
						if(SNPPicker.verbose) {
							System.out.println("No In Soln when NotIn="+notInSoln.length);System.out.flush();
						}						
					} else {
						edu.mayo.genotype.PowerSet<Integer>.BitMaskSet bm = it.next();
						// The first loop considers taking out no SNP from current solution.
						nelems = bm.size();
						inSolnSubset = new int[nelems];
						{
							Iterator<Integer> itp = bm.iterator();
							int ii=0;
							while(itp.hasNext()) {
								inSolnSubset[ii++] = itp.next().intValue();
							}
						}
					} 

					if(cpunow<cpulast) {
						int nInSolnSubset =inSolnSubset.length;

//						int[] inSolnBinCoverage= this.clusterBinCoverageForSnps(inSoln,inSolnSubset);

						/*
						 * Subset loop: over notInSolution Subsets.
						 */

						Iterator<PowerSet<Integer>.BitMaskSet> itN = powN.iterator();
						int notinloop=0;
						boolean noNotInSolution = (notInSoln.length==0);
						while((cpunow<cpulast) && (itN.hasNext()
								|| ((!itN.hasNext()) && notinloop==0 && noNotInSolution  && !noInSolutionSnps)) 
						) {// Loop over Not-in-solution-and-overlapping patterns
							
							notinloop++;
							innerloop++;
							int nNotInSolutionSubsets=0;
							int[] notInSolnSubset = null;
							if(notinloop==1 && noNotInSolution) {
								notInSolnSubset = new int[0];
							} else {
								edu.mayo.genotype.PowerSet<Integer>.BitMaskSet bmN = itN.next();
								nNotInSolutionSubsets = bmN.size();
								notInSolnSubset = new int[nNotInSolutionSubsets];
								{
									Iterator<Integer> itpN = bmN.iterator();
									{
										int ii=0;
										while(itpN.hasNext()) {
											notInSolnSubset[ii++] = itpN.next().intValue();
										}
									}
								}
							}
							// Compute notInSoln bin Coverage vector
							boolean keepSoln=true;

						/*
							// Rapidly check the solution to see if it needs to be rejected
							// for making coverage worst.						 						
						    int[] notInSolnBinCoverage= this.clusterBinCoverageForSnps(notInSoln,notInSolnSubset);
							ntried++;
							for(int m=0;keepSoln && m<this.bins.length;m++) {
								Bin b = this.bins[m];
								if(b!=null) {// skip deleted bins.
									int b_actual = b.getNSelected(); // currently selected.
//									double p_actual = b.getProb();
									
									// this will then cover the case where
									// we selected less tagSNPs because
									// of the Probability option (-minP)
									int b_in = inSolnBinCoverage[m]; // Lost Coverage of in solution subset of already chosen SNPs
									int b_notIn = notInSolnBinCoverage[m]; // replacement coverage not in solution, but choosable (not excluded)
									if(SNPPicker.minP<1.0d) { // May get away with less coverage, so cannot exclude solutions based on coverage.
										if(b_actual-b_in==0 && b_actual>0) {// Exclude solutions that completely drop bins.
											keepSoln=false;
										}
									} else if((b_actual<b.getNSB() && b_notIn-b_in>=0) // Not full coverage and not worst
											|| (b_actual>=b.getNSB() && (b_actual+b_notIn-b_in>=b.getNSB())) // Pre-swap coverage was OK and so is post-swap coverage (may have gone down)
											|| (b_notIn-b_in)>=0 // better or equal coverage
											) {
										keepSoln=true;// stop point for debugger.
										// Cannot exclude this solution without checking the score/probability
									} else {
										keepSoln=false;
									}
								}
							}
							*/
							
							if(keepSoln) {
								// Score this solution

								for(int m=0;m<nInSolnSubset;m++) {
									Snp sin=  inSoln[inSolnSubset[m]];
									if(sin!=null) {
										sin.setTmpSelected(false, true);
									}
								}
								for(int m=0;m<nNotInSolutionSubsets;m++) {
									Snp snotin= notInSoln[notInSolnSubset[m]];
									if(snotin!=null) {
										snotin.setTmpSelected(true, true);
									}
								}
								int retcode = SolutionTracker.updateTmpScore(bins2pick,soln,false);
								if(retcode==1) {
									// solution was transfered to {Snp. and Bin.}->best-so-far if get here.
									boolean solutionOK=true;

									// Make sure this solution does not violate the nOPA constraint.
									
									/* THis is now checked in the SolutionTracker
									if(SNPPicker.tooClose!=0 && SNPPicker.nOPA>=0) {
										int needOPA = this.nOPAForTmpSolution(); // Count Number of OPA needed to store without conflict
										if(needOPA>SNPPicker.nOPA) {
											solutionOK=false;
										}
									}
									*/
									
									if(solutionOK) {
//										System.out.println("m");System.out.flush();

										SolutionTracker.moveToBestSoFar(bins2pick);
//										bestScore=soln.getScore();
										//bestNSnps = baselineNSnps-nInSoln+nNotInSoln;
										this.finalPick(); // move best-so-far solution to Snps and Bins.
										bestSubsetIn=inSolnSubset.length;
										bestSubsetNotIn=notInSolnSubset.length;
										noptimals++;
										this.eraseTmpSolution(); //?? redundant ??
									} else {
										this.eraseTmpSolution(); // No SNPs tmpSelected.
										
										/* Was needed when old SolutionTracker did not check for nOPA
										soln = new SolutionTracker();
										SolutionTracker.updateTmpScore(bins2pick,soln,false);
										*/
									} 
								} else {
									this.eraseTmpSolution(); // Resets Tmp Solution to same as .isSelected()
								}
							} //if(keepSoln)
							cpunow = System.currentTimeMillis();
						}//while() // loop over notinSoln subsets
						cpunow = System.currentTimeMillis();
						if(cpunow>cpulast) {
							System.err.println("SNPPicker: ran out of cputime ("+SNPPicker.maxCPU+" ms) for optimal search for cluster cid="+Integer.toString(this.clusterid)+" : "+this.getString()+" : examined roughly "+innerloop+" solns out of 2^("+(inSoln==null ? 0: inSoln.length)+"+"+(notInSoln==null ? 0: notInSoln.length)+") possible substitutions combinations");
							System.err.flush();
							semiOptimalSoln(inSoln,notInSoln,soln,bins2pick,overlapDiscarded,lastCall);
							noptimals=0;
							break; 
						}
					} else {
						System.err.println("SNPPicker: ran out of cputime ("+SNPPicker.maxCPU+" ms) for optimal search for cluster cid="+Integer.toString(this.clusterid)+" : "+this.getString()+" : examined roughly "+innerloop+" solns out of 2^("+(inSoln==null ? 0: inSoln.length)+"+"+(notInSoln==null ? 0: notInSoln.length)+") possible substitutions combinations");
						System.err.flush();
						semiOptimalSoln(inSoln,notInSoln,soln,bins2pick,overlapDiscarded,lastCall);
						noptimals=0;
						break; 
					}
				} // while(it.hasNext) : loop over inSoln Subsets
				

				// Make the Change for this set of connected SNPs

				if(noptimals>0) {
					nchanged +=(bestSubsetNotIn-bestSubsetIn); // Statistics on Total number of SNPs
					//						this.pickBestSoFarAndReset(); XXX-! the best was already picked
					nbetterkept++;
				}
			} // end of loop over connected sub clusters. for(Map.Entry)
		} catch (Exception e) {
			e.printStackTrace();
			System.err.println(e.getMessage());System.err.flush(); System.out.flush();
		}
// 		if(SNPPicker.verbose) {
 			if(nsolns>512) {
				System.out.println("tried "+nsolns+" solns");System.out.flush();
//			}
 		}
//			System.out.println("X");System.out.flush();

		return nbetterkept;


	}
	/**
	 * Return the cluster of Snps using recursive clustering.
	 * @param discarded_snps
	 * @param excludedMap - Exclude Map disactivated
	 * @return
	 */
	public static HashMap<Integer,HashMap<Snp,Snp>>clusterDiscarded(ArrayList<Snp> discarded_snps,HashMap<Snp,Snp> excludedMap,HashMap<Snp,ArrayList<Snp>> overlapDiscarded) {
		HashMap<Integer,HashMap<Snp,Snp>> clusteredDiscarded = new HashMap<Integer,HashMap<Snp,Snp>>();
		if(discarded_snps==null) {
			return clusteredDiscarded;
		}
		int curr_cid=0;
		boolean[] clustered= new boolean[discarded_snps.size()];
		for(int i=0;i<discarded_snps.size();i++) {
			clustered[i]=false;
		}
		for(int i=0;i<discarded_snps.size();i++) {
			if(!clustered[i]) {
				Snp csnp = discarded_snps.get(i);
				if(csnp!=null) {
					if(excludedMap==null || !(excludedMap.containsKey(csnp))) {// skip singly connected SNPs will try them later
						HashMap<Snp,Snp> clusteredSnps = new HashMap<Snp,Snp>();
						// Recursive clustering.
						clusterForward(clusteredSnps,csnp,overlapDiscarded,discarded_snps,i,i+1,clustered,excludedMap);	
						if(clusteredSnps.size()>0) {
							curr_cid++;
							clusteredDiscarded.put(Integer.valueOf(curr_cid), clusteredSnps);
						}
					}
				}
			}
		}

/*
		for(int i=0;i<discarded_snps.size();i++) {
			if(!clustered[i]) {
				Snp csnp = discarded_snps.get(i);
				if(csnp!=null) {

					if(excludedMap==null || !(excludedMap.containsKey(csnp))) {// skip singly connected SNPs will try them later
						ArrayList<Snp> od =  overlapDiscarded.get(csnp);
						if(od!=null && od.size()>0) {
							HashMap<Snp,Snp> clusteredSnps = new HashMap<Snp,Snp>();
							clusteredSnps.put(csnp, null);
							curr_cid++;
							clusteredDiscarded.put(Integer.valueOf(curr_cid), clusteredSnps);
						}
					}
				}
			}
		}
*/
		
		
		// Create Single-SNP clusters for non-selected SNPs that touch more than one selected SNPs.
		// and that weren't clustered by previous step.
		// XXX We could have a more optimal clustering, but in most cases, since these SNPs
		// come from unFixing tooClose SNPs, the SNPs would be linked.
		HashMap<Snp,Snp> singletons= new HashMap<Snp,Snp>();
		for(int i=0;i<discarded_snps.size();i++) {
			if(!clustered[i]) {
				Snp csnp = discarded_snps.get(i);
				if(csnp!=null) {
					singletons.put(csnp, null);
					curr_cid++;
					clusteredDiscarded.put(Integer.valueOf(curr_cid), singletons);
				}
			}
		}
		
		return clusteredDiscarded;
	}
	/**
	 * Recursive clustering. Clusters non-selected SNPs (based on shared overlap of with selected SNPs). For any overlapping SNP it finds, it also scans forward recursively
	 * @param HashMap clusteredSnps Input HashMap that keeps track of SNps clustered
	 * @param Snp csnp Discarded SNP to be clustered with
	 * @param HashMap overlapDiscarded  Contains selected SNps overlapping with discarded Snps overlap of these 
	 * @param ArrayList discarded_snps 
	 * @param int root_indx = starting index of recursion.
	 * @param int indx         Start position in discarded_snps for the recursive search.
	 * @param boolean[] clustered    boolean array to keep track of which Discarded SNP is already clustered.
	 * @param excludedMap  HashMap of Discarded SNPs to skip for clustering
	 * @return
	 */
	public static void clusterForward(HashMap<Snp,Snp> clusteredSnps,Snp csnp,HashMap<Snp,ArrayList<Snp>> overlapDiscarded,ArrayList<Snp> discarded_snps,int root_indx,int indx, boolean[] clustered,HashMap<Snp,Snp> excludedMap) {	
		if(discarded_snps!=null && indx<discarded_snps.size()) {
			ArrayList<Snp> clist = overlapDiscarded.get(csnp); // For each non-selected SNP List of overlapping selected SNPS	
			if(clist!=null && clist.size()>0) {//got list of selected SNPs overlapping with discarded csnp
				for(int j=indx;j<discarded_snps.size();j++) {// look for match in all "forward" snps
					if(!clustered[j]) {
						Snp tsnp =  discarded_snps.get(j);
						if(tsnp!=null && (excludedMap==null || !(excludedMap.containsKey(tsnp)))) {
							boolean disconnected=true;
							if(csnp.getId()!=tsnp.getId()) {
								ArrayList<Snp> tlist =  overlapDiscarded.get(tsnp); // For each non-selected SNP List of overlapping selected SNPS
								if(tlist!=null) {
									for(int k=0;(k<clist.size()) && disconnected;k++) {
										Snp ccsnp =  clist.get(k);
										if(ccsnp!=null) {
											for(int l=0;(l<tlist.size()) && disconnected;l++) {
												Snp ttsnp =  tlist.get(l);
												if(ttsnp!=null) {
													if (ccsnp.getId()==ttsnp.getId()) {
														disconnected=false;
													}
												}
											}
										}
									}
								}
							}
							if(!disconnected) {
								clusteredSnps.put(tsnp,null);// Only Key matters... XXX should use HashSet instead
								clusteredSnps.put(csnp, null);
								disconnected=true;
								clustered[j]=true;
								clustered[root_indx]=true;
								// insure that tSNP also has no additional overlaps within space scanned by source csnp.
								if(root_indx<discarded_snps.size()-2) {
									if(j==root_indx+1) { // e.g. root_indx and root_indx+1 are already in cluster
										// no "backward" indices to cover
										j=j+1-1;// spot for debugger
									} else {
										clusterForward(clusteredSnps,tsnp,overlapDiscarded,discarded_snps,root_indx,root_indx+1,clustered,excludedMap);	
									}
								} else {
									// these are the last two snps in list.. nothing more to scan.
								}
							}
						}
					}
				}
			}
		}
	}
	
	/**
	 * Find all possible subsets of the input objects (including the empty set)
	 * @param snplist
	 * @return an ArrayList of int[] containing indexes into input array.
	 */
	public static ArrayList<int[]> findSubsets(Object[] snplist) throws Exception {
		// Assume that we have Snp objects.
		ArrayList<int[]> l = null;
		if(snplist!=null && snplist.length>0) {
			int size = snplist.length; // You probably don't want more than 32
			if(size>25) {
				System.err.println("Cluster too big(>25), limiting seeking of optimal solution to 25 first SNPs out of "+size+"\n");
				System.err.flush();
				size=25;// 32~ 1000 seconds for powerser creation.. a lot longer for java objects.
			}
			l = new ArrayList<int[]>(2^size);
			LinkedHashSet<Integer> s = new LinkedHashSet<Integer>();
			for (int i = 0;i<size;i++ ) {
				s.add(Integer.valueOf(i));
			}
			PowerSet<Integer> pow = new PowerSet<Integer>(s);
			Iterator<PowerSet<Integer>.BitMaskSet> it = pow.iterator();
			while(it.hasNext()) {
				edu.mayo.genotype.PowerSet<Integer>.BitMaskSet bm = it.next();
				int nelems = bm.size();
				int[] indexes = new int[nelems];
				Iterator<Integer> itp = bm.iterator();
				int i=0;
				while(itp.hasNext()) {
					indexes[i++] = itp.next().intValue();
				}
				l.add(indexes);
			}
		}
		return l;
	}

	/**
	 * Compute bin coverage for subset of SNPs inSoln array, indexes by subsetting array
	 * @param snpsset Snp[] array
	 * @param subsetIndexes   index of subset of inSoln
	 * @return int[] containing the bin coverage (array of length Bins.getNBins())
	 */
	public int[]  binCoverageForSnps(Snp[] snpset, int[] subsetIndexes) {
		if(snpset==null || subsetIndexes==null) {return null;}
		int[] bc = new int[Bin.getNbins()+1];
		for(int i=0;i<subsetIndexes.length;i++) {
			Snp s =   snpset[subsetIndexes[i]];
			if(s!=null) {
				int[] bins4snp = s.getBins();
				for(int j=0;j<bins4snp.length;j++) {bc[bins4snp[j]]++;}
			}
		}
		bc[0]=0; // no binid=0 bin.
		return bc;
	}
	
	/**
	 * Compute bin coverage indexes wrt to bins in this cluster.
	 * @param snpsset Snp[] array
	 * @param subsetIndexes   index of subset of inSoln
	 * @return int[] containing the bin coverage (array of length Bins.getNBins())
	 */
	private int[]  clusterBinCoverageForSnps(Snp[] snpset, int[] subsetIndexes) {
		if(snpset==null || subsetIndexes==null) {return null;}
		int[] bc = new int[this.bins.length+1];
		for(int i=0;i<subsetIndexes.length;i++) {
			Snp s =   snpset[subsetIndexes[i]];
			if(s!=null) {
				int[] bins4snp = s.getBins();
				for(int j=0;j<bins4snp.length;j++) {
					int bid = bins4snp[j];
					if(bid>0) {
						int bindex = this.binId2BinIndex[bid];
						bc[bindex]++;
					}
				}
			}
		}
		bc[0]=0; // no binid=0 bin.
		return bc;
	}
	private void cacheBinId2BinIndex() {
		if(this.binId2BinIndex==null) {
			this.binId2BinIndex = new int[Bin.getNbins()+1];
			for(int i=0;i<this.binId2BinIndex.length;i++) {
				this.binId2BinIndex[i]=-1;
			}
			for(int bindex=0;bindex<this.bins.length;bindex++) {
				Bin b = this.bins[bindex];
				if(b!=null) {
					this.binId2BinIndex[b.getId()]=bindex;
				}
			}
		}
		
	}
	
	public double getScore() {
		double cscore=0.0d;
		int nsel=0;
		for(int i=0;i<this.bins.length;i++) {
			if(this.bins[i]!=null) {
				cscore+=this.bins[i].getBinScore();
				int selcount = 0;
				if(SNPPicker.infinium) {
					selcount = this.bins[i].getNSelectedBeadTypes();
				} else {
					selcount = this.bins[i].getNSelected();
				}
				nsel+=selcount;
			}
		}
		if(nsel==0) {
			return 0.0d;
		}
		return cscore/nsel;
	}
	
	
	public double getTmpScore() {
		double cscore=0.0d;
		int nsel=0;
		for(int i=0;i<this.bins.length;i++) {
			if(this.bins[i]!=null) {
				cscore+=this.bins[i].getBinTmpScore();
				int selcount = 0;
				if(SNPPicker.infinium) {
					selcount = this.bins[i].getTmpNSelectedBeadTypes();
				} else {
					selcount = this.bins[i].getTmpNSelected();
				}
				nsel+=selcount;
			}
		}
		if(nsel==0) {
			return 0.0d;
		}
		return cscore/nsel;
	}
	
	
	public int countSelectedSnps() {
		int nsel=0;
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b = this.bins[i];
				if(b!=null) {
						nsel+=b.getNSelected();
				}
			}
		}
		return nsel;
	}
	
	public void finalPick() throws Exception {
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b = this.bins[i];
				if(b!=null) {
						b.finalPick(true,false);
				}
			}
		}

	}
	
	
	/**
	 * Reset Tmp Solution to already selected and obligates.
	 *
	 */
	public void resetTmpSolution() {
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b= this.bins[i];
				if(b!=null) {
					b.resetTmpSolution(true);
				}
			}
		}
	}
	
	/**
	 * Erase Tmp Solution to already selected and obligates.
	 *
	 */
	public void eraseTmpSolution() {
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b= this.bins[i];
				if(b!=null) {
					b.eraseTmpSolution(true);
				}
			}
		}
	}
	
	/**
	 * Erase Tmp Solution to already selected and obligates.
	 *
	 */
	public void copySolutionToTmp() {
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b= this.bins[i];
				if(b!=null) {
					b.copySolutionToTmp(true);
				}
			}
		}
	}
	
	/*
	 * Does not evaluate solution, but just transfers current TMP solution to best-so-far
	 * Does not pick any SNPs. Call this when you want to store a multi-SNP temporary selection as
	 * the best so far.. and want to temporarily select another batch of SNPs for a new solution.
	 * 
	 * The SNPs are not "picked" until finalPick is called.
	 * 
	 * returns true if one better solution was found.
	 */
	public void resetBestSoFarAndTmp() {
	
		for(int i=0;i<this.bins.length;i++) {
			if(this.bins[i]!=null) {
				this.bins[i].resetBestSoFarAndTmp(true);
			}
		}
		
	}
	

	public boolean pickBestSoFarAndReset() {
		boolean abetterSolution=false;
		for(int i=0;i<this.bins.length;i++) {
			if(this.bins[i]!=null) {
				if(this.bins[i].pickBestSoFarAndReset(true)) {
					abetterSolution=true;
				}
			}
		}
		return abetterSolution;
		
	}
	public String getString() {
		if(this.bins==null) {
			return Integer.toString(this.clusterid);
		} else {
			StringBuffer str = new StringBuffer();
			for(int i=0;i<this.bins.length;i++) {
				if(this.bins[i]!=null) {
					str.append(";");
					str.append(this.bins[i].getOutputName());
				}
			}
			return str.substring(1);
		}
	}
	

	/**
	 * 
	 * @return false if aborted because of cpu time limit.
	 * 
     * New (SNPPicker 2) Recursive function to exhaustively pick SNPs that are in multiple bins.
     * The algorithm assumes that there is no conflict, e.g. if a SNP is chosen, this doesn't
     *  prevent another SNP from being chosen.
	 */
	private boolean pickBinEdgesExhaustive() throws Exception {
		if(SNPPicker.tooClose!=1000000000) {
			throw new Exception("pickBinEdgesExhaustive Function is not finished and may no longer be relevant");
		}
/*
		  Explore all possible ways of selecting the tag SNPs in the edges first
		  and then picking the best set of tag SNPs (not in the edges) in each bin.
		  		Edges are either cause by multipop tag SNPs
	       or because bins contained SNPs with proximity conflicts above and beyond the
        number allowed.

		In order to use this function, we will assume that there is no possibility
		of conflicts. 
		THe difficulty in the algorithm design is because some tag SNPs
		can be chosen in multiple bin edges

		Algorithm: Find all tagSNPs that are shared by at least one pair of bins.(and pickable in at least one bin)
					Once we specify the state of all those shared tag SNP (picked /not picked)
					then exploration of the each bin is greedy.
*/							

	
		
		
		if(this.edges==null || this.edges.length==0 || this.bins==null || this.bins.length==0) {
			return true; // SInce there are no tag SNPs to pick, this function did go through all solns.
		}
		if(this.snps==null) {
			this.populateSnpArrayInCluster();
		}
		if(this.snps==null || this.snps.length==0) {
			return true;
		}
		// INitialize Solution Tracker
		SolutionTracker st = new SolutionTracker();
		// Set OPA id of obligates that may be too close. (this clears Tmp Selected)
		{
			ArrayList<Snp> snpsInCluster = new ArrayList<Snp>();
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null) {
					snpsInCluster.add(this.snps[i]);
				}
			}
			st.computeMaxObligates(snpsInCluster);
		}
		
//		BinEdgeComp c = new BinEdgeComp();
		
		ArrayList<BinEdge> ar2Choose = new ArrayList<BinEdge>();
		HashMap<Boolean, Integer> haveSnp = new HashMap<Boolean,Integer>();
		
		// Find SNPs that are the only possible choices for at least one bin.
		HashSet<Integer> snpInBinWithNoChoice = this.snpsThatAreOnlyChoice(); // Only for SNPs in edges.
		
		// if a SNP is cosmopolitan, it has to be in at least one bin edge and
		// hence must be choosable by two bins.
		// We are looking for any tag SNP that is choosable and that is needed by
		// at least one of the bins. If it is needed by no bins in any bin-edge,
		// then we have no reason to choose it because this implies that every bin
		// that this SNP is part of is full.
		int[] nToChooseList = new int[this.edges.length];
		int[] nToChooseFromList = new int[this.edges.length];
		ArrayList<int[]> snpsToChooseList= new ArrayList<int[]>(); 
		ArrayList<Integer> snpIdsToIterate = new ArrayList<Integer>();
		HashMap<Integer,Integer> snpMap = new HashMap<Integer,Integer>();
		int nSNPs2Loop=0;
		int nMin=0;
		for(int i=0;i<this.edges.length;i++) {
			// Only include edge if BOTH bins still need SNPs.
			if(this.edges[i]!=null) {
				// Check if bin edge has more unselected tagSNPs (which should be common to both bins
				// and at least one bin needs more tag SNPs 
				// and both bins have tag snps that can be picked.
				BinEdge be = this.edges[i];
				if(be!=null) {
					if(be.isReal()) {// Real Edges corresponds to ld bins
						// Get list of SNPs to choose from (shared betwen bins)
						// and number of tag SNPs needed.
						int[] toChoose= be.getSnpIdsToChoose(false); // false means not to count TMP selected tag SNP
						int nToChoose = toChoose[1]; // Number of tag SNPs in common that we have to choose in at least one of the bins.
						int nToChooseInBothBins = toChoose[2]; // number of tag SNPs in common that are not yet selected or excluded.

			
						// generate the list of Bin Edges that have to be iterated
						//  over while all the other SNPs get fixed.
						//
						if(nToChoose>0) {
							int nToChooseFrom = toChoose[0]; // number of tag SNPs in common that are not yet selected or excluded.
							assert nToChoose <= nToChooseFrom : "Error in be.getSnpIdsToChoose, nToChoose>nToChooseFrom" ;
							// We have figured out (in snpsThatAreOnlyChoice) that some tag SNPs
							// have to be picked because there will be no other choice for some bin.
							// Subtract out those and figure out the real number of tag SNPs to choose from

							int[] toReallyChooseFrom = new int[nToChooseFrom];
							int nToReallyChooseFrom=0;
							for(int k=2;k<nToChooseFrom+2;k++) {
								int snpid = toChoose[k];
								if(! snpInBinWithNoChoice.contains(Integer.valueOf(snpid))) {
									toReallyChooseFrom[nToReallyChooseFrom++]=snpid;
								}
							}
							int nChosen = nToChooseFrom-nToReallyChooseFrom;
							int nToReallyChoose = nToChoose-nChosen;
							if(nToChooseFrom>0 && nToReallyChooseFrom>0) {
								if(ar2Choose.contains(be)) {
									ar2Choose.add(be); // might have been added in a previous iteration
								}
								int idx = ar2Choose.indexOf(be);
								nToChooseList[idx]=nToReallyChoose;
								nToChooseFromList[idx]=nToReallyChooseFrom;
								snpsToChooseList.add(idx,toReallyChooseFrom);
								int firstSNPInEdge = nSNPs2Loop;
								nSNPs2Loop+=nToReallyChooseFrom;
								nMin+=nToReallyChoose;
								for(int j=0;j<nToReallyChooseFrom;j++) {
									Integer key = Integer.valueOf(toReallyChooseFrom[j]);
									if(!snpIdsToIterate.contains(key)) {
										snpIdsToIterate.add(key);
									}
									int snppos = snpIdsToIterate.indexOf(key);
									snpMap.put(Integer.valueOf(firstSNPInEdge+j),Integer.valueOf(snppos));
								}
							} 
						}
					}
				}
			}
		}
		// Count the total number of (non-unique) SNPs to Choose.
		if(snpInBinWithNoChoice!=null) {
			Iterator<Integer> it = snpInBinWithNoChoice.iterator();
			while(it.hasNext()) {
				Integer sid = it.next();
				Snp s = Snp.getSnpById(sid.intValue());
				s.setSelected(true, true);
			}
		}
		
		if(ar2Choose.size()==0) {
			//There are only (perhaps) snps to fix in snpInBinWithNoChoice
			//and we can then greedily pick all bins with remaining tag SNPs to pick.
			// Loop over all bins and tmp select independently best
			for(int ii=0;ii<bins.length;ii++) {
				Bin b = bins[ii];
				if(b!=null) {
					boolean tmpToo=false; 
					b.pickBinByProb(false,tmpToo); // pick all tag SNPs, not just 1
				}
			}
			return true; // nothing to pick ==> explored all solutions..
		} else {
			// a Bin-Edge is only in this list if one has to pick k out of n (n>k,n>0,k>0)
			// tag SNPs in the edge. all only-choice tag SNPs and k==n binEdges
			
			// Generate the list of (non-unique) snpids.
			if(snpIdsToIterate.size()>31) { // 32
				System.err.println("SNPPicker:Cluster:WARNING: Too many tag SNPs in common across bins for exhaustive solution, resorting to heuristic for cluster: "+ this.toString());
				this.pickBins(); // XXX If bin is singleton Cluster, then too close should be unfixed
//				System.out.println("pbe_gt31");System.out.flush();
				return false;
			} else {

				ClusterState cs = new ClusterState(this);
				
				// In order to consider best functional SNPs first, we
				// need to
				//	   1- Order the best SNPs in reverse order (so they end up in the most
				//         significant bits
				//     2 - Invert the mask (e.g. a "0" means we select the tag SNP), so that in effect we
				//         are counting backward
				
				int[] sortPos = sortSnpsFromSnpIds(snpIdsToIterate,false);
				// Create Ordered (Prob & Function) List of SNPs in BinEdge.
				Snp[] sortedSnps = new Snp[sortPos.length];
				for(int j=0;j<sortPos.length;j++) {
					Integer sid = snpIdsToIterate.get(sortPos[j]);
					Snp s = Snp.getSnpById(sid.intValue());
					sortedSnps[sortPos[j]]=s;
				}
				// Create maps fromBit and toBit, that map a SNP to a position in  the bin-edge.
				int[] fromBit = new int[snpMap.size()];
				int[] toBit = new int[snpMap.size()];
				{
					Set<Map.Entry<Integer, Integer>> ftme = snpMap.entrySet();
					Iterator<Map.Entry<Integer, Integer>> it = ftme.iterator();
					int ii=0;
					while(it.hasNext()) {
						Map.Entry<Integer, Integer> me = it.next();
						Integer iit = me.getKey();
						Integer ito = me.getValue();
						if(ito!=null) {
							fromBit[ii]=sortPos[ito.intValue()];
							toBit[ii++]=iit.intValue();
						}
					}
				}
				int nSnpsInMultipleBinEdges=fromBit.length-sortPos.length;
				int wantedMinSnps = nMin-nSnpsInMultipleBinEdges; // Absolute minimum possible SNPS to select
				LinkedHashSet<Integer> sss = new LinkedHashSet<Integer>();
				int nsnps = sortedSnps.length;
				for (int i = 0;i<nsnps;i++ ) {
					sss.add(Integer.valueOf(i));
				}
				PowerSet<Integer> pow = new PowerSet<Integer>(sss);
//				System.out.println("po");System.out.flush();
				Iterator<PowerSet<Integer>.BitMaskSet> it = pow.iterator();
				while(it.hasNext()) {
					edu.mayo.genotype.PowerSet<Integer>.BitMaskSet bm = it.next();
					int nelems = nsnps-bm.size(); // number of zeros
					int mask = bm.mask; 

					mask = ~mask; // swap 1's and zeros.
					long mapped_mask = Utils.mapBits(mask, fromBit,  toBit);


//					int nOn = Utils.countBits(mapped_mask);
					// Filter on at most nToChoose bits
					if(Utils.tooManyBitsSetInMask(mapped_mask, nToChooseFromList, nToChooseList)<=nSnpsInMultipleBinEdges) {
						// Coarse filter for acceptable solutions.
						
						for(int is=0;is<sortedSnps.length;is++) {
							sortedSnps[is].setTmpSelected(true, true);
						}
						
						Iterator<Integer> itp = bm.iterator();
						int i=0;
						while(itp.hasNext()) {
							int idx= itp.next().intValue();
							// unset those with a "1" (since we are inverting the sort)
							sortedSnps[idx].setTmpSelected(false, true);						
						}
	
						// Loop over all bins and tmp select independently best
						for(int ii=0;ii<bins.length;ii++) {
							Bin b = bins[i];
							if(b!=null) {
								b.pickTmpBinByProb(false);
							}
						}
						SolutionTracker.updateTmpScore(this.bins,st);
						//update Best Solution
						this.nsolns++;
						this.pickBestSoFar();// Evaluate solution and move to best solution if warranted.
						this.restoreState(cs);
						// Since we do not touch the bin or edge status, we do not need
						//    to restore the cluster state.
					}
				} // while (it.hasNext)
//				System.out.println("poF");System.out.flush();
				this.finalPick();// move best solution to final solution.
			} // else
		}// else
	
		return true; // explored all solutions..
	}
	
	/*
	 * @param snpIdsToIterate ArrayList of Integer snpids
	 * @param sortForward if true sort with best prob and function first, if false, reverse sort order.
	 * @return an int[] array specifying the new index of the jth SNP in input snpIdsToIterate
	 */

	private int[] sortSnpsFromSnpIds(ArrayList<Integer> snpIdsToIterate,boolean sortForward) {
		if(snpIdsToIterate==null) {
			return new int[0];
		}
		HashMap<Integer,Integer> snpId2Index = new HashMap<Integer,Integer>();
		int nsnps = snpIdsToIterate.size();
		if(nsnps==0) {
			return null;
		}
		int[] ret = new int[nsnps];
		if(nsnps==1) {
			ret[0]=0;
			return ret;
		}
		for(int i=0;i<nsnps;i++) {
			ret[i]=-1;
		}


		// Sort currently unselected SNPs by their probability.(obligates are not assumed to be selected)
		TreeSet<Snp> sortedByProb = new TreeSet<Snp>(new SnpScoreComparator());// TreeSet sorts the data..

		for(int i=0;i<nsnps;i++) {
			Integer is = snpIdsToIterate.get(i);
			if(is!=null) {
				int iis = is.intValue();
				snpId2Index.put(is, Integer.valueOf(i));
				Snp s = Snp.getSnpById(iis);
				if(s!=null) {
					sortedByProb.add(s);
				}
			}
		}

		Iterator<Snp> it = sortedByProb.iterator();
		int i=0;
		int last_elem=sortedByProb.size()-1;
		while(it.hasNext()) {
			Snp s =  it.next();
			int sid = s.getId();
			Integer posI = snpId2Index.get(Integer.valueOf(sid));
			if(sortForward) {
				ret[posI.intValue()]=i++;
			} else {
				ret[posI.intValue()]=last_elem-i;
				i++;
			}
		}
		return ret;

	}
	
	
	/*
	 * Find SNPs that are the only possible choices for a bin (for SNPs and bins in edges only)
	 * @return a HashSet with the Integer ids of the SNPs that should be picked no matter what.
	 */

	
	public HashSet<Integer> snpsThatAreOnlyChoice() {

		// Loop over all SNPs and find those who are part of at least one binEdge
		//        with no choice.
		HashSet<Integer> snpInBinWithNoChoice = new HashSet<Integer>();
		int nfixed=1;
		while(nfixed>0) {// The loop gets redone as long as new SNPs keep getting fixed.
			nfixed=0;
			
			// XXX Should rewrite this function to loop over this.snp2Edges
			// but this function is no longer needed
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null && this.snp2Edges!=null) {
					ArrayList<BinEdge> al = this.snp2Edges.get(String.valueOf(s.getId()));
					if(al!=null) {
						for(int j=0;j<al.size();j++) {
							BinEdge be = al.get(j);
							if((be!=null) && be.isReal()) {
								int[] toChoose= be.getSnpIdsToChoose(false);
								int nToChooseFrom = toChoose[0]; // Number of tag SNPs to choose from in overlap region
								int nToChoose = toChoose[1];// Max number of Tag SNPs to choose in overlap region
								int nToChooseBetween2Bins = toChoose[2];
								if(nToChoose>=nToChooseBetween2Bins) {// => Means that to fill those two bins, have to pick all overlapping tag SNPs.
									// Have to pick all those tag SNPs because there is no choice to make.

									boolean skipSNPs=true;
									if(SNPPicker.minP<1.0) {
										// ... unless there is a probability cutoff on one
										//  of the bins tied to this bin edge.. and
										//  the compound probability of the k-1 top prob SNPs are above
										//  the cutoff
										//						XXX This is an optimization that we have not finished coded for yet.
										//						int nmin =be.computeMinNumberOfSnpsLeftToChooseFrom(toChoose);// in both bin.
										//						if(nmin<nToChooseFrom) {
										//							skipSNPs=false;
										//						}
										skipSNPs=false;
									}
									if(skipSNPs) {
										for(int k=3;k<(nToChooseFrom+3);k++) {
											Integer key = Integer.valueOf(toChoose[k]);
											if(!snpInBinWithNoChoice.contains(key)) {
												snpInBinWithNoChoice.add(key);
												// We could check if this SNP is part of another Bin and not in SNP Edge.
												// because this is the only reason to restart the loop.
												nfixed++;
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		return snpInBinWithNoChoice;
	}
	

	/*
	 * unfix (fixed) beadTypeII SNPs in this cluster and return list of SNPs.
	 * 
	 */
	public int unFixAndUnexcludeTypeII() {
		HashSet<Snp> unFixedSnps = new HashSet<Snp>();
		if(this.bins!=null) {
			for(int i=0;i<this.length;i++) {
				Bin b = this.bins[i];
				if(b!=null) {
					Snp[] ss= b.getSnps();
					if(ss!=null) {
						for(int j=0;j<ss.length;j++) {
							Snp s = ss[j];
							if(s!=null) {
								if (s.getNI2beads()>1 && ! (s.isObligate() || s.isObligateButNotGenotyped())) {
									if(!(s.isExcluded() && s.isFixed())) {// Don't fix SNPs from Exclude file.
										if(s.isFixed() && !s.isExcluded()) {
											if(!unFixedSnps.contains(s)) {
												unFixedSnps.add(s);
												s.setFixed(false);
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
//		Snp[] toret = new Snp[unFixedSnps.size()];
//		return  unFixedSnps.toArray(toret);
		return unFixedSnps.size();
	}
	
	/*
	 * unfix beadTypeI  SNPs in this cluster and return list of SNPs.
	 *                 The only reason a SNP is Fixed "off" is because it is too close to another one.
	 */
	public int unFixAndUnexcludeTooCloseTypeI() {
		HashSet<Snp> unFixedSnps = new HashSet<Snp>();
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b =  this.bins[i];
				if(b!=null) {
					Snp[] ss= b.getSnps();
					if(ss!=null) {
						for(int j=0;j<ss.length;j++) {
							Snp s = ss[j];
							if(s!=null) {
								if (s.getNI2beads()==1 && ! (s.isObligate() || s.isObligateButNotGenotyped())) {
									if(!(s.isExcluded() && s.isFixed())) {// Don't fix SNPs from Exclude file.
										if(s.isFixed() && ! s.isExcluded()) {
											if(!unFixedSnps.contains(s)) {
												unFixedSnps.add(s);
												if(this.tooCloseSnpToClusterId.containsKey(s)) {
													//System.out.println("Deselected tc "+s.getDbsnpid());
												}
												s.setFixed(false);
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		//Snp[] toret = new Snp[unFixedSnps.size()];
		// return unFixedSnnps.toArray(toret);
		return unFixedSnps.size();
	}
	


	/*
	 * get non-selected nor fixed SNPs in this cluster and return list of SNPs.
	 * 
	 */
	public Snp[] getNotSelectedNorFixedSnps() {
		HashSet<Snp> unSelectedSnps = new HashSet<Snp>();
		if(this.bins!=null) {
			for(int i=0;i<this.bins.length;i++) {
				Bin b =  this.bins[i];
				if(b!=null) {
					Snp[] ss= b.getSnps();
					if(ss!=null) {
						for(int j=0;j<ss.length;j++) {
							Snp s = ss[j];
							if(s!=null) {
								if(!(s.isFixed() ||  s.isExcluded() ||  s.isSelected() || s.isObligate() || s.isObligateButNotGenotyped())) {
									if(!unSelectedSnps.contains(s)) {
										unSelectedSnps.add(s);
									}
								}
							}
						}
					}
				}
			}
		}
		Snp[] toret = new Snp[unSelectedSnps.size()];
		return  unSelectedSnps.toArray(toret);
	}

	/*
	 * fix and unfix SNPs in this cluster.
	 * Only fix unselected SNPs that are not obligate nor excludes
	 */
	private void fixUnselectedSnps(Snp[] ss,boolean newStatus) {
		if(ss!=null) {
			for(int j=0;j<ss.length;j++) {
				Snp s = ss[j];
				if(s!=null) {
					if(!(s.isExcluded())) {// Don't fix SNPs from Exclude file.
						if(! (s.isSelected() || s.isObligate() || s.isObligateButNotGenotyped())) {
							s.setFixed(newStatus);
						}
					}
				}
			}
		}
	}

	public void populateSnpArrayInCluster() {
		HashSet<Snp> clusterSnps = new HashSet<Snp>();
		HashMap<String,ArrayList<BinEdge>> snp2EdgesList= new HashMap<String,ArrayList<BinEdge>>();
		if(this.edges!=null) {
			for(int i=0;i<this.edges.length;i++) {
				BinEdge be = this.edges[i];
				if(be!=null) {
					Snp[] ss = be.getSnpids();
					if(ss!=null) {
						for(int j=0;j<ss.length;j++) {
							Snp s =  ss[j];
							if(s!=null && !(((s.isObligateButNotGenotyped() || s.isExcluded()) && SNPPicker.nOPA!=0) )) {
								if(!clusterSnps.contains(s)) {
									clusterSnps.add(s);
									this.orderedSnps.add(s);
									ArrayList<BinEdge> al = new ArrayList<BinEdge>();
									al.add(be);
									snp2EdgesList.put(String.valueOf(s.getId()),al);
								} else {
									ArrayList<BinEdge> al = snp2EdgesList.get(String.valueOf(s.getId()));
									al.add(be);
									snp2EdgesList.put(String.valueOf(s.getId()),al);
								}
							}
						}
					}
				}
			}
		}
		if (this.bins!=null) {
	// Fill in SNPs in the BIN, but not in the edges.
			for(int i=0;i<this.bins.length;i++) {
				Bin b = this.bins[i];
				if(b!=null) {
					Snp[] ss = b.getSnps();
					if(ss!=null) {
						for(int j=0;j<ss.length;j++) {
							Snp s =  ss[j];
							if(s!=null && !(((s.isExcluded() || s.isObligateButNotGenotyped()) && SNPPicker.nOPA!=0))) {
								if(!clusterSnps.contains(s)) {
									clusterSnps.add(s);
									this.orderedSnps.add(s);
								}
							}
						}
					}
				}
			}

		}
		
		int n_allsnps_in_cluster = clusterSnps.size();
		this.snps = clusterSnps.toArray(new Snp[n_allsnps_in_cluster]);
		this.snp2Edges = snp2EdgesList;
	}
	
	/**
	 * Function to output a uniquely identifying string for cluster.
	 */
	public String toString() {
		if(this.bins==null || this.bins.length==0) {
			if(this.snps==null || this.snps.length==0) {
				this.populateSnpArrayInCluster();
				if(this.snps==null || this.snps.length==0) {
					return "Empty Cluster";
				}
				StringBuffer sb = new StringBuffer();
				for(int i=0;i<this.snps.length;i++) {
					Snp s = this.snps[i];
					sb.append(";");
					if(s!=null) {
						sb.append(s.toString());
					}
				}
				return (String) ("SnpsCluster:"+sb.substring(1));
			}
		} else {
			StringBuffer sb = new StringBuffer();
			for(int i=0;i<this.bins.length;i++) {
				Bin b = this.bins[i];
				sb.append(";");
				if(b!=null) {
					sb.append(b.getName());
				}
			}
			return (String) ("BinsCluster:"+sb.substring(1));
		}
		return ""; // This line only to quiet the Eclipse warnings.
	}

	
	
	public void  fixSnps(Snp[] snpsToChangeFixedState,boolean newFixedState) {

		int nToChange = snpsToChangeFixedState.length;
		for(int i=0;i<nToChange;i++) {
				Snp s= snpsToChangeFixedState[i];
				if(s!=null) {
					s.setFixed(newFixedState);
				}
		}
	}

	/**
	 * @return the nclusters
	 */
	public static int getNclusters() {
		return nclusters;
	}
	public static Cluster getClusterForSnp(Snp s) {
		if(s!=null) {
			return snpId2Cluster.get(Integer.toString(s.getId()));
		} else{
			return null;
		}
	}


	private int semiOptimalSoln(Snp[] inSoln, Snp[] notInSoln,SolutionTracker soln,HashMap<Bin,Integer> bins2pick, HashMap<Snp,ArrayList<Snp>> overlapDiscarded,boolean lastCall ) throws Exception {
		isSemiOptimal=true;
		
		//	int[] inSolnOrder = Snp.getGreedyOrder(inSoln,true); // reverse order
		ArrayList<Snp> inSolnArr = new ArrayList<Snp>();
		if(inSoln!=null) {
			for(int i=0;i<inSoln.length;i++) {
				Snp s = inSoln[i];
				if(s!=null && !(s.isFixed() || s.isObligate())) {
					inSolnArr.add(s);
				}
			}
		}
		ArrayList<Snp> notInThisSolnArr = new ArrayList<Snp>();
		if(notInSoln!=null) {
			for(int i=0;i<notInSoln.length;i++) {
				Snp s = notInSoln[i];
				if(s!=null && !(s.isFixed() || s.isObligate())) {
					notInThisSolnArr.add(s);
				}
			}
		}

		int foundAtLeastOneBetter=1;
		int iteration=-1;
		System.out.println("semiOptimalSoln: starting for cluster "+Integer.toString(this.clusterid)+"\n");
		System.out.flush();
		long cpunow = System.currentTimeMillis();
		long lastcpu = cpunow + SNPPicker.maxCPU;
		while(foundAtLeastOneBetter>0 && (cpunow<lastcpu)) { // loop each time the reference solution changes
			foundAtLeastOneBetter=0;
			iteration++;
			Snp[] notInThisSoln = new Snp[notInThisSolnArr.size()];
			notInThisSoln = notInThisSolnArr.toArray(notInThisSoln);
			
			int[] notInSolnOrder = Snp.getGreedyOrder(notInThisSoln,false);
			for(int m=0;m<notInThisSoln.length;m++) {
				// Add one at a time.
				Snp snotin= notInThisSoln[notInSolnOrder[m]];
				int foundOne=0;
				boolean wasAlreadyTmpSelected=false;
				if(snotin!=null) {
					if(snotin.isFixed()) {
						continue;
					}
					if(snotin.isTmpSelected()) {
						wasAlreadyTmpSelected=true;
					} else {
						snotin.setTmpSelected(true, true);
					}
				}
				int retcode = SolutionTracker.updateTmpScore(bins2pick,soln,false);
				if(retcode==1) {// Corresponds to adding one and not remove any other
					SolutionTracker.moveToBestSoFar(bins2pick);
					foundOne=1;
					foundAtLeastOneBetter=1;
				}
				int[] whichBetter = null;
				int maxToChoose=inSolnArr.size();
				if(maxToChoose>SNPPicker.maxBinPop) {
					maxToChoose=SNPPicker.maxBinPop; // max number to simultaneously remove.
					if(maxToChoose<3) {
						maxToChoose=3;
					} else if (maxToChoose>10) {
						maxToChoose=10;
					}
				}
				for(int nToChoose =1;nToChoose<=maxToChoose && (cpunow<lastcpu);nToChoose++) {
					binomialCombinations bc = new binomialCombinations(nToChoose,inSolnArr.size());
					
					Iterator<int[]> itn= bc.iterator();

					while(itn.hasNext() && (cpunow<lastcpu) ) {
						int[] keepFlag= itn.next();
						int nFixed=0;
						for(int n=0;n<inSolnArr.size();n++) {
							if(keepFlag[n]==1) {
								Snp sin=inSolnArr.get(n);
								if(sin!=null) {
									if(sin.isFixed()) {
										nFixed++;
									} else {
										sin.setTmpSelected(false, true);
									}
								}
							}
						}	
						retcode = SolutionTracker.updateTmpScore(bins2pick,soln,false);
						if(retcode==1 && nFixed==0) {
							// solution was transfered to {Snp. and Bin.}->best-so-far if get here.
							SolutionTracker.moveToBestSoFar(bins2pick);
							whichBetter=keepFlag;
							foundOne=1;
							foundAtLeastOneBetter=1;
						}
						for(int n=0;n<inSolnArr.size();n++) {
							if(keepFlag[n]==1) {
								Snp sin=null;
								sin=  inSolnArr.get(n);
								if(sin!=null && !sin.isFixed()) {
									sin.setTmpSelected(true, true);
								}
							}
						}
						cpunow = System.currentTimeMillis();
					} //while
					cpunow = System.currentTimeMillis();
					
				} // for (nToChoose

				if(foundOne==0) { // if adding that SNP did not generate a better solution, de-select it.
					if(snotin!=null)  {
						if(!wasAlreadyTmpSelected) {
							snotin.setTmpSelected(false, true);
						}
					}
				} else { // if adding and removing some SNPs generated a better score, now de-tmpselect again the ones that had to be deleted.
					if(whichBetter!=null) {
						int ndeselected=0;
						for(int n=inSolnArr.size()-1;n>=0;n--) {// have to loop backwards to remove
							if(whichBetter[n]==1) {
								Snp sin=null;
								sin=  inSolnArr.get(n);
								if(sin!=null) {
									ndeselected++;
									sin.setTmpSelected(false, true);
									inSolnArr.remove(n); // no longer part of the solution
									notInThisSolnArr.add(sin);
								}
							}
						}
						System.out.println("semiOptimalSoln: iteration "+iteration+" added "+m+(m==2 ? "nd": (m==1 ? "st" : (m==3 ? "rd" : "th")))+" SNP and removed "+ndeselected+" SNPs");
						System.out.flush();
					} else {
						System.out.println("semiOptimalSoln: iteration "+iteration+" added "+m+(m==2 ? "nd": (m==1 ? "st" : (m==3 ? "rd" : "th")))+" SNP and removed none");
						System.out.flush();
					}
					inSolnArr.add(snotin); // Now part of the solution, so it is fair game to try to remove int.
					// remove from the not in this soln
					// delete the one we just added from the notInThisSoln
					notInThisSolnArr.remove(snotin); // p.s. cannot remove it by index
				
				} // if (foundOne)
			}// loop over nToChoose
			if(foundAtLeastOneBetter>0) {
				this.finalPick();
				this.eraseTmpSolution(); // No SNPs tmpSelected.
			}
			cpunow = System.currentTimeMillis();
		} // loop over while(foundAtLeastOneBetter)

		if(lastCall) {
			double Pvalue = compareSemiOptimalSolnWithRandom( soln,bins2pick,inSoln, notInSoln );
			if(Pvalue>0.0d) {
				this.finalPick();
				this.eraseTmpSolution(); // No SNPs tmpSelected.
				if(foundAtLeastOneBetter==0) {
					foundAtLeastOneBetter++;
				}
			}
		}
		return foundAtLeastOneBetter; // this will force relooping.
	}	

	/*
	 * return a P-value, except that this P-value is biased toward good solutions.
	 */
	private double compareSemiOptimalSolnWithRandom(SolutionTracker optimsoln,HashMap<Bin,Integer> bins2pick, Snp[] inSoln, Snp[] notInSoln ) throws Exception {
	
		// Don't use fixed SNPs.
		if(SNPPicker.nRandomComplex==0) {
			return 0.0d;
		}
		nSemiOptimal++;

		HashSet<Snp> snpsToChooseFrom = new HashSet<Snp>(2*this.snps.length+1);
		if(inSoln!=null) {
				for(int i=0;i<inSoln.length;i++) {
					Snp s=inSoln[i];
					if(s!=null && ! s.isFixed()) {
						snpsToChooseFrom.add(s);
					}
				}
		}
		if(notInSoln!=null) {
			for(int i=0;i<notInSoln.length;i++) {
				Snp s=notInSoln[i];
				if(s!=null && ! (s.isFixed() || s.isExcluded())) {
					snpsToChooseFrom.add(s);
				}
			}
		}	

		
		HashSet<Snp> theseSnps = new HashSet<Snp>(2*this.snps.length+1);
		HashSet<Snp> fixedSnpsSet = new HashSet<Snp>(2*this.snps.length+1);
		HashMap<Snp,Boolean> snps2Selected = new HashMap<Snp,Boolean>(2*this.snps.length+1);
		HashSet<Snp> toUnfix0 = new HashSet<Snp>(2*this.snps.length+1);
		
		{// make SNP list
			Iterator<Bin> it = bins2pick.keySet().iterator();
			while(it.hasNext()) {
				Bin b = it.next();
				Snp[] snps = b.getSnps();
				if(snps!=null) {
					for(int i=0;i<snps.length;i++) {
						Snp s = snps[i];
						if(s!=null) {
							if(snpsToChooseFrom.contains(s)) {
								theseSnps.add(s);
								if(s.isFixed()) {
									fixedSnpsSet.add(s);
								} else {
									snps2Selected.put(s, Boolean.valueOf(s.isSelected()));
								}
							} else {
								fixedSnpsSet.add(s);
								if(!s.isFixed()) {
									toUnfix0.add(s);
									s.setFixed(true);
								}
							}
						}
					}
				}
			}
		}
		int[] fixedsnpids = new int[fixedSnpsSet.size()];
		{ // Make fixed SNPs list, that has to be refixed during random solutions generation
			Iterator<Snp> itf = fixedSnpsSet.iterator();
			int i=0;
			while(itf.hasNext()) {
				Snp s = itf.next();
				fixedsnpids[i++]=s.getId();
			}
		}
	
		int nPanel=0;
		{ // count selected (among inSoln and notInSoln)
			Iterator<Snp> it = theseSnps.iterator();
			while(it.hasNext()) {
				Snp s = it.next(); 
				if(s.isSelected() || s.isObligate()) {
					nPanel++;
				}
			}
		}
		int nsnps = Snp.getNsnps();
		int nbins = Bin.getNbins();
	
		int nBetterSolutions=0;

		boolean[] randomlyPickedBin = new boolean[nbins]; // index by binid. - Keep Track of Bins Picked. (each Bin is only "visited" once.. though multiple tag SNPs could be selected).
		boolean[] randomlyPickedSnp = new boolean[nsnps]; // index by SNPid - Keep Track of Snps Picked (each SNP is only picked once, even when across bins).
	
		String[] splitFile = Utils.splitFile(SNPPicker.foutname);
	
		int[] excludedIndex = Snp.getExcludedIndex();// snpid is 1+index
		int[] obligateIndex =  Snp.getObligatesIndex();
		int[] obligateNotGenotypedIndex = Snp.getObligatesNotGenotypedIndex();
	
		// Save current optimized solution
		// Gotta copy the solution to TMP in order to score it.
	
		Bin[] binsToPickList = bins2pick.keySet().toArray(new Bin[bins2pick.size()]);
	
	
		SolutionTracker bestSolution = new SolutionTracker();

		SolutionTracker.updateTmpScore(bins2pick,bestSolution);
		// Figure out score for top nPanel SNPs.
		double optimizedScore=bestSolution.getScore(); // Score of first nSNPsToPick Snps.
		int bestCoverage = bestSolution.getNbins();
		int bestNMissing = bestSolution.getNMissing();
		//int optimNecessaryNotPicked = bestSolution.getNNecessaryButNotPicked();
		int optimObligates = bestSolution.getNObligates();
	
		
		
		java.util.Random randg = new java.util.Random();
	
	
		String randomScoreFile =splitFile[0]+splitFile[1]+ "_randomscores_"+Integer.toString(this.clusterid)+"_"+Integer.toString(nSemiOptimal)+"."+splitFile[2];
		PrintStream randomTagStream=null;
		try {
			randomTagStream = new PrintStream(new BufferedOutputStream(new FileOutputStream(randomScoreFile)));
		} catch (FileNotFoundException fnfe) {
			String sele = fnfe.getMessage();
			System.err.println(sele);
			System.err.flush();
			System.exit(-1);
		}
		randomTagStream.println(Double.toString(optimizedScore)+","+Integer.toString(nPanel)+","+Integer.toString(bestCoverage)+","+Integer.toString(bestNMissing));
		java.util.Random randbin=new java.util.Random();
		HashSet<Snp> toUnfix= new HashSet<Snp>(2*this.length);
	
		for(int i=0;i<SNPPicker.nRandomComplex;i++) {
	
			int nRandomlyPicked=0;
			//reset solutions and counter
			Bin.resetBinsExceptBestAndFixed(binsToPickList); // Reset All Solutions (keep obligates and excluded by Score)
			for(int ib=0;ib<binsToPickList.length;ib++) {
				randomlyPickedBin[ib]=false; //  indexed wrt to binsToPickList
			}
			for(int is=0;is<nsnps;is++) { // indexed by snpid
				randomlyPickedSnp[is]=false;
			}
			for(int is=0;is<fixedsnpids.length;is++) {
				randomlyPickedSnp[fixedsnpids[is]-1]=true; // don't pick fixed (leave as is).
			}
	
			try {
				// Pick Obligates, Exclude Excluded
				for(int ii=0;ii<excludedIndex.length;ii++) {
					int snpindex = excludedIndex[ii];
					Snp s = Snp.getSnpById(snpindex+1);
					s.setExcluded(true);
					randomlyPickedSnp[snpindex]=true; // Set it to true, so we we cannot pick it again.
				}
				for(int ii=0;ii<obligateIndex.length;ii++) {
					int snpindex = obligateIndex[ii];
					Snp s = Snp.getSnpById(snpindex+1);
	
					s.setObligate(true);
					s.setSelected(true, true);
					s.setTmpSelected(true, true);
					randomlyPickedSnp[snpindex]=true; // Obligate is picked
					if(theseSnps.contains(s)) { // only count it in inSoln and notInSoln
						nRandomlyPicked++;
					}
				}					
				for(int ii=0;ii<obligateNotGenotypedIndex.length;ii++) {
					int snpindex = obligateNotGenotypedIndex[ii];
					Snp s = Snp.getSnpById(snpindex+1);
					s.setObligateButNotGenotyped(true);
					s.setSelected(true, true);
					s.setTmpSelected(true, true);
					randomlyPickedSnp[snpindex]=true; // So we cannot pick it
				}
				{
					// unfix SNP fixed during previous random solution generation
					Iterator<Snp> it = toUnfix.iterator();
					while(it.hasNext()) {
						Snp s = it.next();
						if(s!=null && s.isFixed()) {
							s.setFixed(false);
						}
					}
					toUnfix.clear();
				}
			} catch (Exception e) {
				e.printStackTrace();
				String sele = e.getMessage();
				System.err.println(sele);
				System.err.flush();
				System.exit(-1);
			}
			{ // Shuffle all the bins
				if(binsToPickList!=null && binsToPickList.length>0) {
					for(int ir =0;ir<binsToPickList.length;ir++) {
						int jj = randg.nextInt(binsToPickList.length);
						Bin tmpB = binsToPickList[jj];
						binsToPickList[jj]=binsToPickList[ir];
						binsToPickList[ir]=tmpB;
					}
				} 
			}
			int ii=0;
			int[] topick =null;
			int nreshuffle=0;
			while(ii<binsToPickList.length && nRandomlyPicked<nPanel) {
				if(randomlyPickedBin[ii]==false) {
					randomlyPickedBin[ii]=true; // only visit each bin once.
					Bin b = binsToPickList[ii];
					if(b!=null) { // could be full from obligates or from overlapping
						topick = b.getAvailableSnpIds(true);
						int needed = b.getNSB();
						int nsel = b.getTmpNSelected();
						int nToSel =0;
	
						int irandom=0;
						if(SNPPicker.multipleRandomSnpsPerBin) {
							nToSel = Math.min(Math.min(needed-nsel,topick.length),nPanel-nRandomlyPicked);
							if(nToSel>0 && topick.length>0) {
								irandom = randbin.nextInt(topick.length);
							} else {
								nToSel=0;
							}
						} else if (nsel==0) {// if none selected, pick at least one
							if(topick.length>0) {
								nToSel=Math.min(1,needed);
								irandom = randbin.nextInt(topick.length);
							} else {
								nToSel=0;
							}
						}
						while(nToSel>=1) {
							if(topick.length>0) {
								Snp s = Snp.getSnpById(topick[irandom]);
								if(s!=null) {
									int sid = s.getId();
									try {
										if(randomlyPickedSnp[sid-1]==false) {
											if(s.canISelect(true)) {// check for tooClose
												s.setTmpSelected(true,true);
												nRandomlyPicked++;
												nToSel--;
											} else {
												s.setFixed(true);
												toUnfix.add(s);
											}
											randomlyPickedSnp[sid-1]=true; // exclude
											topick=b.getAvailableSnpIds(true);
										} else {
											s.setFixed(true);
											toUnfix.add(s);
											topick=b.getAvailableSnpIds(true);
										}
									} catch (Exception e) {
										e.printStackTrace();
										String sele = e.getMessage();
										System.err.println(sele);
										System.err.flush();
										System.exit(-1);
									}
								} else {
									topick=b.getAvailableSnpIds(true);
								}
								if(topick.length==0) {
									nToSel=0;
								} else {
									irandom = randbin.nextInt(topick.length);
								}
							} else {
								nToSel=0;
							}
						}
					}
				}
				ii++;
				if(binsToPickList!=null && ii==binsToPickList.length && nRandomlyPicked<nPanel && nreshuffle<10) {
					nreshuffle++;
					if(binsToPickList.length>0) {
						for(int ir =0;ir<binsToPickList.length;ir++) {
							int jj = randg.nextInt(binsToPickList.length);
							Bin tmpB = binsToPickList[jj];
							binsToPickList[jj]=binsToPickList[ir];
							binsToPickList[ir]=tmpB;
						}
					}
					for(int ib=0;ib<binsToPickList.length;ib++) {
						randomlyPickedBin[ib]=false;
					}
					for(int is=0;is<nsnps;is++) {
						randomlyPickedSnp[is]=false;
					}
				}
			}
			SolutionTracker randomSolution = new SolutionTracker();
			SolutionTracker.updateTmpScore(bins2pick,randomSolution);
			// Figure out score for top nPanel SNPs.
			double randomScore=				randomSolution.getScore(); // Score of first nSNPsToPick Snps.
			int randomCoverage = 			randomSolution.getNbins();
			int randomNMissing = 			randomSolution.getNMissing();
			//			int randomNecessaryNotPicked =	randomSolution.getNNecessaryButNotPicked();
			int randomObligates = 			randomSolution.getNObligates();
			if(randomCoverage>=bestCoverage && randomNMissing<=bestNMissing && nRandomlyPicked<=nPanel &&   randomScore>optimizedScore 
					//					&& randomNecessaryNotPicked <=optimNecessaryNotPicked 
					&& randomObligates >=optimObligates) {
				nBetterSolutions++;
				int retcode = SolutionTracker.updateTmpScore(bins2pick,bestSolution);
	
				if(retcode==1) {
					// solution was transfered to {Snp. and Bin.}->best-so-far if get here.
					SolutionTracker.moveToBestSoFar(bins2pick);
					System.out.println("Clusterid="+Integer.toString(this.clusterid)+",Nsnps="+this.snps.length+",nbins="+this.length+",Random_score=" + randomScore+"("+optimizedScore+"), Random_nsnps=" +nRandomlyPicked+"("+nPanel+"), randomCoverage="+randomCoverage+"("+bestCoverage+"), randomNMissing="+randomNMissing+"("+bestNMissing+"),inSoln="+(inSoln==null? 0: inSoln.length)+", notInSoln="+(notInSoln==null? 0 : notInSoln.length));
				}
			}
	
			randomTagStream.println(Double.toString(randomScore)+","+Integer.toString(nRandomlyPicked)+","+Integer.toString(randomCoverage)+","+Integer.toString(randomNMissing));
	
		}
		randomTagStream.flush();
		randomTagStream.close();
		double Pvalue = (1.0d*nBetterSolutions)/(SNPPicker.nRandomComplex+1.0d);
		System.out.println("Clusterid="+Integer.toString(this.clusterid)+",Nsnps="+this.snps.length+",nbins="+this.length+",Better_Solutions=" +Integer.toString(nBetterSolutions)+" / "+SNPPicker.nRandomComplex+", Pvalue="+Pvalue);
		System.out.flush();
	
		if(toUnfix!=null){
			// unfix SNP fixed during random solution generation
			Iterator<Snp> it = toUnfix.iterator();
			if(it!=null) {
				while(it.hasNext()) {
					Snp s = it.next();
					if(s!=null) {
						s.setFixed(false);
					}
				}
				toUnfix.clear();
			}
		}
		if(toUnfix0!=null){
			// unfix SNP that were not in inSoln or notInSoln and were not already fixed
			Iterator<Snp> it = toUnfix0.iterator();
			if(it!=null) { 
				while(it.hasNext()) {
					Snp s = it.next();
					if(s!=null) {
						s.setFixed(false);
					}
				}
			}
			toUnfix0.clear();
		}
		
		if(nBetterSolutions>0) {
		// either restore best-so-far solution or take new one.
			this.finalPick();
			this.eraseTmpSolution(); // No SNPs tmpSelected.
		} else {
			// restore to original Solution
			Iterator<Map.Entry<Snp, Boolean>> it = snps2Selected.entrySet().iterator();
			while(it.hasNext()) {
				Map.Entry<Snp, Boolean> me = it.next();
				Snp s=me.getKey();
				Boolean status = me.getValue();
				boolean sel = status.booleanValue();
				if(sel && !s.isSelected()) {
					s.setSelected(true, true);
				} else if((!sel) && s.isSelected()) {
					s.setSelected(false, true);
				}
			}
		}
		//maxcoverate and maxScore is messed up by this function.
	
		return Pvalue; // return a P-value and the updated so
	
}


}