package edu.mayo.genotype;

/**
 * This Class converts input scores into probabilities and probability bins.
 * a SNP is defined by a score, and a Designability category, and a Validation_Class, and a location Class
 * This function stores and create a way to convert the scores into a probability
 * @author Hugues Sicotte
 *
 */

import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.io.BufferedReader;
public class ProbabilityFromScore {
	private static double risk_aversion=1.0; // risk_neutral = 1.0
	// Defaults are for Illumina 2006 Golgen Gate Design
	// data is extracted/infered from figures from R. Ingersoll et all "Factors associated with Success of Custom SNP Assays"
	//       The Core SNP Center, Institute of Genetic Medecine, John Hopkins
	// http://snpcenter.grcf.jhmi.edu/downloads/ILMNposter.pdf, retrieved jan 29 2007
	
	private static String[] validationClass = {"1","2","3","4","5"}; // For Illumina 1-not validated, 2-validated, 3 golden-gate validated, 5-iNFINIUM ii
	private static String[] scoreClass = {"0","0.25","0.5","0.75","1"};
	// Score Class >= Lower and <ruleUpper
	private static double[] ruleLower = {0.0,0.2,0.4,0.6,0.8};
	private static double[] ruleUpper = {0.2,0.4,0.6,0.8,1.11};

	private  static HashMap<String,Double> probs = new HashMap<String,Double>();
	static {		// Probability of Failure given Class and score Class.
		probs.put("P(F|0,5)", new Double(3*0.51*7/17));
		probs.put("P(F|0,4)", new Double(3*0.51*7/17));
		probs.put("P(F|0,3)", new Double(3*0.51*7/17));
		probs.put("P(F|0,2)", new Double(3*0.51*13/37));
		probs.put("P(F|0,1)", new Double(3*0.51*17/37));

		probs.put("P(F|0.25,5)", new Double(3*0.41*7/17));
		probs.put("P(F|0.25,4)", new Double(3*0.41*7/17));
		probs.put("P(F|0.25,3)", new Double(3*0.41*7/17));
		probs.put("P(F|0.25,2)", new Double(3*0.41*13/37));
		probs.put("P(F|0.25,1)", new Double(3*0.41*17/37));
		
		probs.put("P(F|0.5,5)", new Double(3*0.4*7/17));
		probs.put("P(F|0.5,4)", new Double(3*0.4*7/17));
		probs.put("P(F|0.5,3)", new Double(3*0.4*7/17));
		probs.put("P(F|0.5,2)", new Double(3*0.4*13/37));
		probs.put("P(F|0.5,1)", new Double(3*0.4*17/37));

		probs.put("P(F|0.75,5)", new Double(3*0.24*7/17));
		probs.put("P(F|0.75,4)", new Double(3*0.24*7/17));
		probs.put("P(F|0.75,3)", new Double(3*0.24*7/17));
		probs.put("P(F|0.75,2)", new Double(3*0.24*13/37));
		probs.put("P(F|0.75,1)", new Double(3*0.24*17/37));


		probs.put("P(F|1,5)", new Double(3*0.05*7/37));
		probs.put("P(F|1,4)", new Double(3*0.05*7/37));
		probs.put("P(F|1,3)", new Double(3*0.05*7/37));
		probs.put("P(F|1,2)", new Double(3*0.05*13/37));
		probs.put("P(F|1,1)", new Double(3*0.05*17/37));	
		
		// probability for SNPs without positive score (user can supply a default and avoid this
		
		probs.put("P(F|,5)",new Double(0.0));
		probs.put("P(F|,4)",new Double(0.0));
		probs.put("P(F|,3)",new Double(0.0));
		probs.put("P(F|,2)",new Double(0.0));
		probs.put("P(F|,1)",new Double(0.0));
		
		
	}
	private static int locationClassMax=15;
	
	private static int locationClassMin=-5;
	
	private static HashMap<String,Integer> locationClass=new HashMap<String,Integer>();
	
	static  {
		 locationClass.put("nonsyn-linked",new Integer(15)); //); // Linked to a disease (suggest use obligate instead)
		 locationClass.put("nonsyn-intolerant",new Integer(10)); //); // computational prediction of intolerance
		 locationClass.put("nonsyn-probablydamaging",new Integer(10)); //); // computational prediction of intolerance.
		 locationClass.put("nonsyn-possiblydamaging",new Integer(10)); //); //
		 locationClass.put("coding-nonsyn",new Integer(10)); // ("nonsyn-tolerant",unknown syn)"); 
		 locationClass.put("nonsyn-tolerant",new Integer(8)); //
		 locationClass.put("nonsyn-benign",new Integer(8)); //
		 locationClass.put("nonsyn",new Integer(8)); //
		 locationClass.put("splice",new Integer(8)); //
		 locationClass.put("factor",new Integer(5)); //
		 locationClass.put("polyasignal",new Integer(5)); //
		 locationClass.put("initiation",new Integer(5)); //
		 locationClass.put("genomicconserved",new Integer(5)); //
		 locationClass.put("coding-unknown",new Integer(4)); //// whether SYN or non-syn
		 locationClass.put("3utr",new Integer(4)); //
		 locationClass.put("5utr",new Integer(3)); //
		 locationClass.put("utr",new Integer(3)); // Illumina category.. how come they don't know which UTR?
		 locationClass.put("flanking_3utr",new Integer(2)); // Illumina category
		 locationClass.put("flanking_5utr",new Integer(2)); // Illumina category
		 locationClass.put("flanking",new Integer(2)); // Illumina category
		 locationClass.put("coding-synon",new Integer(2)); // synonymous substitution
		 locationClass.put("synon",new Integer(2)); // synonymous substitution
		 locationClass.put("intron",new Integer(1)); //
		 locationClass.put("unspecified",new Integer(0)); // This is the default for no preference.
		 locationClass.put("unknown",new Integer(0)); // This is the default for no preference.
		 locationClass.put("repeat",new Integer(-1)); // (hard to genotype)
		 locationClass.put("microsat",new Integer(-1)); //(hard to genotype)
		 locationClass.put("lowcomplexity",new Integer(-1)); //(hard to genotype)
		 locationClass.put("indel",new Integer(-5)); // impossible to genotype on most platforms.
		 locationClass.put("msnp",new Integer(-5)); // impossible to genotype on most platforms
	}
	private static String scoreClassFromScore(String score) {


			double f = 0.0;
			try {
				f= Float.parseFloat(score);
			} catch (Exception e) {
				f=0.0;
			}
			int indx=0;
			int closest = -1;
			double dist=Float.MAX_VALUE;
			boolean foundClass=false;
			for(int i=0;i<ruleLower.length;i++) {
				if(f>=ruleLower[i] && f<ruleUpper[i]) {
					indx=i;
					foundClass=true;
					break;
				} else {
					double d = f-ruleLower[i];
					if(d>=0.0 && d<dist) {
						dist=d;
						closest =i;
					}
				}
			}
			if(!foundClass) {
				if(f>=ruleUpper[ruleUpper.length-1]) {
					return scoreClass[ruleUpper.length-1];
				} else if(closest>=0) {
					return scoreClass[closest];
				} else {
					return "";
				}
			}
			return scoreClass[indx];
	}
	public ProbabilityFromScore() {

	}
	
	/**
	 * Validate that a validation class is within defined classes.
	 * @param classId
	 * @return
	 */
	public static boolean validateValidationClass(String classId) {
		for(int i=0;i<validationClass.length;i++) {
			if(validationClass[i].equalsIgnoreCase(classId)) {
				return true;
			}
		}
		return false;
		
	}
	public static double combineFailProb(double[] failProbs) {
		double  fp = 1.0;
		if(failProbs.length==0) {
			return fp; // No SNP in bin ==> 100% failure probability.
		}
		for(int i=0;i<failProbs.length;i++) {
			fp=fp*failProbs[i];
		}
		if(fp>1.0) {
			fp=1.0;
		} else if (fp<0.0) {
			fp=0.0;
		}
		return fp;	
	}
	public static double combineSuccessProb(double[] succProbs) {
		double  fp = 1.0;
		// This is simply Chevalier's problem.
		// To combine the Prob of success 1-product(failure probs))
		if(succProbs.length==0) {
			return 0.0; // No SNP in bin ==> 0% success prob.
		}
		for(int i=0;i<succProbs.length;i++) {
			double f = 1-succProbs[i];
			if(succProbs[i]>1.0) {
				f=1.0;
			} else if (f<0.0) {
				f=0.0;
			}
			fp *=f;
		}
		if(fp>1.0) {
			fp=1.0;
		} else if (fp<0.0) {
			fp=0.0;
		}
		return 1.0-fp;	
	}
	
	public static double getFailProb(double score,String validationClass) {
		String scoreClass = scoreClassFromScore(Double.toString(score));
		if(score == SNPPicker.badscore_double) {
			return 1.0;
		} else {
			return getFailProb(scoreClass,validationClass);
		}
	}
	
	public static double getFailProb(String scoreClass,String validationClass) {
		StringBuffer key = new StringBuffer("P(F|");
		key.append(scoreClass);
		key.append(",");
		key.append(validationClass);
		key.append(")");
		if(probs.containsKey(key.toString())) {
			Double f = (Double) probs.get(key.toString());
			return f.doubleValue();
		} else {
			System.err.println("SNPPicker:WARNING: Could not compute Probability from score "+scoreClass+", and class "+validationClass);
			return 	SNPPicker.DEFAULT_FAIL_PROB ; 
		}
	}
	
	// Parses "baby"-XML. 1 item/line.
	
	public  static void parseParameters(BufferedReader in) throws Exception {
		// This function should be called after having read the "<PROBABILITIES>" tag
		probs.clear();
		String line;
		int vartype=0;
		while((line=in.readLine())!=null) {
			line = line.trim();
			line.replaceAll("[ ,\\t,\\n,\\r]+","");
			if(line.length()>1) {
				if(line.indexOf("</PROBABILITIES>")>=0) {
					return;
				}
				int i=0;
				if((i=line.indexOf("<VALIDATIONCLASS>"))>0) {
					vartype=1;
					line = line.substring(i+"<VALIDATIONCLASS>".length());
				} else if((i=line.indexOf("<SCORECLASS>"))>0) {
					vartype=2;
					line = line.substring(i+"<SCORECLASS>".length());
				} else if((i=line.indexOf("<RULELOWER>"))>0) {
					vartype=3;
					line = line.substring(i+"<RULELOWER>".length());
				}else if((i=line.indexOf("<RULEUPPER>"))>0) {
					vartype=4;
					line = line.substring(i+"<RULEUPPER>".length());
				} else if((i=line.indexOf("<PROBS>"))>0) {
					vartype=5;
					line = line.substring(i+"<PROBS>".length());
				} else if((i=line.indexOf("</"))>0) {
					vartype=0;				
				} else {
					if(vartype==1) {
						validationClass = Utils.split(line,',');
					} else if (vartype==2) {
						scoreClass= Utils.split(line,',');
					} else if (vartype==3) {
						String[] ruleLStr =  Utils.split(line,',');
						ruleLower = new double[ruleLStr.length];
						for(int k=0;k<ruleLStr.length;k++) {
							ruleLower[k]=Double.parseDouble(ruleLStr[k]);
						}
					} else if (vartype==4) {
						String[] ruleUStr =  Utils.split(line,',');
						ruleUpper = new double[ruleUStr.length];
						for(int k=0;k<ruleUStr.length;k++) {
							ruleUpper[k]=Double.parseDouble(ruleUStr[k]);
						}
					} else if(vartype==5) {
						String[] probsStr =  Utils.split(line,',');
						probs.put(probsStr[0],new Double(probsStr[1]));
					}
				}
			}
		}
	}
	public static double getRisk_aversion() {
		return risk_aversion;
	}
	public static void setRisk_aversion(double risk_aversion) {
		ProbabilityFromScore.risk_aversion = risk_aversion;
	}
	public static double[] getRuleLower() {
		return ruleLower;
	}
	public static void setRuleLower(double[] ruleLower) {
		ProbabilityFromScore.ruleLower = ruleLower;
	}
	public static double[] getRuleUpper() {
		return ruleUpper;
	}
	public static void setRuleUpper(double[] ruleUpper) {
		ProbabilityFromScore.ruleUpper = ruleUpper;
	}
	public static String[] getScoreClass() {
		return scoreClass;
	}
	public static void setScoreClass(String[] scoreClass) {
		ProbabilityFromScore.scoreClass = scoreClass;
	}
	public static String[] getValidationClass() {
		return validationClass;
	}
	public static int getLocationClassMax() {
		return locationClassMax;
	}
	public static void setLocationClassMax(int locationClassMax) {
		ProbabilityFromScore.locationClassMax = locationClassMax;
	}
	public static int getLocationClassMin() {
		return locationClassMin;
	}
	public static HashMap<String,Integer> getLocationClass() {
		return ProbabilityFromScore.locationClass;
	}
	public static void setLocationClassMin(int locationClassMin) {
		ProbabilityFromScore.locationClassMin = locationClassMin;
	}
	public static HashMap<String,Double> getProbs() {
		return probs;
	}
	public static void setProbs(HashMap<String,Double> probs) {
		ProbabilityFromScore.probs = probs;
	}
	public static void setLocationClass(HashMap<String,Integer> lc) {
		ProbabilityFromScore.locationClass = lc;
	}
	/**
	 * 
	 * @return a hashmap with the ldselect mappings.
	 */
	public static HashMap<String,String> readClassMappings(Properties props,Set<String> ldselectKeys) throws Exception {
		Set<Object> propKeys = props.keySet();
		HashMap<String,String> ldSelectMappings = new HashMap<String,String>();
		HashMap<String,Integer> newClassMappings = new HashMap<String,Integer>();
		Iterator<Object> it = propKeys.iterator();
		while(it.hasNext()) {
			String key = (String) it.next();
			String val = props.getProperty(key);
			if(ldselectKeys.contains(key)) {
				ldSelectMappings.put(key, val);
			} else {
				try {
					newClassMappings.put(key, Integer.parseInt(val));
				} catch (Exception e) {
					System.err.println("Invalid Int format "+val+" in property file for functional Class Mappings");
					System.err.flush();
					throw new Exception("SNPPicker::edu.mayo.genotype.ProbabilityFromScore.readClassMappings: Invalid Float format "+val+" in property file for functional Class Mappings");
				}
			}
		}
		ProbabilityFromScore.setLocationClass(newClassMappings);
		return ldSelectMappings;
	}

	

	
	
	
}
