package edu.mayo.genotype;

import java.util.HashMap;
import java.lang.Math;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.TreeSet;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.PrintStream;

/**
 *
 * @author Hugues Sicotte
 *
 *
 */
public class Bin {
		// Class level variables for Object Manager
	  private static int nbins=0; // first binid will be 1 
	  private static HashMap<String,Bin> haveBin = new HashMap<String,Bin>(1000);
	  private static HashMap<String,Bin> synBin = new HashMap<String,Bin>(1000);

	  private static int indexSize=1000;
	  public static final int MAXBIN=(int) Math.floor(Math.sqrt((double) Long.MAX_VALUE));
	  private static Object[] binIndex =new Object[indexSize];; // Rebuildable index of all Bins

	  // Instance level variables

	  private String name;
	  private String synonym;

	  private int binid;
	  private int popid;
	  private int basePopId; // if the popid is for a "temp", then save the base PopId.
	  private int nSites=0; // Number of Snps sites tagged by this bin.
	  private String gene="";
	  private String geneId="";
	  private String source="";
	  private String chromosome = SNPPicker.chromosome ==null ? null : SNPPicker.chromosome.toUpperCase();
	  
	  // Number of Tag Snps in this Bin
	  private int nsnps=0;
	  private Snp[] snps=null;

	  // parameters
	  private int NSB=1; // Number of Snp per Bin (NSB) according to rules

      // Helper Variables for Exploring Solutions.
	  private int status=0; // Set to 1 if Bin is Full (NSB==nSelected || nSelected>=nsnps)

	  private int[] maxScoringSnpids=null; // List of snpsids touching that BIN in order of Probability(Score)
	  private Snp[] maxScoringSnps=null; // List of snpsids touching that BIN in order of Probability(Score)

	  private double maxScore=0.0;// Actually Probability.


	  private int[] maxCoverageSnpids=null;
//	  private double maxCoverageScore=0.0d;

	  //	 variables relating to picking SNPs

	  private int nSelected=0;  // Number of SNPs selected.
	  private int[] selectedSnpIds=null; // Snpids of Selected Snps..
	  private int nExcluded=0; // obligates exclude and fixed (by too-close algorithm).
	  private boolean full=false; // e.g. all Snps either excluded or Selected.


	  // Temporary Solution
	  private int tmpNSelected = 0; //Number of SNps selected during descent.
	  private int[] tmpSelectedSnpIds=new int[0]; // Snpids of Selected Snps.
	  private boolean hasBeenReset=false;


	  // Best-so-far
	  private int bestNSelected = 0; //Number of SNps selected for this bin (includes accidental overlap)
	  private int[] bestSelectedSnpIds=null; // Snpids of Selected Snps.
	  private boolean bestWasTransfered=false;

	  // Variables for/from clustering
	  private int nbinedges=0;
	  private BinEdge[] binedges=null; // Store the links to other bins, indexed by binid

	  // Used to Single-link Bins into Clusters.

	  private Bin nextbin=null; // reference to next object in the cluster
	  private Bin prevbin=null; // reference to the previous object in the cluster.

	/**
	   *
	   * @param npop requires total number of population to sim
	   * @param name e.g.dbsnpid
	   */
	  public Bin(String binname, int binPopid,int basePopid) throws Exception {
		  this.name=binname;
		  this.popid=binPopid;
		  nbins++;
		  this.binid=nbins;
		  haveBin.put(binname, this);
		  this.basePopId =basePopid;
		  updateIndex(this);
	  }
	  private static void updateIndex(Bin newBin) throws Exception{
		  // Index by binid, some elements may be null.
		  if(binIndex==null) {
			  indexSize=1000;
			  binIndex = new Object[indexSize];
			  for(int i=0;i<indexSize;i++) {
				  binIndex[i]=null;
			  }
		  } else if (nbins>indexSize) {
			  int newIndexSize=indexSize+1000;
			  Object[] tmp = new Object[newIndexSize];
			  int i=0;
			  for(;i<indexSize;i++) {
				  tmp[i]=binIndex[i];
			  }
			  for(;i<newIndexSize;i++) {
				  tmp[i]=null;
			  }
			  binIndex=tmp;
			  indexSize=newIndexSize;
		  }
		  binIndex[nbins-1]=newBin;
		  if(indexSize>Bin.MAXBIN) {
			  throw new Exception("SNPPicker: SEVERE ERROR: Number of Bins exceeds maximum number of bins "+Bin.MAXBIN);
		  }
	  }

	  public static boolean haveBin(String name) {
		  if(haveBin!=null && ( haveBin.containsKey(name) || synBin.containsKey(name) )) {
			  return true;
		  } else {
			  return false;
		  }
	  }
	  public void setSynonym(String newbinname) {
		  this.synonym = newbinname;
		  synBin.put(newbinname,this);
		  
	  }
	  public void changeBinName(String newbinname) {
		  if(this.name.indexOf("DupFile")>0) {
			  Exception e = new Exception();
			  e.printStackTrace();
			  System.err.println("SNPPicker: Error calling changeBinName, was called twice");
			  System.err.flush();
			  System.exit(-1);
		  }
		  this.synonym=this.name;
		  haveBin.remove(this.name);
		  haveBin.put(newbinname,this);
		  synBin.put(this.name,this);
		  this.name=newbinname;
	  }

	  public static Bin getBinByName(String binname) {
		  	Bin b =(Bin) haveBin.get(binname);
		  	if(b==null) {
		  		b = (Bin)synBin.get(binname);
		  	}
		  	return b;
	  }
	  public static Bin getBinById(int id) {
		    if(binIndex==null) {
		    	return null;
		    }
		  	if(id>0 && id<=Bin.nbins) {
		  		return (Bin) binIndex[id-1];
		  	} else {
		  		return null;
		  	}
	  }
	  
	 /**
	  * Delete Bin by Name - will take care of BinEdges
	  * @param binname
	  * @param recursive
	  */
	  public static void deleteByName(String binname,boolean recursive) {
		  Bin b = Bin.getBinByName(binname); //al searches synonym index.
		  if(b!=null) {
			  Snp[] bsnps = b.getSnps();
			  int binid=b.getId();
			  if(bsnps!=null) {
				  for(int i=0;i<bsnps.length;i++) {
					  Snp s = bsnps[i];
					  if(s!=null) {
						  s.deleteBin(binid,recursive); // will take care of the BinEdge deletion too
					  }
				  }
			  }
			  b.snps=null;
			  if(haveBin.containsKey(binname)) {
				  haveBin.remove(binname);
			  }
			  if(synBin.containsKey(binname)) {
				  synBin.remove(binname);
			  }
			  binIndex[binid-1]=null;
			  
			 
		  }
		  // do NOT decrement nbins!
	  }

	  public static void deleteSmallBins(int minSites) {
		  if(binIndex!=null) {
			  for(int j=0;j<binIndex.length;j++) {
				  Bin b = (Bin)Bin.binIndex[j];
				  if(b!=null) {
					  if(b.getNSites()<minSites) {
						  Snp[] bsnps = b.getSnps();
						  int binid=b.getId();
						  if(bsnps!=null) {
							  for(int i=0;i<bsnps.length;i++) {
								  Snp s = bsnps[i];
								  if(s!=null) {
									  s.deleteBin(binid,true); // recursive=true..will take care of the BinEdge deletion too
								  }
							  }
						  }
						 
						  if(haveBin.containsValue(b)) {
							  String bname =b.getName();
							  haveBin.remove(bname);
						  }
						  binIndex[binid-1]=null;
					  }
				  }
			  }
		  }
	  }
	  public static void clear() {
		  deleteSmallBins(0);
		  nbins=0;
		  haveBin.clear();
	  }


	  public void setPrevBin(Bin bin) {prevbin=bin;}
	  public Bin getPrevBin() {return prevbin;}

	  public int getId(){return binid;}

	  /**
	   * function to add a Bin without creating "collisions" if there are multiple Bins for that population
	   * @param binname the name of the bin.
	   * @param  snpNames a list of snpnames in this bin.
	   * @param areTag wether the list of SNPs are tagSNPs or non-tagSNPs
	   * @param the default (base) population id of this binname.
	   * @param  fileid Unique integer (String) or String (without "-") to identify the file
	   * @param filebinname bin name in file, e.g. an integer
	   * @param geneId Entrez gene id
	   * @param source Source of the genotype
	   */
	  
		public static Bin getNonCollidingBin(String binname,String[] snpNames,boolean areTag,int popid,String fileid,String filebinname,String geneId,String source,boolean fullbin) throws Exception {
			
			// even if bin already exists, we have to doublecheck wether or NOT we
			// have to swap out it's population for a "Dupfile" population.
			//
			Bin b = Bin.getBinByName(binname);
			// new bin, make sure that are no bins assigned for any of those SNPs in the
			// current population
			// currently, each SNP can only have one BIN per population.
			if(areTag && snpNames!=null && snpNames.length>0) {
				HashSet<Integer> popids = new HashSet<Integer>();
				for (int j = 0; j <snpNames.length; j++) {
					if (snpNames[j]!=null && snpNames[j].length() > 0) {
						Snp s=Snp.getSnpByName(snpNames[j]);
						if(s!=null) {
							int[] binids = s.getBins();
							if(binids!=null) {
								for(int k=0;k<binids.length;k++) {
									int bin_id = binids[k];
									if(bin_id>0) {
										Bin bb = Bin.getBinById(bin_id);
										if(bb!=null) {
											int bpopid = bb.getBasePopId();
											if(bpopid==popid) {
												popids.add(Integer.valueOf(popid));
												popids.add(Integer.valueOf(bb.getPopid()));
											}
										}
									}
								}
							}
						}
					}
				}
				int npopids=popids.size();
				/*
				 * Problem happens when loading binning data one SNP at a time.
				 * So we do not know the extent of overlap of a bin.
				 * Imagine that we load a bin that partially overlaps an existing bin.
				 * then some early SNPs could be loaded, and without any overlap, a new bin would
				 * get created with the Bin's name.
				 * If on the other hand, the bin is loaded as "Duplicate" right from the first SNP,
				 * we have to have  a mechanism to find match the input file binname to the
				 * modified "Dupfile" bin-name
				 * 
				 * We created the "Dupfile" mechanism because ldselect names bins with "integers", so
				 *  if the bin source is in the same population, the full bin name would be identical.
				 *  
				 *  Let's create a calling parameter called FULLbin which basically tells us wether 
				 *  we know about all the tagSNPs in that bin (true) or wether we cannot make that assumption.
				 *  
				 *  So when parameter FULLBIN==true && npopids>0, then assume that if we find the binname already existing
				 *  this is an overlap that needs to be addressed by renaming the population using the suffix Dupfile.
				 *  
				 *  If FULLBIN==false, we have to assume that the bin naming convention does NOT allow
				 *  name collisions.. 
				 *  
				 *  but that we will use the "Dupfile" mechanism to make sure that
				 *  SNPPicker treats the overlapping bin appropriately.
				 *  In that case, we will match a bin-name using the full name which could be a synonym for the
				 *  "Dupfile" modified gene.
				 *  
				 */
				
				String popname = Population.getPopById(popid).getName();
				String newpop = popname+":DupFile"+Integer.toString(npopids);
				String prefix = newpop + "-" + fileid+":";
				String newbinname = prefix+filebinname;
				if(fullbin) { // ld-select style treat input data assuming that bin names can collide between different invocations of this code.
					if(npopids>0) {// found >0 population existing with that bin name.
						Population new_t_pop=Population.getPopByName(newpop); 						
						if(new_t_pop==null) {
							new_t_pop = new Population();
							new_t_pop.setName(newpop);
						}
						int currpopid = new_t_pop.getId();
						b = new Bin(newbinname, currpopid,popid);
					} else {
						// do nothing, b already contains default bin or is null(in which case a new one will be created)
					}
				} else { // Treat input data assuming no possible name collision. Output of SNPPicker, Mayo Tabular Style.
					if(npopids==0) {
						// do nothing, there are no existing bins with that name for any of those SNPs.(a new Bin will be created down there)
					} else {
						// snps to be added to this binname are already in a bin
						// for this population.
						if(b!=null) { // then this binname exists
							// Check if this bin had a population collision
							if(b.getBasePopId()!=b.getPopid()) {
								// do nothing, already got a modified bin-popid name,
							} else {
							// If no pre-existing population collision then
							// create new population
								Population new_t_pop=Population.getPopByName(newpop); 						
								if(new_t_pop==null) {
									new_t_pop = new Population();
									new_t_pop.setName(newpop);
								}
								int currpopid = new_t_pop.getId();
							// change bin-name, but leave regular binname as synonym
								b.changeBinName(newbinname);
								System.out.println("SNPPicker:INFO:"+binname+":->:"+newbinname+":");System.out.flush();
								b.setPopId(currpopid);

							}
						} else { // This bin doesn't exist, but some of the SNPs in this bin
								 // are already in a bin for this population popid, so create a new one
								 // with new popid as well as old bin-name
							Population new_t_pop=Population.getPopByName(newpop); 
							if(new_t_pop==null) {
								new_t_pop = new Population();
								new_t_pop.setName(newpop);
							} else {
								// OK to take lowest population id with no collision, even if pre-existing.
							}
							int currpopid = new_t_pop.getId();
							b = new Bin(newbinname, currpopid,popid);
							b.setSynonym(binname);// Synonym should not go in main Hash (in case of looping, just used by the getBinByName function
						}
						
					}
				}
				
				// Only create a bin if SNPs are tagSNPs
				if(b==null && areTag && snpNames!=null && snpNames.length>0) {
					b = new Bin(binname, popid,popid);
				}
				if(b!=null) {
					b.setGeneId(geneId);
					b.setSource(source);
				}
			} else {
				// if none of those SNPs were tagSNPs(or no SNPS), do not create a bin if it did not exist.
			}
			return b;

}
/**
 * function to add a link to another bin.
 *
 * @param bindid
 * @param snpid
 */ 
	  public void addBin(int obinid, int snpid) {
		  addBinInternal(obinid, snpid, true);

	  }

	  /**
	   * Function to be called by addBin from other instance of Bin only!
	   *
	   * @param obinid
	   * @param snpid
	   */
	   private void addBinInternal(int obinid, int snpid, boolean processOther) {
		   Snp s = Snp.getSnpById(snpid);
		   if(s==null) {
			   System.err.println("SNPPicker:WARNING: addBin called with unknown snpid "+snpid);
			   Exception ee = new Exception();
			   ee.printStackTrace();
			   System.err.flush();
			   return;
		   }
		   if(this.binedges==null) {
			   this.binedges = new BinEdge[5];
			   BinEdge be = BinEdge.getBinEdge(this.binid,obinid);// returns existing or adds
			   if(SNPPicker.verbose) {
				   System.out.println("Bin Edge added for snp="+snpid+",bin1="+Bin.getBinById(this.binid).getName()+",bin2="+Bin.getBinById(obinid).getName());
			   }
			   this.binedges[0]=be;
			   for(int i=1;i<5;i++) {
				   this.binedges[i]=null;
			   }
			   be.addSnp(snpid);
			   this.nbinedges=1;
		   } else {
			   BinEdge be=null;
			   boolean notfound=true;
			   long binidprod = ((long)this.binid)*((long)obinid);
			   int firstFree=-1;
			   for(int i=0;i<this.binedges.length;i++) {
				   BinEdge bi =this.binedges[i];
				   if(bi==null && firstFree==-1) {
					  firstFree=i;
				   } else if(bi!=null) {
					   long binidprodi = ((long)bi.getBinid1())*((long)bi.getBinid2());
					   if(binidprodi==binidprod) {
						   notfound=false;
						   be = bi;
						   break;
					   }
				   }
			   }
			   if(notfound) {
				   if (firstFree==-1 && this.nbinedges>=this.binedges.length) {
					   BinEdge[] tmp = new BinEdge[2*this.nbinedges+10];
					   int i=0;
					   for(;i<this.binedges.length;i++) {
						   tmp[i]=this.binedges[i];
					   }
					   for(;i<tmp.length;i++) {
						   tmp[i]=null;
					   }
					   this.binedges=tmp;
				   }
				   if(firstFree==-1) {
					   firstFree=this.nbinedges;
				   }

				   this.nbinedges++;
				   be = BinEdge.getBinEdge(this.binid,obinid);
				   this.binedges[firstFree]=be;
			   }
			   if(SNPPicker.verbose) {
				   System.out.println("Bin Edge added for snp="+snpid+",bin1="+Bin.getBinById(this.binid).getName()+",bin2="+Bin.getBinById(obinid).getName());
			   	System.out.flush();
			   }
			   be.addSnp(snpid);
		   }
		   if(processOther) {
			   Bin b2 = Bin.getBinById(obinid);
			   b2.addBinInternal(this.binid,snpid,false);
		   }
	  }

	  public void addSnp(Snp s) {
		  if(s==null) {
			  return;
		  }
		  if(this.snps==null) {
			  this.snps = new Snp[5];
			  this.nsnps=1;
			  this.snps[0]=s;
			  for(int i=1;i<5;i++) {
				  this.snps[i]=null;
			  }
		  } else {
			  boolean already=false;
			  int sid= s.getId();
			  int firstFree=-1;
			  for(int i=0;i<this.snps.length;i++) {
				  if(this.snps[i]!=null && this.snps[i].getId()==sid) {
					  already=true;break;
				  }
				  if(firstFree==-1 && this.snps[i]==null) {
					  firstFree=i;
				  }
			  }
			  if(!already) {
				  if (firstFree==-1 && this.nsnps>=this.snps.length) {
					  Snp[] tmp = new Snp[Math.max(10,2*this.nsnps)];
					  for(int i=0;i<this.snps.length;i++) {
						  tmp[i]=this.snps[i];
					  }
					  this.snps=tmp;
				  } else if (firstFree==-1 && this.nsnps<this.snps.length) {
					  System.err.println("SNPPicker:addSNP: BUG this.snps is full but nsnps too small\n");
					  java.lang.Exception e = new java.lang.Exception("");
					  e.printStackTrace();
					  System.err.flush();
					  System.out.flush();
					  System.exit(-1);
				  }
				  if(firstFree==-1) {
					  firstFree = this.nsnps;
				  }
				  snps[firstFree]=s;
				  this.nsnps++;
			  }
		  }
	  }

	  public void deleteSnp(Snp s,boolean recursive) {
		  if(s==null || this.snps==null) {
			  return;
		  }
		  int indexOfSnp=-1;
		  for(int i=0;i<this.snps.length;i++) {
			  if(this.snps[i]!=null && this.snps[i].getId()==s.getId()) {
				indexOfSnp=i;
				break;
			  }
		  }

		  if (indexOfSnp>=0) {
			  if(recursive) {
				  Snp.deleteByName(s.getSnpName(), false);
			  }
			  for(int i=indexOfSnp;i<this.snps.length-1;i++) {
				 this.snps[i]=this.snps[i+1];
			  }
			  this.snps[this.snps.length-1]=null;
			  this.nsnps--;
			  if(this.nsnps==0) {
				  System.err.println("SNPPicker:deleteSnp: WARNING: Deleted all snps in bin "+this.getOutputName());
				  System.err.flush();
				  this.snps=null;
			  }
		  }
	  }

	  /**
	   * Function to be called by addBin from other instance of Bin only!
	   *
	   * @param obinid
	   * @param snpid
	   */
	   public void deleteBinEdge(int obinid, int snpid,boolean processOtherBin) {
			  if(this.binedges==null) {
				  return;
			  } else {
				  boolean notfound=true;
				  long binidprod = ((long)this.binid)*((long)obinid);
				  BinEdge bi =null;
				  int bipos=0;
				  for(int i=0;i<this.binedges.length;i++) {
					  bi =binedges[i];
					  if(bi==null) {
						  break;
					  } else if(bi!=null && ( ((long)bi.getBinid1())*((long)bi.getBinid2()))==binidprod) {
						  notfound=false;bipos=i;
						  break;
					  }
				  }
				  if(!notfound) {
					  Snp[] besnps = bi.getSnpids();
					  if(besnps==null) {
						  if(processOtherBin) {
							  Bin b2 = Bin.getBinById(obinid);
							  b2.deleteBinEdge(binid, snpid, false);
						  }
						  BinEdge.delete(bi);
						  bi=null;
					  } else {
						  boolean foundSnp=false;
						  int nbesnps = bi.getNsnps();
						  for(int i=0;i<nbesnps;i++) {
							  Snp s = besnps[i];
							  if(s!=null && s.getId()==snpid) {foundSnp=true;break;}
						  }
						  if(foundSnp) {
							  if(bi.getNsnps()==1) {
								  if(processOtherBin) {
									  Bin b2 = Bin.getBinById(obinid);
									  b2.deleteBinEdge(binid, snpid, false);
								  }
								  BinEdge.delete(bi);
								  bi=null;
							  } else { // more than one SNP, be will not be deleted
								  bi.deleteSnp(snpid);
							  }
						  }
					  }
					  if(bi==null && this.binedges!=null) {// deleted the binEdge because removed last SNP

						  this.binedges[bipos]=null;
						  for(int i=bipos+1;i<this.nbinedges;i++){
							  this.binedges[i-1]=this.binedges[i];
						  }
						  this.binedges[this.nbinedges-1]=null;
						  this.nbinedges--;

					  }
				  }
			  }
	  }




	  // Cache maximally scoring snps yet unselected or excluded. Does not include Fixed
	  public void cacheMaxScore(boolean tmpToo) {
		  if(nsnps<1) {
			  this.maxScore=0.0d;
			  this.maxScoringSnpids=null;
			  this.maxScoringSnps=null;
			  return;
		  }
		  // Sort currently unselected SNPs by their probability.(obligates are not assumed to be selected)
		  TreeSet<Snp> sortedByProb = new TreeSet<Snp>(new SnpScoreComparator());// TreeSet sorts the data..

		  for(int i=0;i<this.snps.length;i++) {
			  if(this.snps[i]!=null) {
				  Snp ss = this.snps[i];
				  if( !(ss.isExcluded() || ((!tmpToo) &&ss.isSelected())  || (tmpToo && ss.isTmpSelected()) || ss.isFixed())) {
					  int tnbins = ss.getNbins();
					  if(tnbins>0) {
						  sortedByProb.add(ss);
					  }
				  }
			  }
		  }
		  if(sortedByProb==null || sortedByProb.size()==0) {
			  this.maxScoringSnpids=null;
			  this.maxScoringSnps=null;
			  this.maxScore=0.0d;
			  return;
		  }

		  if(this.maxScoringSnpids==null || this.maxScoringSnpids.length!=sortedByProb.size()) {
			  this.maxScoringSnpids = new int[sortedByProb.size()];
			  this.maxScoringSnps = new Snp[sortedByProb.size()];
		  }
		  Iterator<Snp> it = sortedByProb.iterator();
		  int i=0;
		  while(it.hasNext()) {
			  Snp s = (Snp)it.next();
			  this.maxScoringSnpids[i]=s.getId();
			  this.maxScoringSnps[i++]=s;
		  }
		  this.maxScore = 1.0-(Snp.getSnpById(maxScoringSnpids[0])).getFailProb();

	  }

	  public void cacheMaxCoverage(boolean tmpToo) {
		  if(this.nsnps==0) {
			  boolean wasNULL=true;
			  if(this.snps!=null) {
				  wasNULL=false;
			  }
			  this.snps=null;
			  this.maxCoverageSnpids=null;
//			  this.maxCoverageScore=0.0d;
			  if(!wasNULL) {
				  	System.err.println("SNPPicker:cacheMaxCoverage: WARNING : setting this.snps==null for bin "+this.getOutputName());
			  }
			  return;
		  }
		  // Sort currently unselected SNPs by the number of Bins they touch.
		  TreeSet<Snp> sortedByCoverage = new TreeSet<Snp>(new SnpCoverageComparator());// TreeSet sorts the data..
		  int navail=0;
		  Snp rememberAvail=null;
		  for(int i=0;i<snps.length;i++) {
			  Snp s = this.snps[i];
			  if(s!=null ) {
				  if(!(s.isExcluded())) {
					  navail++; // Count any SNP not excluded, this included SNPs that are fixed
					  rememberAvail=s;
				  }
				  if(!(s.isExcluded() || ((!tmpToo) && s.isSelected()) || s.isFixed() || (tmpToo && s.isTmpSelected()))) {
					  sortedByCoverage.add(s);
				  }
			  }
		  }
		  if(sortedByCoverage==null || sortedByCoverage.size()==0) {
			  this.maxCoverageSnpids=null;
//			  this.maxCoverageScore=0.0d;
			  return;
		  }
		  Iterator<Snp> it = sortedByCoverage.iterator();
		 
		  if(this.maxCoverageSnpids==null || this.maxCoverageSnpids.length!=sortedByCoverage.size()) {
			  this.maxCoverageSnpids = new int[sortedByCoverage.size()];
		  }
		  {
			  int i=0;
			  Snp s=null;
			  while(it.hasNext()) {
				  s = (Snp)it.next();
				  this.maxCoverageSnpids[i++]=s.getId();
			  }
		  }
		  if(navail==1) {
			  // once SNPS get fixed by the too-close test, this doesn't modify the
			  // uniquelyCovers Status.. so that we can compare all solutions on the same footing.
			  // Note that an obligate in a bin will prevent any SNPs from being set as uniquely covering.
			 rememberAvail.setUniquelyCovers(this.binid);
		  }
//		  this.maxCoverageScore = 1.0-Snp.getSnpById(this.maxCoverageSnpids[0]).getFailProb();
	  }


	  /**
	   * Function to compute the maximum scoring snp in the bin if we exclude the input snp list
	   * @param snplist
	   */
	  public double getMaxScoreNotInList(Snp[] snplist) {
		  double lmaxscore=0.0;
		  if(snps==null) {
			  return lmaxscore;
		  }
		  for (int i=0;i<snps.length;i++) {
			  Snp s = snps[i];
			  if(s!=null) {
				  boolean foundSnp=false;
				  if(snplist!=null) {
					  int sid=s.getId();
					  for(int j=0;j<snplist.length;j++) {
						if(snplist[j]!=null && snplist[j].getId()==sid) {
							foundSnp=true;
							break;
						}
					  }
				  }
				  if(!foundSnp) {
					  double sco = s.getScore();
					  if(sco>lmaxscore) {
						  lmaxscore=sco;
					  }
				  }
			  }
		  }
		  return lmaxscore;
	  }

	  /**
	   * Sort the Snps on all Bins so each bin has a sorted list of Snps to consider (in score sorting order)
	   *
	   *
	   */
	  public static void buildMaxinums() {
		  for(int i=0;i<binIndex.length;i++) {
			  Bin b = (Bin)binIndex[i];
			  if(b!=null) {
				  b.cacheMaxScore(false);// Sort Snps in maximal Scoring Order.
				  b.cacheMaxCoverage(false);
			  }
		  }
	  }



	/**
	 * @return the nextbin
	 */
	public Bin getNextbin() {
		return nextbin;
	}
	/**
	 * @param the nextbin to set
	 */
	public void setNextbin(Bin bin) {
		this.nextbin = bin;
	}
	/**
	 * @return the prevbin
	 */
	public Bin getPrevbin() {
		return prevbin;
	}
	/**
	 * @param prevbin the prevbin to set
	 */
	public void setPrevbin(Bin prevbin) {
		this.prevbin = prevbin;
	}
	public Bin getLastBin() {
		Bin b1 = this;
		while(b1.nextbin!=null) {
			b1=b1.nextbin;
		}
		return b1;
	}
	public Bin getFirstBin() {
		Bin b1 = this;
		int a=1;
		if(b1.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:38") ||
				b1.getName().equals("CEU-GPC6|10082;SNPApp_EUR_0.90_0.05_1228_1231Hapmap10082ldselect.out/13:34") ||
				b1.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:536") ||
				b1.getName().equals("CEU-STAT1|6772;SNPApp_EUR_0.90_0.05_1228_1231Hapmap6772ldselect.out/2:18") ||
				b1.getName().equals("AFR-STAT1|6772;SNPApp_AFR_0.90_0.05_1228_1232Hapmap6772ldselect.out/2:35")
		) {
			a++; // debugger breakpoint.
		}
		while(b1.prevbin!=null) {
			b1=b1.prevbin;
			if(b1.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:38") ||
					b1.getName().equals("CEU-GPC6|10082;SNPApp_EUR_0.90_0.05_1228_1231Hapmap10082ldselect.out/13:34") ||
					b1.getName().equals("AFR-GPC6|10082;SNPApp_AFR_0.90_0.05_1228_1232Hapmap10082ldselect.out/13:536") ||
					b1.getName().equals("CEU-STAT1|6772;SNPApp_EUR_0.90_0.05_1228_1231Hapmap6772ldselect.out/2:18") ||
					b1.getName().equals("AFR-STAT1|6772;SNPApp_AFR_0.90_0.05_1228_1232Hapmap6772ldselect.out/2:35")
			) {
				a++; // debugger breakpoint.
			}
		}
		return b1;
	}
	public static Bin[] getBinHeads() {
		Bin[] heads =null;
		if(nbins==0) {
			return heads;
		}
		if(haveBin!=null) {
			int nheads=0;
			Set<Map.Entry<String, Bin>> s = haveBin.entrySet();
			for (Map.Entry<String, Bin> me : s) {
				Bin b = me.getValue();
				if(b.prevbin==null) {
					nheads++;
				}
			}
			heads = new Bin[nheads];
			nheads=0;
			s = haveBin.entrySet();
			for (Map.Entry<String, Bin> me : s) {
				Bin b = me.getValue();
				if(b.prevbin==null) {
					heads[nheads++]=b;
				}
			}
		}
		return heads;
	}


	/**
	 * @return the nbins
	 */
	public static int getNbins() {
		return nbins;
	}
	/**
	 * @return the binedges
	 */
	public BinEdge[] getBinedges() {
		return this.binedges;
	}
	/**
	 * @return the name
	 */
	public String getName() {
		return name;
	}
	
	public String getPublicName() {
		String bname = this.getName();
		int ppos = bname.indexOf(":");
		int dpos = bname.indexOf(":DupFile");
		if(dpos>=ppos) {
			// format is DupFilenn 
			int p2 = bname.indexOf("-",dpos+":DupFile".length());
			bname = bname.substring(0,dpos) + bname.substring(p2);
		}
		return bname;
	}
	
	public String getOutputName() {
		assignGene(); // look at SNPs genes.
		String bname = this.getName();
		int ppos = bname.indexOf(":");
		int dpos = bname.indexOf(":DupFile");
		if(dpos>=ppos) {
			// format is DupFilenn 
			int p2 = bname.indexOf("-",dpos+":DupFile".length());
			bname = bname.substring(0,dpos) + bname.substring(p2);
		}
		int dashpos = bname.indexOf("-");
		ppos = bname.indexOf(":");
		String chr = this.getChromosome();
		String fileIdStr = bname.substring(dashpos+1,ppos);
		try {// if file string is a simple integer .. no longer done..11/10/2009
			if(fileIdStr!=null && fileIdStr.length()>0) {
				int fileid = Integer.parseInt(fileIdStr);
			}
			fileIdStr = this.getGene()+"|"+this.getGeneId()+";"+fileIdStr+"/"+chr;
		} catch (Exception nfe){
			if(fileIdStr.indexOf("/")<0) {
				fileIdStr = fileIdStr + "/"+chr;
			}
			int semicpos = fileIdStr.indexOf(";");
			if(semicpos<0) {
				fileIdStr = this.getGeneId() + "|" + this.getGene()+";"+fileIdStr;
			} else {
				fileIdStr = this.getGeneId() + "|" + this.getGene()+";"+fileIdStr.substring(semicpos+1);
			}
		}
		bname = bname.substring(0,dashpos+1)+fileIdStr +bname.substring(ppos);
		return bname;
	}
	
	public void assignGene() {
		if(this.gene!=null && this.gene.length()>0 && this.geneId!=null && this.geneId.length()>0) {
			return; // Already assigned.
		}
		if((this.gene== null || this.gene.length()==0) && this.geneId!=null && this.geneId.length()>0) {
			if(SNPPicker.GENEID2GENE.containsKey(this.geneId)) {
				this.gene =  SNPPicker.GENEID2GENE.get(this.geneId);
			}
			return;
		}
		if((this.geneId== null || this.geneId.length()==0) && this.gene!=null && this.gene.length()>0) {
			if(SNPPicker.GENE2GENEID.containsKey(this.gene)) {
				this.geneId =  SNPPicker.GENE2GENEID.get(this.gene);
			}
			return;
		}
		if(this.snps!=null) {
				int maxGidCnt=0;
				HashMap<String,String> gid2gene = new HashMap<String,String>();
				HashMap<String,String> gene2gid = new HashMap<String,String>();
				HashMap<String,Integer> geneIdsCnt = new HashMap<String,Integer>();
				HashSet<String> geneConflict = new HashSet<String>();
				HashSet<String> gidConflict = new HashSet<String>();
				for(int i=0;i<this.snps.length;i++) {
					Snp s = this.snps[i];
					if(s!=null) {
						String gid = s.getGeneId();
						String gene = s.getGene();
						boolean oktomap=true;
						// First Fill up missing SNPs who only have one or the other annotation
						if((gid!=null && gid.length()>0) && (gene==null ||  gene.length()==0)) {
							if(SNPPicker.GENEID2GENE.containsKey(gid)) {
								gene = SNPPicker.GENEID2GENE.get(gid);
								s.setGene(gene);
							}
						}
						if((gene!=null && gene.length()>0) && (gid==null ||  gid.length()==0)) {
							if(SNPPicker.GENE2GENEID.containsKey(gene)) {
								gid =  SNPPicker.GENE2GENEID.get(gene);
								s.setGeneId(gid);
							}
						}
						// Then make sure there are no conflicts.
						if(gid!=null && gid.length()>0 && gene!=null && gene.length()>0) {
								if(gid2gene.containsKey(gid)) {
									String oldgene =  gid2gene.get(gid);
									if(oldgene!=null && gene!=null && oldgene.length()>0 && gene.length()>0 && !oldgene.equalsIgnoreCase(gene)) {
										gidConflict.add(gid);
										geneConflict.add(gene);
										oktomap=false;
									}
								}
								if(gene2gid.containsKey(gene)) {
									String oldgeneid =  gene2gid.get(gene);
									if(oldgeneid!=null && gid!=null && oldgeneid.length()>0 && gid.length()>0 && !oldgeneid.equalsIgnoreCase(gid)) {
										gidConflict.add(gid);
										geneConflict.add(gene);
										oktomap=false;
									}								
								}
								if(oktomap) {
									gid2gene.put(gid, gene);
									gene2gid.put(gene,gid);
								}
						}
					}
				}
				int nOKtomap=0;
				nOKtomap = gid2gene.size();
				String bestGeneId=this.geneId;
				if(nOKtomap>0 && (this.geneId ==null || this.geneId.length()==0 || this.gene==null || this.gene.length()==0)) {
					if((this.geneId ==null || this.geneId.length()==0) &&( this.gene!=null && this.gene.length()>0)) {
						if(!geneConflict.contains(this.gene)) {
							if(gene2gid.containsKey(this.gene)) {
								this.geneId= gene2gid.get(this.gene);
								return;
							}
						}
					}
					if((this.gene ==null || this.gene.length()==0) &&( this.geneId!=null && this.geneId.length()>0)) {
						if(!gidConflict.contains(this.geneId)) {
							if(gid2gene.containsKey(this.geneId)) {
								this.gene=  gid2gene.get(this.geneId);
								return;
							}
						}
					}
					if(geneConflict.contains(this.gene) || gidConflict.contains(this.geneId)) {
						return;
					}
					
					for(int i=0;i<this.snps.length;i++) {
						Snp s = this.snps[i];
						if(s!=null) {
							String gid = s.getGeneId();
							String gene = s.getGene();
							if(gid==null || gid.length()==0) {
								if(gene!=null && gene.length()>0) {
									gid =  gene2gid.get(gene);
								}
							}
							if(gene==null || gene.length()==0) {
								if(gid!=null && gid.length()>0) {
									gene =  gid2gene.get(gid);
								}
							}
							int ngids=0;
							if(geneIdsCnt.containsKey(gid)) {
								ngids = 1+ geneIdsCnt.get(gid).intValue();
							}
							geneIdsCnt.put(gid, Integer.valueOf(ngids));
							if(ngids>maxGidCnt && !gidConflict.contains(gid)) {
								maxGidCnt = ngids;
								bestGeneId = gid;
							}
						}
					}
					
					this.geneId=bestGeneId;
				}
				if(this.geneId!=null && this.geneId.length()>0 && gid2gene.containsKey(this.geneId)) {
					this.gene =  gid2gene.get(bestGeneId);
				}
				
				
		}
		
		
		if(this.geneId==null || this.geneId.length()==0) {
			if(gene!=null && gene.length()>0 && Gene.containsGene(gene)) {
				this.geneId= Gene.getGene(this.geneId);
			}
		}
		if(this.gene==null || this.gene.length()==0) {
			if(this.geneId!=null && this.geneId.length()>0 && Gene.containsGeneid(this.geneId)) {
				this.gene= Gene.getGeneId(this.gene);
			}
		}

		
	}
	
	/**
	 * @return the nbinedges
	 */
	public int getNbinedges() {
		return nbinedges;
	}

	/**
	 * @return the nsnps
	 */
	public int getNsnps() {
		return nsnps;
	}
	/**
	 * @return the popid
	 */
	public int getPopid() {
		return popid;
	}
	/**
	 * @return the basePopid
	 */
	public int getBasePopId() {
		return this.basePopId;
	}
	/**
	 * @return the basePopid
	 */
	public void setBasePopId(int basePopid) {
		basePopId=basePopid;
	}
	/**
	 * @return the score
	 */

	public void setPopId(int popid) {
		this.popid = popid;
	}
	/**
	 * @return the snps
	 */
	public Snp[] getSnps() {
		return this.snps;
	}
	/**
	 * @param score the score to set
	 */


	public int getNSB() {
		return NSB;
	}
	public void setNSB(int nsb) {
		this.NSB = nsb;
/*
		if(selectedSnpIds!=null) {
			if(selectedSnpIds[0]!=0 && ! SNPPicker.computeUtilityOnly) {
				int sid = selectedSnpIds[0];
				Snp s = Snp.getSnpById(sid);
				if(s!=null && s.isObligateButNotGenotyped() && SNPPicker.obligateIncludeNGFileOverRule) {
					// OK to reset NSB here
				} else {
					System.err.println("SNPPicker: Potential Bug: Shouldn't reset NSB after have selected SNPs");
				}
				if(nsb>selectedSnpIds.length) {
					int[] new_selectedSnpIds=new int[this.NSB];
					for(int i=0;i<this.NSB;i++) {new_selectedSnpIds[i]=0;}
					for(int i=0;i<selectedSnpIds.length;i++) {new_selectedSnpIds[i]=selectedSnpIds[i];}
					selectedSnpIds = new_selectedSnpIds;
					
				}
			}
		} else {
			selectedSnpIds=new int[this.NSB];
			for(int i=0;i<this.NSB;i++) {selectedSnpIds[i]=0;}
		}
		*/
	}
	public int getNSites() {
		return nSites;
	}
	public void setNSites(int sites) {
		nSites = sites;
	}
	public int getNSelected() {
		return nSelected;
	}
	public int setNSelected(int nsel) {
		return this.nSelected=nsel;
	}
	public int getNFixedOrObligateOrSelected() {
		int nsel=0;
		if(snps!=null) {
			for(int i=0;i<snps.length;i++) {
				Snp s = snps[i];
				if(s!=null) {
					if((!s.isExcluded()) && (s.isTmpSelected())) {
						nsel++;
					}
				}
			}
		}
		return nsel;
	}
	
	public int getNSelectedBeadTypes() {
		if(this.snps==null) {
			return 0;
		}
		int count =0;
		for (int i=0;i<this.snps.length;i++) {
			Snp s = snps[i];
			if(s!=null) {
				if(!s.isExcluded() &&(s.isTmpSelected())) {
						count +=s.getNI2beads();	
				}
			}
		}
		return count;
	}
	
	public int getTmpNSelectedBeadTypes() {
		if(this.snps==null) {
			return 0;
		}
		int count =0;
		for (int i=0;i<this.snps.length;i++) {
			Snp s = snps[i];
			if(s!=null) {
				// Only count SNPs that will be genotyped
				if((!s.isExcluded()) && (s.isTmpSelected() || (s.isObligate() && !s.isExcluded()) || (s.isFixed() && !s.isExcluded()))) {
					count +=s.getNI2beads();
				}
			}
		}
		return count;
	}


	public int[] getMaxScoringSnpids() {
		return this.maxScoringSnpids;
	}
	public Snp[] getMaxScoringSnps() {
		return this.maxScoringSnps;
	}
	public int[] getSelectedSnpIds() {
		return this.selectedSnpIds;
	}
	/**
	 *
	 * @param sid
	 * @param newstate true if selecting, false if de-selecting.
	 * @param processSnpToo
	 * @return index of Selected SNP in snps array on bin. -1 if SNP not in Bin.
	 */
	public int selectSnpId(int sid, boolean newstate,boolean processSnpToo) throws Exception {
		// make sure SNP is in bin
		if(sid==0) {
			sid=0; // breakpoint for debugger.
		}
		if(this.snps==null || nsnps==0 || sid<=0) {
			return -1; // SNP is not (no longer?) in bin
		}
		Snp s=null;
		int nleft=0;
		for(int i=0; i<snps.length;i++) {
			if(snps[i]!=null) {
				if(snps[i].getId()==sid) {
					s=snps[i];
				}
				if(!(snps[i].isSelected() || snps[i].isExcluded() || snps[i].isObligate() || snps[i].isObligateButNotGenotyped())) {
					nleft++;
				}
			}
		}
		if(s==null) {
			return -1;
		}
		if(SNPPicker.verbose) {
			System.out.println("SNPPicker::INFOL: selectSnpId selecting "+s.getSnpName()+ " to " +(newstate ? " selected " : "not selected "));
		}
		if(newstate && s.isObligate() && s.isObligateButNotGenotyped() && SNPPicker.obligateIncludeNGFileOverRule) {
			this.setStatus(1);
			this.setNSB(1);
		}
		// make sure SNP was not already selected
		if(selectedSnpIds!=null) {
			for(int i=0;i<selectedSnpIds.length;i++) {
				if(sid>0) {
					if( selectedSnpIds[i]==sid) {
						if(newstate) {
							// already selected
							this.hasBeenReset=false; // no longer properly reset

						} else {
							// we're de-selecting this.
							selectedSnpIds[i]=0;
							for(int k=i+1;k<selectedSnpIds.length;k++) {
								selectedSnpIds[k-1]=selectedSnpIds[k];
							}
							selectedSnpIds[selectedSnpIds.length-1]=0;
							this.nSelected--;
							if(s.isObligate() && s.isObligateButNotGenotyped() && SNPPicker.obligateIncludeNGFileOverRule) {
								System.err.println("SNPPicker:ERROR: Cannot deselect an obligate not-genotyped that is forced to be selected\n");
							}
						}
						if(processSnpToo) {
							s.setSelected(newstate,false);
						}
						int _nExcluded = this.getNExcluded();
						if ((this.NSB<=this.nSelected) || nleft==0  ) {
							this.setStatus(1); // no more snps to pick.
						} else {
							this.setStatus(0);
						}
						return i;
					}
				}
			}
		}
		if(!newstate) {
			// deselecting a SNP, but this SNP was not selected. (otherwise it would have already "returned" from function)
			return -1;
		}
		//
		// new SNP to select.
		// make sure it fits in selectedSnpIds array (may choose more than NSB)
		if(selectedSnpIds==null || selectedSnpIds.length==0 || this.nSelected>=selectedSnpIds.length) {
			int[] newsel = new int[this.nSelected+5];// make a little extra room.
			int i=0;
			if(this.selectedSnpIds!=null) {
				for(;i<this.nSelected;i++) {
					newsel[i]=selectedSnpIds[i];
				}
			}
			selectedSnpIds=newsel;
		}
		// increment nSelected
		//System.err.println("nSelected="+nSelected+", sid="+sid);System.err.flush();
		//if(nSelected<0) {
		//	nSelected=nSelected;
		//}
		selectedSnpIds[this.nSelected]=sid;
		this.nSelected++;
		if(processSnpToo) {
			s.setSelected(newstate,false);
		}

		if ((this.NSB<=this.nSelected) || nleft==0  ) {
			this.setStatus(1); // no more snps to pick.
		} else if(this.getProb()>=SNPPicker.minP) {
			this.setStatus(1); // no more snps to pick.
		} else {
			this.setStatus(0); 
		}

		return nSelected;
	}
	/**
	 * Function to temporarely Select or De-select a SNP from a bin. (won't process other bins which contain this SNP!)
	 * @param sid
	 * @param processSnpToo
	 * @return  -1 if not found in bin, or index in tmpSelectedSnpIds array
	 */
	public int tmpSelectSnpId(boolean select,int sid,boolean processSnpToo) {
		if(sid==0) {
			System.err.println("SNPPicker:Bin.tmpSelectSnpId:: BUG  was called with sid==0");
			System.err.flush();
			return -1;
		}
		this.hasBeenReset=false;
		this.bestWasTransfered=false;
		if(this.snps==null || this.nsnps==0) {
			return -1;
		}
		// make sure SNP is in bin
		Snp s=null;
		for(int i=0;i<this.snps.length;i++) {
			if(this.snps[i]!=null && this.snps[i].getId()==sid) {
				s=this.snps[i];
				break;
			}
		}
		if(s==null) {
			return -1;
		}
		int firstFree=-1;
		// make sure SNP was not already selected
		if(select) {
			if(tmpSelectedSnpIds!=null){
				for(int i=0;i<tmpSelectedSnpIds.length;i++) {
					if(sid>0 && tmpSelectedSnpIds[i]==sid) {
							if(processSnpToo) {
								s.setTmpSelected(true,false);// no twice recursive on Snp
							}
							return i;
					} else if(firstFree==-1 && tmpSelectedSnpIds[i]==0) {
						firstFree=i; // find an empty slot in array.
					}
				}
			}
		} else {//de-select a snpid
			if(tmpSelectedSnpIds!=null){
				for(int i=0;i<tmpSelectedSnpIds.length;i++) {
					if(sid>0 && tmpSelectedSnpIds[i]==sid) {
							tmpNSelected--;
							for(int j=i+1;j<tmpSelectedSnpIds.length;j++) {
								tmpSelectedSnpIds[j-1]=tmpSelectedSnpIds[j];
							}
							tmpSelectedSnpIds[tmpSelectedSnpIds.length-1]=0;
							if(processSnpToo) {
								s.setTmpSelected(false,false);// no recursive on Snp
							}
							return -1;
					}
				}
			} // else nothing to deselect
			if(s!=null && processSnpToo) {
				s.setTmpSelected(false,false);// no recursive on Snp
			}
			return -1; // Didn't have to unset the SNP, it wasn't in the bin
		}



		if(firstFree==-1) {
			//			 make sure it fits in selectedSnpIds array (may choose more than NSB)
			if(tmpSelectedSnpIds==null || tmpNSelected>=tmpSelectedSnpIds.length) {
				int[] newsel = new int[tmpNSelected+5];// make a little extra room.
				int i=0;
				if(tmpSelectedSnpIds!=null) {
					for(;i<tmpNSelected;i++) {
						newsel[i]=tmpSelectedSnpIds[i];
					}
				}
				for(;i<tmpNSelected+5;i++) {
					newsel[i]=0;
				}
				tmpSelectedSnpIds=newsel;
			}
			firstFree=tmpNSelected;// put at the end.
		}
		// increment nSelected
		tmpSelectedSnpIds[firstFree]=sid;
		tmpNSelected++;
		if(processSnpToo) {
			s.setTmpSelected(select,false);// no recursive on Snp
		}
		return tmpNSelected;
	}
	/*
	 * reset tmp solution vector and prepare for next step.
	 */

    public void eraseTmpSolution(boolean processSnp) {
    		
	    	this.tmpNSelected=0;
	    	this.tmpSelectedSnpIds= new int[0];
	    	if(processSnp && this.snps!=null) {
				// Loop over bins and
				// mark BINS tha contain this SNP .. and call with firstCall=false;
				for(int i=0;i< this.snps.length;i++) {
					// Call finalPick for Bins.
					Snp s = this.snps[i];
					if(s!=null) {
							s.setTmpSelected(s.isSelected(),false);
					}
				}
			}
    }

    
	/*
	 * Copy current solution to tmp for scoring.
	 */

    public void copySolutionToTmp(boolean processSnpToo) {
    		
	    	this.tmpNSelected=0;
	    	this.tmpSelectedSnpIds=new int[0];
	    	
	    	if(processSnpToo && this.snps!=null) {
				// Loop over bins and
				// mark BINS tha contain this SNP .. and call with firstCall=false;
				for(int i=0;i< this.snps.length;i++) {
					// Call finalPick for Bins.
					Snp s = this.snps[i];
					if(s!=null) {
						if(s.isSelected()) {
							try {
							this.tmpSelectSnpId(true,s.getId(), processSnpToo);
							} catch (Exception e) {
								e.printStackTrace();
								System.err.println("SNPPicker:ERROR:"+ e.getMessage());
								System.err.flush();
								System.exit(-1);
							}
						}
					}
				}
			}
    }
    
	/*
	 * reset tmp solution vector and prepare for next step.
	 */

    public void resetTmpSolution(boolean processSnp) {
    //	if(!this.hasBeenReset) {
    	// e.g. pick up obligates and already selected
    	
    		int tlen =0;
			if(this.selectedSnpIds!=null) {
				tlen = this.selectedSnpIds.length;
			}	
	    	
	    	this.tmpSelectedSnpIds = new int[tlen];
	    	int ib=0;
	    	for(int ii=0;ii<tlen;ii++) {
	    		int sid = this.selectedSnpIds[ii];
	    		if(sid>0) {
	    			this.tmpSelectedSnpIds[ib++]=sid;
	    		}
	    	}

	    	this.tmpNSelected=ib;
	    	if(nSelected!=ib) {
	    		this.tmpNSelected=nSelected; // breakpoint for debugger
	    	}
	    	this.hasBeenReset=true;
			if(processSnp && this.snps!=null) {
				// Loop over bins and
				// mark BINS tha contain this SNP .. and call with firstCall=false;
				for(int i=0;i< this.snps.length;i++) {
					// Call finalPick for Bins.
					Snp s = this.snps[i];
					if(s!=null) {
							s.resetTmpSolution(false);
					}
				}
			}
    	//}
    }
	/**
	 * Reset best-so-far
	 *
	 */
	public void resetBestSoFarAndTmp(boolean processSnp) {

			this.bestWasTransfered=false;
			this.bestSelectedSnpIds = null;
			this.bestNSelected = 0;
			resetTmpSolution(false); // it's going to create a new tmp Solution.. so we could use the old storage space.

			if(processSnp && this.snps!=null) {
				// Loop over bins and
				// mark BINS tha contain this SNP .. and call with firstCall=false;
				for(int i=0;i< this.snps.length;i++) {
					// Call finalPick for Bins.
					Snp s = this.snps[i];
					if(s!=null) {
							s.resetTmpSolution(false);
					}
				}
			}
	}
	/**
	 * Move temporarely best solution to best-so-far solution and reset
	 *
	 */
	public boolean pickBestSoFarAndReset(boolean processSnp) {
		boolean abetterSolution=false;
		if(!this.bestWasTransfered) {
			this.bestWasTransfered=true;
			this.bestSelectedSnpIds = this.tmpSelectedSnpIds;
			this.bestNSelected = this.tmpNSelected;
			abetterSolution=true;
			resetTmpSolution(false); // it's going to create a new tmp Solution.. so we could use the old storage space.

			if(processSnp && this.snps!=null) {
				// Loop over bins and
				// mark BINS tha contain this SNP .. and call with firstCall=false;
				for(int i=0;i< this.snps.length;i++) {
					// Call finalPick for Bins.
					Snp s = this.snps[i];
					if(s!=null) {
							s.pickBestSoFarAndReset(false);
					}
				}
			}
		}
		return abetterSolution;
	}
	/**
	 * Move temporary solution to best-so-far solution
	 *
	 */
	public void pickBestSoFar(boolean processSnp) {
		//	if(!this.bestWasTransfered) {
		this.bestWasTransfered=true;
		this.bestSelectedSnpIds = null;
		int ib=0;
		if(this.tmpSelectedSnpIds !=null && this.tmpNSelected>0) {
			this.bestSelectedSnpIds = new int[this.tmpNSelected];	

			for(int i=0;i<this.tmpSelectedSnpIds.length;i++) {
				int sid =this.tmpSelectedSnpIds[i];
				if(sid>0) {
					this.bestSelectedSnpIds[ib++] =sid; 
				}
			}
			for(int i=ib;i<this.bestSelectedSnpIds.length;i++) {
				this.bestSelectedSnpIds[i] =0;
			}
		}
		this.bestNSelected = ib;
		if(processSnp && this.snps!=null) {
			// Loop over bins and
			// mark BINS that contain this SNP .. and call with firstCall=false;
			for(int i=0;i< this.snps.length;i++) {
				// Move the tmp selected SNPs over
				Snp s=this.snps[i];
				if(s!=null && ! s.isFixed()) {
					s.pickBestSoFar(false);
				}
			}
		}
		//	}
	}

	/**
	 * Move best-so-far solution to the final solution.
	 * if fixSNPOnly==true ==> only fix SNP, but do NOT select it... e.g. reversible "selection"
	 *
	 */
	public void finalPick(boolean processSnp, boolean fixSnpOnly) throws Exception  {
		// mark this SNP as selected..
		if(fixSnpOnly && !processSnp) {
			System.err.println("SNPPicker:BUG Should not call finalPick function with !processSNP && fixSnpOnly\n");
			System.out.flush();
			System.err.flush();System.exit(-1);
		}
		int countSelected=0;
		if(processSnp && this.snps!=null) {
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null) {
					if(!(s.isObligate() || s.isFixed() || s.isExcluded() || s.isObligateButNotGenotyped())) {
						if(processSnp) {
							if(s.isBestSelected()) {
								if(fixSnpOnly) {
									s.setFixed(true);
								} else {
									s.finalPick(false,false,true); // set SNP to selected is .bestSelected==true
									if(s.isSelected()) { // make tmpSelected match the Selected
										s.setTmpSelected(true,true);
									} else {
										s.setTmpSelected(false, true);
									}
								}
							} else if(!fixSnpOnly){
								s.setSelected(false, true); // unselect all the non-best (non-obligate) SNPs in the bin
								s.setTmpSelected(false, true); // unTmpSelect all the SNPs in the bin
							} 
						}
					}
					if(s.isSelected() || s.isObligate() ||s.isObligateButNotGenotyped()) {
						countSelected++;
					}
				}
			}
		} 
		if(!fixSnpOnly) {
			if(processSnp) {
				this.selectedSnpIds = new int[countSelected];
				int ib=0;
				for(int i=0;i<this.snps.length;i++) {
					Snp s = this.snps[i];
					if(s!=null) {
						if(s.isSelected() || s.isObligate() || s.isObligateButNotGenotyped()) {
							this.selectedSnpIds[ib++]=s.getId();
						}
					}
				}
				this.nSelected=countSelected;
			} else { //trust bestSelectedSnpIds array
				if((this.bestSelectedSnpIds!=null) && bestNSelected>0) {
					this.selectedSnpIds = new int[bestNSelected];
					for(int i=0;i<bestNSelected;i++) {
						this.selectedSnpIds[i]=this.bestSelectedSnpIds[i];
					}
				}
				this.nSelected=bestNSelected;
			}
			
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null) {
					if(s.isSelected() && (s.isObligate() || s.isFixed() || s.isObligateButNotGenotyped())) {
						this.selectSnpId(s.getId(), true, false); // Just Make sure the SNP is on the selectedSnpIds.
					}
				}
			}
		}


		if(!fixSnpOnly) {
			if ((this.getNSB()<=this.nSelected) || (this.nSelected>=(this.nsnps-this.getNExcluded()))) {
				this.setStatus(1);
			} else {
				this.setStatus(0);
			}
		}
	}

	public int[] getAvailableSnpIds(boolean tmpToo) {
		int nAvailable =0;
		if(this.snps==null) {
			return new int[0];
		}
		for(int i=0;i<this.snps.length;i++) {
			Snp s = this.snps[i];
			try {
			if(s!=null && !( ((!tmpToo) && s.isSelected()) || (tmpToo && s.isTmpSelected()) 
					||  s.isExcluded() || s.isFixed() || s.isObligateButNotGenotyped() )) {
				nAvailable++;
			}
			} catch (Exception e) {
				e.printStackTrace();
				System.err.println("SNPPicker:WARNING: "+e.getMessage());
			}
		}

		int[] avail = new int[nAvailable];
		if(nAvailable==0) {
			return avail;
		}
		nAvailable=0;
		for(int i=0;i<this.nsnps;i++) {
			Snp s = this.snps[i];
			try {
			if(s!=null && !( ((!tmpToo) && s.isSelected()) || (tmpToo && s.isTmpSelected()) ||  s.isExcluded() || s.isFixed() )) {
				avail[nAvailable++]=snps[i].getId();
			}
			} catch (Exception e) {
				e.printStackTrace();
				System.err.println("SNPPicker:WARNING: "+e.getMessage());
			}
		}
		return avail;
	}

	/*
	 * Only difference between getAvailableSnpIds and getNotSelectedOrExcludedSnpIds is that the latter
	 * includes "fixed SNPs" or tmpSelected. This is needed for setting the uniquelyCovers Flag.
	 * 
	 */

	public int[] getNotSelectedOrExcludedSnpIds() {
		int nAvailable =0;
		if(this.snps==null) {
			return null;
		}
		for(int i=0;i<this.snps.length;i++) {
			Snp s = this.snps[i];
			if(s!=null && !s.isSelected() && !(s.isExcluded() || s.isObligateButNotGenotyped())) {
				nAvailable++;
			}
		}
		if(nAvailable==0) {
			return null;
		}
		int[] avail = new int[nAvailable];
		nAvailable=0;
		for(int i=0;i<this.nsnps;i++) {
			Snp s = this.snps[i];
			if(s!=null && !s.isSelected() && !(s.isExcluded() || s.isObligateButNotGenotyped())) {
				avail[nAvailable++]=snps[i].getId();
			}
		}
		return avail;
	}



	 /**
	  * Set a SNP to Obligate Exclude (if exist in this Bin)
	  *
	  * @param s
	  */
	 public void excludeSnp(Snp s,boolean recursive) throws Exception {
		 if(s!=null) {
			 boolean newExcluded=!s.isExcluded();
			 if(newExcluded) {
				 this.nExcluded++;
			 }
			 if(recursive) {
				 s.setExcluded(false); // false means no recursive
			 }
			 if(this.selectedSnpIds!=null) {
				 int sid = s.getId();
				 if(sid>0) {
					 for(int i=0;i<selectedSnpIds.length;i++) {
						 if(sid>0 && sid==selectedSnpIds[i]) {
							 System.err.println("SNPPicker:WARNING: bin.excludeSnp() called after having selected that SNP\n");
							 (new Exception()).printStackTrace();
							 System.err.flush();
							 System.out.flush();
							 this.selectSnpId(sid, false, recursive);  // also sets this.nSelected
						 }
					 }
				 }
			 }
			 if(this.nSelected==0 && this.getNSB()>0) {
				int[] remainingSnpids = this.getNotSelectedOrExcludedSnpIds();
				if(remainingSnpids!=null && remainingSnpids.length==1) {
					Snp su = Snp.getSnpById(remainingSnpids[0]);
					if(su!=null) {
						su.setUniquelyCovers(this.binid);
					}
				}
			}
		 }
	 }
	 public void incNExcluded() {
		 this.nExcluded++; // can only be called by Snp->setExcluded;
		 
	 }
	 
	 public void setObligate(Snp s, boolean recursive) {
		 if(recursive) {
			 s.setObligate(false); // no recursive on the SNP
		 }
		 //selectSnpId(s.getId(), true,recursive);
	 }

	public int getBestNSelected() {
		return bestNSelected;
	}	
	public int getUnPickedObligates() {
		int nObligsToPick=0;
		if(this.snps!=null) {
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null) {
					if(s.isObligate() && !(s.isSelected() || s.isExcluded() || s.isFixed())) {
						nObligsToPick++;
					}
				}
			}
		}
		return nObligsToPick;
	}
	/**
	 * Tells whether there are any more SNPs available for Selection.
	 * @return
	 */
	public boolean isFull(boolean pickAllUnpickedObligates) {
		this.full=false;
		if(this.snps!=null) {
			if(SNPPicker.verbose) {
				System.out.println("isFull: this.snps.length="+this.snps.length+", nsnps="+nsnps+",NSB="+
					this.getNSB()+", nExcluded ="+this.nExcluded+", nSelected="+this.nSelected);
			}
			int nObligsToPick=0;
			if(pickAllUnpickedObligates) {
				nObligsToPick= this.getUnPickedObligates();
				if(nObligsToPick>0) {
					if(SNPPicker.verbose) {
						System.out.println("isFull: Bin "+this.getName()+" not full by virtue of unselected obligates");
					}
					return false;
				}
			}
			if(this.getNSB()<=nSelected) {
				if(SNPPicker.verbose) {
					System.out.println("isFull=true from NSB");
				}
				return true;
			}
			// maybe there are no more to pick.
			int _nExcluded = this.getNExcluded();
			if(this.nsnps==this.nSelected+_nExcluded) {
				this.full=true;
				if(SNPPicker.verbose) {
					System.out.println("isFull=true from Excluded");
				}

			} else if (this.nsnps<this.nSelected+_nExcluded) {
					System.err.println("SNPPIcker:BUG: Bin.isFull:bug: snps.length<nSelected+nExcluded for bin "+this.getId()+"\n");
					System.err.println("nsnps="+this.nsnps+",nSelected="+this.nSelected+", nExcluded="+_nExcluded);
					for(int i=0;i<this.snps.length;i++) {
						Snp s = this.snps[i];
						if(s!=null) {
							System.err.println("SNPPicker:ERROR: snp "+s.getSnpName()+", selected ="+(s.isSelected()? "yes":"no ")+", obligate="+(s.isObligate()? "yes":"no ")+
									", exclude ="+(s.isExcluded()? "yes":"no ")+", fixed="+(s.isFixed()? "yes":"no ")+"");
						}
					}
					System.err.flush();
					System.out.flush();
					System.exit(-1);
			}
		} else {
			if(SNPPicker.verbose) {
				System.out.println("isFull=true from snps==null");
			}
			this.full=true;
		}
		return this.full;
	}
	
	public int getNExcluded() {
		// HS- THere is a bug in setting/unsetting NExcluded.. so computed it from SNPs.
		int _nExcluded=0;
		if(this.snps!=null) {
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null) {
					if(s.isExcluded()) {
						_nExcluded++;
					}
				}
			}
		}
		return _nExcluded;
	}
	public int getTmpNSelected() {
		return this.tmpNSelected;
	}
	public int[] getTmpSelectedSnpIds() {
		return this.tmpSelectedSnpIds;
	}
	public void warning(String msg) {
		
		
		System.err.println("SnpsTooClose::WARNING :"+msg+": Bin "+this.getId()+",  with snps {"+ this.getSnpString()+"} in Pop "+Population.getPopById(this.getPopid()).getName());

	
	}
	
	public String getSnpString() {
		if(this.snps ==null ) {
			return "";
		}
		StringBuffer snpString = new StringBuffer();
		for(int l=0;l<this.snps.length;l++) {
			Snp s = this.snps[l];
			if(s!=null) {
				 snpString.append(";");
				 snpString.append(s.getSnpName());
				 snpString.append("[");
				 snpString.append(s.getDbsnpid());
				 snpString.append("]");
				 snpString.append("(pos=");					
				 snpString.append(s.getPosition());
				 snpString.append(")");
			}
		}
		if(snpString.length()>0) {
			return snpString.substring(1);
		} else {
			return "";
		}
	}
	
	public void error(String msg) {
		System.err.println("SnpsTooClose::Error :"+msg+": Bin "+this.getId()+" in Pop "+Population.getPopById(this.getPopid()).getName());
		System.out.flush();
		System.err.flush();
		System.exit(-1);
	}
	/**
	 *
	 * @param snpsInDangerZoneByPos
	 * @return Best Scoring SNP outside Danger Zone.
	 */
	public int findBestUnpickedSnpOutsideDangerZone(HashMap<String,Snp> snpsInDangerZoneByPos) {


		Snp[] bestsnps = this.getMaxScoringSnps(); // already sorted Snps
		if(bestsnps==null || bestsnps.length==0) {
			return -1;
		}
		// Favor obligates
		for (int i=0;i<bestsnps.length;i++) {
			Snp s = bestsnps[i];
			if(s!=null && s.isObligate() && !s.isExcluded()) {
				int sspos = s.getPosition();
				if(snpsInDangerZoneByPos==null || !snpsInDangerZoneByPos.containsKey(Integer.toString(sspos))) {
					if((!( s.isExcluded() || s.isTmpSelected()))) {
						return i;
					}
				}
			}
		}
		
		for (int i=0;i<bestsnps.length;i++) {
			Snp s = bestsnps[i];
			if(s!=null && !(s.isObligate() || s.isExcluded())) {
				int sspos = s.getPosition();
				if(snpsInDangerZoneByPos==null || !snpsInDangerZoneByPos.containsKey(Integer.toString(sspos))) {
					if((!(s.isExcluded() || s.isTmpSelected()))) {
						return i;
					}
				}
			}
		}
		return -1;
	}

	/**
	 * returns the success Prob for the selected + temporarely Selected Snps in this Bin 
	 * @return
	 */
	public double getTmpProb() {
		if(snps==null) {
			return 0.0d;
		}
		double pf=1.0;
		for(int i=0;i<this.snps.length ;i++) {
			Snp s = this.snps[i];
			if(s!=null && (!s.isExcluded())&&
					(s.isObligate() || s.isObligateButNotGenotyped() || s.isTmpSelected()) ) {
				pf *= s.getFailProb();
			}
		}
		return 1.0d-pf;
	}
	
	
	/**
	 * returns the Prob for the obligates AND fixed-selected AND the list of snpss to select.
	 * Strong assumption that snps to select were not already fixed or obligates
	 * @return
	 */
	public double getProbForSnpsInThisBin(Snp[] snps_in_cluster) {
		if(snps==null) {
			return 0.0d;
		}
		double pf=1.0;
		for(int i=0;i<this.snps.length;i++) {
			Snp s = this.snps[i];
			if(s!=null) {
				if((((s.isObligate() || s.isObligateButNotGenotyped()) && !s.isExcluded()) || (s.isFixed() && s.isSelected())  )) {
					pf *= s.getFailProb();
				} else if(snps_in_cluster!=null) {
					boolean selectIt=false;
					for(int j=0;(selectIt==false) && j<snps_in_cluster.length;j++) {
						Snp s2 = snps_in_cluster[j];
						if(s2!=null && s.getId() == s2.getId() && s2.isExcluded()==false) {
							selectIt=true;
						}
					}
					if(selectIt) {
						pf *= s.getFailProb();
					}
				}
			}
		}
		
		return 1.0d-pf;
	}

	
	/**
	 * returns the Prob for the selected and currently printed Snps in this Bin * the number of tag snps
	 * @return
	 */
	public double getPrintProb() {
		if(snps==null) {
			return 0.0d;
		}
		double pf=1.0;
		for(int i=0;i<this.snps.length;i++) {
			Snp s = this.snps[i];
			if(s!=null && s.isWasPrinted() && (s.isSelected()  || (s.isObligate() && ! s.isExcluded()) || s.isObligateButNotGenotyped()) ) {
				pf *= s.getFailProb();
			}
		}

		//System.out.print("print prob="+(1.0-pf)+"\n");
		return 1.0d-pf;
	}
	
	public int countPrintSelectedSnps() {
		if(snps==null) {
			return 0;
		}
		int npsel=0;
		for(int i=0;i<this.snps.length;i++) {
			Snp s = this.snps[i];
			if(s!=null && !s.isExcluded() && s.isWasPrinted() && (s.isSelected() || s.isObligate() || s.isObligateButNotGenotyped())) {
				npsel++;
			}
		}
		//System.out.print("print selected="+npsel+"\n");
		return npsel;
	}
	/**
	 * returns the Prob for the Selected Snps in this Bin * the number of tag snps
	 * @return
	 */
	public double getProb() {
		if(snps==null) {
			return 0.0d;
		}
		double pf=1.0;
		for(int i=0;i<this.snps.length;i++) {
			Snp s = this.snps[i];
			if(s!=null && (!s.isExcluded()) && (s.isSelected() ||  s.isObligate() || s.isObligateButNotGenotyped())) {
				pf*= s.getFailProb();
			}
		}
		return 1.0d-pf;
	}









	//   The best scoring solution maximizes ( bang for the buck)
	// (Expected # of SNPS represented)/ # of tag-snps genotyped)
	//   To penalize solutions with low-design score SNPs, we can add a risk-averse factor.
	//   risk_factor*Expected_Failure_prob*bin_size + expected_success*bin_size

	//      If a bin is excluded, the probability of failure is 1 and the prob of success is 0.
	//      This takes care of comparing partial solutions.

	//    All other things being equal, take solution with smallest sum(functional category) wins.


	/**
	 * @return the score for the Selected+Temporarely Selected Snps.
	 */
	public double getBinTmpScore() {
		int n_in_bin = this.nSites;
		double p=this.getTmpProb(); // P_success
		double bscore = (SNPPicker.riskFactor*(1-p)*n_in_bin+p*n_in_bin);
		return bscore;
	}
	/**
	 * @return the score for the SNPs that were already Selected and Printed.
	 */
	public double getBinPrintScore() {
		int n_in_bin = this.nSites;
		double p=this.getPrintProb(); // P_success for selected and wasPrinted Snps.
		double bscore = (SNPPicker.riskFactor*(1-p)*n_in_bin+p*n_in_bin);
		return bscore;
	}
	
	/**
	 *
	 * @return the score for the Selected Snps
	 */
	public double getBinScore() {
		int n_in_bin = this.nSites;
		double p=this.getProb(); // P_success
		double bscore = (SNPPicker.riskFactor*(1-p)*n_in_bin+p*n_in_bin);
		return bscore;
	}

	/**
	 *
	 * @return The Sum of the Functional Location Class (smallest Wins) for Temporarily Selected SNPs. Ignore obligate tag SNPs as this is a constant.
	 */
	public double getTmpSumFuncClass() {
		double sum=0.0;
		if(this.snps!=null) {
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null && (s.isTmpSelected())) {
					sum+=s.getLocation_class();
				}
			}
		}
		return sum;
	}
	/**
	 *
	 * @return The Sum of the Functional Location Class (biggest Wins) for Selected Snps. Ignore obligate tag SNPs as this is a constant.
	 */
	public double getSumFuncClass() {
		double sum=0.0;
		if(this.snps!=null) {
			for(int i=0;i<this.snps.length;i++) {
				Snp s = this.snps[i];
				if(s!=null && s.isSelected()) {  // We don't need to consider obligates since all solutions include them.
					sum+=s.getLocation_class();
				}
			}
		}
		return sum;
	}
	public double getMaxScore() {
		return maxScore;
	}
	public int getStatus() {
		return status;
	}
	public void setStatus(int _status) {
		this.status = _status;
		if(SNPPicker.verbose) {
			System.out.println("Status="+_status+" for bin="+this.getName());
		}
	}
	public void computeStatus() {
		if(this.nSelected >= this.NSB) { // includes included obligates.
			this.setStatus(1); // Bin is full.
			return;
		}
//		int _nExcluded = this.getNExcluded();
		if(this.nSelected<this.nsnps-this.nExcluded) { // Still some tags left to pick in that bin.
			if(SNPPicker.minP<1.0) {// Threshold option was activated.
				if(this.getProb()>=SNPPicker.minP) {
					this.setStatus(1);// b1 doesn't need any more tagSNPs
				} else {
					int[] topick = this.getAvailableSnpIds(false);
					if(topick.length>0) {
						this.setStatus(0); // some more tag SNPs available for picking and  need more.
					} else {
						this.setStatus(1);
					}
				}
			} else {
				int[] topick = this.getAvailableSnpIds(false);
				if(topick.length>0) {
					this.setStatus(0); // some more tag SNPs available for picking and  need more.
				} else {
					this.setStatus(1);
				}
			}
		} else {// no more tagSNPs available to pick... even if wanted more.
			this.setStatus(1); // When SNPs are fixed, this status does not reflect that some fixed SNPs are unfixable.
		}
	}

	public int getTmpStatus() {
		if(this.tmpNSelected>= this.NSB) { // includes included obligates.
			this.setStatus(1); // Bin is full.
			return 1;
		}
		// There is no tmpExcluded Statuus
		if(this.nSelected<this.nsnps-this.nExcluded) { // Still some tags left to pick in that bin.
			
			if(SNPPicker.minP<1.0) {// Threshold option was activated.
				if(this.getProb()>=SNPPicker.minP) {
					return 1;// b1 doesn't need any more tagSNPs
				} else {
					int[] topick = this.getAvailableSnpIds(true);
					if(topick.length>0) {
						return 0; // some more tag SNPs available for picking and  need more.
					} else {
						return 1; // No more to pick..
					}
				}
			} else {
				int[] topick = this.getAvailableSnpIds(true);
				if(topick.length>0) {
					return 0; // some more tag SNPs available for picking and  need more.
				} else {
					return 1; // No more to pick.. bins is full.
				}
			}
		} else {// no more tagSNPs available to pick... even if wanted more.
			return 1;
		}
	}
	
	
	public int[] getMaxCoverageSnpids() {
		return this.maxCoverageSnpids;
	}
	public void setMaxCoverageSnpids(int[] maxCoverageSnpids) {
		this.maxCoverageSnpids = maxCoverageSnpids;
	}
	public static void outEmpty(PrintStream outStream) {
		for(int i=0;i<indexSize;i++) {
			Bin b = (Bin) binIndex[i];
			if(b!=null) {
				b.outEmpty("",outStream);
			}
		}
	}

	public void outEmpty(String msg,PrintStream outStream) {
		if(this.snps==null || this.nsnps==0) {
			Snp.outEmpty(this.name,nSites,"","","",outStream);
			if(this.nSites==0) {
				System.err.println("ERROR:ERROR "+this.name+" has no sites or SNPs in it\n");
			}
		} else {
//			boolean debugthis=false;
			int nSelected = 0;
			Snp sref=null;
			for(int i=0;i<this.snps.length;i++) {
				Snp s = snps[i];
				if(s!=null) {
					sref=s;
//					if(debugthis) {
//						System.out.println("GOOGLE: for snp rs17860264, this index="+i);
//					}
//					if(s.getDbsnpid().equals("rs17860264")) {
//							debugthis=true;
//							System.out.println("GOOGLE: for snp rs17860264, bin size ="+this.nsnps+", this index="+i+", for this bin="+this.getName());
//					}
					if((s.isSelected() || s.isObligate()) && ! s.isExcluded()) {
						nSelected++;
//						if(debugthis) {
//							System.out.println("GOOGLE: for snp rs17860264, bin name ="+this.getName()+", this index="+i+", is selected for snp ="+s.getSnpName());
//						}
//						if(!debugthis) {
							break;
//						}
					}
				}
			}
			if(nSelected==0) {
				String gene="";
				String chromosome="";
				String position ="";
				if(sref!=null) {
					gene = sref.getGene();
					chromosome = sref.getChromosome();
					position = Integer.toString(sref.getPosition());
				}
//				 if (debugthis) {
//						System.out.println("GOOGLE: found a Selected SNP, for bin "+this.getName()+" for snp rs17860264");
//					}
				Snp.outEmpty(this.name,this.nSites,gene,  chromosome,position,outStream);
			} //else if (debugthis) {
			//	System.out.println("GOOGLE: Did not find any selected snps for bin "+this.getName()+" for snp rs17860264");
			//}
		}
	}
	public String getGene() {
		if(this.gene!=null && this.gene.length()>0) {
			return this.gene;
		} else {
			return "";
			// fill in matching gene-geneid
			/*
			if(snps!=null && snps.length>0) {
				int ng=0;
				String newgene=null;
				String newgeneid=null;
				for(int i=0;i<snps.length;i++) {
					Snp s = snps[i];
					if(s!=null) {
						String tgene = s.getGene();
						if(tgene!=null && tgene.length()>0) {
							String tgeneId = s.getGeneId();
							if(!tgene.equals(newgene)) {
								ng++;
								newgene=tgene;
								newgeneid = tgeneId;
							}
							if(geneId!=null && geneId.length()>0) {
								if(geneId.equals(tgeneId)) {
									//same geneid on bin as SNP ==> only fixing annotation.
									this.gene = tgene;
									return this.gene;
								}
							}
						}
					}
				}
				// Only found one gene and this bin did not have a geneid.
				if(ng==1 && (this.geneId==null || this.geneId.length()==0)) {
					this.gene=newgene;
					this.geneId=newgeneid;
					return this.gene;
				} else {
					// Use-Majority No.. Fix the code to include gene annotation.
				}
			}
			*/
		}
		//return "";
	}
	public void setGene(String gene) {
		this.gene = gene;
	}
	public String getGeneId() {
		if(geneId!=null && geneId.length()>0) {
			return this.geneId;
		} else {
			return "";
			/*
			// fill in matching gene-geneid
			if(snps!=null && snps.length>0) {
				int ng=0;
				String newgene=null;
				String newgeneid=null;
				for(int i=0;i<snps.length;i++) {
					Snp s = snps[i];
					if(s!=null) {
						String tgeneId = s.getGeneId();
						if(tgeneId!=null && tgeneId.length()>0) {
							String tgene = s.getGene();
							if(!tgeneId.equals(newgeneid)) {
								ng++;
								newgene=tgene;
								newgeneid = tgeneId;
							}
							if(this.gene!=null && this.gene.length()>0) {
								if(this.gene.equals(tgene)) {
									//same geneid on bin as SNP ==> only fixing annotation.
									this.geneId = tgeneId;
									return this.geneId;
								}
							}
						}
						
						
					}
				}
				// Only found one gene and this bin did not have a geneid.
				if(ng==1 && (this.geneId==null || this.geneId.length()==0)) {
					this.gene=newgene;
					this.geneId=newgeneid;
					return this.geneId;
				} else {
					// Use-Majority No.. Fix the code to include gene annotation.
				}
			}
			*/
		}
		//return "";
	}
	public void setGeneId(String geneid) {
		this.geneId = geneid;
	}
	public String getSource() {
		return source;
	}
	public void setSource(String source) {
		this.source = source;
	}
	public String getChromosome() {
		return chromosome;
	}
	public void setChromosome(String this_chromosome) {
		this.chromosome = this_chromosome;
	}
	
	public int getNRemainingToChoose(boolean tmpToo)  {
		int nToChoose1=0;
		
		if(tmpToo) { 
			nToChoose1=this.getNSB()-this.getTmpNSelected();
		} else {
			nToChoose1=this.getNSB()-this.getNSelected();
		}
		int[] ssids1 = null;
		if(nToChoose1>0) {
			ssids1 = this.getAvailableSnpIds(tmpToo);
			if( ssids1==null || ssids1.length==0) {
				nToChoose1=0;
			} else if (ssids1.length<nToChoose1) {
				nToChoose1=ssids1.length; // can't choose more than you have
			}
		}
		return nToChoose1;
	}
	/*
	 *  Pick The NSB best Probability tagSNPs in this Bin
	 */
	public void pickBinByProb(boolean pickOnlyOne,boolean tmpToo) throws Exception {
		this.cacheMaxScore(tmpToo);//Does not include Fixed
		this.cacheMaxCoverage(tmpToo);//Does not include Fixed
		int _status =0;
		if(tmpToo) {
			_status = this.getTmpStatus();
		} else {
			this.computeStatus(); 	// set status to 1 if all NSB snps were picked.
			_status = this.getStatus();
		}
		int[] s = this.getMaxScoringSnpids(); // sorted by Prob. does not Includesselected (obligates) and excluded
		int i=0;
		while(s!=null && _status==0 && s.length>i ) {
			int t_sid= s[0];
			Snp snp = Snp.getSnpById(t_sid);
			if(snp!=null && snp.canISelect(tmpToo)) {
				this.selectSnpId(t_sid,true,true); // new state=true, pick bin too

				this.cacheMaxScore(tmpToo);
				this.cacheMaxCoverage(tmpToo);
				if(tmpToo) {
					_status = this.getTmpStatus();
				} else {
					this.computeStatus(); 	// set status to 1 if all NSB snps were picked.
					_status = this.getStatus();
				}
				if(pickOnlyOne) {
					break;
				}
				s=this.getMaxScoringSnpids();
			} else {
				i++;
			}
		}
		this.computeStatus();
	}
/*
 *  Pick The NSB best Probability tagSNPs in this Bin
 */
	public void pickTmpBinByProb(boolean pickOnlyOne) throws Exception {
		boolean tmpToo=true;
		this.cacheMaxScore(tmpToo);
		this.cacheMaxCoverage(tmpToo);
		this.computeStatus(); 	// set status to 1 if all NSB snps were picked.
		int[] sids = this.getMaxScoringSnpids(); // sorted by Prob. does not Includesselected (obligates) and excluded
		int i=0;
		while(sids!=null && sids.length>0 && this.getStatus()==0 && i<sids.length ) {
			int t_sid= sids[0];
			if(t_sid>0) {
				Snp s = Snp.getSnpById(t_sid);
				 // Check if no proximity constraint and if no SNPs too close.
				if(s!=null && s.canISelect(true)==true) {
					this.tmpSelectSnpId(true,t_sid,true); // new state=true, pick bin too
					this.cacheMaxScore(tmpToo);
					this.cacheMaxCoverage(tmpToo);
					this.computeStatus(); 	// set status to 1 if all NSB snps were picked.
					if(pickOnlyOne) {
						break;
					}
					sids=this.getMaxScoringSnpids();
					i=0;
				} else {
					i++;
				}
			} else {
				i++;
			}
		}
	}

/*
 *  Pick The SNPs with the most leftover coverage
 *  @param pickOnlyOne If true, only pick One SNP, otherwise pick all NSB Snps needed.
 */
public void pickBinGreedy(boolean pickOnlyOne,boolean tmpToo) throws Exception {

	this.cacheMaxCoverage(tmpToo);
	this.computeStatus(); 	// set status to 1 if all NSB SNPs were picked .. or if all pickable SNPs are picked.
	if(SNPPicker.verbose) {
		System.out.println("SNPPicker:INFO: Picking bin "+this.getName());System.out.flush();
	}
	int[] ss = this.getMaxCoverageSnpids(); // Remaining unselected SNPs, sorted by Max Coverage, then prob, then score
	if(SNPPicker.verbose) {
		System.out.println("SNPPicker:INFO: "+this.getName()+" started with "+(ss==null ? 0: ss.length)+" max coverage SNPids");System.out.flush();
		System.out.println("SNPPicker:INFO: "+this.getName()+" start status="+this.getStatus());System.out.flush();
	}
	int i=0;
	while((ss!=null) && (ss.length>0) && (i<ss.length) &&
			   ((tmpToo &&this.getTmpStatus()==0) 
		   || ((!tmpToo) && this.getStatus()==0))) {
		int t_sid=ss[i];
		if(t_sid<=0) {
				System.out.println("SNPPicker:INFO: "+this.getName()+" has snpid==0 max coverage SNP!");System.out.flush();
				i++;
		} else {
			Snp s = Snp.getSnpById(t_sid);
			if(s!=null && s.canISelect(tmpToo)) {
				if(SNPPicker.verbose) {
					System.out.println("SNPPicker:INFO: Bin "+this.getName()+" : greedy selected snp="+s.getSnpName());System.out.flush();
				}
				this.selectSnpId(t_sid,true,true);// permanently select snp. and process SNP too.

				this.cacheMaxCoverage(tmpToo);
				this.computeStatus(); // set status to 1 if all NSB snps were picked.
				if(pickOnlyOne) {
					break;
				}
				ss = this.getMaxCoverageSnpids(); // includes the effect of tmpToo
				i=0;
				if(SNPPicker.verbose) {
					System.out.println("SNPPicker:INFO: "+this.getName()+" now has "+(ss==null ? 0: ss.length)+" max coverage SNPids");System.out.flush();
					System.out.println("SNPPicker:INFO: "+this.getName()+" loop status="+this.getStatus());System.out.flush();
				}
			} else {
				i++;
			}
		}
	}

	this.computeStatus(); // All SNPs that were pickable were picked OR all needed SNPs were picked.
}
public void reset(){
	  this.status =0;
	  this.maxCoverageSnpids = new int[0];
	  this.maxScoringSnpids=new int[0];
	  this.maxScoringSnps=new Snp[0];
	  this.maxScore=0.0d;
	  this.maxCoverageSnpids=new int[0];
	  this.nSelected=0;
	  this.nExcluded=0;
	  this.selectedSnpIds=new int[0];
	  this.full=false;
	  this.tmpNSelected=0;
	  this.tmpSelectedSnpIds=new int[0];
	  this.hasBeenReset=false;
	  this.bestNSelected=0;
	  this.bestSelectedSnpIds=null;
	  this.bestWasTransfered=false;
	  
	  Snp[] ss =this.snps;
	  if(ss!=null) {
		  for(int ii=0;ii<ss.length;ii++) {
			  Snp s = ss[ii];
			  if(s!=null) {
				  s.reset();
			  }
		  }
	  }
}
public void resetExceptBestAndFixed(){
	  this.status =0;
	  this.maxCoverageSnpids = new int[0];
	  this.maxScoringSnpids=new int[0];
	  this.maxScoringSnps=new Snp[0];
	  this.maxScore=0.0d;
	  this.maxCoverageSnpids=new int[0];
	  this.nSelected=0;
	  this.nExcluded=0;
	  this.selectedSnpIds=new int[0];
	  this.full=false;
	  this.tmpNSelected=0;
	  this.tmpSelectedSnpIds=new int[0];
	  this.hasBeenReset=false;

	  
	  Snp[] ss =this.snps;
	  if(ss!=null) {
		  for(int ii=0;ii<ss.length;ii++) {
			  Snp s = ss[ii];
			  if(s!=null) {
				  	if(!s.isFixed()) {
				  		s.resetExceptBest();
				  	} else {
				  		if(s.isSelected()) {
				  			nSelected++;
				  			if(this.selectedSnpIds.length<nSelected) {
				  				int [] newsel = new int[nSelected+5];
				  				for(int i=0;i<nSelected;i++) {
				  					newsel[i]=this.selectedSnpIds[i];
				  				}
				  				newsel[nSelected-1]=s.getId();
				  				this.selectedSnpIds=newsel;
				  			}
				  		} else if (s.isExcluded()) {
				  			nExcluded++;
				  		}
				  	}
			  }	
		  }
	  }
}
public static void resetAll() {
	  for(int i=0;i<Bin.binIndex.length;i++) {
		  Bin  b = (Bin)binIndex[i]; 
		  if(b!=null) {
			  b.reset();
		  }  
	  }
}
public static void resetBins(Bin[] binslist) {
	if(binslist!=null) {
	  for(int i=0;i<binslist.length;i++) {
		  Bin  b = (Bin)binslist[i]; 
		  if(b!=null) {
			  b.reset();
		  }  
	  }
	}
}
public static void resetBinsExceptBestAndFixed(Bin[] binslist) {
	if(binslist!=null) {
	  for(int i=0;i<binslist.length;i++) {
		  Bin  b = (Bin)binslist[i]; 
		  if(b!=null) {
			  b.resetExceptBestAndFixed();
		  }  
	  }
	}
}
public static Bin[] getAllBins() {
	Bin[] allbins = new Bin[nbins];
	int ib=0;
	for(int i=0;i<binIndex.length;i++) {
		Bin b = (Bin) binIndex[i];
		if(b!=null && b.snps!=null) {
			allbins[ib++]=b;
		}
	}
	return allbins;
}

public static void dumpBinProbFile(String file) {
	PrintStream bpf =null;
	try {
		bpf = new PrintStream(new BufferedOutputStream(
			new FileOutputStream(file)));

	} catch (Exception e) {
		System.err.println(e.getMessage());
		System.err.flush();
		return;
	}
	bpf.println("binid\tbinname\tgene\tchromosome\tnselected\tntags\tbinprob\n");

	for(int i=0;i<binIndex.length;i++) {
		Bin b = (Bin) binIndex[i];
		if(b!=null && b.snps!=null) {
			if(!Population.getPopById(b.getPopid()).getName().equals("obligate")) {
				b.getNSelected();
				bpf.println(b.binid+"\t"+b.getName()+"\t"+b.getGene()+"\t"+b.chromosome+"\t"+b.getNSelected()+"\t"+b.getNsnps()+"\t"+b.getProb());
			}
		}
	}
	bpf.close();
}
	
	
} // end of Bin class
