/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.interval;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import net.sf.picard.reference.ReferenceSequenceFile;
import net.sf.picard.util.Interval;
import net.sf.picard.util.IntervalList;
import net.sf.samtools.SAMFileHeader;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.gatk.datasources.reference.ReferenceDataSource;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.GenomeLocParser;
import org.broadinstitute.sting.utils.GenomeLocSortedSet;
import org.broadinstitute.sting.utils.Utils;
import org.broadinstitute.sting.utils.collections.Pair;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.interval.IntervalMergingRule;
import org.broadinstitute.sting.utils.interval.IntervalSetRule;
import org.broadinstitute.sting.utils.text.XReadLines;

public class IntervalUtils {
    private static Logger logger = Logger.getLogger(IntervalUtils.class);

    public static List<GenomeLoc> parseIntervalArguments(GenomeLocParser parser, List<String> argList) {
        ArrayList<GenomeLoc> rawIntervals = new ArrayList<GenomeLoc>();
        if (argList != null) {
            for (String argument : argList) {
                rawIntervals.addAll(IntervalUtils.parseIntervalArguments(parser, argument));
            }
        }
        return rawIntervals;
    }

    public static List<GenomeLoc> parseIntervalArguments(GenomeLocParser parser, String arg) {
        ArrayList<GenomeLoc> rawIntervals = new ArrayList<GenomeLoc>();
        if (arg.indexOf(59) != -1) {
            throw new UserException.BadArgumentValue("-L " + arg, "The legacy -L \"interval1;interval2\" syntax is no longer supported. Please use one -L argument for each interval or an interval file instead.");
        }
        if (IntervalUtils.isUnmapped(arg)) {
            rawIntervals.add(GenomeLoc.UNMAPPED);
        } else if (IntervalUtils.isIntervalFile(arg)) {
            try {
                rawIntervals.addAll(IntervalUtils.intervalFileToList(parser, arg));
            }
            catch (UserException.MalformedGenomeLoc e) {
                throw e;
            }
            catch (Exception e) {
                throw new UserException.MalformedFile(arg, "Interval file could not be parsed in any supported format.", e);
            }
        } else {
            rawIntervals.add(parser.parseGenomeLoc(arg));
        }
        return rawIntervals;
    }

    public static List<GenomeLoc> intervalFileToList(GenomeLocParser glParser, String file_name) {
        File inputFile = new File(file_name);
        ArrayList<GenomeLoc> ret = new ArrayList<GenomeLoc>();
        if (file_name.toUpperCase().endsWith(".BED")) {
            throw new ReviewedStingException("BED files must be parsed through Tribble; parsing them as intervals through the GATK engine is no longer supported");
        }
        boolean isPicardInterval = false;
        try {
            IntervalList il = IntervalList.fromFile((File)inputFile);
            isPicardInterval = true;
            int nInvalidIntervals = 0;
            for (Interval interval : il.getIntervals()) {
                if (glParser.isValidGenomeLoc(interval.getSequence(), interval.getStart(), interval.getEnd(), true)) {
                    ret.add(glParser.createGenomeLoc(interval.getSequence(), interval.getStart(), interval.getEnd(), true));
                    continue;
                }
                ++nInvalidIntervals;
            }
            if (nInvalidIntervals > 0) {
                logger.warn((Object)("Ignoring " + nInvalidIntervals + " invalid intervals from " + inputFile));
            }
        }
        catch (Exception e) {
            if (isPicardInterval) {
                throw new UserException.CouldNotReadInputFile(inputFile, e);
            }
            try {
                XReadLines reader = new XReadLines(new File(file_name));
                for (String line : reader) {
                    if (line.trim().length() <= 0) continue;
                    ret.add(glParser.parseGenomeLoc(line));
                }
                reader.close();
            }
            catch (IOException e2) {
                throw new UserException.CouldNotReadInputFile(inputFile, (Exception)e2);
            }
        }
        return ret;
    }

    public static boolean isUnmapped(String interval) {
        return interval != null && interval.trim().toLowerCase().equals("unmapped");
    }

    public static List<GenomeLoc> mergeListsBySetOperator(List<GenomeLoc> setOne, List<GenomeLoc> setTwo, IntervalSetRule rule) {
        if (setOne == null || setOne.size() == 0 || setTwo == null || setTwo.size() == 0) {
            return setOne == null || setOne.size() == 0 ? setTwo : setOne;
        }
        if (rule == IntervalSetRule.UNION) {
            setOne.addAll(setTwo);
            return setOne;
        }
        int iOne = 0;
        int iTwo = 0;
        LinkedList<GenomeLoc> retList = new LinkedList<GenomeLoc>();
        while (iTwo < setTwo.size() && iOne < setOne.size()) {
            if (setTwo.get(iTwo).isBefore(setOne.get(iOne))) {
                ++iTwo;
                continue;
            }
            if (setOne.get(iOne).isBefore(setTwo.get(iTwo))) {
                ++iOne;
                continue;
            }
            retList.add(setOne.get(iOne).intersect(setTwo.get(iTwo)));
            if (setOne.get(iOne).getStop() < setTwo.get(iTwo).getStop()) {
                ++iOne;
                continue;
            }
            ++iTwo;
        }
        if (retList.size() == 0) {
            throw new UserException.BadInput("The INTERSECTION of your -L options produced no intervals.");
        }
        return retList;
    }

    public static GenomeLocSortedSet sortAndMergeIntervals(GenomeLocParser parser, List<GenomeLoc> intervals, IntervalMergingRule mergingRule) {
        Collections.sort(intervals);
        intervals = IntervalUtils.mergeIntervalLocations(intervals, mergingRule);
        return GenomeLocSortedSet.createSetFromList(parser, intervals);
    }

    public static String equateIntervals(List<GenomeLoc> masterArg, List<GenomeLoc> testArg) {
        LinkedList<GenomeLoc> master = new LinkedList<GenomeLoc>(masterArg);
        LinkedList<GenomeLoc> test = new LinkedList<GenomeLoc>(testArg);
        while (!master.isEmpty()) {
            GenomeLoc masterHead = master.pop();
            GenomeLoc testHead = test.pop();
            if (testHead.overlapsP(masterHead)) {
                for (GenomeLoc masterPart : Utils.reverse(masterHead.subtract(testHead))) {
                    master.push(masterPart);
                }
                continue;
            }
            return "Incompatible locs detected masterHead=" + masterHead + ", testHead=" + testHead;
        }
        if (test.isEmpty()) {
            return null;
        }
        return "Remaining elements found in test: first=" + test.peek();
    }

    public static boolean isIntervalFile(String str) {
        return IntervalUtils.isIntervalFile(str, true);
    }

    public static boolean isIntervalFile(String str, boolean checkExists) {
        File file = new File(str);
        if (str.toUpperCase().endsWith(".BED") || str.toUpperCase().endsWith(".LIST") || str.toUpperCase().endsWith(".PICARD") || str.toUpperCase().endsWith(".INTERVAL_LIST") || str.toUpperCase().endsWith(".INTERVALS")) {
            if (!checkExists) {
                return true;
            }
            if (file.exists()) {
                return true;
            }
            throw new UserException.CouldNotReadInputFile(file, "The interval file does not exist.");
        }
        if (file.exists()) {
            throw new UserException.CouldNotReadInputFile(file, String.format("The interval file %s does not have one of the supported extensions (.bed, .list, .picard, .interval_list, or .intervals). Please rename your file with the appropriate extension. If %s is NOT supposed to be a file, please move or rename the file at location %s", str, str, file.getAbsolutePath()));
        }
        return false;
    }

    public static Map<String, Integer> getContigSizes(File reference) {
        ReferenceDataSource referenceSource = new ReferenceDataSource(reference);
        List<GenomeLoc> locs = GenomeLocSortedSet.createSetFromSequenceDictionary(referenceSource.getReference().getSequenceDictionary()).toList();
        LinkedHashMap<String, Integer> lengths = new LinkedHashMap<String, Integer>();
        for (GenomeLoc loc : locs) {
            lengths.put(loc.getContig(), loc.size());
        }
        return lengths;
    }

    public static void scatterContigIntervals(SAMFileHeader fileHeader, List<GenomeLoc> locs, List<File> scatterParts) {
        IntervalList intervalList = null;
        int fileIndex = -1;
        int locIndex = 0;
        String contig = null;
        for (GenomeLoc loc : locs) {
            if (!(fileIndex + 1 >= scatterParts.size() || contig != null && contig.equals(loc.getContig()))) {
                if (intervalList != null) {
                    intervalList.write(scatterParts.get(fileIndex));
                    intervalList = null;
                }
                ++fileIndex;
                contig = loc.getContig();
            }
            if (intervalList == null) {
                intervalList = new IntervalList(fileHeader);
            }
            intervalList.add(IntervalUtils.toInterval(loc, ++locIndex));
        }
        if (intervalList != null) {
            intervalList.write(scatterParts.get(fileIndex));
        }
        if (fileIndex + 1 != scatterParts.size()) {
            throw new UserException.BadArgumentValue("scatterParts", String.format("Only able to write contigs into %d of %d files.", fileIndex + 1, scatterParts.size()));
        }
    }

    public static List<List<GenomeLoc>> splitIntervalsToSubLists(List<GenomeLoc> locs, List<Integer> splits) {
        int start = 0;
        ArrayList<List<GenomeLoc>> sublists = new ArrayList<List<GenomeLoc>>(splits.size());
        for (Integer stop : splits) {
            ArrayList<GenomeLoc> curList = new ArrayList<GenomeLoc>();
            for (int i = start; i < stop; ++i) {
                curList.add(locs.get(i));
            }
            start = stop;
            sublists.add(curList);
        }
        return sublists;
    }

    public static void scatterFixedIntervals(SAMFileHeader fileHeader, List<List<GenomeLoc>> splits, List<File> scatterParts) {
        if (splits.size() != scatterParts.size()) {
            throw new UserException.BadArgumentValue("splits", String.format("Split points %d does not equal the number of scatter parts %d.", splits.size(), scatterParts.size()));
        }
        int fileIndex = 0;
        int locIndex = 1;
        for (List<GenomeLoc> split : splits) {
            IntervalList intervalList = new IntervalList(fileHeader);
            for (GenomeLoc loc : split) {
                intervalList.add(IntervalUtils.toInterval(loc, locIndex++));
            }
            intervalList.write(scatterParts.get(fileIndex++));
        }
    }

    public static List<List<GenomeLoc>> splitFixedIntervals(List<GenomeLoc> locs, int numParts) {
        if (locs.size() < numParts) {
            throw new UserException.BadArgumentValue("scatterParts", String.format("Cannot scatter %d locs into %d parts.", locs.size(), numParts));
        }
        long locsSize = IntervalUtils.intervalSize(locs);
        ArrayList<Integer> splitPoints = new ArrayList<Integer>();
        IntervalUtils.addFixedSplit(splitPoints, locs, locsSize, 0, locs.size(), numParts);
        Collections.sort(splitPoints);
        splitPoints.add(locs.size());
        return IntervalUtils.splitIntervalsToSubLists(locs, splitPoints);
    }

    @Requires(value={"locs != null", "numParts > 0"})
    @Ensures(value={"result != null"})
    public static List<List<GenomeLoc>> splitLocusIntervals(List<GenomeLoc> locs, int numParts) {
        long bp = IntervalUtils.intervalSize(locs);
        long idealSplitSize = Math.max((long)Math.floor((double)bp / (1.0 * (double)numParts)), 1L);
        ArrayList<List<GenomeLoc>> splits = new ArrayList<List<GenomeLoc>>(numParts);
        LinkedList<GenomeLoc> locsLinkedList = new LinkedList<GenomeLoc>(locs);
        while (!locsLinkedList.isEmpty()) {
            if (splits.size() + 1 == numParts) {
                splits.add(new ArrayList<GenomeLoc>(locsLinkedList));
                locsLinkedList.clear();
                continue;
            }
            SplitLocusRecursive one = IntervalUtils.splitLocusIntervals1(locsLinkedList, idealSplitSize);
            splits.add(one.split);
            locsLinkedList = one.remaining;
        }
        return splits;
    }

    @Requires(value={"remaining != null", "!remaining.isEmpty()", "idealSplitSize > 0"})
    @Ensures(value={"result != null"})
    static SplitLocusRecursive splitLocusIntervals1(LinkedList<GenomeLoc> remaining, long idealSplitSize) {
        ArrayList<GenomeLoc> split = new ArrayList<GenomeLoc>();
        long size = 0L;
        while (!remaining.isEmpty()) {
            GenomeLoc head = remaining.pop();
            long newSize = size + (long)head.size();
            if (newSize == idealSplitSize) {
                split.add(head);
                break;
            }
            if (newSize > idealSplitSize) {
                long remainingBp = idealSplitSize - size;
                long cutPoint = (long)head.getStart() + remainingBp;
                GenomeLoc[] parts = head.split((int)cutPoint);
                remaining.push(parts[1]);
                remaining.push(parts[0]);
                continue;
            }
            split.add(head);
            size = newSize;
        }
        return new SplitLocusRecursive(split, remaining);
    }

    public static List<GenomeLoc> flattenSplitIntervals(List<List<GenomeLoc>> splits) {
        ArrayList<GenomeLoc> locs = new ArrayList<GenomeLoc>();
        for (List<GenomeLoc> split : splits) {
            locs.addAll(split);
        }
        return locs;
    }

    private static void addFixedSplit(List<Integer> splitPoints, List<GenomeLoc> locs, long locsSize, int startIndex, int stopIndex, int numParts) {
        if (numParts < 2) {
            return;
        }
        int halfParts = (numParts + 1) / 2;
        Pair<Integer, Long> splitPoint = IntervalUtils.getFixedSplit(locs, locsSize, startIndex, stopIndex, halfParts, numParts - halfParts);
        int splitIndex = (Integer)splitPoint.first;
        long splitSize = (Long)splitPoint.second;
        splitPoints.add(splitIndex);
        IntervalUtils.addFixedSplit(splitPoints, locs, splitSize, startIndex, splitIndex, halfParts);
        IntervalUtils.addFixedSplit(splitPoints, locs, locsSize - splitSize, splitIndex, stopIndex, numParts - halfParts);
    }

    private static Pair<Integer, Long> getFixedSplit(List<GenomeLoc> locs, long locsSize, int startIndex, int stopIndex, int minLocs, int maxLocs) {
        int splitIndex = startIndex;
        long splitSize = 0L;
        for (int i = 0; i < minLocs; ++i) {
            splitSize += (long)locs.get(splitIndex).size();
            ++splitIndex;
        }
        long halfSize = locsSize / 2L;
        while (splitIndex < stopIndex - maxLocs && splitSize < halfSize) {
            splitSize += (long)locs.get(splitIndex).size();
            ++splitIndex;
        }
        return new Pair<Integer, Long>(splitIndex, splitSize);
    }

    private static Interval toInterval(GenomeLoc loc, int locIndex) {
        return new Interval(loc.getContig(), loc.getStart(), loc.getStop(), false, "interval_" + locIndex);
    }

    public static List<GenomeLoc> mergeIntervalLocations(List<GenomeLoc> raw, IntervalMergingRule rule) {
        if (raw.size() <= 1) {
            return raw;
        }
        ArrayList<GenomeLoc> merged = new ArrayList<GenomeLoc>();
        Iterator<GenomeLoc> it = raw.iterator();
        GenomeLoc prev = it.next();
        while (it.hasNext()) {
            GenomeLoc curr = it.next();
            if (prev.overlapsP(curr)) {
                prev = prev.merge(curr);
                continue;
            }
            if (prev.contiguousP(curr) && rule == IntervalMergingRule.ALL) {
                prev = prev.merge(curr);
                continue;
            }
            merged.add(prev);
            prev = curr;
        }
        merged.add(prev);
        return merged;
    }

    public static long intervalSize(List<GenomeLoc> locs) {
        long size = 0L;
        for (GenomeLoc loc : locs) {
            size += (long)loc.size();
        }
        return size;
    }

    public static void writeFlankingIntervals(File reference, File inputIntervals, File flankingIntervals, int basePairs) {
        ReferenceDataSource referenceDataSource = new ReferenceDataSource(reference);
        GenomeLocParser parser = new GenomeLocParser((ReferenceSequenceFile)referenceDataSource.getReference());
        List<GenomeLoc> originalList = IntervalUtils.intervalFileToList(parser, inputIntervals.getAbsolutePath());
        if (originalList.isEmpty()) {
            throw new UserException.MalformedFile(inputIntervals, "File contains no intervals");
        }
        List<GenomeLoc> flankingList = IntervalUtils.getFlankingIntervals(parser, originalList, basePairs);
        if (flankingList.isEmpty()) {
            throw new UserException.MalformedFile(inputIntervals, "Unable to produce any flanks for the intervals");
        }
        SAMFileHeader samFileHeader = new SAMFileHeader();
        samFileHeader.setSequenceDictionary(referenceDataSource.getReference().getSequenceDictionary());
        IntervalList intervalList = new IntervalList(samFileHeader);
        int i = 0;
        for (GenomeLoc loc : flankingList) {
            intervalList.add(IntervalUtils.toInterval(loc, ++i));
        }
        intervalList.write(flankingIntervals);
    }

    public static List<GenomeLoc> getFlankingIntervals(GenomeLocParser parser, List<GenomeLoc> locs, int basePairs) {
        List<GenomeLoc> sorted = IntervalUtils.sortAndMergeIntervals(parser, locs, IntervalMergingRule.ALL).toList();
        if (sorted.size() == 0) {
            return Collections.emptyList();
        }
        LinkedHashMap<String, List<GenomeLoc>> locsByContig = IntervalUtils.splitByContig(sorted);
        ArrayList<GenomeLoc> expanded = new ArrayList<GenomeLoc>();
        for (String contig : locsByContig.keySet()) {
            GenomeLoc stopLoc;
            List<GenomeLoc> contigLocs = locsByContig.get(contig);
            int contigLocsSize = contigLocs.size();
            GenomeLoc startLoc = parser.createGenomeLocAtStart(contigLocs.get(0), basePairs);
            if (startLoc != null) {
                expanded.add(startLoc);
            }
            for (int i = 0; i < contigLocsSize - 1; ++i) {
                stopLoc = parser.createGenomeLocAtStop(contigLocs.get(i), basePairs);
                startLoc = parser.createGenomeLocAtStart(contigLocs.get(i + 1), basePairs);
                if (stopLoc.getStop() + 1 >= startLoc.getStart()) {
                    GenomeLoc merged = parser.createGenomeLoc(stopLoc.getContig(), stopLoc.getStart(), startLoc.getStop());
                    expanded.add(merged);
                    continue;
                }
                expanded.add(stopLoc);
                expanded.add(startLoc);
            }
            stopLoc = parser.createGenomeLocAtStop(contigLocs.get(contigLocsSize - 1), basePairs);
            if (stopLoc == null) continue;
            expanded.add(stopLoc);
        }
        return expanded;
    }

    private static LinkedHashMap<String, List<GenomeLoc>> splitByContig(List<GenomeLoc> sorted) {
        LinkedHashMap<String, List<GenomeLoc>> splits = new LinkedHashMap<String, List<GenomeLoc>>();
        GenomeLoc last = null;
        ArrayList<GenomeLoc> contigLocs = null;
        for (GenomeLoc loc : sorted) {
            if (GenomeLoc.isUnmapped(loc)) continue;
            if (last == null || !last.onSameContig(loc)) {
                contigLocs = new ArrayList<GenomeLoc>();
                splits.put(loc.getContig(), contigLocs);
            }
            contigLocs.add(loc);
            last = loc;
        }
        return splits;
    }

    private static final class SplitLocusRecursive {
        final List<GenomeLoc> split;
        final LinkedList<GenomeLoc> remaining;

        @Requires(value={"split != null", "remaining != null"})
        private SplitLocusRecursive(List<GenomeLoc> split, LinkedList<GenomeLoc> remaining) {
            this.split = split;
            this.remaining = remaining;
        }
    }
}

