/* $Author: sinnwell $ */
/* $Date: 2011/02/25 16:37:24 $ */
/* $Header: /projects/genetics/cvs/cvsroot/mend.err/src/geno.c,v 1.3 2011/02/25 16:37:24 sinnwell Exp $ */
/* $Locker:  $ */
/* $Log: geno.c,v $
/* Revision 1.3  2011/02/25 16:37:24  sinnwell
/* long to int, add x.sexcheck, T to TRUE
/*
/* Revision 1.2  2011/02/16 16:40:00  sinnwell
/* fixed a small bug in sge.q, small fixes to c code, but still error
/*
/* Revision 1.1.1.1  2011/02/16 14:33:20  sinnwell
/* initial for mend.err package
/*
 * Revision 1.3  2004/04/13 20:36:46  folie
 * Child allele typo corrected.
 *
 * Revision 1.2  2003/09/11 21:07:37  folie
 * *** empty log message ***
 *
 * Revision 1.1  2003/09/11 17:19:46  folie
 * Initial revision
 * */
#include <stdlib.h>
#include <stdio.h>
#include "mcmemory.h"
#include <setjmp.h>
#include <signal.h>

/******************************************************************************/
/******************************************************************************/
/*

                                    OVERVIEW

     This source file contains functions used to implement genotype elimination
upon pedigree data.  It is assumed that the pedigrees being processed are 
acyclic in nature, but they may contain multiple pairs of ancestoral founders.
While the pedigrees are assumed to have no loops, pedigrees containing loops 
can still be processed without causing abnormal program termination.  Unfortu-
nately, the results for cyclic pedigrees are not guaranteed to be correct.  
Specifically, Mendelian Errors encountered in cyclic pedigrees are usually 
valid.  But in the presence of loops, there exist Mendelian Errors that will 
not be detected.  For acyclic pedigrees, though, all Mendelian Errors should
be caught and reported appropriately.
    All the functions below are static (not visible to programs defined outside
this file) with the exceptions of the following:

genotype_elimination(): 

     Peforms genotype elimination upon a pedigree.

genotype_elimination_jackknife():

     Performs genotype elimination upon pedigrees in which each untyped member
     is systematically jackknifed in an attempt to find candidates for the 
     causation of Mendelian Errors.

     Furthermore, the genotype elimination algorithm is based upon The 
Sequential Genotype-Elimination Algorithm outlined in O'Connell and Weeks [1], 
which is a modifed version of the Lange-Goradia Genotype-Elimination Algorithm
specified in Lange and Goradia [2].

                                    AUTHORS

     This program was primarily written by Daniel K. Folie with significant 
contributions from Daniel J. Schaid, Ph.D.  Correspondence is 

Daniel K. Folie                           Daniel J. Schaid, Ph.D.
Analyst/Programmer                        Editor-in-Chief, Genetic Epidemiology
Statistical Systems, Harwick 7            Division of Biostatistics, Harwick 7
Mayo Clinic                               Mayo Clinic
200 First St., SW                         200 First St., SW
Rochester, MN 55905                       Rochester, MN 55905

email: folie@mayo.edu                     email: schaid@mayo.edu


                                   REFERENCES

[1] Jeffrey R. O'Connell and Daniel E. Weeks.  An Optimal Algorithm for 
    Automatic Genotype Elimination.  "American Journal of Human Genetics,"
    65:1733-1740, 1999.

[2] Kenneth Lange and Tushar Madhu Goradia.  An Algorithm for Automatic
    Genotype Elimination.  "American Journal of Human Genetics," 
    40:250-256, 1987.

*/
/******************************************************************************/
/******************************************************************************/

static int     DONE     = 1;
static int     NOT_DONE = 0;
static int     FLAG     = 0;
static int    *STATUS   = NULL;
static jmp_buf  buf;

#define MALE         1 
#define FEMALE       0
#define CONSISTENT   1
#define INCONSISTENT 0
#define MENDELIAN_INCONSISTENCY -1
#define SORT_FLAG  0
#define PRINT_FLAG 1

/* Represents a person in a generated pedigree                                */

static struct person
{ struct marriage  *parents;              /* Parents' marriage node           */
  struct marriage  *nuclearFamily;        /* Person's first marriage node     */
  int              sex;		  /* M or F depending on sex          */
  int              status;		  /* Has this node had data generated */
  int              id;			  /* Person ID                        */
  int             *pheno;                /* This person phenotypes           */
  int            **genoLists;            /* Genotype list at each locus      */
  int             *nGenoLists;           /* Length of each genotype list     */
  int            **genoListsCon;         /* Genotype consistency vectors     */
};

/* Represents a marriage in a generated pedigree                              */

static struct marriage
{ struct person    *husband;	          /* Husband in this marriage         */
  struct person    *wife;                 /* Wife in this marriage	      */
  struct children  *child;                /* Children from this marriage      */
  struct marriage  *nextHusbandMarriage;  /* Husband's other marriage(s)      */
  struct marriage  *nextWifeMarriage;     /* Wife's other marriage(s)	      */
  int              done;                 /* Marriage processation indicator  */
};

/* Represents a linked list of children in a marriage                         */

static struct children
{ struct person    *firstChild;		  /* First child		      */
  struct children  *nextChild;		  /* Next child			      */
};

/* Represents a pedigree                                                      */

static struct pedigree
{ struct person    *proband;              /* Proband of the pedigree          */
  int              n;                    /* Number of people in the pedigree */
  struct person   **personNodes;          /* People in the pedigree           */
  struct marriage **marriageNodes;        /* Marriages in the pedigree        */
  int              nMarriageNodes;       /* Length of marriageNodes          */
  int             *marriageIndex;        /* Family weight index              */
  int             *oldStatus;            /* Initial genotype list status     */
  int             *status;               /* New genotype list status         */
  double          **Weights;              /* Informativness of each family    */
};

/* Represents a conglomeration of all the data required/generated for a       */
/* pedigree                                                                   */

static struct pedigreeData
{ int  *father;                    /* Father ID's                            */
  int  *mother;	            /* Mother ID's                            */
  int  *person;                    /* Person ID's                            */
  int  *sex;                       /* Sex status                             */
  int   n;                         /* Length of the above five vectors       */
  int   nLoci;                     /* Number of loci used for elimination    */
  int  *pheno;                     /* Phenotypes at the genotype loci        */
  int  *nAllele;                   /* Number of allele types at each loci    */
  int  *phenoCats;                 /* Phenotype categories at each locus     */
  int  *lengthPhenoCats;           /* Length of each phenotype category      */
  int   nLengthPhenoCats;          /* Length of lengthPhenoCats              */
  int  *genoMaps;                  /* Genotype lists for each phenotype      */
  int  *lengthGenoMaps;            /* Length of each genotype list           */
  int   nLengthGenoMaps;           /* Length of lengthGenoMaps               */
  int  *errLocus;                  /* Indicates Menelian Errors (ME)         */
  int  *errRelation;               /* Indicates how ME's occurred            */
  int  *errSpouse;                 /* Indicates spouse during ME's, if known */
  int  *typed;                     /* The person ID's to be jackknifed       */
  int  *lengthTyped;               /* Length of typed                        */
  int  *nLengthTyped;              /* Length lengthTyped                     */
  int **p2g;                       /* Phenotype-to-genotype ragged array     */
  int **unknownGenotypeList;       /* Unknown genotype ragged array          */
  int  *lengthUnknownGenotypeList; /* Unknown genotype ragged array          */
  void  (*sfunc)();                 /* Status function for the genotype lists */ 
  void  (*wfunc)();                 /* Weighting function for informativeness */
  struct pedigree *ped;             /* Points to current pedigree structure   */
};

static void sigsegv_error_handler(int code);
static struct pedigree *generate_pedigree(struct pedigreeData *pedData);
static struct pedigree *generate_pedigree_jackknife(struct 
                                                    pedigreeData 
                                                    *pedData);
static void remove_type(struct pedigreeData *pedDataDynamic,
                        int personID,
                        int locus);
static void restore_ped(struct pedigreeData *pedDataDynamic,
                        struct pedigreeData *pedDataStatic,
                        int locus);
static int genotype_elimination_engine(struct pedigreeData *pedData, 
                                        int locus);
static int preprocess(struct pedigreeData *pedData, int locus);
static void informativeness(struct pedigreeData *pedData, 
                            int locus, 
                            double *weights);
static void genotype_status(struct pedigreeData *pedData, 
                           int locus, 
                           int *status);
static void compact(struct marriage *family, int people, 
                    int action,             int locus);
static void traverse(struct pedigreeData *pedData, void (*func)());
static void traverse_engine(struct person *subject, 
                            struct pedigreeData *pedData,
                            void(*func)());
static int quicksort_ordinal(int n, double array_sort[], int array_permute[],
		             int temp[]);
static int quicksort_permute(int n, double array_sort[], int array_permute[]);
static int quicksort_lexicographic(int n, int arr1[], int arr2[]);
static void qs(int left, int right, double array_sort[], 
               int array_permute[]);
static void qs_lex(int left, int right, int arr1[], int arr2[]);

/******************************/
/***** DEBUGGING function *****/
/******************************/

static void print_geno_lists(struct pedigreeData *pedData)
{
 int i, j, k;
 struct person *currentPerson;

 for(i = 1; i <= pedData->n; i++)
   {
    currentPerson = pedData->ped->personNodes[i];
    printf("Person %d:\n", currentPerson->id);
    for(j = 0; j < pedData->nLoci; j++)
      { 
       printf("Phenotype at locus %d is %d\n", j, currentPerson->pheno[j]);
       printf("Corresponding genotype list is\n");
       for(k = 0; k < currentPerson->nGenoLists[j]; k++)
         {
          printf("%d %d\n", currentPerson->genoLists[2*j][k], 
		            currentPerson->genoLists[2*j+1][k]);
         }
       printf("\n");
     }
    printf("\n");
   }
 return;
}

/******************************/
/***** DEBUGGING function *****/
/******************************/

static void print_personID(struct pedigreeData *pedData, 
                           int personID, 
                           int locus,
                           char *label)
{
 int i;
 struct person *currentPerson = pedData->ped->personNodes[personID];
    
 printf("%s:\n",label);
 printf("Person %d at locus %d:\n", personID, locus);
 printf("Corresponding genotype list is\n");
 for(i = 0; i < currentPerson->nGenoLists[locus]; i++)
   {
    printf("%d %d %s %s\n", currentPerson->genoLists[2*locus][i], 
                            currentPerson->genoLists[2*locus+1][i],
    currentPerson->genoListsCon[2*locus][i] ? "CONSISTENT" : "INCONSISTENT", 
    currentPerson->genoListsCon[2*locus][i] ? "CONSISTENT" : "INCONSISTENT");
   }
 printf("\n");
      
 return;
}

/* sigsegv_error_handler

                                   FUNCTION ARGUMENTS                         

sigsegv_error_handler() takes one argument:

   -code:  An integer containing the signal code passed by the operating system.

                                   SIDE EFFECTS

sigsegv_error_handler() will print a message to stdout that a segmentation 
violation signal has been received and then make a call to longjmp() with
an error code of 1.

                                   RETURN VALUE

sigsegv_error_handler() returns no value.

*/

static void sigsegv_error_handler(int code)
{ 
  printf("SIGSEGV signal detected: aborting program\n");
  longjmp(buf,1);
  return;
}

/* GENOTYPE_ELIMINATION

                                   FUNCTION ARGUMENTS                         

genotype_elimination() takes the following arguments:

-person:       An array of int such that for each integer i between 1 and n,
               inclusive, we have personID[i] as the logical identifier for 
               person i in the pedigree.

-father:       An array of int integers such that fatherID[i] denotes the 
               person that is the father of the person given by personID[i].  
               An entry of zero denotes that the person identifed by personID[i]
               has no father in this pedigree.

-mother:       An array of int integers such that motherID[i] denotes the 
               person that is the mother of the person given by personID[i].  
               An entry of zero denotes that the person identifed by personID[i]
               has no mother in this pedigree.

-sex:          An array of int such that sex[i] denotes the sex of the person
               identified by personID[i].  A value of 1 denotes male, and a 
               value of zero denotes female.

-n:            A int integer pointer such that *n gives the length of each of
               the preceding n-by-1 arrays. 

-nLoci:        The number of loci that genotype elimination will be performed
               upon - one genotype elimination run(s) per locus.

-nAllele:     nAllelei] is the number of alleles possible at locus i, which
              is pheno[1i:ni] above.  For example, if nAllele[10] is three, then
              locus 10 has the 0.5*3*(3+1)=6 possible genotypes 11,12,13,22,23, 
              and 33.  This value is useful when a phenotype is unknown and all
              possible values must be assumed in the initial genotype list.

-pheno:        The list of phenotypes for each person at each locus.  The 
               format is as follows:  Let m=nLoci and n=length(person), then 
               pheno[11:n1] is the phenotypes of persons 1,2,...,n at locus 1,
               pheno[12:n2] is the phenotypes of persons 1,2,...,n at locus 2,
                                          ...,
                                          ...,
                                          ...,
              and pheno[1m:nm] is the phenotypes of persons 1,2,...,n at locus
              m where pheno is ordered as pheno[11:n1, 12:n2, ... 1m:nm] and 
              the order of the indices is 11,n1,12,n2,...,1m,nm,  Note that we 
              will be able to use zero for the case when the phenotype is 
              unknown. 

-phenoCats:   Phenotype categories at each locus with the phenotype values 
              ranging from 0,1,2,...,k, where k is the number of distinct
              observed phenotypes at a locus.  These category enumerations will 
              be present nLoci times.

-lengthPhenoCats:  Number of phenotypes in each category, including the
                   unknown phenotype, zero.

-nLengthPhenoCats: The length of the lengthPhenoCats vector.

-genoMaps:         A vector containing all of the genotype lists that map to
                   the above phenotype categories in phenoCats.

-lengthGenoMaps:   The length of each genotype map in genoMaps.

-nLengthGenoMaps:  The length of lengthGenoMaps.

-flags:            A int array of logical (zero or non-zero) flags that control
                   the behavior of genotype_elimination:
                   
                   flag[0] - If true, alleles are sortd into lexicographic 
                             order before they are returned.

                   flag[1] - If true, a verbose before and after view of the
                             pedigree genotypes are output to stdout.

-errLocus:         A int array of length nLoci.  At program termination, each 
                   array element will contain a zero if no Mendelian Errors 
                   occurred at the corresponding locus.  If a Mendelian Error
                   did occur, the person ID of the person that was being 
                   processed when the error was first detected will be placed
                   in errLocus at the position corresponding to the locus:

                   errLocus[0]            ->             Locus 1
                   errLocus[1]            ->             Locus 2                   
                                          ...
                                          ...
                                          ...
                   errLocus[nLoci-1]      ->             Locus nLoci

-errRelation:      A int array of length nLoci.  At program termination, if a
                   Mendelian Error is signified in errLocus above, say locus 
                   k+1, the corresponding value in errRelation[k] will tell 
                   whether the error was detected when the person ID in 
                   errLocus[k] was being considered as a parent or a child in 
                   a nuclear family.  If errRelation[k] is 1, the person ID in
                   errLocus[k] was a child in the nuclear family where the error
                   was first detected.   And if errRelation[k] is 2, the person 
                   ID in errLocus[k] was a parent in the nuclear where the 
                   aforementioned error was first seen.

-errSpouse:        A int array of length nLoci.  At program termination, if a
                   Mendelian Error is signified in errLocus above, say locus 
                   k+1, and the corresponding value in errRelation[k] was 2,
                   errSpouse[k] will give the person ID of the spouse of the
                   person specified in errLocus[k].  This is mainly to identify
                   the nuclear family in which a Mendelian Error was first 
                   detected when one of the parents has been married more than
                   once.

-errFatal:         A int pointer to an integer.  If genotype_elimination() 
                   receives a segmentation fault signal, errFatal will set to 
                   one upon program exit.

-returnIndex:     A int array of length n*nLoci that will store the number
                  of alleles in returnValue below that correspond to each 
                  locus of each person in the pedigree after genotype elmination
                  has been performed.  In other words, if this pedigree contains
                  nLoci loci, then the following correspondence is observed:

                  returnIndex[0:nLoci-1]              -> Person 1's loci
                  returnIndex[nLoci:2*nLoci-1]        -> Person 2's loci
                  returnIndex[2*nLoci:3*nLoci-1]      -> Person 3's loci
                                        ...
                                        ...
                                        ...
                  returnIndex[(n-1)*nLoci:n*nLoci-1]  -> Person n's loci

-returnValue:     A int array that will store the alleles for each locus of
                  each person in the pedigree after genotype elimination has
                  been performed.  Let the number of pedigree members with known
                  genotypes at locus i before genotype elimation be denoted by 
                  ST[i] (Strict Types), the number of pedigree members with one 
                  allele known at locus i before genotype elimation be denoted 
                  by HT[i] (Half Types), and the number of pedigree members with
                  unknown genotypes at locus i before genotype elimation be 
                  denoted by UT[i] (Unknown Types).  Then the required length 
                  of returnValue to guarantee ample space under all circumstances
                  is 
                  
                  length = 0;
                  for(i = 0; i < *nLoci; i++)
                  length += ST[i] * 2 + 
                            HT[i] * nAllele[i] + 
                            UT[i] * (nAllele[i] * (nAllele[i]+1) / 2);

                  It should be noted that returnValue can be made larger than 
                  the above without causing harm, but the extra space won't be 
                  used.

                                   SIDE EFFECTS

The arrays flags, errLocus, errRelation, errSpouse, returnIndex, and returnValue
will be populated with the values generated by the genotype elimaintion process.
All other function parameters will not be modifed.

                                   RETURN VALUE

genotype_elimination() returns no value.

                                   SYNOPSIS

genotype_elimination() generates the pedigree specified by its arguments, and 
then populates the initial genotype lists at each loci for each person in the 
pedigree.  Upon completion of this, genotype_elimination() then proceeds to 
traverse the pedigree and perform genotype elimination until no further 
genotypes may be eliminated.

*/
     
void genotype_elimination(int   *person,  int   *father,   int   *mother,
                          int   *sex,     int   *n,        int   *nLoci,
                          int   *nAllele, int   *pheno,    int   *phenoCats,
                          int   *lengthPhenoCats,
                          int   *nLengthPhenoCats,
                          int   *genoMaps,
                          int   *lengthGenoMaps,
                          int   *nLengthGenoMaps,
                          int   *flags,
                          int   *errLocus,
                          int   *errRelation,
                          int   *errSpouse,
                          int   *errFatal,
                          int   *returnIndex,
                          int   *returnValue) 
{
  int locus = 0, i = 0, j = 0, k = 0, m1 = 0, m2 = 0, size = 0;
  int nPersonNodes = 0, err = 0;
  struct pedigreeData temp;
  struct pedigreeData *pedData = &temp;
  struct person **personNodes;
  struct person  *currentPerson;

  /* Store input parameters into a single struct which can then be passed to  */
  /* all called functions - simplifies the calling conventions.               */

  pedData->father           =    father;
  pedData->mother           =    mother;
  pedData->person           =    person;
  pedData->sex              =    sex;
  pedData->n                =   *n;
  pedData->nLoci            =   *nLoci;
  pedData->nAllele          =    nAllele;
  pedData->pheno            =    pheno;
  pedData->phenoCats        =    phenoCats;
  pedData->lengthPhenoCats  =    lengthPhenoCats;
  pedData->nLengthPhenoCats =   *nLengthPhenoCats;
  pedData->genoMaps         =    genoMaps;
  pedData->lengthGenoMaps   =    lengthGenoMaps;
  pedData->nLengthGenoMaps  =   *nLengthGenoMaps;
  pedData->errLocus         =    errLocus;
  pedData->errRelation      =    errRelation;
  pedData->errSpouse        =    errSpouse;
  pedData->sfunc            =    genotype_status; 
  pedData->wfunc            =    informativeness;

  /* If we run out of memory or we receive a segmentation fault signal, we    */
  /* will crash "nicely."                                                     */

  if(err=setjmp(buf))
    {
     *errFatal = 1;
     return;
    }

  /* Register a signal handler to handle SIGSEGV signals from the operating   */
  /* system:                                                                  */
 
  signal(SIGSEGV,sigsegv_error_handler);

  /* Initialize the Mendelian Error detection vectors errLocus, errRelation,  */
  /* and errSpouse to zero to guard against a non-zero initialized arrays     */
  /* being passed accidentally to genotype_elimination:                       */

  for(i = 0; i < *nLoci; i++)
    {
     errLocus[i]    = 0;  
     errRelation[i] = 0;  
     errSpouse[i]   = 0;  
    }

  /*** Generate the pedigree: ***/
  generate_pedigree(pedData);

  if(flags[PRINT_FLAG])
    {
     printf("***** Initial genotype list: *****\n");
     print_geno_lists(pedData);
    }

  /*** Perform genotype elimination at all loci: ***/
  for(locus = 0; locus < *nLoci; locus++)
    {
     preprocess(pedData, locus);
     while(STATUS[locus] == NOT_DONE)
       genotype_elimination_engine(pedData,locus);
    }
 
  if(flags[PRINT_FLAG])
    {
     printf("***** Finished *****\n");
     print_geno_lists(pedData);
    }

  nPersonNodes = *n;
  personNodes  = pedData->ped->personNodes;

  /*** Sort the genotypes into lexicographic  ***/
  /*** order if requested:                    ***/

  if(flags[SORT_FLAG])
    {
     for(i = 1; i <= nPersonNodes; i++)
       {
	 currentPerson = personNodes[i];
         for(j = 0; j < *nLoci; j++)
 	    quicksort_lexicographic(currentPerson->nGenoLists[j],
                                    currentPerson->genoLists[2*j],
                                    currentPerson->genoLists[2*j+1]);
       }
    }

  /*** Copy the return values back to S-Plus: ***/
  nPersonNodes = *n;
  personNodes  = pedData->ped->personNodes;
  for(i = 1; i <= nPersonNodes; i++)
    {
     currentPerson = personNodes[i];
     for(j = 0; j < *nLoci; j++)
       {
        size =  currentPerson->nGenoLists[j];
        if(size == 0)
	  {
	    returnIndex[m1++] = 2;
       	    returnValue[m2++] = MENDELIAN_INCONSISTENCY;
	    returnValue[m2++] = MENDELIAN_INCONSISTENCY;
            continue;
	  }
	returnIndex[m1++] = 2*size;
        for(k = 0; k < size; k++)
	  {
     	   returnValue[m2++] =  currentPerson->genoLists[2*j][k]; 
	   returnValue[m2++] =  currentPerson->genoLists[2*j+1][k];
	  }
       }
    }

  mc_free_all();
  return;  
}

/* GENOTYPE_ELIMINATION_JACKKNIFE

                                   FUNCTION ARGUMENTS                         

genotype_elimination_jackknife() takes the following arguments:

-person:       An array of int such that for each integer i between 1 and n,
               inclusive, we have personID[i] as the logical identifier for 
               person i in the pedigree.

-father:       An array of int integers such that fatherID[i] denotes the 
               person that is the father of the person given by personID[i].  
               An entry of zero denotes that the person identifed by personID[i]
               has no father in this pedigree.

-mother:       An array of int integers such that motherID[i] denotes the 
               person that is the mother of the person given by personID[i].  
               An entry of zero denotes that the person identifed by personID[i]
               has no mother in this pedigree.

-sex:          An array of int such that sex[i] denotes the sex of the person
               identified by personID[i].  A value of 1 denotes male, and a 
               value of zero denotes female.

-n:            A int integer pointer such that *n gives the length of each of
               the preceding n-by-1 arrays. 

-nLoci:        The number of loci that genotype elimination will be performed
               upon - one genotype elimination run(s) per locus.

-nAllele:     nAllelei] is the number of alleles possible at locus i, which
              is pheno[1i:ni] above.  For example, if nAllele[10] is three, then
              locus 10 has the 0.5*3*(3+1)=6 possible genotypes 11,12,13,22,23, 
              and 33.  This value is useful when a phenotype is unknown and all
              possible values must be assumed in the initial genotype list.

-pheno:        The list of phenotypes for each person at each locus.  The 
               format is as follows:  Let m=nLoci and n=length(person), then 
               pheno[11:n1] is the phenotypes of persons 1,2,...,n at locus 1,
               pheno[12:n2] is the phenotypes of persons 1,2,...,n at locus 2,
                                          ...,
                                          ...,
                                          ...,
              and pheno[1m:nm] is the phenotypes of persons 1,2,...,n at locus
              m where pheno is ordered as pheno[11:n1, 12:n2, ... 1m:nm] and 
              the order of the indices is 11,n1,12,n2,...,1m,nm,  Note that we 
              will be able to use zero for the case when the phenotype is 
              unknown. 

-phenoCats:   Phenotype categories at each locus with the phenotype values 
              ranging from 0,1,2,...,k, where k is the number of distinct
              observed phenotypes at a locus.  These category enumerations will 
              be present nLoci times.

-lengthPhenoCats:  Number of phenotypes in each category, including the
                   unknown phenotype, zero.

-nLengthPhenoCats: The length of the lengthPhenoCats vector.

-genoMaps:         A vector containing all of the genotype lists that map to
                   the above phenotype categories in phenoCats.

-lengthGenoMaps:   The length of each genotype map in genoMaps.

-nLengthGenoMaps:  The length of lengthGenoMaps.

-flags:            A int array of logical (zero or non-zero) flags that control
                   the behavior of genotype_elimination_jackknife:
                   
                   flag[0] - If true, alleles are sortd into lexicographic 
                             order before they are returned.

                   flag[1] - If true, a verbose before and after view of the
                             pedigree genotypes are output to stdout.

-typed:           A int array of the person ID's of people that are fully typed
                  or partially typed in the pedigree.  These subjects are 
                  the candidates that will be jackknifed.

-lengthTyped:     A int array giving the number of jackknife candidates at
                  each locus that will be jackkknifed.

-nLengthTyped:    A int integer pointer giving the length of lengthTyped above.

-errFatal:         A int pointer to an integer.  If genotype_elimination() 
                   receives a segmentation fault signal, errFatal will set to 
                   one upon program exit.

-results:         A int array of identical length to the array typed defined
                  above.  Each entry in results will tell if jackknifing the 
                  corresponding entry in typed, with respect to the loci 
                  boundaries implied in lengthTyped, will remove the Mendelian
                  Errors that were previously present in the pedigree at the 
                  given locus.


                                   SIDE EFFECTS

The arrays flags, errLocus, errRelation, errSpouse, returnIndex, and returnValue
will be populated with the values generated by the genotype elimaintion process.
All other function parameters will not be modifed.

                                   RETURN VALUE

genotype_elimination_jackknife() returns no value.

                                   SYNOPSIS

genotype_elimination_jackknife() generates the pedigree specified by its 
arguments, and then populates the initial genotype lists at each loci for each 
person in the pedigree.  Upon completion of this, 
genotype_elimination_jackknife() then proceeds to traverse the pedigree and 
perform genotype elimination until no further genotypes may be eliminated.

*/
     
void genotype_elimination_jackknife(int   *person,  int   *father,   
                                    int   *mother,  int   *sex,     
                                    int   *n,       int   *nLoci,
                                    int   *nAllele, int   *pheno,    
                                    int   *phenoCats,
                                    int   *lengthPhenoCats,
                                    int   *nLengthPhenoCats,
                                    int   *genoMaps,
                                    int   *lengthGenoMaps,
                                    int   *nLengthGenoMaps,
                                    int   *flags,
                                    int   *typed,
                                    int   *lengthTyped,
                                    int   *nLengthTyped,
                                    int   *errFatal,
                                    int   *results) 
{
  int locus = 0, i = 0, j = 0, k = 0, m1 = 0, m2 = 0, size = 0;
  int nPersonNodes = 0, err = 0;
  int *errLocus, *errRelation, *errSpouse;
  struct pedigreeData temp, temp2;
  struct pedigreeData *pedDataStatic  = &temp;
  struct pedigreeData *pedDataDynamic = &temp2;
  struct person **personNodes;
  struct person  *currentPerson;

  /* The following will not be used, but they are required for preprocess()   */
  /* and genotype_elimination_engine() to complete successfully:              */

  errLocus    = (int *)Salloc(*nLoci,int);
  errRelation = (int *)Salloc(*nLoci,int);
  errSpouse   = (int *)Salloc(*nLoci,int);

  /* Store input parameters into a single struct which can then be passed to  */
  /* all called functions - simplifies the calling conventions.               */

  pedDataStatic->father            =    father;
  pedDataStatic->mother            =    mother;
  pedDataStatic->person            =    person;
  pedDataStatic->sex               =    sex;
  pedDataStatic->n                 =   *n;
  pedDataStatic->nLoci             =   *nLoci;
  pedDataStatic->nAllele           =    nAllele;
  pedDataStatic->pheno             =    pheno;
  pedDataStatic->phenoCats         =    phenoCats;
  pedDataStatic->lengthPhenoCats   =    lengthPhenoCats;
  pedDataStatic->nLengthPhenoCats  =   *nLengthPhenoCats;
  pedDataStatic->genoMaps          =    genoMaps;
  pedDataStatic->lengthGenoMaps    =    lengthGenoMaps;
  pedDataStatic->nLengthGenoMaps   =   *nLengthGenoMaps;
  pedDataStatic->errLocus          =    errLocus;
  pedDataStatic->errRelation       =    errRelation;
  pedDataStatic->errSpouse         =    errSpouse;
  pedDataStatic->sfunc             =    genotype_status; 
  pedDataStatic->wfunc             =    informativeness;

  pedDataDynamic->father           =    father;
  pedDataDynamic->mother           =    mother;
  pedDataDynamic->person           =    person;
  pedDataDynamic->sex              =    sex;
  pedDataDynamic->n                =   *n;
  pedDataDynamic->nLoci            =   *nLoci;
  pedDataDynamic->nAllele          =    nAllele;
  pedDataDynamic->pheno            =    pheno;
  pedDataDynamic->phenoCats        =    phenoCats;
  pedDataDynamic->lengthPhenoCats  =    lengthPhenoCats;
  pedDataDynamic->nLengthPhenoCats =   *nLengthPhenoCats;
  pedDataDynamic->genoMaps         =    genoMaps;
  pedDataDynamic->lengthGenoMaps   =    lengthGenoMaps;
  pedDataDynamic->nLengthGenoMaps  =   *nLengthGenoMaps;
  pedDataDynamic->errLocus         =    errLocus;
  pedDataDynamic->errRelation      =    errRelation;
  pedDataDynamic->errSpouse        =    errSpouse;
  pedDataDynamic->typed            =    typed;
  pedDataDynamic->lengthTyped      =    lengthTyped;
  pedDataDynamic->nLengthTyped     =    nLengthTyped;
  pedDataDynamic->sfunc            =    genotype_status; 
  pedDataDynamic->wfunc            =    informativeness;

  /* If we run out of memory or we receive a segmentation fault signal, we    */
  /* will crash "nicely."                                                     */

  if(err=setjmp(buf))
    {
     *errFatal = 1;
     return;
    }

  /* Register a signal handler to handle SIGSEGV signals from the operating   */
  /* system:                                                                  */
 
  signal(SIGSEGV,sigsegv_error_handler);

  /*** Generate the pedigrees: ***/
  generate_pedigree(pedDataStatic);
  generate_pedigree_jackknife(pedDataDynamic);

  if(flags[PRINT_FLAG])
    {
     printf("***** Initial genotype list: *****\n");
     print_geno_lists(pedDataDynamic);
    }       
        
  /*** Perform jackknifing and genotype elimination and at all loci: ***/
  j = 0;
  for(locus = 0; locus < *nLengthTyped; locus++)
    for(i = 0; i < lengthTyped[locus]; i++)
          {
           remove_type(pedDataDynamic,typed[j],locus);
           results[j] = preprocess(pedDataDynamic, locus);
           while(STATUS[locus] == NOT_DONE)
              results[j] = genotype_elimination_engine(pedDataDynamic,locus);
	   restore_ped(pedDataDynamic,pedDataStatic,locus);
           STATUS[locus] = NOT_DONE;
           j++;
          }

  if(flags[PRINT_FLAG])
    {
     printf("***** Finished *****\n");
     print_geno_lists(pedDataDynamic);
    }

  mc_free_all();
  return;  
}

/* GENERATE_PEDIGREE

                                   FUNCTION ARGUMENTS                         

generate_pedigree() takes a pointer to a struct of type pedigreeData called 
"pedData" and the following fields of pedData are used:

pedData->father:    An array of int integers such that fatherID[i] denotes 
                    the person that is the father of the person given by 
                    personID[i].  An entry of zero denotes that the person 
                    identifed by personID[i] has no father in this pedigree.

pedData->mother:    An array of int integers such that motherID[i] denotes 
                    the person that is the mother of the person given by 
                    personID[i].  An entry of zero denotes that the person 
                    identifed by personID[i] has no mother in this pedigree.

pedData->person:    An array of int such that for each integer i between 1 
                    and n, inclusive, we have personID[i] as the logical 
                    identifier for person i in the pedigree.

pedData->sex:       An array of int such that sex[i] denotes the sex of the 
                    person identified by personID[i].  A value of 1 denotes 
                    male, and a value of zero denotes female.

pedData->n:         A int integer pointer such that *n gives the length of 
                    each of the preceding five arrays. 

pedData->nLoci:     The number of loci that genotype elimination will be 
                    performed upon - one genotype elimination run(s) per locus.

pedData->pheno:     The list of phenotypes for each person at each locus.  The 
                    format is as follows:  Let m=nLoci and n=length(person), 
                    then pheno[11:n1] is the phenotypes of persons 1,2,...,n at
                    locus 1, pheno[12:n2] is the phenotypes of persons 1,2,...,n
                    at locus 2,
                                               ...,
                                               ...,
                                               ...,
                    and pheno[1m:nm] is the phenotypes of persons 1,2,...,n at 
                    locus m where pheno is ordered as pheno[11:n1, 12:n2, 
                    ..., 1m:nm] and the order of the indices is 11,n1,12,
                    n2,...,1m,nm,  Note that we  will be able to use zero for 
                    the case when the phenotype is unknown. 

pedData->nAllele:   nAllelei] is the number of alleles possible at locus i, 
                    which is pheno[1i:ni] above.  For example, if nAllele[10] is
                    three, then locus 10 has the 0.5*3*(3+1)=6 possible 
                    genotypes 11,12,13,22,23, and 33.  This value is useful when
                    a phenotype is unknown and all possible values must be 
                    assumed in the initial genotype list.

pedData->phenoCats: Phenotype categories at each locus with the phenotype values
                    ranging from 0,1,2,...,k, where k is the number of distinct
                    observed phenotypes at a locus.  These category enumerations
                    will be present nLoci times.

pedData->lengthPhenoCats:  Number of phenotypes in each category, including the
                           unknown phenotype, zero.

pedData->nLengthPhenoCats: The length of the lengthPhenoCats vector.

pedData->genoMaps:         A vector containing all of the genotype lists that 
                           map to the above phenotype categories in phenoCats.

pedData->lengthGenoMaps:   The length of each genotype map in genoMaps.

pedData->nLengthGenoMaps:  The length of lengthGenoMaps.

                                   SIDE EFFECTS

generate_pedigree() will generate a pedigree based upon the father, mother, 
person, and sex vectors above.  Furthermore, genotype lists for each person in
the pedigree will be initialized and populated in accordance with the data 
specifiied by nLoci, pheno, nAllele, phenoCats, lengthPhenoCats, 
nLengthPhenoCats, genoMaps, lengthGenoMaps, and nLengthGenoMaps.

                                   RETURN VALUE

generate_pedigree() returns a pointer to a struct of type pedigree.
*/

static struct pedigree *generate_pedigree(struct pedigreeData *pedData)
{
 int             *father            = pedData->father;
 int             *mother            = pedData->mother; 
 int             *person            = pedData->person;
 int             *sex               = pedData->sex;
 int              n                 = pedData->n;
 int              nLoci             = pedData->nLoci;
 int             *pheno             = pedData->pheno;
 int             *phenoCats         = pedData->phenoCats;
 int             *lengthPhenoCats   = pedData->lengthPhenoCats;
 int              nLengthPhenoCats  = pedData->nLengthPhenoCats;
 int             *genoMaps          = pedData->genoMaps;
 int             *lengthGenoMaps    = pedData->lengthGenoMaps;
 int              nLengthGenoMaps   = pedData->nLengthGenoMaps;
 int              nMarriageNodes = 1;
 int              phenoIndex;
 int              i, j, k, p, action, temp, *offset, *index, **p2g;
 int             *oldStatus, *status;
 double          **Weights, *phenoCatsDbl;
 struct person   **personNodes;
 struct marriage **marriageNodes;
 struct marriage  *marriageNode;
 struct children  *childNode;
 struct person    *currentPerson;
 struct person    *currentFather;
 struct person    *currentMother;
 struct marriage  *currentSpouse;
 struct marriage  *currentMarriage; 
 struct marriage  *currentFatherMarriage; 
 struct marriage  *currentMotherMarriage;
 struct children  *currentChild;
 struct pedigree  *Pedigree;

 /*** Set up person nodes and fill in basic information      ***/
 personNodes = (struct person **)Salloc(n+1, struct person *);
 personNodes[0] = NULL;
 for(i=1; i <= n; i++)
   {
    personNodes[i] = (struct person *)Salloc(1L, struct person);
    currentPerson                = personNodes[i];
    currentPerson->id            = person[i-1];
    currentPerson->sex           = sex[i-1];
    currentPerson->status        = DONE;
    currentPerson->parents       = NULL;
    currentPerson->nuclearFamily = NULL;
   }

/********************** Set up marriage nodes *********************************/

 marriageNodes           = (struct marriage **)Salloc(n+1, struct marriage *);
 marriageNodes[0]        = NULL;
 for(i=1; i<=n; i++)
   marriageNodes[i]      = NULL;

/********************** Set up the pedigree return value *********************/

 Pedigree                = (struct pedigree *)Salloc(1L, struct pedigree);
 Pedigree->n             = n;
 Pedigree->personNodes   = personNodes;
 Pedigree->marriageNodes = marriageNodes;

/********************* Set pedData->ped to the above pedigree *****************/

 pedData->ped = Pedigree;

/***** Set up the STATUS global array, in which STATUS[i] will give the   *****/
/***** status of genotype elimination at locus i                          *****/

 STATUS = (int *)Salloc(nLoci, int);
 for(i = 0; i < nLoci; i++)
   STATUS[i] = NOT_DONE;

/********************* Link person nodes into a pedigree **********************/

 for(i = 1; i <= n; i++)
   {
    currentFather = personNodes[father[i-1]];
    currentMother = personNodes[mother[i-1]];
    currentPerson = personNodes[i];

/* The variable "action" may take on the values 0,1,2,...,6.  For a given     */
/* person, these values are interpreted as follows:                           */
/*                                                                            */
/* 0: The current person has no parents in this pedigree.                     */
/* 1: This is the first marriage and child for both parents.                  */
/* 2: This is the first marriage for the father, the mother has been          */
/*    previously married, and this is the first child in this marriage.       */
/* 3: This is the first marriage for the mother, the father has been          */
/*    previously married, and this is the first child in this marriage.       */
/* 4: The father and mother are already married to each other, each parent's  */
/*    first marriage points to this marriage, and this is the kth child in    */
/*    this marriage, with k >= 2.                                             */
/*                                                                            */
/* For value 5, the father and mother have each been married before and at    */
/* least one of them, possibly both, has multiple spouses.                    */
/*                                                                            */
/* 5: At least one of the mother and father has multiple marriages.  This     */
/*    may be the first child in this marriage, if the marriage doesn't        */
/*    already exist, or the kth child in an existing marriage, k >= 2.        */



  if(   (currentFather != NULL) &&
        (currentMother != NULL)   )
           if(   (currentFather->nuclearFamily == NULL) &&
                 (currentMother->nuclearFamily == NULL)   )
            action = 1;
           else if(   (currentFather->nuclearFamily == NULL) &&
                      (currentMother->nuclearFamily != NULL)   ) 
                action = 2;
           else if(   (currentFather->nuclearFamily != NULL) &&
                      (currentMother->nuclearFamily == NULL)   ) 
                action = 3;
/**************  We now know that the following is true:         **************/
/**************  (   (currentFather->nuclearFamily != NULL) &&   **************/
/**************      (currentMother->nuclearFamily != NULL)   )  **************/
           else if(currentFather->nuclearFamily == currentMother->nuclearFamily)
	        action = 4;
/*************  We now know that at least one of the father and mother    *****/
/*************  has more than one marriage:                               *****/
           else 
	        action = 5;
  else
        action = 0;

 switch(action)
   {
    case 0:
      break;

    case 1:
      {
       marriageNode = (struct marriage *)Salloc(1L, struct marriage);
       marriageNode->husband = currentFather;
       marriageNode->wife = currentMother;
       marriageNode->nextHusbandMarriage = NULL;
       marriageNode->nextWifeMarriage = NULL;

       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       marriageNode->child = childNode;
             
       currentFather->nuclearFamily = marriageNode;
       currentMother->nuclearFamily = marriageNode;
       currentPerson->parents = marriageNode;
       marriageNodes[nMarriageNodes] = marriageNode;
       nMarriageNodes++;
      }  
       break;

    case 2: 
      {
       marriageNode = (struct marriage *)Salloc(1L, struct marriage);
       marriageNode->husband = currentFather;
       marriageNode->wife = currentMother;
       marriageNode->nextHusbandMarriage = NULL;
       marriageNode->nextWifeMarriage = NULL;
 
       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       marriageNode->child = childNode;
       
       currentFather->nuclearFamily = marriageNode;
       /*** Put this marriage at the end of the mother's marriage list ***/
       currentSpouse = currentMother->nuclearFamily;
       while(currentSpouse->nextWifeMarriage != NULL)
         { 
	   currentSpouse = currentSpouse->nextWifeMarriage; 
         }
       currentSpouse->nextWifeMarriage = marriageNode;
       currentPerson->parents = marriageNode;
       marriageNodes[nMarriageNodes] = marriageNode;
       nMarriageNodes++;
      }
       break;
 
    case 3:
      {
       marriageNode = (struct marriage *)Salloc(1L, struct marriage);
       marriageNode->husband = currentFather;
       marriageNode->wife = currentMother;
       marriageNode->nextHusbandMarriage = NULL;
       marriageNode->nextWifeMarriage = NULL;
 
       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       marriageNode->child = childNode;
         
       /*** Put this marriage at the end of the father's marriage list ***/
       currentSpouse = currentFather->nuclearFamily;
       while(currentSpouse->nextHusbandMarriage != NULL)
         { 
	   currentSpouse = currentSpouse->nextHusbandMarriage; 
         }
       currentSpouse->nextHusbandMarriage = marriageNode;
       currentMother->nuclearFamily = marriageNode;
       currentPerson->parents = marriageNode;
       marriageNodes[nMarriageNodes] = marriageNode;
       nMarriageNodes++;
      }
       break;

    case 4:
      {
       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       currentChild = currentFather->nuclearFamily->child;
       while(currentChild->nextChild != NULL)
         { 
          currentChild = currentChild->nextChild; 
         }
       currentChild->nextChild = childNode;   
       currentPerson->parents = currentFather->nuclearFamily;
      }
       break;

    case 5:
      { 
       currentMarriage = NULL;
       currentFatherMarriage = currentFather->nuclearFamily;
       currentMotherMarriage = currentMother->nuclearFamily;
       while(currentFatherMarriage != NULL && currentMarriage == NULL)
         {
          while(currentMotherMarriage != NULL && currentMarriage == NULL)
            {
	      if(currentFatherMarriage == currentMotherMarriage)
                currentMarriage = currentFatherMarriage;
              currentMotherMarriage = currentMotherMarriage->nextWifeMarriage;
            }
          currentFatherMarriage = currentFatherMarriage->nextHusbandMarriage;
         }

       if(currentMarriage != NULL)
         {
	  /*** Marriage already exists - add the child: ***/
          childNode = (struct children *)Salloc(1L, struct children);
          childNode->firstChild = currentPerson;
          childNode->nextChild = NULL;
          currentChild = currentMarriage->child;
          while(currentChild->nextChild != NULL)
            { 
             currentChild = currentChild->nextChild; 
            }
          currentChild->nextChild = childNode;   
          currentPerson->parents = currentMarriage;
         }
       else
         {
	  /*** Marriage doesn't exist yet- create it: ***/
          marriageNode = (struct marriage *)Salloc(1L, struct marriage);
          marriageNode->husband = currentFather;
          marriageNode->wife = currentMother;
          marriageNode->nextHusbandMarriage = NULL;
          marriageNode->nextWifeMarriage = NULL;

	  /*** Add the child: ***/
          childNode = (struct children *)Salloc(1L, struct children);
          childNode->firstChild = currentPerson;
          childNode->nextChild = NULL;
          marriageNode->child = childNode;

	  /*** Put this marriage at the end of the father's marriage list ***/
          currentFatherMarriage = currentFather->nuclearFamily;
          while(currentFatherMarriage->nextHusbandMarriage != NULL)
	    currentFatherMarriage = currentFatherMarriage->nextHusbandMarriage;
          currentFatherMarriage->nextHusbandMarriage = marriageNode;

	  /*** Put this marriage at the end of the mother's marriage list ***/
          currentMotherMarriage = currentMother->nuclearFamily;
          while(currentMotherMarriage->nextWifeMarriage != NULL)
	    currentMotherMarriage = currentMotherMarriage->nextWifeMarriage;   
          currentMotherMarriage->nextWifeMarriage = marriageNode;

          currentPerson->parents = marriageNode;
          marriageNodes[nMarriageNodes] = marriageNode;
          nMarriageNodes++;
         }
       break;
      }

   default:
     {}
   }
}

 /****  nMarriageNodes is one more than the actual number after the above  ****/
 /****  for loop, so we decrement it to the correct number:                ****/

  nMarriageNodes--;

  /**** Set up the double array containing the informativeness weights for ****/
  /**** each family at each locus with weights[locus][family] storing the  ****/
  /**** appropriate value:                                                 ****/

  Weights = (double **)Salloc(nLoci, double *);
  for(i = 0; i < nLoci; i++)
    Weights[i] = (double *)Salloc(nMarriageNodes, double);
  
  pedData->ped->Weights = Weights;

  /**** Set up the status and oldStatus vectors used for determining if    ****/
  /**** genotype elimination has occurred on the last run of genotype      ****/
  /**** elimination:                                                       ****/

  oldStatus               = (int *)Salloc(nLoci, int);
  pedData->ped->oldStatus = oldStatus;  

  status                  = (int *)Salloc(nLoci, int);
  pedData->ped->status    = status;  

  /**** Record the number of nuclear families in this pedigree and         ****/
  /**** allocate an array of ints called marriageIndex such that          ****/
  /**** marriageIndex[i] will give the ith most informative family.        ****/
  /**** Note: marriageIndex will be populated later in the function        ****/
  /**** genotype_elimination().                                            ****/

  pedData->ped->nMarriageNodes = nMarriageNodes;
  pedData->ped->marriageIndex  = (int *)Salloc(nMarriageNodes, int);

  /**** Construct p2g, which is the phenotype-to-genotype mapping *************/

  p = 0; /* p is the total number of phenotype categories */
  for(i = 0; i < nLengthPhenoCats; i++)
     p += lengthPhenoCats[i];

  p2g = (int **)Salloc(p, int *);

  k = 0;
  for(i = 0; i < p; i++)
    {
     p2g[i] = (int *)Salloc(lengthGenoMaps[i], int);
     for(j = 0; j < lengthGenoMaps[i]; j+=2)
       {
        p2g[i][j]   = genoMaps[k];	k++;
        p2g[i][j+1] = genoMaps[k];	k++;
        if(p2g[i][j] > p2g[i][j+1])
          {
           temp = p2g[i][j];
           p2g[i][j] = p2g[i][j+1];
           p2g[i][j+1] = temp;
          }
       }
    }

  pedData->p2g = p2g;

  /*** Populate the phenotype vectors for each person ************************/
  
  for(i = 1; i <= n; i++)
    {
     currentPerson = personNodes[i];
     currentPerson->pheno = (int *)Salloc(nLoci, int);
     for(j = 0; j < nLoci; j++)
        currentPerson->pheno[j] = pheno[j*n + i - 1];
    }

  /*** Populate the genotype lists for each person ***************************/

  offset = (int *)Salloc(nLengthPhenoCats, int);
  offset[0] = 0L;
  for(i = 1; i < nLoci; i++)
    offset[i] = offset[i-1] + lengthPhenoCats[i-1];

  /*** Get the index of each phenotype category in phenoCats: *****************/
  index = (int *)Salloc(p, int);
  phenoCatsDbl = (double *)Salloc(p, double);

  for(i = 0; i < p; i++)
      phenoCatsDbl[i] = (double)phenoCats[i];

  for(i = 0; i < nLengthPhenoCats; i++)
    {
      quicksort_permute(lengthPhenoCats[i],
                        phenoCatsDbl + offset[i],
			index + offset[i]);
    }

  /*** index is one-based, but C uses zero-based indexing, so we modify index */
  /*** to also be zero-based.                                                 */

  for(i = 0; i < p; i++)
    index[i] -= 1;
  
  for(i = 1; i <= n; i++)
    {
     currentPerson = personNodes[i];
     currentPerson->genoLists    = (int **)Salloc(2*nLoci, int *);
     currentPerson->nGenoLists   = (int *)Salloc(nLoci, int);
     currentPerson->genoListsCon = (int **)Salloc(2*nLoci, int *);
     for(j = 0; j < nLoci; j++)
       {
        phenoIndex = offset[j] + index[offset[j] + currentPerson->pheno[j]];
        currentPerson->nGenoLists[j] = 
                       lengthGenoMaps[phenoIndex]/2;
        /*** Allocate memory: ***/
        currentPerson->genoLists[2*j] = 
                       (int *)Salloc(currentPerson->nGenoLists[j], int);
        currentPerson->genoLists[2*j+1] = 
                       (int *)Salloc(currentPerson->nGenoLists[j], int);
        currentPerson->genoListsCon[2*j] = 
                       (int *)Salloc(currentPerson->nGenoLists[j], int);
        currentPerson->genoListsCon[2*j+1] = 
                       (int *)Salloc(currentPerson->nGenoLists[j], int);

        for(k = 0; k < currentPerson->nGenoLists[j]; k++)
          {
           /*** Store the genotype list: ***/
           currentPerson->genoLists[2*j][k] = 
                          p2g[phenoIndex][2*k];
           currentPerson->genoLists[2*j+1][k] = 
                          p2g[phenoIndex][2*k+1];

           /*** Set initial status to INCONSISTENT: ***/
           currentPerson->genoListsCon[2*j][k] = INCONSISTENT;
           currentPerson->genoListsCon[2*j+1][k] = INCONSISTENT;
          }
      }
    }  

  /*** Populate the initial genotype status at each locus: ********************/
  for(i = 0; i < nLoci; i++)
    genotype_status(pedData, i, oldStatus);

 return Pedigree;
}

/* GENERATE_PEDIGREE_JACKKNIFE

                                   FUNCTION ARGUMENTS                         

generate_pedigree_jackknife() takes a pointer to a struct of type pedigreeData 
called "pedData" and the following fields of pedData are used:

pedData->father:    An array of int integers such that fatherID[i] denotes 
                    the person that is the father of the person given by 
                    personID[i].  An entry of zero denotes that the person 
                    identifed by personID[i] has no father in this pedigree.

pedData->mother:    An array of int integers such that motherID[i] denotes 
                    the person that is the mother of the person given by 
                    personID[i].  An entry of zero denotes that the person 
                    identifed by personID[i] has no mother in this pedigree.

pedData->person:    An array of int such that for each integer i between 1 
                    and n, inclusive, we have personID[i] as the logical 
                    identifier for person i in the pedigree.

pedData->sex:       An array of int such that sex[i] denotes the sex of the 
                    person identified by personID[i].  A value of 1 denotes 
                    male, and a value of zero denotes female.

pedData->n:         A int integer pointer such that *n gives the length of 
                    each of the preceding five arrays. 

pedData->nLoci:     The number of loci that genotype elimination will be 
                    performed upon - one genotype elimination run(s) per locus.

pedData->pheno:     The list of phenotypes for each person at each locus.  The 
                    format is as follows:  Let m=nLoci and n=length(person), 
                    then pheno[11:n1] is the phenotypes of persons 1,2,...,n at
                    locus 1, pheno[12:n2] is the phenotypes of persons 1,2,...,n
                    at locus 2,
                                               ...,
                                               ...,
                                               ...,
                    and pheno[1m:nm] is the phenotypes of persons 1,2,...,n at 
                    locus m where pheno is ordered as pheno[11:n1, 12:n2, 
                    ..., 1m:nm] and the order of the indices is 11,n1,12,
                    n2,...,1m,nm,  Note that we  will be able to use zero for 
                    the case when the phenotype is unknown. 

pedData->nAllele:   nAllelei] is the number of alleles possible at locus i, 
                    which is pheno[1i:ni] above.  For example, if nAllele[10] is
                    three, then locus 10 has the 0.5*3*(3+1)=6 possible 
                    genotypes 11,12,13,22,23, and 33.  This value is useful when
                    a phenotype is unknown and all possible values must be 
                    assumed in the initial genotype list.

pedData->phenoCats: Phenotype categories at each locus with the phenotype values
                    ranging from 0,1,2,...,k, where k is the number of distinct
                    observed phenotypes at a locus.  These category enumerations
                    will be present nLoci times.

pedData->lengthPhenoCats:  Number of phenotypes in each category, including the
                           unknown phenotype, zero.

pedData->nLengthPhenoCats: The length of the lengthPhenoCats vector.

pedData->genoMaps:         A vector containing all of the genotype lists that 
                           map to the above phenotype categories in phenoCats.

pedData->lengthGenoMaps:   The length of each genotype map in genoMaps.

pedData->nLengthGenoMaps:  The length of lengthGenoMaps.

                                   SIDE EFFECTS

generate_pedigree_jackknife() will generate a pedigree based upon the father, 
mother, person, and sex vectors above.  Furthermore, genotype lists for each 
person in the pedigree will be initialized and populated in accordance with 
the data specifiied by nLoci, pheno, nAllele, phenoCats, lengthPhenoCats, 
nLengthPhenoCats, genoMaps, lengthGenoMaps, and nLengthGenoMaps.

                                   RETURN VALUE

generate_pedigree_jackknife() returns a pointer to a struct of type pedigree.
*/

static struct pedigree *generate_pedigree_jackknife(struct 
                                                    pedigreeData 
                                                    *pedData)
{
 int             *father            = pedData->father;
 int             *mother            = pedData->mother; 
 int             *person            = pedData->person;
 int             *sex               = pedData->sex;
 int              n                 = pedData->n;
 int              nLoci             = pedData->nLoci;
 int             *nAllele           = pedData->nAllele;
 int             *pheno             = pedData->pheno;
 int             *phenoCats         = pedData->phenoCats;
 int             *lengthPhenoCats   = pedData->lengthPhenoCats;
 int              nLengthPhenoCats  = pedData->nLengthPhenoCats;
 int             *genoMaps          = pedData->genoMaps;
 int             *lengthGenoMaps    = pedData->lengthGenoMaps;
 int              nLengthGenoMaps   = pedData->nLengthGenoMaps;
 int             *typed             = pedData->typed;
 int             *lengthTyped       = pedData->lengthTyped;
 int             *nLengthTyped      = pedData->nLengthTyped;
 int              nMarriageNodes = 1;
 int              phenoIndex, nAlleles, num, iter;
 int              i, j, k, p, action, temp, *offset, *index, **p2g;
 int             *oldStatus, *status;
 int             *lengthUnknownGenotypeList;
 int            **unknownGenotypeList;
 double          **Weights, *phenoCatsDbl;
 struct person   **personNodes;
 struct marriage **marriageNodes;
 struct marriage  *marriageNode;
 struct children  *childNode;
 struct person    *currentPerson;
 struct person    *currentFather;
 struct person    *currentMother;
 struct marriage  *currentSpouse;
 struct marriage  *currentMarriage;
 struct marriage  *currentFatherMarriage; 
 struct marriage  *currentMotherMarriage;
 struct children  *currentChild;
 struct pedigree  *Pedigree;

/* Set up person nodes and fill in basic information      */
 personNodes = (struct person **)Salloc(n+1, struct person *);
 personNodes[0] = NULL;
 for(i=1; i <= n; i++)
   {
    personNodes[i] = (struct person *)Salloc(1L, struct person);
    currentPerson                = personNodes[i];
    currentPerson->id            = person[i-1];
    currentPerson->sex           = sex[i-1];
    currentPerson->status        = DONE;
    currentPerson->parents       = NULL;
    currentPerson->nuclearFamily = NULL;
   }

/***********************Set up marriage nodes**********************************/

 marriageNodes           = (struct marriage **)Salloc(n+1, struct marriage *);
 marriageNodes[0]        = NULL;
 for(i=1; i<=n; i++)
   marriageNodes[i]      = NULL;

/***********************Set up the pedigree return value**********************/

 Pedigree                = (struct pedigree *)Salloc(1L, struct pedigree);
 Pedigree->n             = n;
 Pedigree->personNodes   = personNodes;
 Pedigree->marriageNodes = marriageNodes;

/**********************Set pedData->ped to the above pedigree******************/

 pedData->ped = Pedigree;

/***** Set up the STATUS global array, in which STATUS[i] will give the   *****/
/***** status of genotype elimination at locus i                          *****/

 STATUS = (int *)Salloc(nLoci, int);
 for(i = 0; i < nLoci; i++)
   STATUS[i] = NOT_DONE;

/**********************Link person nodes into a pedigree***********************/

 for(i = 1; i <= n; i++)
   {
    currentFather = personNodes[father[i-1]];
    currentMother = personNodes[mother[i-1]];
    currentPerson = personNodes[i];

/* The variable "action" may take on the values 0,1,2,...,6.  For a given     */
/* person, these values are interpreted as follows:                           */
/*                                                                            */
/* 0: The current person has no parents in this pedigree.                     */
/* 1: This is the first marriage and child for both parents.                  */
/* 2: This is the first marriage for the father, the mother has been          */
/*    previously married, and this is the first child in this marriage.       */
/* 3: This is the first marriage for the mother, the father has been          */
/*    previously married, and this is the first child in this marriage.       */
/* 4: The father and mother have each been married only once (to each other), */
/*    and this is the kth child in their marriage, with k >= 2                */
/*                                                                            */
/* For value 5, the father and mother have each been married before and at    */
/* least one of them, possibly both, has multiple spouses.                    */
/*                                                                            */
/* 5: At least one of the mother and father has multiple marriages.  This     */
/*    may be the first child in this marriage, if the marriage doesn't        */
/*    already exist, or the kth child in an existing marriage, k >= 2.        */


  if(   (currentFather != NULL) &&
        (currentMother != NULL)   )
           if(   (currentFather->nuclearFamily == NULL) &&
                 (currentMother->nuclearFamily == NULL)   )
            action = 1;
           else if(   (currentFather->nuclearFamily == NULL) &&
                      (currentMother->nuclearFamily != NULL)   ) 
                action = 2;
           else if(   (currentFather->nuclearFamily != NULL) &&
                      (currentMother->nuclearFamily == NULL)   ) 
                action = 3;
/**************  We now know that the following is true:         **************/
/**************  (   (currentFather->nuclearFamily != NULL) &&   **************/
/**************      (currentMother->nuclearFamily != NULL)   )  **************/
           else if(currentFather->nuclearFamily == currentMother->nuclearFamily)
	        action = 4;
/*************  We now know that at least one of the father and mother    *****/
/*************  has more than one marriage:                               *****/
           else 
	        action = 5;
  else
        action = 0;

 switch(action)
   {
    case 0:
      break;

    case 1:
      {
       marriageNode = (struct marriage *)Salloc(1L, struct marriage);
       marriageNode->husband = currentFather;
       marriageNode->wife = currentMother;
       marriageNode->nextHusbandMarriage = NULL;
       marriageNode->nextWifeMarriage = NULL;

       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       marriageNode->child = childNode;
             
       currentFather->nuclearFamily = marriageNode;
       currentMother->nuclearFamily = marriageNode;
       currentPerson->parents = marriageNode;
       marriageNodes[nMarriageNodes] = marriageNode;
       nMarriageNodes++;
      }  
       break;

    case 2: 
      {
       marriageNode = (struct marriage *)Salloc(1L, struct marriage);
       marriageNode->husband = currentFather;
       marriageNode->wife = currentMother;
       marriageNode->nextHusbandMarriage = NULL;
       marriageNode->nextWifeMarriage = NULL;
 
       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       marriageNode->child = childNode;
       
       currentFather->nuclearFamily = marriageNode;
       /*** Put this marriage at the end of the mother's marriage list ***/
       currentSpouse = currentMother->nuclearFamily;
       while(currentSpouse->nextWifeMarriage != NULL)
         { 
	   currentSpouse = currentSpouse->nextWifeMarriage; 
         }
       currentSpouse->nextWifeMarriage = marriageNode;
       currentPerson->parents = marriageNode;
       marriageNodes[nMarriageNodes] = marriageNode;
       nMarriageNodes++;
      }
       break;
 
    case 3:
      {
       marriageNode = (struct marriage *)Salloc(1L, struct marriage);
       marriageNode->husband = currentFather;
       marriageNode->wife = currentMother;
       marriageNode->nextHusbandMarriage = NULL;
       marriageNode->nextWifeMarriage = NULL;
 
       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       marriageNode->child = childNode;
         
       /*** Put this marriage at the end of the father's marriage list ***/
       currentSpouse = currentFather->nuclearFamily;
       while(currentSpouse->nextHusbandMarriage != NULL)
         { 
	   currentSpouse = currentSpouse->nextHusbandMarriage; 
         }
       currentSpouse->nextHusbandMarriage = marriageNode;
       currentMother->nuclearFamily = marriageNode;
       currentPerson->parents = marriageNode;
       marriageNodes[nMarriageNodes] = marriageNode;
       nMarriageNodes++;
      }
       break;

    case 4:
      {
       childNode = (struct children *)Salloc(1L, struct children);
       childNode->firstChild = currentPerson;
       childNode->nextChild = NULL;
       currentChild = currentFather->nuclearFamily->child;
       while(currentChild->nextChild != NULL)
         { 
          currentChild = currentChild->nextChild; 
         }
       currentChild->nextChild = childNode;   
       currentPerson->parents = currentFather->nuclearFamily;
      }
       break;

    case 5:
      { 
       currentMarriage = NULL;
       currentFatherMarriage = currentFather->nuclearFamily;
       currentMotherMarriage = currentMother->nuclearFamily;
       while(currentFatherMarriage != NULL && currentMarriage == NULL)
         {
          while(currentMotherMarriage != NULL && currentMarriage == NULL)
            {
	      if(currentFatherMarriage == currentMotherMarriage)
                currentMarriage = currentFatherMarriage;
              currentMotherMarriage = currentMotherMarriage->nextWifeMarriage;
            }
          currentFatherMarriage = currentFatherMarriage->nextHusbandMarriage;
         }

       if(currentMarriage != NULL)
         {
	  /*** Marriage already exists - add the child: ***/
          childNode = (struct children *)Salloc(1L, struct children);
          childNode->firstChild = currentPerson;
          childNode->nextChild = NULL;
          currentChild = currentMarriage->child;
          while(currentChild->nextChild != NULL)
            { 
             currentChild = currentChild->nextChild; 
            }
          currentChild->nextChild = childNode;   
          currentPerson->parents = currentMarriage;
         }
       else
         {
	  /*** Marriage doesn't exist yet- create it: ***/
          marriageNode = (struct marriage *)Salloc(1L, struct marriage);
          marriageNode->husband = currentFather;
          marriageNode->wife = currentMother;
          marriageNode->nextHusbandMarriage = NULL;
          marriageNode->nextWifeMarriage = NULL;

	  /*** Add the child: ***/
          childNode = (struct children *)Salloc(1L, struct children);
          childNode->firstChild = currentPerson;
          childNode->nextChild = NULL;
          marriageNode->child = childNode;
          
	  /*** Put this marriage at the end of the father's marriage list ***/
          currentFatherMarriage = currentFather->nuclearFamily;
          while(currentFatherMarriage->nextHusbandMarriage != NULL)
	    currentFatherMarriage = currentFatherMarriage->nextHusbandMarriage;
          currentFatherMarriage->nextHusbandMarriage = marriageNode;

	  /*** Put this marriage at the end of the mother's marriage list ***/
          currentMotherMarriage = currentMother->nuclearFamily;
          while(currentMotherMarriage->nextWifeMarriage != NULL)
	    currentMotherMarriage = currentMotherMarriage->nextWifeMarriage;   
          currentMotherMarriage->nextWifeMarriage = marriageNode;

          currentPerson->parents = marriageNode;
          marriageNodes[nMarriageNodes] = marriageNode;
          nMarriageNodes++;
         }
       break;
      }
   default:
     {}
   }
}

  /**** nMarriageNodes is one more than the actual number after the above  ****/
  /**** for loop, so we decrement it to the correct number:                ****/

  nMarriageNodes--;

  /**** Set up the double array containing the informativeness weights for ****/
  /**** each family at each locus with weights[locus][family] storing the  ****/
  /**** appropriate value:                                                 ****/

  Weights = (double **)Salloc(nLoci, double *);
  for(i = 0; i < nLoci; i++)
    Weights[i] = (double *)Salloc(nMarriageNodes, double);
  
  pedData->ped->Weights = Weights;

  /**** Set up the status and oldStatus vectors used for determining if    ****/
  /**** genotype elimination has occurred on the last run of genotype      ****/
  /**** elimination:                                                       ****/

  oldStatus               = (int *)Salloc(nLoci, int);
  pedData->ped->oldStatus = oldStatus;  

  status                  = (int *)Salloc(nLoci, int);
  pedData->ped->status    = status;  

  /**** Record the number of nuclear families in this pedigree and         ****/
  /**** allocate an array of ints called marriageIndex such that          ****/
  /**** marriageIndex[i] will give the ith most informative family.        ****/
  /**** Note: marriageIndex will be populated later in the function        ****/
  /**** genotype_elimination().                                            ****/

  pedData->ped->nMarriageNodes = nMarriageNodes;
  pedData->ped->marriageIndex  = (int *)Salloc(nMarriageNodes, int);

  /**** Construct p2g, which is the phenotype-to-genotype mapping *************/

  p = 0; /**** p is the total number of phenotype categories ****/
  for(i = 0; i < nLengthPhenoCats; i++)
     p += lengthPhenoCats[i];

  p2g = (int **)Salloc(p, int *);

  k = 0;
  for(i = 0; i < p; i++)
    {
     p2g[i] = (int *)Salloc(lengthGenoMaps[i], int);
     for(j = 0; j < lengthGenoMaps[i]; j+=2)
       {
        p2g[i][j]   = genoMaps[k];	k++;
        p2g[i][j+1] = genoMaps[k];	k++;
        if(p2g[i][j] > p2g[i][j+1])
          {
           temp = p2g[i][j];
           p2g[i][j] = p2g[i][j+1];
           p2g[i][j+1] = temp;
          }
       }
    }

  pedData->p2g = p2g;

  /**** Find the number of alleles in an unknown phenotype at each locus: ****/

  unknownGenotypeList               = (int **)Salloc(nLoci,int*);
  lengthUnknownGenotypeList         = (int *)Salloc(nLoci,int);
  for(i = 0; i < nLoci; i++)
    {
     nAlleles                        = nAllele[i];
     num                             = nAlleles*(nAlleles+1);
     unknownGenotypeList[i] = (int *)Salloc(num,int);
     lengthUnknownGenotypeList[i]    = num;

     iter = 0;
     for(j = 1; j <= nAlleles; j++)
       for(k = j; k <= nAlleles; k++)
	 {
          unknownGenotypeList[i][iter++] = j;
          unknownGenotypeList[i][iter++] = k;
	 }
    }
  
  pedData->unknownGenotypeList        = unknownGenotypeList;
  pedData->lengthUnknownGenotypeList  = lengthUnknownGenotypeList;

  /**** Populate the phenotype vectors for each person ***********************/
  
  for(i = 1; i <= n; i++)
    {
     currentPerson = personNodes[i];
     currentPerson->pheno = (int *)Salloc(nLoci, int);
     for(j = 0; j < nLoci; j++)
        currentPerson->pheno[j] = pheno[j*n + i - 1];
    }

  /**** Populate the genotype lists for each person **************************/

  offset = (int *)Salloc(nLengthPhenoCats, int);
  offset[0] = 0L;
  for(i = 1; i < nLoci; i++)
    offset[i] = offset[i-1] + lengthPhenoCats[i-1];

  /**** Get the index of each phenotype category in phenoCats: ****************/
  index = (int *)Salloc(p, int);
  phenoCatsDbl = (double *)Salloc(p, double);

  for(i = 0; i < p; i++)
      phenoCatsDbl[i] = (double)phenoCats[i];

  for(i = 0; i < nLengthPhenoCats; i++)
    {
      quicksort_permute(lengthPhenoCats[i],
                        phenoCatsDbl + offset[i],
			index + offset[i]);
    }

  /**** index is one-based, but C uses zero-based indexing, so we modify ****/
  /**** index to also be zero-based.                                     ****/

  for(i = 0; i < p; i++)
    index[i] -= 1;
  
  for(i = 1; i <= n; i++)
    {
     currentPerson = personNodes[i];
     currentPerson->genoLists    = (int **)Salloc(2*nLoci, int *);
     currentPerson->nGenoLists   = (int *)Salloc(nLoci, int);
     currentPerson->genoListsCon = (int **)Salloc(2*nLoci, int *);
     for(j = 0; j < nLoci; j++)
       {
        phenoIndex = offset[j] + index[offset[j] + currentPerson->pheno[j]];
        currentPerson->nGenoLists[j] = 
                       lengthGenoMaps[phenoIndex]/2;
        /*** Allocate memory: ***/
        currentPerson->genoLists[2*j] = 
                       (int *)Salloc(lengthUnknownGenotypeList[j], int);
        currentPerson->genoLists[2*j+1] = 
                       (int *)Salloc(lengthUnknownGenotypeList[j], int);
        currentPerson->genoListsCon[2*j] = 
                       (int *)Salloc(lengthUnknownGenotypeList[j], int);
        currentPerson->genoListsCon[2*j+1] = 
                       (int *)Salloc(lengthUnknownGenotypeList[j], int);

        for(k = 0; k < currentPerson->nGenoLists[j]; k++)
          {
           /*** Store the genotype list: ***/
           currentPerson->genoLists[2*j][k] = 
                          p2g[phenoIndex][2*k];
           currentPerson->genoLists[2*j+1][k] = 
                          p2g[phenoIndex][2*k+1];

           /*** Set initial status to INCONSISTENT: ***/
           currentPerson->genoListsCon[2*j][k] = INCONSISTENT;
           currentPerson->genoListsCon[2*j+1][k] = INCONSISTENT;
          }
      }
    }  

  /*** Populate the initial genotype status at each locus: ********************/
  for(i = 0; i < nLoci; i++)
    genotype_status(pedData, i, oldStatus);

 return Pedigree;
}

/* REMOVE_TYPE

                                   FUNCTION ARGUMENTS                         

remove_type() takes the following arguments:

   -pedDataDynamic:  A struct pedigreeData pointer to the pedigree structure 
                     being processed.

   -personID:        A int integer denoting the person that will be updated.

   -locus:           A int integer denoting the locus that will be removed.

                                   SIDE EFFECTS

The person given by "personID" will have his genotype list and genotype 
consistency list updated at the locus given by "locus" to be unknown.

                                   RETURN VALUE

remove_type() returns no value.

*/

static void remove_type(struct pedigreeData *pedDataDynamic,
                        int personID,
                        int locus)
{
 int i, j;
 struct person *currentPerson   
               = pedDataDynamic->ped->personNodes[personID];
 int *unknownGenotypeList      
               = pedDataDynamic->unknownGenotypeList[locus];
 int lengthUnknownGenotypeList 
               = pedDataDynamic->lengthUnknownGenotypeList[locus];
 
 j = 0;
 for(i = 0; i < lengthUnknownGenotypeList;)
   {
    /**** Store the genotype list: ****/
    currentPerson->genoLists[2*locus][j]     = unknownGenotypeList[i++];
    currentPerson->genoLists[2*locus+1][j]   = unknownGenotypeList[i++];

    /**** Set initial status to INCONSISTENT: ****/
    currentPerson->genoListsCon[2*locus][j]   = INCONSISTENT;
    currentPerson->genoListsCon[2*locus+1][j] = INCONSISTENT;
    j++;
   }

 currentPerson->nGenoLists[locus] = j;
 return;
}         

/* RESTORE_PED

                                   FUNCTION ARGUMENTS                         

restore_ped() takes the following arguments:

   -pedDataDynamic:  A struct pedigreeData pointer to the pedigree structure 
                     being restored.

   -pedDataStatic:   A struct pedigreeData pointer to the pedigree structure 
                     being copied from.

   -personID:        A int integer denoting the person that will be updated.

   -locus:           A int integer denoting the locus that will be removed.

                                   SIDE EFFECTS

The person in pedDataDynamic given by "personID" will have his genotype list 
and genotype consistency list updated at the locus given by "locus" to his
original genotype list that is stored in pedDataStatic.

                                   RETURN VALUE

restore_ped() returns no value.

*/

static void restore_ped(struct pedigreeData *pedDataDynamic,
                        struct pedigreeData *pedDataStatic,
                        int locus)
{
 int i, j, nGenoLists;
 int n = pedDataDynamic->n;
 struct person **currentPersonNodes  = pedDataDynamic->ped->personNodes; 
 struct person **originalPersonNodes = pedDataStatic->ped->personNodes;
 struct person *currentPerson, *originalPerson; 

 for(i = 1; i <= n; i++)
   {
    currentPerson  = currentPersonNodes[i];
    originalPerson = originalPersonNodes[i];
    nGenoLists     = originalPerson->nGenoLists[locus];
    for(j = 0; j < nGenoLists; j++)
      {
       /**** Store the genotype list: ****/
       currentPerson->genoLists[2*locus][j]     
                   =  originalPerson->genoLists[2*locus][j]; 
       currentPerson->genoLists[2*locus+1][j] 
                   =  originalPerson->genoLists[2*locus+1][j];

       /**** Set initial status to INCONSISTENT: ****/
       currentPerson->genoListsCon[2*locus][j]   = INCONSISTENT;
       currentPerson->genoListsCon[2*locus+1][j] = INCONSISTENT;
      } /*** end for(j) *****/
    currentPerson->nGenoLists[locus] = nGenoLists;
   }  /***** end for(i) *****/


 return;
}         

/* INFORMATIVENESS

                                   FUNCTION ARGUMENTS                         

informativeness() takes the following arguments:

   -pedData:  A struct pedigreeData pointer to the pedigree structure being 
              processed.

   -locus:    A int integer denoting the locus upon which the nuclear families
              will be ranked.

   -weights:  A double array of length nMarriageNodes that will hold the weight
              given to each nuclear family.

                                   SIDE EFFECTS

The argument array weights will be updated so that weights[i] gives the 
informativeness rank of nuclear family i+1, with smaller values corresponding
to more informative families, and larger weights corresponding to less 
informative familes.

                                   RETURN VALUE

informativeness() returns no value.

                                   SYNOPSIS

     The function informativeness() allows genotype_elimination_engine() to
gauge which nuclear families are the most informative.  More informative nuclear
families tend to have shorter genotype lists, on average, than less informative
nuclear families (i.e. their genotypes are more "known").  Thus, the value of
weights should be looked at as measures of disorder with larger values denoting
more disorder, smaller values indicating less disorder, and the smallest value
in weights belonging the nuclear family with the least disorder.  The values
contained in weights will be used by genotype_elimanation_engine() to process
nuclear families from the most informative to the least informative so that
information may be propagated downwind from the most informative pedigree members 
to the least informative pedigree members.

                                   NOTES

     The current implementation of informativeness() weights a nuclear family 
with the average genotype list length of the family members.  This weighting 
action does not necessarily have to rank families in this way for genotype 
elimination to succeed sucessfully.  It merely specifies an order to process
families which will hopefully speed up the genotype elimination process.   A 
more complicated function may be inserted for informativeness() as int as 
the function signature and SIDE EFFECT behavior stated above are preserved.

*/

static void informativeness(struct pedigreeData *pedData, 
                            int locus, 
                            double *weights)
{
 double weight = 0.0; 
 int n, i; 
 int nMarriageNodes = pedData->ped->nMarriageNodes;
 struct children *currentChild = NULL;
 struct marriage *family;
 struct marriage **marriageNodes = pedData->ped->marriageNodes;

 for(i = 1; i <= nMarriageNodes; i++)
   {
     n = 0; weight = 0;
     family = marriageNodes[i];

     weight += family->husband->nGenoLists[locus];
     n++;
     weight += family->wife->nGenoLists[locus];
     n++;
     currentChild = family->child;
     while(currentChild != NULL)
       {
        weight += currentChild->firstChild->nGenoLists[locus];
        currentChild = currentChild->nextChild;
        n++;
       }
     weights[i-1] = weight/n;
   }
 return;
}

/* GENOTYPE_STATUS 
                                   FUNCTION ARGUMENTS                         

genotype_status() takes the following arguments:

   -pedData:  A struct pedigreeData pointer to the pedigree structure being 
              processed.

   -locus:    A int integer denoting the locus upon which a pedigree's status
              will be ascertained

   -status:   A double array of length nLoci that will hold the status given
              to the pedigree at the locus specifed by the argument locus 
              above.

                                   SIDE EFFECTS

The argument array status will be updated so that status[locus] gives the genotype
status of the pedigree at the locus specified by locus.

                                   RETURN VALUE

genotype_status() returns no value.

                                   SYNOPSIS

     The function genotype_status() allows genotype_elimination_engine() to
gauge whether genotype elimination was successfully obtained at the end of the
current run.  If genotype elimination was obtained, it is then necessary to 
perform further rounds of genotype elimination until no further genotype 
elimination is garnered.  Once no further superfluous genotypes have been 
removed, we know that further attempts at genotype elimination are ineffective
and we halt execution.

                                   NOTES

     The current implementation of genotype_status() weights a pedigree with  
the total number of genotypes present in the pedigree members.  When the total
number of genotypes does not change after a genotype elimination run, all of the
superfluous genotypes have been trimmed, and program execution should halt at 
current locus being considered, and the next locus should be processed.  Unlike
the function informativeness() above, it is not recommended that this function
be modified.

*/

static void genotype_status(struct pedigreeData *pedData, 
                           int locus, 
                           int *status)
{
 int i, weight; 
 int nMarriageNodes = pedData->ped->nMarriageNodes;
 struct children *currentChild = NULL;
 struct marriage *family;
 struct marriage **marriageNodes = pedData->ped->marriageNodes;

 weight = 0;
 for(i = 1; i <= nMarriageNodes; i++)
   {
     family = marriageNodes[i];

     weight += family->husband->nGenoLists[locus];
     weight += family->wife->nGenoLists[locus];
     currentChild = family->child;
     while(currentChild != NULL)
       {
        weight += currentChild->firstChild->nGenoLists[locus];
        currentChild = currentChild->nextChild;
       }
    }
   status[locus] = weight;
 return;
}

/* COMPACT
                                   FUNCTION ARGUMENTS                         

compact() takes the following arguments:

   -family:   A struct marriage pointer to the nuclear family being 
              processed.

   -people:   A int integer that will be bit masked to determine which
              members of a nuclear family should be processed. See CALLING 
              CONVENTIONS below for more information.

   -action:   A int integer that will be bit masked to determine which
              actions should be perfromed.  See CALLING CONVENTIONS below 
              for more information.

   -locus:    A int integer denoting the locus upon which genotype
              elimination will occur. 

                                   SIDE EFFECTS

Members of the nuclear family being provessed will either have their 
INCONSISTENT genotypes trimmed from their genotype lists or the will have
their CONSISTENT genotypes moved to the end of their genotype lists.  See
CALLING CONVENTIONS below for more information.

                                   CALLING CONVENTIONS

Define the following:

struct marriage *family = &myfamily;
int FATHER = 4, MOTHER = 2, CHILDREN = 1;
int MOVE = 2, TRIM = 1;
int locus = mylocus;

Then you may trim INCONSISTENT genotypes from the genotype lists of specific 
members of a nuclear family using TRIM as the argument to the "action" parameter
and the sum of of your family members as the argument to the "people" parameter:

Trim the genotype list of the mother:

     compact(family, MOTHER, TRIM, locus);

Trim the genotype list of the mother and the father:

     compact(family, MOTHER+FATHER, TRIM, locus);

Trim the genotype list of the mother and the children:

     compact(family, MOTHER+CHILDREN, TRIM, locus);

Trim the genotype list of the mother, father, and children::

     compact(family, MOTHER+FATHER+CHILDREN, TRIM, locus);

Of course, other combinations of family members are also possible.  In addition,
you may move the CONSISTENT genotypes to the end of the genotype lists of 
specified family members by using MOVE as the argument to the "action" parameter
as follows: 

Move the CONSISTENT genotypes to the end of the genotype list of the mother:

     compact(family, MOTHER, MOVE, locus);

Move the CONSISTENT genotypes to the end of the genotype lists of the mother and
the father:

     compact(family, MOTHER+FATHER, MOVE, locus);

Move the CONSISTENT genotypes to the end of the genotype lists of the mother and
the children:

     compact(family, MOTHER+CHILDREN, MOVE, locus);

Move the CONSISTENT genotypes to the end of the genotype lists of the mother, 
the father, and the children:

     compact(family, MOTHER+FATHER+CHILDREN, MOVE, locus);

And as before, other combinations of family members are also possible.

                                   RETURN VALUE

compact() returns no value.

*/

static void compact(struct marriage *family, int people, 
                    int action,             int locus)
{
 int FATHER = 4, MOTHER = 2, CHILDREN = 1, MOVE = 2, TRIM = 1;
 int start, end, temp, numConsistent, i;
 struct person *subject;
 struct children *currentChild;

/*** Move the CONSISTENT genotypes to the end of the genotype list, and the ***/
/*** INCONSISTENT genotypes to the beginning:                               ***/

 if(action & MOVE)
   {
     if(people & FATHER)
       {
	subject = family->husband;
        for(start=0, end=subject->nGenoLists[locus]-1; start <  end;)
          {
           if(subject->genoListsCon[2*locus][start] == INCONSISTENT)
             {
              if(subject->genoListsCon[2*locus][end] == CONSISTENT)
                 end--;
              start++;
              continue;
             }
           if(subject->genoListsCon[2*locus][end] == CONSISTENT)
             {
              end--;
              continue; 
             }
           else /* start is CONSISTENT, end is INCONSISTENT, swap them: */
             {
              /*** Swap genotypes: ****/
              temp = subject->genoLists[2*locus][start];
              subject->genoLists[2*locus][start] = 
                       subject->genoLists[2*locus][end];
              subject->genoLists[2*locus][end] = temp;

              temp = subject->genoLists[2*locus+1][start];
              subject->genoLists[2*locus+1][start] = 
                       subject->genoLists[2*locus+1][end];
               subject->genoLists[2*locus+1][end] = temp;

              /*** Swap genotype consistency markers: ****/
              temp = subject->genoListsCon[2*locus][start];
              subject->genoListsCon[2*locus][start] = 
                       subject->genoListsCon[2*locus][end];
              subject->genoListsCon[2*locus][end] = temp;

              temp = subject->genoListsCon[2*locus+1][start];
              subject->genoListsCon[2*locus+1][start] = 
                       subject->genoListsCon[2*locus+1][end];
              subject->genoListsCon[2*locus+1][end] = temp;
              start++;
              end--;
	     }
	  } /* end for(start,end)    ***/
       }   /** end if(people,FATHER) ***/
 
     if(people & MOTHER)
       {
	subject = family->wife;
        for(start=0, end=subject->nGenoLists[locus]-1; start <  end;)
          {
           if(subject->genoListsCon[2*locus][start] == INCONSISTENT)
             {
              if(subject->genoListsCon[2*locus][end] == CONSISTENT)
                 end--;
              start++;
              continue;
             }
           if(subject->genoListsCon[2*locus][end] == CONSISTENT)
             {
              end--;
              continue; 
             }
           else /* start is CONSISTENT, end is INCONSISTENT, swap them: */
             {
              /*** Swap genotypes: ****/
              temp = subject->genoLists[2*locus][start];
              subject->genoLists[2*locus][start] = 
                       subject->genoLists[2*locus][end];
              subject->genoLists[2*locus][end] = temp;

              temp = subject->genoLists[2*locus+1][start];
              subject->genoLists[2*locus+1][start] = 
                       subject->genoLists[2*locus+1][end];
               subject->genoLists[2*locus+1][end] = temp;

              /*** Swap genotype consistency markers: ****/
              temp = subject->genoListsCon[2*locus][start];
              subject->genoListsCon[2*locus][start] = 
                       subject->genoListsCon[2*locus][end];
              subject->genoListsCon[2*locus][end] = temp;

              temp = subject->genoListsCon[2*locus+1][start];
              subject->genoListsCon[2*locus+1][start] = 
                       subject->genoListsCon[2*locus+1][end];
              subject->genoListsCon[2*locus+1][end] = temp;
              start++;
              end--;
	     }
	  } /* end for(start,end)    ***/
       }   /** end if(people,MOTHER) ***/

     if(people & CHILDREN)
       {
        currentChild = family->child; 
	 while(currentChild != NULL)
	   {
	    subject = currentChild->firstChild;
            for(start=0, end=subject->nGenoLists[locus]-1; start <  end;)
              {
               if(subject->genoListsCon[2*locus][start] == INCONSISTENT)
                 {
                  if(subject->genoListsCon[2*locus][end] == CONSISTENT)
                     end--;
                  start++;
                  continue;
                 }
               if(subject->genoListsCon[2*locus][end] == CONSISTENT)
                 {
                  end--;
                  continue; 
		 }
               else /* start is CONSISTENT, end is INCONSISTENT, swap them: */
                 {
                  /*** Swap genotypes: ****/
                  temp = subject->genoLists[2*locus][start];
                  subject->genoLists[2*locus][start] = 
                           subject->genoLists[2*locus][end];
                  subject->genoLists[2*locus][end] = temp;

                  temp = subject->genoLists[2*locus+1][start];
                  subject->genoLists[2*locus+1][start] = 
                           subject->genoLists[2*locus+1][end];
                  subject->genoLists[2*locus+1][end] = temp;

                  /*** Swap genotype consistency markers: ****/
                  temp = subject->genoListsCon[2*locus][start];
                  subject->genoListsCon[2*locus][start] = 
                           subject->genoListsCon[2*locus][end];
                  subject->genoListsCon[2*locus][end] = temp;

                  temp = subject->genoListsCon[2*locus+1][start];
                  subject->genoListsCon[2*locus+1][start] = 
                           subject->genoListsCon[2*locus+1][end];
                  subject->genoListsCon[2*locus+1][end] = temp;
                 start++;
                 end--;
		 }
	      } /* end for(start,end)      ***/
            currentChild = currentChild->nextChild;
	   }  /**  end while               ***/
       }     /***  end if(people,CHILDREN) ***/
   }        /****  end if(action,MOVE)     ***/

/*** Move the CONSISTENT genotypes to the beginning of the genotype list,   ***/
/*** remove the INCONSISTENT genotypes:                                     ***/

 if(action & TRIM)
   {
     if(people & FATHER)
       {
	subject = family->husband;
        if(subject->nGenoLists[locus] == 1 &&
           subject->genoListsCon[2*locus][0] == INCONSISTENT)
          {
	    subject->nGenoLists[locus] = 0;
	  }
        for(start=0, end=subject->nGenoLists[locus]-1; start <  end;)
          {
           if(subject->genoListsCon[2*locus][start] == CONSISTENT)
             {
              if(subject->genoListsCon[2*locus][end] == INCONSISTENT)
                 end--;
              start++;
              continue;
             }
           if(subject->genoListsCon[2*locus][end] == INCONSISTENT)
             {
              end--;
              continue; 
             }
           else /* start is INCONSISTENT, end is CONSISTENT, swap them: */
             {
              /*** Swap genotypes: ****/
              temp = subject->genoLists[2*locus][start];
              subject->genoLists[2*locus][start] = 
                       subject->genoLists[2*locus][end];
              subject->genoLists[2*locus][end] = temp;

              temp = subject->genoLists[2*locus+1][start];
              subject->genoLists[2*locus+1][start] = 
                       subject->genoLists[2*locus+1][end];
               subject->genoLists[2*locus+1][end] = temp;

              /*** Swap genotype consistency markers: ****/
              temp = subject->genoListsCon[2*locus][start];
              subject->genoListsCon[2*locus][start] = 
                       subject->genoListsCon[2*locus][end];
              subject->genoListsCon[2*locus][end] = temp;

              temp = subject->genoListsCon[2*locus+1][start];
              subject->genoListsCon[2*locus+1][start] = 
                       subject->genoListsCon[2*locus+1][end];
              subject->genoListsCon[2*locus+1][end] = temp;
              start++;
              end--;
	     }
	  } /* end for(start,end) */

           numConsistent = 0;
           for(i = 0; i < subject->nGenoLists[locus]; i++)
	     {
	      if(subject->genoListsCon[2*locus][i] == CONSISTENT)
	        {
                 numConsistent++;
                 continue;
	        }
              else /* INCONSISTENT genotypes are at the end: */ 
                {
	  	 break;
	        }
	     }
           subject->nGenoLists[locus] = numConsistent;
       } /* end if(people,FATHER) */
   
     if(people & MOTHER)
       {
	subject = family->wife;
        if(subject->nGenoLists[locus] == 1 &&
           subject->genoListsCon[2*locus][0] == INCONSISTENT)
          {
	    subject->nGenoLists[locus] = 0;
	  }
        for(start=0, end=subject->nGenoLists[locus]-1; start <  end;)
          {
           if(subject->genoListsCon[2*locus][start] == CONSISTENT)
             {
              if(subject->genoListsCon[2*locus][end] == INCONSISTENT)
                 end--;
              start++;
              continue;
             }
           if(subject->genoListsCon[2*locus][end] == INCONSISTENT)
             {
              end--;
              continue; 
             }
           else /* start is INCONSISTENT, end is CONSISTENT, swap them: */
             {
              /*** Swap genotypes: ****/
              temp = subject->genoLists[2*locus][start];
              subject->genoLists[2*locus][start] = 
                       subject->genoLists[2*locus][end];
              subject->genoLists[2*locus][end] = temp;

              temp = subject->genoLists[2*locus+1][start];
              subject->genoLists[2*locus+1][start] = 
                       subject->genoLists[2*locus+1][end];
               subject->genoLists[2*locus+1][end] = temp;

              /*** Swap genotype consistency markers: ****/
              temp = subject->genoListsCon[2*locus][start];
              subject->genoListsCon[2*locus][start] = 
                       subject->genoListsCon[2*locus][end];
              subject->genoListsCon[2*locus][end] = temp;

              temp = subject->genoListsCon[2*locus+1][start];
              subject->genoListsCon[2*locus+1][start] = 
                       subject->genoListsCon[2*locus+1][end];
              subject->genoListsCon[2*locus+1][end] = temp;
              start++;
              end--;
	     }
	  } /* end for(start,end) */

           numConsistent = 0;
           for(i = 0; i < subject->nGenoLists[locus]; i++)
	     {
	      if(subject->genoListsCon[2*locus][i] == CONSISTENT)
	        {
                 numConsistent++;
                 continue;
	        }
              else /* INCONSISTENT genotypes are at the end: */ 
                {
	  	 break;
	        }
	     }
           subject->nGenoLists[locus] = numConsistent;
       } /* end if(people,MOTHER) */
       
     if(people & CHILDREN)
       {
	 currentChild = family->child;
	 while(currentChild != NULL)
	   {
	    subject = currentChild->firstChild;
            if(subject->nGenoLists[locus] == 1 &&
               subject->genoListsCon[2*locus][0] == INCONSISTENT)
              {
	       subject->nGenoLists[locus] = 0;
	      }
            for(start=0, end=subject->nGenoLists[locus]-1; start <  end;)
              {
               if(subject->genoListsCon[2*locus][start] == CONSISTENT)
                 {
                  if(subject->genoListsCon[2*locus][end] == INCONSISTENT)
                     end--;
                  start++;
                  continue;
                 }
               if(subject->genoListsCon[2*locus][end] == INCONSISTENT)
                 {
                  end--;
                  continue; 
		 }
               else /* start is INCONSISTENT, end is CONSISTENT, swap them: */
                 {
                  /*** Swap genotypes: ****/
                  temp = subject->genoLists[2*locus][start];
                  subject->genoLists[2*locus][start] = 
                           subject->genoLists[2*locus][end];
                  subject->genoLists[2*locus][end] = temp;

                  temp = subject->genoLists[2*locus+1][start];
                  subject->genoLists[2*locus+1][start] = 
                           subject->genoLists[2*locus+1][end];
                  subject->genoLists[2*locus+1][end] = temp;

                  /*** Swap genotype consistency markers: ****/
                  temp = subject->genoListsCon[2*locus][start];
                  subject->genoListsCon[2*locus][start] = 
                           subject->genoListsCon[2*locus][end];
                  subject->genoListsCon[2*locus][end] = temp;

                  temp = subject->genoListsCon[2*locus+1][start];
                  subject->genoListsCon[2*locus+1][start] = 
                           subject->genoListsCon[2*locus+1][end];
                  subject->genoListsCon[2*locus+1][end] = temp;
                  start++;
                  end--;
		 }
	      } /* end for(start,end) */

            numConsistent = 0;
            for(i = 0; i < subject->nGenoLists[locus]; i++)
	      {
 	      if(subject->genoListsCon[2*locus][i] == CONSISTENT)
	         {
                  numConsistent++;
                  continue;
	         }
               else /* INCONSISTENT genotypes are at the end: */ 
                 {
	   	 break;
	         }
	      }
            subject->nGenoLists[locus] = numConsistent;
            currentChild = currentChild->nextChild;
	   } /* end while                ***/
       }    /** end if(people,CHILDREN)  ***/
   }       /*** end if(action,TRIM)      ***/

 return;
}

/* GENOTYPE_ELIMINATION_ENGINE

                                   FUNCTION ARGUMENTS                         

genotype_elimination_engine() takes the following arguments:

   -pedData:  A struct pedigreeData pointer to the pedigree structure being 
              processed.

   -locus:    A int integer denoting the locus upon which genotype
              elimination will occur. 

                                   SIDE EFFECTS

Pedigree members will have genotype elimination at the locus specified by the
locus argument above.  Superfluous genotypes will be removed.

                                   RETURN VALUE

genotype_elimination_engine() returns the following values:

   0: Successful completion - genotype elimination has proceeded without 
      incident.

   1: One person in the pedigree has a genotype candidate list of length zero.
      This indicates a Mendelian inconsistency in the pedigree that must be 
      corrected.  As a footnote, once a Menelian inconsistency is found in a
      nuclear family, genotype_elimination_engine() returns without further 
      processing of any additional nuclear families in the pedigree.

*/

static int genotype_elimination_engine(struct pedigreeData *pedData, 
                                        int locus)
{
 struct marriage  *family;
 struct person    *mother;
 struct person    *father;
 struct children  *currentChild;
 struct marriage **marriageNodes = pedData->ped->marriageNodes;
 void  (*weightFunction)()       = pedData->wfunc;
 void  (*statusFunction)()       = pedData->sfunc;
 int    nMarriageNodes          = pedData->ped->nMarriageNodes;
 int   *marriageIndex           = pedData->ped->marriageIndex;
 int   *oldStatus               = pedData->ped->oldStatus;
 int   *status                  = pedData->ped->status;
 int   *errLocus                = pedData->errLocus;
 int   *errRelation             = pedData->errRelation;
 int   *errSpouse               = pedData->errSpouse;
 double *weights                 = pedData->ped->Weights[locus];
 int i,j,k,p;
 int childAlleleOne, childAlleleTwo, fatherAlleleOne, fatherAlleleTwo,
      motherAlleleOne, motherAlleleTwo, numChildren = 0, 
      numConsistentChildren = 0, outerLoopContinue = 0,
      allChildrenConsistent = 0, statusChange = 0;
 int FATHER = 4, MOTHER = 2, CHILDREN = 1;
 int MOVE = 2, TRIM = 1;

/*** We will set STATUS[locus] to DONE initially.  Any genotype will cause  ***/
/*** STATUS[locus] to be marked as NOT_DONE.                                ***/

 STATUS[locus] = DONE;

/*** Weight the family's based upon their genotype informativeness and then ***/
/*** processed the families from most informative to least informative.     ***/
/*** Also obtain the intitial genotype status prior to this genotype        ***/
/*** elimination run:                                                       ***/

 weightFunction(pedData, locus, weights);
 quicksort_permute(nMarriageNodes, weights, marriageIndex);

 for(i = 1; i <= nMarriageNodes; i++)
  {
   family = marriageNodes[marriageIndex[i-1]];
   mother = family->wife;
   father = family->husband;

/************ Check for family members with empty genotype lists: *************/

   if(mother->nGenoLists[locus] == 0)
    {
     STATUS[locus]      = DONE;
     errLocus[locus]    = mother->id;
     errRelation[locus] = 2;
     errSpouse[locus]   = father->id;
     return 1;
    }

   if(father->nGenoLists[locus] == 0)
    {
     STATUS[locus]      = DONE;
     errLocus[locus]    = father->id;
     errRelation[locus] = 2;
     errSpouse[locus]   = mother->id;
     return 1;
    }

   currentChild = family->child;
   while(currentChild != NULL)
    {
     if(currentChild->firstChild->nGenoLists[locus] == 0)
      {
       STATUS[locus]      = DONE;
       errLocus[locus]    = currentChild->firstChild->id;
       errRelation[locus] = 1;
       return 1;
      }
     currentChild = currentChild->nextChild;
    }

/*************** Eliminate the mother's INCONSISTENT genotypes: ***************/

   for(j = 0; j < mother->nGenoLists[locus]; j++)
    {
     motherAlleleOne = mother->genoLists[2*locus][j];
     motherAlleleTwo = mother->genoLists[2*locus+1][j];
     for(k = 0; k < father->nGenoLists[locus]; k++)
      {
       /**** Check to see if the mother's genotype was found to be ****/
       /**** CONSISTENT on the previous run of this loop:          ****/
       if(outerLoopContinue)
	{
         outerLoopContinue = 0;
         break;
        }
       fatherAlleleOne = father->genoLists[2*locus][k];
       fatherAlleleTwo = father->genoLists[2*locus+1][k];           
       currentChild = family->child;
       numChildren = 0;
       numConsistentChildren = 0;
       allChildrenConsistent = 1;
       while(currentChild != NULL)
        {
         /**** The following checks to see if this child is CONSISTENT ****/
         /**** with the above parental genotypes:                      ****/
         for(p = 0; p < currentChild->firstChild->nGenoLists[locus]; p++)
	  {
  	   childAlleleOne = currentChild->firstChild->genoLists[2*locus][p];
           childAlleleTwo = currentChild->firstChild->genoLists[2*locus+1][p];
           if( ( ((childAlleleOne == motherAlleleOne)    
                                  || 
                  (childAlleleOne == motherAlleleTwo))             
                                  &&
                 ((childAlleleTwo == fatherAlleleOne) 
                                  || 
                  (childAlleleTwo == fatherAlleleTwo)))

                                  ||

               ( ((childAlleleTwo == motherAlleleOne)    
                                  || 
                  (childAlleleTwo == motherAlleleTwo))             
                                  &&
                 ((childAlleleOne == fatherAlleleOne) 
                                  || 
                  (childAlleleOne == fatherAlleleTwo))  ))
            { 
             numConsistentChildren++;
             break;
            } /* end if     *****/
	  }  /** end for(p) *****/
         numChildren++;
        if(numConsistentChildren != numChildren)
	  {
	    allChildrenConsistent = 0;
            break;
	  }
         currentChild = currentChild->nextChild;
        } /****  end while  *****/
       /**** The following checks to see if all children are CONSISTENT ****/
       /**** with the above parental genotypes:                         ****/
       if(allChildrenConsistent)
        {
         currentChild = family->child;
         while(currentChild != NULL)
          {
           for(p = 0; p < currentChild->firstChild->nGenoLists[locus]; p++)
            {
             childAlleleOne = 
               currentChild->firstChild->genoLists[2*locus][p];
             childAlleleTwo = 
               currentChild->firstChild->genoLists[2*locus+1][p];
             if( ( ((childAlleleOne == motherAlleleOne)    
                                    || 
                    (childAlleleOne == motherAlleleTwo))             
                                    &&
                   ((childAlleleTwo == fatherAlleleOne) 
                                    || 
                    (childAlleleTwo == fatherAlleleTwo)))
 
                                    ||
 
                 ( ((childAlleleTwo == motherAlleleOne)    
                                    || 
                    (childAlleleTwo == motherAlleleTwo))             
                                    &&
                   ((childAlleleOne == fatherAlleleOne) 
                                    || 
                    (childAlleleOne == fatherAlleleTwo))  ))
	      { 
               /* mark this child genotype as CONSISTENT */
       	       currentChild->firstChild->genoListsCon[2*locus][p]   =
	         CONSISTENT;
	       currentChild->firstChild->genoListsCon[2*locus+1][p] =
	         CONSISTENT;
	      } /* end if     *****/
	    }  /** end for(p) *****/
           currentChild = currentChild->nextChild;
          }  /**** end while  *****/
         /* Mark the mother and father genotypes as CONSISTENT */
         mother->genoListsCon[2*locus][j]    = CONSISTENT;
         mother->genoListsCon[2*locus+1][j]  = CONSISTENT;
         father->genoListsCon[2*locus][k]    = CONSISTENT;
         father->genoListsCon[2*locus+1][k]  = CONSISTENT;
         STATUS[locus] = NOT_DONE;
         if(k < father->nGenoLists[locus]-1)
            outerLoopContinue = 1;
        } /* end if     *****/
      } /*** end for(k) *****/
    } /***** end for(j) *****/

  outerLoopContinue = 0;

  /**** Remove (TRIM) the mother's INCONSISTENT genotypes and move     ****/
  /**** (MOVE) the father's and childrens' CONSISTENT genotypes to the ****/
  /**** end of their respective genotype lists:                        ****/

  compact(family, MOTHER, TRIM, locus);

  if(STATUS[locus] == NOT_DONE)
   {
    compact(family, FATHER+CHILDREN, MOVE, locus);
   }

  if(mother->nGenoLists[locus] == 0)
   {
    STATUS[locus]      = DONE;
    errLocus[locus]    = mother->id;
    errRelation[locus] = 2;
    errSpouse[locus]   = father->id;
    return 1;
   }
 
/*************** Eliminate the father's INCONSISTENT genotypes: ***************/
 
   for(j = 0; j < father->nGenoLists[locus]; j++)
    {
     /**** Father genotypes that were found to be CONSISTENT when the ****/
     /**** mother's genotypes were eliminated above populate the end  ****/
     /**** portion of the father's genotype list:                     ****/
     if(father->genoListsCon[2*locus][j] == CONSISTENT)
      {
       break;
      }
     fatherAlleleOne = father->genoLists[2*locus][j];
     fatherAlleleTwo = father->genoLists[2*locus+1][j];
     for(k = 0; k < mother->nGenoLists[locus]; k++)
      {
       /**** Check to see if the father's genotype was found to be ****/
       /**** CONSISTENT on the previous run of this loop:          ****/
       if(outerLoopContinue)
        {
         outerLoopContinue = 0;
         break;
        }
       motherAlleleOne = mother->genoLists[2*locus][k];
       motherAlleleTwo = mother->genoLists[2*locus+1][k];           
       currentChild = family->child;
       numChildren = 0;
       numConsistentChildren = 0;
       allChildrenConsistent = 1;
       while(currentChild != NULL)
        {
         /**** The following checks to see if this child is CONSISTENT ****/
         /**** with the above parental genotypes:                      ****/
         for(p = 0; p < currentChild->firstChild->nGenoLists[locus]; p++)
          {
           childAlleleOne = currentChild->firstChild->genoLists[2*locus][p];
           childAlleleTwo = currentChild->firstChild->genoLists[2*locus+1][p];
           if( ( ((childAlleleOne == motherAlleleOne)    
                                  || 
                  (childAlleleOne == motherAlleleTwo))             
                                  &&
                 ((childAlleleTwo == fatherAlleleOne) 
                                  || 
                  (childAlleleTwo == fatherAlleleTwo)))

                                  ||

               ( ((childAlleleTwo == motherAlleleOne)    
                                  || 
                  (childAlleleTwo == motherAlleleTwo))             
                                  &&
                 ((childAlleleOne == fatherAlleleOne) 
                                  || 
                  (childAlleleOne == fatherAlleleTwo))  ))
            { 
             numConsistentChildren++;
             break;
            } /* end if     *****/
          }  /** end for(p) *****/
         numChildren++;
         if(numConsistentChildren != numChildren)
	  {
	    allChildrenConsistent = 0;
            break;
	  }
         currentChild = currentChild->nextChild;
        } /****  end while  *****/
       /**** The following checks to see if all children are CONSISTENT ****/
       /**** with the above parental genotypes:                         ****/
       if(allChildrenConsistent)
        {
         currentChild = family->child;
         while(currentChild != NULL)
          {
	   for(p = 0; p < currentChild->firstChild->nGenoLists[locus]; p++)
	    {
  	     childAlleleOne = 
               currentChild->firstChild->genoLists[2*locus][p];
             childAlleleTwo = 
               currentChild->firstChild->genoLists[2*locus+1][p];
             if( ( ((childAlleleOne == motherAlleleOne)    
                                    || 
                    (childAlleleOne == motherAlleleTwo))             
                                    &&
                   ((childAlleleTwo == fatherAlleleOne) 
                                    || 
                    (childAlleleTwo == fatherAlleleTwo)))
 
                                    ||
  
                 ( ((childAlleleTwo == motherAlleleOne)    
                                    || 
                    (childAlleleTwo == motherAlleleTwo))             
                                    &&
                   ((childAlleleOne == fatherAlleleOne) 
                                    || 
                    (childAlleleOne == fatherAlleleTwo))  ))
              { 
               /* mark this child genotype as CONSISTENT */
               currentChild->firstChild->genoListsCon[2*locus][p]   =
	         CONSISTENT;
	       currentChild->firstChild->genoListsCon[2*locus+1][p] =
	         CONSISTENT;
	      } /* end if     *****/
	    }  /** end for(p) *****/
           currentChild = currentChild->nextChild;
          }  /**** end while  *****/
         /* Mark the father's genotype  as CONSSISTENT:        */
         father->genoListsCon[2*locus][j]    = CONSISTENT;
         father->genoListsCon[2*locus+1][j]  = CONSISTENT;
         STATUS[locus] = NOT_DONE;
         if(k < mother->nGenoLists[locus]-1)
            outerLoopContinue=1;
        } /* end if     *****/
      } /*** end for(k) *****/
    } /***** end for(j) *****/

  outerLoopContinue = 0;

  /**** Remove (TRIM) the father's INCONSISTENT genotypes and move     ****/
  /**** (MOVE) childrens' CONSISTENT genotypes to the end of their     ****/
  /**** respective genotype lists:                                     ****/

   compact(family, FATHER, TRIM, locus);

   if(STATUS[locus] == NOT_DONE)
   {
    compact(family, CHILDREN, MOVE, locus);
   }

  if(father->nGenoLists[locus] == 0)
   {
    STATUS[locus]      = DONE;
    errLocus[locus]    = father->id;
    errRelation[locus] = 2;
    errSpouse[locus]   = mother->id;
    return 1;
   }

/************** Eliminate the childrens' INCONSISTENT genotypes: **************/

   currentChild = family->child;
   while(currentChild != NULL)
    {
     for(j = 0; 
         j < currentChild->firstChild->nGenoLists[locus] &&
         currentChild->firstChild->genoListsCon[2*locus][j] != CONSISTENT; 
         j++)
      {
       childAlleleOne = currentChild->firstChild->genoLists[2*locus][j];
       childAlleleTwo = currentChild->firstChild->genoLists[2*locus+1][j];       
       for(k = 0; k < mother->nGenoLists[locus]; k++)
        {
         if(outerLoopContinue)
	  {
	   outerLoopContinue = 0;
           break;
	  }
         motherAlleleOne = mother->genoLists[2*locus][k];
         motherAlleleTwo = mother->genoLists[2*locus+1][k];
         for(p = 0; p < father->nGenoLists[locus]; p++)
          {
           fatherAlleleOne = father->genoLists[2*locus][p];
           fatherAlleleTwo = father->genoLists[2*locus+1][p];       
           if( ( ((childAlleleOne == motherAlleleOne)    
                                  || 
                  (childAlleleOne == motherAlleleTwo))             
                                  &&
                 ((childAlleleTwo == fatherAlleleOne) 
                                  || 
                  (childAlleleTwo == fatherAlleleTwo)))

                                  ||
  
               ( ((childAlleleTwo == motherAlleleOne)    
                                  || 
                  (childAlleleTwo == motherAlleleTwo))             
                                  &&
                 ((childAlleleOne == fatherAlleleOne) 
                                  || 
                  (childAlleleOne == fatherAlleleTwo))  ))
	    {
             currentChild->firstChild->genoListsCon[2*locus][j]   =
               CONSISTENT;
	     currentChild->firstChild->genoListsCon[2*locus+1][j] =
	       CONSISTENT;
             STATUS[locus] = NOT_DONE;
             if(k < mother->nGenoLists[locus]-1)
                outerLoopContinue = 1;
             break; 
	    } /* end if     ***/
	  }  /** end for(p) ***/
	}   /*** end for(k) ***/
      }    /**** end for(j) ***/
     currentChild = currentChild->nextChild;
    }     /***** end while  ***/

  outerLoopContinue = 0;

  /**** Remove (TRIM) the childrens' INCONSISTENT genotypes from their ****/
  /**** genotype lists:                                                ****/

  if(STATUS[locus] == NOT_DONE)
    compact(family, CHILDREN, TRIM, locus);

  /**** Check for a child with a zero-length genotype list:            ****/

  currentChild = family->child;
  while(currentChild != NULL)
   {
    if(currentChild->firstChild->nGenoLists[locus] == 0)
     {
      STATUS[locus]      = DONE;
      errLocus[locus]    = currentChild->firstChild->id;
      errRelation[locus] = 1;
      return 1;
     }
    currentChild = currentChild->nextChild;
   }

  /**** Set the genotype lists to INCONSISTENT because these people    ****/
  /**** may be processed again at a future time:                       ****/

  /**** Mother:   ****/
  for(j = 0; j < mother->nGenoLists[locus]; j++)
   {
    mother->genoListsCon[2*locus][j]   = INCONSISTENT;
    mother->genoListsCon[2*locus+1][j] = INCONSISTENT;
   }

  /**** Father:   ****/
  for(j = 0; j < father->nGenoLists[locus]; j++)
   {
    father->genoListsCon[2*locus][j]   = INCONSISTENT;
    father->genoListsCon[2*locus+1][j] = INCONSISTENT;
   }
 
  /**** Children: ****/
  currentChild = family->child;
  while(currentChild != NULL)
   {
    for(j = 0; j < currentChild->firstChild->nGenoLists[locus]; j++)
     {
      currentChild->firstChild->genoListsCon[2*locus][j]   = INCONSISTENT;
      currentChild->firstChild->genoListsCon[2*locus+1][j] = INCONSISTENT;
     }
    currentChild = currentChild->nextChild;
   }
  }  /******* end for(i) *****/

 statusChange = 0;
 statusFunction(pedData, locus, status);

 if(status[locus] != oldStatus[locus])
    statusChange = 1;

    
 if(statusChange)
   {
     STATUS[locus] = NOT_DONE;
     oldStatus[locus] = status[locus];
     return 0;
   }
 else
   { 
    STATUS[locus] = DONE;
    return 0;
   }
}

/* PREPROCESS

                                   FUNCTION ARGUMENTS                         

preprocess() takes the following arguments:

   -pedData:  A struct pedigreeData pointer to the pedigree structure being 
              processed.

   -locus:    A int integer denoting the locus upon which genotype
              elimination will occur. 

                                   SIDE EFFECTS

If the mother or father has a genotype list of length 1, this list will be used
to eliminate superfluous genotypes in the childrens' genotype list.  Similarily,
if a child has a genotype list of length one, his genotype will be used to 
eliminate superfluous genotypes in his mother and father.

                                   RETURN VALUE

preprocess() returns the following values:

   0: Successful completion - genotype elimination has proceeded without 
      incident.

   1: One person in the pedigree has a genotype candidate list of length zero.
      This indicates a Mendelian inconsistency in the pedigree that must be 
      corrected.  As a footnote, once a Menelian inconsistency is found in a
      nuclear family, preprocess() returns without further processing of any 
      additional nuclear families in the pedigree.

*/
   
static int preprocess(struct pedigreeData *pedData, int locus)
{
 struct marriage *family;
 struct person *mother;
 struct person *father;
 struct children *currentChild;
 struct marriage **marriageNodes = pedData->ped->marriageNodes;
 int nMarriageNodes = pedData->ped->nMarriageNodes;
 int   *errLocus    = pedData->errLocus;
 int   *errRelation = pedData->errRelation;
 int   *errSpouse   = pedData->errSpouse;
 int i,j,k;
 int childrenMotherTrim, childrenFatherTrim, 
      childrenMotherTrimRecord, childrenFatherTrimRecord,
      motherChildrenTrim, fatherChildrenTrim;
 int childAlleleOne, childAlleleTwo, fatherAlleleOne, fatherAlleleTwo,
      motherAlleleOne, motherAlleleTwo;
 int FATHER = 4, MOTHER = 2, CHILDREN = 1;
 int TRIM = 1;

 for(i = 1; i <= nMarriageNodes; i++)
   {
    family = marriageNodes[i];
    mother = family->wife;
    father = family->husband;

/******************************************************************************/
/***** Check if the mother has a genotype list of length 1.  If she does, *****/
/***** then eliminate the childrens' superfluous genotypes:               *****/
/******************************************************************************/

    motherChildrenTrim = 0;

    if(mother->nGenoLists[locus] == 1)
      { 
       /**** The mother has a genotype list of length one *****/
       motherChildrenTrim = 1;
       motherAlleleOne = mother->genoLists[2*locus][0];
       motherAlleleTwo = mother->genoLists[2*locus+1][0];  
       currentChild = family->child;
       while(currentChild != NULL)
         {
	  for(j = 0; j < currentChild->firstChild->nGenoLists[locus]; j++)
	    {
  	     childAlleleOne = 
               currentChild->firstChild->genoLists[2*locus][j];
             childAlleleTwo = 
               currentChild->firstChild->genoLists[2*locus+1][j];
             if( (childAlleleOne == motherAlleleOne)
                                 ||
                 (childAlleleOne == motherAlleleTwo)
                                 ||
                 (childAlleleTwo == motherAlleleOne)
                                 ||
                 (childAlleleTwo == motherAlleleTwo) )
                                 
              { 
               /* mark this child genotype as CONSISTENT */
               currentChild->firstChild->genoListsCon[2*locus][j]   =
	         CONSISTENT;
	       currentChild->firstChild->genoListsCon[2*locus+1][j] =
	         CONSISTENT;
	      } /* end if     *****/
	    }  /** end for(j) *****/
           currentChild = currentChild->nextChild;
          }  /**** end while  *****/
      }     /***** end if     *****/

    if(motherChildrenTrim)
      {
       compact(family, CHILDREN, TRIM, locus);
       currentChild = family->child;
       while(currentChild != NULL)
         {
          if(currentChild->firstChild->nGenoLists[locus] == 0)
            {
             STATUS[locus]      = DONE;
             errLocus[locus]    = currentChild->firstChild->id;
             errRelation[locus] = 1;
             return 1;
	    }
          for(j = 0; j < currentChild->firstChild->nGenoLists[locus]; j++)
            {
             currentChild->firstChild->genoListsCon[2*locus][j]   = 
                                       INCONSISTENT;
             currentChild->firstChild->genoListsCon[2*locus+1][j] = 
                                       INCONSISTENT;
            } /* end for(j) *****/
          currentChild = currentChild->nextChild;
         }  /*** end while  *****/
      }    /**** end if     *****/

/******************************************************************************/
/***** Check if the father has a genotype list of length 1.  If he does,  *****/
/***** then eliminate the childrens' superfluous genotypes:               *****/
/******************************************************************************/

    fatherChildrenTrim = 0;

    if(father->nGenoLists[locus] == 1)
      { 
       /**** The father has a genotype list of length one *****/
       fatherChildrenTrim = 1;
       fatherAlleleOne = father->genoLists[2*locus][0];
       fatherAlleleTwo = father->genoLists[2*locus+1][0];  
       currentChild = family->child;
       while(currentChild != NULL)
         {
	  for(j = 0; j < currentChild->firstChild->nGenoLists[locus]; j++)
	    {
  	     childAlleleOne = 
               currentChild->firstChild->genoLists[2*locus][j];
             childAlleleTwo = 
               currentChild->firstChild->genoLists[2*locus+1][j];
             if( (childAlleleOne == fatherAlleleOne)
                                 ||
                 (childAlleleOne == fatherAlleleTwo)
                                 ||
                 (childAlleleTwo == fatherAlleleOne)
                                 ||
                 (childAlleleTwo == fatherAlleleTwo) )
                                 
              { 
               /* mark this child genotype as CONSISTENT */
               currentChild->firstChild->genoListsCon[2*locus][j]   =
	         CONSISTENT;
	       currentChild->firstChild->genoListsCon[2*locus+1][j] =
	         CONSISTENT;
               fatherChildrenTrim = 1;
	      } /* end if     *****/
	    }  /** end for(j) *****/
           currentChild = currentChild->nextChild;
          }  /**** end while  *****/
      }     /***** end if     *****/

    if(fatherChildrenTrim)
      {
       compact(family, CHILDREN, TRIM, locus);
       currentChild = family->child;
       while(currentChild != NULL)
         {
          if(currentChild->firstChild->nGenoLists[locus] == 0)
            {
             STATUS[locus]      = DONE;
             errLocus[locus]    = currentChild->firstChild->id;
             errRelation[locus] = 1;
             return 1;
	    }
          for(j = 0; j < currentChild->firstChild->nGenoLists[locus]; j++)
            {
             currentChild->firstChild->genoListsCon[2*locus][j]   = 
                                       INCONSISTENT;
             currentChild->firstChild->genoListsCon[2*locus+1][j] = 
                                       INCONSISTENT;
            } /* end for(j) *****/
          currentChild = currentChild->nextChild;
         }  /*** end while  *****/
      }    /**** end if     *****/

/******************************************************************************/
/***** Check if a child has a genotype list of length 1.  If he does,     *****/
/***** then eliminate the parents' superfluous genotypes:                 *****/
/******************************************************************************/

    childrenMotherTrim       = childrenFatherTrim       = 0;
    childrenMotherTrimRecord = childrenFatherTrimRecord = 0;

    currentChild = family->child;
    while(currentChild != NULL)
      {
       if(currentChild->firstChild->nGenoLists[locus] == 1)
	 {
          childrenMotherTrim = 1;
          childrenFatherTrim = 1;
          childAlleleOne = 
            currentChild->firstChild->genoLists[2*locus][0];
          childAlleleTwo = 
            currentChild->firstChild->genoLists[2*locus+1][0];

          /***** Check for consistency in the mother: *****/
          for(j = 0; j < mother->nGenoLists[locus]; j++)
	    {
             motherAlleleOne = mother->genoLists[2*locus][j];
             motherAlleleTwo = mother->genoLists[2*locus+1][j];  
             if( (childAlleleOne == motherAlleleOne)
                                 ||
                 (childAlleleOne == motherAlleleTwo)
                                 ||
                 (childAlleleTwo == motherAlleleOne)
                                 ||
                 (childAlleleTwo == motherAlleleTwo) )
	       {
                mother->genoListsCon[2*locus][j]   = CONSISTENT;
	        mother->genoListsCon[2*locus+1][j] = CONSISTENT;
	       } /* end if     *****/
	    }   /** end for(j) *****/

          if(childrenMotherTrim)
	    {
	     compact(family, MOTHER, TRIM, locus);
             if(mother->nGenoLists[locus] == 0)
               {
		STATUS[locus] = DONE;
                errLocus[locus]    = mother->id;
                errRelation[locus] = 2;
                errSpouse[locus]   = father->id;
                return 1;
	       }
             for(k = 0; k < mother->nGenoLists[locus]; k++)
               {
                mother->genoListsCon[2*locus][k]   = INCONSISTENT;
                mother->genoListsCon[2*locus+1][k] = INCONSISTENT;
               } /* end for(k) *****/
             childrenMotherTrim = 0;
             childrenMotherTrimRecord = 1;
	    } /* end if *****/

          /***** Check for consistency in the father: *****/
          for(j = 0; j < father->nGenoLists[locus]; j++)
	    {
             fatherAlleleOne = father->genoLists[2*locus][j];
             fatherAlleleTwo = father->genoLists[2*locus+1][j];  
             if( (childAlleleOne == fatherAlleleOne)
                                 ||
                 (childAlleleOne == fatherAlleleTwo)
                                 ||
                 (childAlleleTwo == fatherAlleleOne)
                                 ||
                 (childAlleleTwo == fatherAlleleTwo) )
	       {
                father->genoListsCon[2*locus][j]   = CONSISTENT;
	        father->genoListsCon[2*locus+1][j] = CONSISTENT;
	       } /* end if     *****/
	    }   /** end for(j) *****/

          if(childrenFatherTrim)
	    {
             compact(family, FATHER, TRIM, locus);
             if(father->nGenoLists[locus] == 0)
               {
        	STATUS[locus] = DONE;
                errLocus[locus]    = father->id;
                errRelation[locus] = 2;
                errSpouse[locus]   = mother->id;
                return 1;
	       }
             for(k = 0; k < father->nGenoLists[locus]; k++)
               {
                father->genoListsCon[2*locus][k]   = INCONSISTENT;
                father->genoListsCon[2*locus+1][k] = INCONSISTENT;
               } /* end for(k) *****/
             childrenFatherTrim = 0;
             childrenFatherTrimRecord = 1;
	    } /* end if     *****/

	 }  /*** end if     *****/
          currentChild = currentChild->nextChild;
      }   /***** end while  *****/

     /**** Set the genotype lists to INCONSISTENT because these people    ****/
     /**** may be processed again at a future time:                       ****/

    if(childrenMotherTrimRecord)
      {
       /**** Mother:   ****/
       for(j = 0; j < mother->nGenoLists[locus]; j++)
         {
          mother->genoListsCon[2*locus][j]   = INCONSISTENT;
          mother->genoListsCon[2*locus+1][j] = INCONSISTENT;
         } /* end for(j) *****/
      }

    if(childrenFatherTrimRecord)
      {
       /**** Father:   ****/
       for(j = 0; j < father->nGenoLists[locus]; j++)
         {
          father->genoListsCon[2*locus][j]   = INCONSISTENT;
          father->genoListsCon[2*locus+1][j] = INCONSISTENT;
         } /* end for(j) *****/
      }
   }
 return 0;
}
  
/* TRAVERSE

                                   FUNCTION ARGUMENTS                         

traverse() takes the following arguments:

   -pedData:  A struct pedigreeData pointer to the pedigree structure being 
              processed.

   -func:     A function pointer to a function that returns type void.  The 
              function that is assigned to this argument should accept a struct 
              person pointer and a struct pedigreeData pointer, respectively, 
              such that the struct pedigreeData pointer points to a pedigree 
              that contains the person being pointed to by the struct person 
              pointer.

                                   SIDE EFFECTS

The traverse() function will update the values of the static global variables 
"DONE" and "FLAG".  This is DONE to allow the a pedigree to be traversed more
than one time.  This traversal is actually performed by traverse_engine().

Note:  This pedigree traversal algorithm is for use with non-acyclic pedigrees 
with only one pair of ancestoral founders.

                                   RETURN VALUE

traverse() returns no value.

*/

static void traverse(struct pedigreeData *pedData,
                     void (*func)())
{
 if(FLAG == 0)
   {
    DONE++;
    traverse_engine(pedData->ped->proband,pedData,func);
    DONE--;
    FLAG = 1;
      }
 else
   {
    DONE--;
    traverse_engine(pedData->ped->proband,pedData,func);
    DONE++;
    FLAG = 0;
   }

 return;
}

/* TRAVERSE_ENGINE

                                   FUNCTION ARGUMENTS                         

traverse_engine() takes the following arguments:

   -proband:  A struct person pointer to the proband in a pedigree structure.

   -pedData:  A struct pedigreeData pointer to the pedigree structure being 
              processed.

   -func:     A function pointer to a function that returns type void.  The 
              function that is assigned to this argument should accept a struct 
              person pointer and a struct pedigreeData pointer, respectively, 
              such that the struct pedigreeData pointer points to a pedigree 
              that contains the person being pointed to by the struct person 
              pointer.

	                           SIDE EFFECTS

traverse_engine() updates the value of the struct data member person.DONE to the
current global value of "DONE".  Any side effects of functions assigned the the
"func" function pointer argument above will also be side effects of the 
traverse_engine() function.

Note:  This pedigree traversal algorithm is for use with non-acyclic pedigrees 
with only one pair of ancestoral founders.

                                   RETURN VALUE

traverse_engine() returns no value.

                                   SYNOPSIS

traverse_engine() is a recursive function that traverses through a pedigree and 
performs arbitrary operations upon the members of the pedigree, with these 
actions being specified by the function assigned to the"func" parameter.  

Upon entry, traverse_engine() invokes the function pointed to by "func" and 
passes to this function a struct person pointer, subject, to the current 
person, along with a struct pedigreeData pointer, pedData, containing the 
pedigree and associated pedigree data for the pedigree to which the 
aforementioned subject belongs.  Upon completion, the person.status variable 
of subject is marked as DONE.

Once the current person has been processed, all of the current person's spouses,
children, and parents are checked for their status and processed if their status 
is not marked as DONE.

*/

static void traverse_engine(struct person *subject, struct pedigreeData
*pedData,
                            void(*func)())
{
 struct children *currentChild;
 struct marriage *currentSpouse;

 /**Process this person**/

 if(subject->status != DONE)
   {
    (*func)(subject,pedData);
    subject->status = DONE;
   }

 /********Spouse********/

 if(subject->sex == MALE)
   { currentSpouse = subject->nuclearFamily;
     while(currentSpouse != NULL)
       { 
         if(currentSpouse->wife->status != DONE)
            traverse_engine(currentSpouse->wife,pedData,func);
         currentSpouse = currentSpouse->nextHusbandMarriage;
       }
   }

 if(subject->sex == FEMALE)
   { currentSpouse = subject->nuclearFamily;
     while(currentSpouse != NULL)
       { 
         if(currentSpouse->husband->status != DONE)
            traverse_engine(currentSpouse->husband,pedData,func);
         currentSpouse = currentSpouse->nextWifeMarriage;
       }
   }


 /*******Children********/

 if(subject->nuclearFamily != NULL)
   { currentChild = subject->nuclearFamily->child;
     while(currentChild != NULL)
       { 
         if(currentChild->firstChild->status != DONE)
            traverse_engine(currentChild->firstChild,pedData,func);
         currentChild = currentChild->nextChild;
       }
   }


 /********Parents********/

 if(subject->parents != NULL && 
    subject->parents->husband->status != DONE)
   { 
    traverse_engine(subject->parents->husband,pedData,func);
   }

return;
}


/* QUICKSORT_PERMUTE and QUICKSORT_ORDINAL 
  
				   FUNCTION ARGUMENTS 
These functions, quicksort_permute and quicksort_ordinal, take the following 
arguments:

n: 	       A int integer giving the length of the array to be sorted
array_sort:    A double array of length n to be sorted
array_permute: A int array of length n that will contain the 
	       sequence 1,2,...n permuted to the order corresponding
	       to the sorting of array_sort (see user's manual below).
	       This order will vary slightly depending on whether 
	       quicksort_permute or quicksort_ordinal is used.
temp           A int array of length *n that is used for scratch space (used in
               quicksort_ordinal() only).

				   SIDE EFFECTS
The array array_sort will be returned in sorted order from lowest to highest. 
And array_permute will be returned containg a permutation of 1,2,...,n.  The 
particular permutation varies depending upon which method is called (see user's
manual).

				   RETURN CODES
The following return codes may be returned:
	0:     Successful completion
       -1:     Array length degenerate (lesss than or equal to zero)

				   USER MANUAL
These functions, quicksort_permute and quicksort_ordinal, both use
the main qs(...) engine to perform a quicksort on the elements of 
array_sort (shorthand for "array to be sorted") and return a permutation
of the elements of array_permute ("array to be permuted").  The primary 
difference should be made apparent by the following:

let a[] = {0.8, 0.4, 0.5, 0.2, 0.7, 0.1, 0.6, 0.3}
    b[] = {1,   2,   3,   4,   5,   6,   7,   8  }  

then 

quicksort_permute(8, a, b,) returns

   a[] = {0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8}
   b[] = {6,   4,   8,   2,   3,   7,   5,   1  }

and quicksort_ordinal(8, a, b) returns 

   a[] = {0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8}
   b[] = {8,   4,   5,   2,   7,   1,   6,   3  }

It should be noted that the array of 1,2,...,n - in this case b[] - does 
not need to contain the integers at the time of the call to either 
of the above functions.  It must simply be a int array of length n.  

*/


static int quicksort_ordinal(int n, double array_sort[], int array_permute[],
		             int temp[])
{
  int i; 
  
  if(n <= 0)
    return -1;			      /* non-sensical array length	       */
  
  for(i = 0; i < n; i++)	      /* Initialize array_permute to 1,2,...,n */
    array_permute[i] = i + 1;
  qs(0, n - 1, array_sort, array_permute);
	
  for(i = 0; i < n; i++)
	temp[array_permute[i] - 1] = i + 1;
  for(i = 0; i < n; i++)
    	array_permute[i] = temp[i];

  return 0;
}

static int quicksort_permute(int n, double array_sort[], int array_permute[])
{
  int i; 
  
  if(n <= 0)
    return -1;			      /* non-sensical array length	       */
  for(i = 0; i < n; i++)	      /* Initialize array_permute to 1,2,...,n */
    array_permute[i] = i + 1;
  qs(0, n - 1, array_sort, array_permute);
  return 0;
}


static void qs(int left, int right, double array_sort[], int array_permute[])
{
  int i,j, temp_permute;
  double mid, temp_sort;

  i = left; j = right;
  mid = array_sort[(left + right) / 2];

  do {
    while(array_sort[i] < mid && i < right) i++;
    while(mid < array_sort[j] && left < j) j--;
    
    if(i<=j) {
      temp_sort = array_sort[i];		/* swap elements	       */
      array_sort[i] = array_sort[j];
      array_sort[j] = temp_sort;
					     
      temp_permute = array_permute[i];		/* comes along for the ride... */
      array_permute[i] = array_permute[j];
      array_permute[j] = temp_permute;
      
      i++; j--;
    }
  } while(i <= j);

  if(left < j) 
    qs(left, j, array_sort, array_permute);
  if(i < right) 
    qs(i, right, array_sort, array_permute);

  return; 
}



/* QUICKSORT_LEXICOGRAPHIC
  
				   FUNCTION ARGUMENTS 
This function, quicksort_lexicographic, takes the following arguments:

n: 	       A int integer giving the length of the array to be sorted
arr1           A int array of length n to be sorted
arr2           A int array of length n that will be sorted subordinate to
               arr1 above in a lexicographic fashion (see USER MANUAL below)

				   SIDE EFFECTS
The array arr1 will be returned in sorted order from lowest to highest. And 
arr2 will be sorted subordinate to arr1 in a lexicographic manner.

				   RETURN CODES
The following return codes may be returned:
	0:     Successful completion
       -1:     Array length degenerate (lesss than or equal to zero)

				   USER MANUAL
This function, quicksort_lexicographic, uses the main qs_lex(...) engine to 
perform a quicksort on the elements of arr1 and arr2 and returns arr1 and arr2
sorted lexicographically.  For example, if arr1 and arr2 are defined as

arr1 = { 1,       arr2 = { 5,
         4,                5,
         2,                3,
         1,                4,
         3,                3,
         3,                5,
         3,                4,
         4,                4,
         2,                5,
         2,                2,
         1,                2,
         1,                3,
         2,                4,
         5,                5,
         1 }               1 }

then quicksort_lexicographic(15,arr1,arr2) will return

arr1 = { 1,       arr2 = { 1,
         1,                2,
         1,                3,
         1,                4,
         1,                5,
         2,                2,
         2,                3,
         2,                4,
         2,                5,
         3,                3,
         3,                4,
         3,                5,
         4,                4,
         4,                5,
         5 }               5 }

*/

static int quicksort_lexicographic(int n, int arr1[], int arr2[])
{
  int i; 
  
  if(n <= 0)
    return -1;			      /* non-sensical array length	       */

  qs_lex(0, n - 1, arr1, arr2);
  return 0;
}

static void qs_lex(int left, int right, int arr1[], int arr2[])
{
  int i,j, temp1;
  int mid1, mid2, temp2;

  i = left; j = right;
  mid1 = arr1[(left + right) / 2];
  mid2 = arr2[(left + right) / 2];

  do {
    while((arr1[i] < mid1 || 
	  (arr1[i] == mid1 && arr2[i] < mid2)) &&
	  i < right)
       i++;
    while((mid1 < arr1[j]   ||
          (mid1 == arr1[j] && mid2 < arr2[j]))  && 
          left < j)
       j--;
    
    if(i<=j) {
      temp1 = arr1[i];		/* swap elements	       */
      arr1[i] = arr1[j];
      arr1[j] = temp1;
					     
      temp2 = arr2[i];		/* comes along for the ride... */
      arr2[i] = arr2[j];
      arr2[j] = temp2;
      
      i++; j--;
    }
  } while(i <= j);

  if(left < j) 
    qs_lex(left, j, arr1, arr2);
  if(i < right) 
    qs_lex(i, right, arr1, arr2);

  return; 
}

