#$Author: sinnwell $
#$Date: 2011/02/25 16:37:23 $
#$Header: /projects/genetics/cvs/cvsroot/mend.err/R/mend.err.engine.q,v 1.2 2011/02/25 16:37:23 sinnwell Exp $
#$Locker:  $
#$Log: mend.err.engine.q,v $
#Revision 1.2  2011/02/25 16:37:23  sinnwell
#long to int, add x.sexcheck, T to TRUE
#
#Revision 1.1.1.1  2011/02/16 14:33:20  sinnwell
#initial for mend.err package
#
#Revision 1.3  2009/02/25 22:13:17  sinnwell
#add keywords
#
mend.err.engine <- 
            function(ped,                   # Pedigree data
                     geno=NULL,             # Genotype data (if not in ped)
                     pedigree.column=1,     # Pedigree ID column
                     person.column=2,       # Person ID column
                     labID.column=3,        # Lab ID column
                     father.column=4,       # Father ID column
                     mother.column=5,       # Mother ID column
                     sex.column=6,          # Sex ID column
                     locus.start=7,         # Starting column of the loci
                     locus.end=ncol(ped),   # Ending column of the loci
                     miss.val=c(NA,0),      # Missing Value code
                     male.code=1,           # Male sex code
                     female.code=2,         # Female sex code
                     warn.untyped=FALSE,    # Warn if a locus is untyped
                     check.missing=TRUE,    # Check if all alleles are missing
                     print.no.error=TRUE,   # List pedigrees with no errors
                     print.error=TRUE,      # Print jackknife results
                     print.summary=TRUE,    # Summary of mendelian errors
                     print.num.processed=TRUE, # Print number of pedigrees handled
                     sort.pedigrees=FALSE,  # Process pedigrees in sorted order
                     locus.labels=NULL,	    # Optional lables for each locus
                     summary.names=NULL)    # labels for return tables
{
 # Check if two datasets are used instead of one:
 if(!is.null(geno))
   {
    locus.start <- ncol(ped)+1
    ped <- cbind(ped,geno)
    locus.end <- ncol(ped)
   }
      
 if(is.data.frame(ped))
   ped <- as.matrix.data.frame(ped, justify.format="none")
 
 # Enumerate the distinct pedigrees and process them in sorted order
 # if requested: 
 distinct.pedigrees <- unique(ped[,pedigree.column])
 if(sort.pedigrees)
   distinct.pedigrees <- sort(distinct.pedigrees)

 if(is.factor(distinct.pedigrees))
   {
    distinct.pedigrees <- as.character(distinct.pedigrees)
   }
             
 n.columns <- ncol(ped)
 errors <- 0

 if(print.no.error || print.error || print.summary || print.num.processed)
    cat("\n")
 
 # Data items used for summary tables:
 loc.missing <- NULL
 ped.err     <- NULL
 loc.err     <- NULL
 ped.no.err  <- NULL
 loc.no.err  <- NULL
 
 nLoci <- (locus.end-locus.start+1)/2
  
 # Process each pedigree:
 for(pedID in distinct.pedigrees)
   {
    # Display on unanticipated error:
    fail.msg <- paste("\nFailed on pedigree ",pedID,".\n\n",sep="")
    on.exit(cat(fail.msg))
    
    # Retrieve the pedigree and genotypes being processed:
    pedigree  <- ped[ped[,pedigree.column] == pedID,]
    genotypes <- pedigree[,locus.start:locus.end]
    
    # Check for all alleles missing:
    if(check.missing)
      {
       if(all(is.na(factor(unlist(genotypes),exclude=miss.val))))
         {
          loc.missing <- c(loc.missing,pedigree[1,pedigree.column])
          on.exit()
          next
         }
      }

    # Perform genotype elimination:
    geno.elim <- sge(ped=pedigree, 
                     geno=genotypes, 
                     miss.val=miss.val,
                     male.code=male.code, 
                     female.code=female.code,
                     pedigree.column=pedigree.column,
                     person.column=person.column,
                     labID.column=labID.column,
                     father.column=father.column,
                     mother.column=mother.column,
                     sex.column=sex.column,
                     warn.untyped=warn.untyped,
                     sort.loci=FALSE,
                     locus.labels=locus.labels)
    
    # Check for the absence or presence of errors and handle each case
    # appropriately:
    if(!any(geno.elim$err.locus))
      {
       # No Mendelian Errors:
       ped.no.err <- rbind(ped.no.err,pedigree[1,pedigree.column])
       loc.no.err <- rbind(loc.no.err,rep(0,nLoci))
       if(print.no.error)
         {
          # If print.no.error=TRUE, use verbose ouput to confirm that
          # errorless pedigrees were indeed processed:
          msg <- paste("PEDIGREE ",
                       geno.elim$ped[1,pedigree.column],
                       ":  No Mendelian Errors detected.\n\n",
                       sep="")
          cat(msg)
         }
      }
    else
      {
       # One or more Mendelian Errors:
       ped.err     <- rbind(ped.err,pedigree[1,pedigree.column])
       tmp         <- rep(0,nLoci)
       tmp[geno.elim$loci.used[as.logical(geno.elim$err.locus)]] <- 1
       loc.err     <- rbind(loc.err,tmp)
       errors      <- errors + 1

       geno.elim.jk <- sge.jackknife(geno.elim)
       if(print.error)
         {
          # If print.error=TRUE, display the jackknife results of
          # pedigrees with Mendelian Errors:   
          msg <- paste("PEDIGREE ",
                       geno.elim$ped[1,pedigree.column],
                       ":\n",
                       sep="")
           cat(msg)
           print(geno.elim.jk)
         }
     }

   # Remove failure message since we have succeeded:
   on.exit()
  }

 # ret will hold the return value of mend.err():
 ret <- list()

 # Print summary tables if requested: 
 if(print.summary)
   {
    # If print.summary=TRUE, then display tables giving a summary of the
    # pedigrees with and without Mendelian Errors along with the
    # presence or absence of errors at each locus:
    cat("\n")
    msg <-paste("################################",
                "#########",
                "################################\n",
                "################################",
                " SUMMARY ",
                "################################\n",
                "################################",
                "#########",
                "################################\n\n",
                sep="")
    cat(msg)
    cat("SUMMARY <0 = No Mendelian Error, 1 = Mendelian Error>:\n\n")

    # Construct and print each table that is needed to display the results:
    old.options                   <- options(warn=-1)
    on.exit(options(old.options))
    ret$ped.no.err <- NULL
    if(!is.null(ped.no.err))
      {
       # Pedigrees without Mendelian Errors:
       summary.no.err            <- data.frame(cbind(ped.no.err,loc.no.err))
       labels.ped                <- list(1:nrow(summary.no.err),summary.names)
       dimnames(summary.no.err)  <- labels.ped
       cat("The following pedigrees had no Mendelian Errors:\n\n")
       print(summary.no.err)
       ret$ped.no.err            <- summary.no.err
      }
    ret$ped.err <- NULL
    if(!is.null(ped.err))
      {
       # Pedigrees with Mendelian Errors:        
       summary.err               <- data.frame(cbind(ped.err,loc.err))
       labels.ped                <- list(1:nrow(summary.err),summary.names)
       dimnames(summary.err)     <- labels.ped
       cat("\nThe following pedigrees had Mendelian Errors:\n\n")
       print(summary.err)
       ret$ped.err               <- summary.err
      }
    options(old.options)
    ret$ped.missing <- NULL
    if(!is.null(loc.missing))
      {
       # Pedigrees with no allele data:        
       summary.missing           <- data.frame(loc.missing)
       labels.ped                <- list(1:nrow(summary.missing),"Pedigree")
       dimnames(summary.missing) <- labels.ped
       cat("\nThe following pedigrees had no genotype data:\n\n")
       print(summary.missing)
       ret$ped.missing           <- summary.missing
      }
   }
 else
   { # Set up return object if summary not requested:
    old.options                   <- options(warn=-1)
    on.exit(options(old.options))
    ret$ped.no.err <- NULL
    if(!is.null(ped.no.err))
      {
       # Pedigrees without Mendelian Errors:
       summary.no.err            <- data.frame(cbind(ped.no.err,loc.no.err))
       labels.ped                <- list(1:nrow(summary.no.err),summary.names)
       dimnames(summary.no.err)  <- labels.ped
       ret$ped.no.err            <- summary.no.err
      }
    ret$ped.err <- NULL
    if(!is.null(ped.err))
      {
       # Pedigrees with Mendelian Errors:        
       summary.err               <- data.frame(cbind(ped.err,loc.err))
       labels.ped                <- list(1:nrow(summary.err),summary.names)
       dimnames(summary.err)     <- labels.ped
       ret$ped.err               <- summary.err
     }
    options(old.options)
    ret$ped.missing <- NULL
    if(!is.null(loc.missing))
      {
       # Pedigrees with no allele data:        
       summary.missing           <- data.frame(loc.missing)
       labels.ped                <- list(1:nrow(summary.missing),"Pedigree")
       dimnames(summary.missing) <- labels.ped
       ret$ped.missing           <- summary.missing
     }
   }

 # Print number of errors and number of pedigrees processed, if requested:
 total.pedigrees <- length(distinct.pedigrees) 
 if(print.num.processed)
   {
    # Display the total number of pedigrees with errors:
    if(errors == 0)
      cat("\n\nNo pedigrees contained Mendelian Errors.\n")
    else if(errors == 1)
      cat(paste("\n\nThere was ",errors,
                " pedigree with Mendelian Errors.\n",
                  sep=""))
    else
      cat(paste("\n\nThere were ",errors,
                " pedigrees with Mendelian Errors.\n",
                  sep=""))

    # Display the total number of pedigrees processed:
    if(total.pedigrees == 1)
       cat(paste("A total of ",
                 total.pedigrees,
                " pedigree was processed.\n\n",
                  sep=""))
    else
       cat(paste("A total of ",
                  total.pedigrees,
                " pedigrees were processed.\n\n",
                  sep=""))
   }
 
 ret$n.ped <- total.pedigrees
 ret$n.err <- errors

 if(exists("is.R") && is.function(is.R) && is.R())
    class(ret) <- "mend.err"   
 else
    oldClass(ret) <- "mend.err"


   return(ret)
}
