#Copyright 2004 Mayo Clinic College of Medicine.
#This software is accepted by users "as is" and without warranties or
#guarantees of any kind.  It may be used for research purposes or in
#relation to projects with commercial applications or included in commercial
#packages, but only so long as it is not relicensed as a stand-alone
#program, and only so long as the first two sentences of this paragraph
#(copyright notice and no warranty) are reproduced with the software.


#This software is free software; you can redistribute it and/or modify it
#under the terms of the GNU General Public License as published by the Free
#Software Foundation; either version 2 of the License, or (at your
#option) any later version.

#This program is distributed in the hope that it will be useful, but
#WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
#or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
#more details.


fastlo.bmo <- function(x, subset, maxit=3, mfun=NULL, log.it=TRUE,verbose=TRUE,
                        epsilon = 0.01, MM=FALSE, parallel=FALSE, ...) {

  #####
  # "quick help"
  #####
    if ( is.null(x) )
    {
      msg <- paste(
        "\n##########",
        "\n# fastlo()  - Version 1.1.0 - Summer 2004 (estimated)",
        "\n#   x       = BufferedMatrix (See documentation for BufferedMatrix package) of values",
        "\n#             (rows are 'probes' and columns are 'chips')",
        "\n#   subset  = Ignored - All rows used in normalization",
        "\n#   maxit   = maximum number of iterations (default is 3)",
        "\n#   mfun    = function to use for estimating yhat (default is 'yhat <- rowMeans(y - smooth)')",
        "\n#   log.it  = do we want to log the PM values before fitting them (default is TRUE)",
        "\n#   verbose = how much information to report back during process (default is TRUE)",
        "\n#   epsilon = convergence criteria (default is 0.01)",
        "\n##########\n",
        sep="")
      cat(msg)
      invisible(return(FALSE))
    }

    ## filename prefix assigned to temporary BufferedMatrix files
    tmp.prefix <- "G0bBleYg0oK";

    lc<-loess.control(surface="interpolate",statistics="approximate",
                      trace.hat="approximate",cell=0.2,iterations=4)

    if (!inherits(x, "BufferedMatrix")) {
  
      cat("x must be a matrix or BufferedMatrix\n")
      return()
    }
    y <- x
    
    ## PV dtemp <- dimnames(y)     #save the dimnames of y for later
    ## dimnames(y) <- NULL      # this speeds things up

    nchip <- ncol(y)
    nspot <- nrow(y)

    if (!missing(subset)) {
      cat("The subset argument is ignored by fastlo.bmo.\n",
          "\tNormalization will take place accross all rows.\n")
    }
    subset <- 1:nspot    
    if(log.it)  {
      log(y, base=2)
    }

    ##browser()
    
    # initialize matrices with same dimensions as of y but filled with zeros.
    ## NEED TO MAKE FileArrays for smooth and old.smooth (use clone method?)
##    smooth <- FileDoubleMatrix(filename=tempfile(),
##                               nrow=nspot, ncol=nchip, byrow=FALSE)
##    old.smooth <- FileDoubleMatrix(filename=tempfile(),
##                                   nrow=nspot, ncol=nchip, byrow=FALSE)

    tmpdir <- tempdir()
    smooth <- createBufferedMatrix(rows=nspot, cols=nchip, bufferrows=1, buffercols=1,prefix="BM",directory=tmpdir)
    old.smooth <- createBufferedMatrix(rows=nspot, cols=nchip, bufferrows=1, buffercols=1,prefix="BM",directory=tmpdir)
    
    w <- c(0,rep(1,length(subset)),0) ##weights for loess fit
    cl <- NULL

    if (parallel && require(snow) && require(rpvm)) {
	system("pvm")

	# How do I get out of pvm?  Following doesn't work
	# system("quit")
	cl <- makeCluster(nchip)

    }
    
    for (iter in 1:maxit) {
        if(verbose) cat("Iteration", iter, "Chip")

	# TODO: parallelize rowMeans as well?
        if (!is.null(mfun)) {
          cat("Warning:  mfun argument not supported by fastlo.huge.\n",
              "The rowMeans function will be used instead.\n")
        }

        ## Should be eq to yhat <- rowMeans(y - smooth)
        ## PV If we add 2 extra elements, make sure they're not in rowMeans
        y.clone <- duplicate(y, prefix=tmp.prefix)
        yhat <- rowMeans(fmCapply(y.clone, '-', smooth))  # this is faster
        
        ## Need to delete y.clone, but rm() doesn't delete underlying files
        rm (y.clone)
        junk.files <- list.files(path=tmpdir, pattern=paste(tmp.prefix, "*", sep=""))
        file.remove(junk.files)
        
        # To avoid NA's due to extrapolation, the lowess has to use
        #   include the largest and smallest yhat
        temp <- order(yhat)

        ## PV Do we need a FileMatrix with 2 more rows?  What about smooth and old.smooth?
        index <- c(temp[1], subset, temp[nspot])

        #browser()
        
	#####
	# Here's where we're going to try to go parallel.
	# We need to wrap the loess model code in a new function
	# that we can pass to parCapply().  Then we can do each
	# loess fit on a separate node in the cluster.
	#####

	if (parallel && require(snow) && require(rpvm)) {

	    smooth <- matrix(data=parCapply(cl, y[index,], loess.fastlo, yhat=yhat[index], degree=1, weights=w, control=lc), ncol=nchip)

	}
	else {
            for(j in 1:nchip) {
              thisCol <- y[,j]
              smooth[,j] <- loess.fastlo(y=thisCol[index],
                                         yhat=yhat[index], degree=1,
                                         weights=w, control=lc)
              if (verbose) cat(" ", j, "\n")               
            }
 	}	

        ## PV How to handle this?  smooth[subset,] evaluates to matrix.
        ## old.smooth?  matrix or FileMatrix?  Just dump subset?

        ## Doesn't work -- do in multiple steps
        ## PV CAN'T CHANGE THE VALUE OF SMOOTH -- need another (temp) FileMatrix

	temp.fm <- duplicate(smooth, prefix=tmp.prefix)

        fmCapply(temp.fm, '-', old.smooth)

        ## This does smooth ^ 2, but we have to give 2 any arg name but 'y'
        fmCapply(temp.fm, '^', blah=2)  
        #change <- max(colMeans(fmCapply(smooth, '-', old.smooth)^2))

	## PV colMeans undefined for FileMatrix objects!
	## Define one?  Also define apply more robustly?  See source code
        change <- max(colMeans(temp.fm))

        ## Need to delete tmp.fm, but rm() doesn't delete underlying files
        rm (temp.fm)
        junk.files <- list.files(path=tmpdir, pattern=paste(tmp.prefix, "*", sep=""))
        file.remove(junk.files)
        
	## Copy smooth to old.smooth
        ## old.smooth <- clone(smooth) ## Too many files floating around
	##fmCapply(old.smooth, '<-', smooth)

        old.smooth <- duplicate(smooth)
        ##smooth <- temp.fm # Try recycling this FileMatrix
        
        if (verbose) cat("\n   Finished, change = ", format(change),"\n") 
        if (change <= epsilon) break
      } ## end for iter

    if(!is.null(cl)) {
      stopCluster(cl)
      system("rm -f /tmp/pvm*")
    }
    if (verbose) cat("\n")

    fmCapply(y, '-', smooth)    # normalized y

    ## PV what to do about dimnames?
    ##dimnames(ynorm) <- dtemp
    if(log.it) {
      ewApply(y, FUN=expb, base=2)
    }
    invisible(y)
  }


loess.fastlo <- function (y, yhat, degree=NULL, weights=NULL, control=NULL) {

    tfit <- loess(y - yhat ~ yhat, degree=degree, weights=weights, control=control)
    smooth <- predict(tfit, yhat[2:( length(yhat) - 1 )])

    return(smooth)
}
