/* $Author: schaid $ */
/* $Date: 2004/10/08 21:01:26 $ */
/* $Header: /people/biostat3/sinnwell/Projects/arp.gee/Make/RCS/geeArp.c,v 1.2 2004/10/08 21:01:26 schaid Exp $ */
/* $Locker:  $ */
/* $Log: geeArp.c,v $
 * Revision 1.2  2004/10/08 21:01:26  schaid
 * changed lower bound for lambda from .333 to 0.0 and added some comments
 *
 * Revision 1.1  2004/10/08 15:43:43  sinnwell
 * Initial revision
 * * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <S.h>

#define EPSILON .000000001    

/********************************************************************************************

   Jason Sinnwell & Dan Schaid   
                            
  Important: smat from S must be sorted according to type of ARP

*********************************************************************************************/


static void arp_func(long type, double pos, double *gamma, long n_gamma, double eps, double *mu,
		     double *var, double *deriv, long indx_c);
static void arp_lambda(long type, double pos, double *gamma, double eps, double *mu,
		       double *var, double *deriv);
static void dMat_to_Vec(double **Ymat, double *Yvec, long nr, long nc);
static double **dMatrix(long nr, long nc);
static double **dVec_to_Mat(double *Yvec, long nr, long nc);
static int cholesky2(double **matrix, int n);
static void chsolve2(double **matrix, int n, double *y);
static void printdVec(double *x, long n, char *label);
static void printdMat(double **x, long nr, long nc, char *label);

void gee_arp(
   double  *upos,         /* unique positions, sorted vector                 */
   long    *n_pos,        /* number unique positions                         */
   double  *Svec,         /* measured allele sharing                         */
   long    *n_pair,       /* number relative pairs in Smat                   */
   long    *pedIndx,      /* vec, integer indices for peds                   */
   long    *nPed,         /* number unique pedigrees                         */
   long    *type,         /* unique ARP types: 1=FS, 2=HS, 3=FC, 4=GP, 5=AV  */ 
   long    *len_type,     /* length  type vector                             */
   long    *n_type,       /* numbers of pairs for each ARP type              */

   /* control parameters for N-R methods                                     */
   long    *iter,         /* number of completed iterations                  */
   long    *max_iter,     /* max iterations                                  */
   double  *tol,          /* tolerance for deciding convergence              */
   double  *eps,          /* parameter from Liang for 'd'                    */

   /* Results                                                                */
   double  *u_scores,      /* final score vector, near 0 if converged        */
   double  *gamma,         /* tau, followed by C coefs for different ARPs    */
   long    *n_gamma,       /* number of parameters, length of  gamma         */
   double  *info_vec,      /* vector form of information matrix              */
   double  *info_robust_vec, /* robust information matrix                    */
   long    *rank,           /* rank of information matrix                    */   
   long    *converge,       /* convergence status                            */
   long    *alias,          /* vector to indicate if a param is aliased      */
                            /* 1 if aliased with another param, 0 o.w.       */
   long    *verbose         /* 1 to print within  iter loop, 0 if no print   */
   ) {

  double pos_min, pos_max, score_max, ratio;

  double *gamma_new, *deriv, *deriv2, *u_scores_ch, *coef_limit;
  long *stop_pair, *start_pair;

  double **Smat, **info_mat, **info_mat_ch, **info_robust_mat;
  double **u_mat;

  double diff_sum, mu_p, var_p, maxDelta, maxU, mu1, mu2, var1, var2;
  long i, k,k1,k2, t, p1, p2, begin, end;  

  double coef_low[5] = {-1.0, -0.5, -0.25, -0.5, -0.5};
  double coef_up[5]  = { 1.0,  0.5,  0.75,  0.5,  0.5};


  /* allocate arrays */
  gamma_new = (double *) Calloc((*n_gamma), double);
  deriv = (double *) Calloc((*n_gamma), double);
  deriv2 = (double *) Calloc((*n_gamma), double);
  stop_pair = (long *) Calloc(*len_type, long);
  start_pair = (long *) Calloc(*len_type, long);
  u_scores_ch = (double *) Calloc(*n_gamma, double);


   /* pos limits needed in N-R to keep tau in range */
  pos_min = upos[0];
  pos_max = upos[*n_pos-1];
  *converge = 0;

  /* fill vectors that keep track of types, and the Smat rows where they start and stop */
  start_pair[0] = 0;
  stop_pair[0] = n_type[0];
 
  for(i=1; i<(*len_type); i++) {
    stop_pair[i] = stop_pair[i-1] + n_type[i];
    start_pair[i] = stop_pair[i-1];
  }


 
  /* create matrices */
  Smat = dVec_to_Mat(Svec, *n_pair, *n_pos);
  info_mat = dVec_to_Mat(info_vec, (*n_gamma), (*n_gamma));
  info_mat_ch = dVec_to_Mat(info_vec, (*n_gamma), (*n_gamma));
  info_robust_mat = dVec_to_Mat(info_robust_vec, (*n_gamma), (*n_gamma));
  u_mat = dMatrix(*nPed, *n_gamma);


  for(p1=0; p1<(*n_gamma); p1++){
    alias[p1] = 0;
  }

  *iter = 0;
  while(*iter <= *max_iter) {    /* all N-R steps within while */

    (*iter)++;
   
    for(p1=0; p1<(*n_gamma); p1++) {  
      /* zero out summed terms, these will be re-computed during N-R step */
      /* make duplicates (_ch) of these that will be used in cholesky steps */
      u_scores[p1] = 0.0;
      u_scores_ch[p1] = 0.0;
      for(p2=0; p2<(*n_gamma); p2++) {
	info_mat[p1][p2] = 0.0;
        info_mat_ch[p1][p2] = 0.0;
      }

      for(t=0; t<(*len_type); t++) {
	
	for(k=0; k<(*n_pos); k++) {

	  var_p = mu_p = 0.0;

	  arp_func(type[t], upos[k], gamma, *n_gamma, *eps, &mu_p, &var_p, deriv, (t+1) );
	 
	  begin = start_pair[t];
	  end = stop_pair[t];
	  diff_sum = 0.0;
	  for(i=begin; i<end; i++) {
	    diff_sum += Smat[i][k] - mu_p;
	  }  /* end for i */

	  u_scores[p1] += (deriv[p1]/var_p) * diff_sum;

	  for(p2=0; p2<(*n_gamma); p2++){

            if(p1 <= p2)
              {
                info_mat[p1][p2] += n_type[t] * (deriv[p1]/var_p)*deriv[p2];
              }
            else
	      {
                info_mat[p1][p2] = info_mat[p2][p1];
	      }
	  }  /* end for p2 */

	} /* end for k in upos */
      } /* end for t in type */
    } /* end p1 in parameters */

 
  
    /* do N-R updates using info matrix and distance */

    for(p1=0; p1 < (*n_gamma); p1++){
      u_scores_ch[p1] = u_scores[p1];
      for(p2=0; p2<(*n_gamma); p2++){
        info_mat_ch[p1][p2] = info_mat[p1][p2];
      }
    }

    
    *rank = cholesky2(info_mat_ch, ((int) *n_gamma));

    if(*rank < *n_gamma){
      for(p1=0; p1< (*n_gamma); p1++){
        if( fabs(info_mat_ch[p1][p1]) < EPSILON) alias[p1] = 1;
      }
    }

 
    chsolve2(info_mat_ch, ((int) *n_gamma), u_scores_ch);

    /* set new values to current, and do another step */
    for(i=0; i<(*n_gamma); i++) {
      gamma_new[i] = gamma[i] + u_scores_ch[i];
    }


    if((*verbose)==1){
      printf("\nIter = %i\n", *iter);
      printdVec(u_scores,(*n_gamma), "u_scores");
      printdVec(gamma_new,(*n_gamma),"gamma");
      printdMat(info_mat,(*n_gamma), (*n_gamma), "info_mat");
    }    

    /* checks on N-R steps, out of range, or meet converge criteria */
  
    if((gamma_new[0] < pos_min) || (gamma_new[0] > pos_max)) {
      *converge = -1;
      break;
    }
  
    for(i=1; i<(*n_gamma); i++) {
      if((gamma_new[i] < coef_low[i-1]) || (gamma_new[i] > coef_up[i-1])){
        *converge = -2;
        break;
      } 
    }

  
    maxDelta = 0.0;
    maxU = 0.0;
    for(i=0; i<(*n_gamma); i++){
      if(fabs(u_scores_ch[i]) > maxDelta) maxDelta = fabs(u_scores_ch[i]) ;
      if(fabs(u_scores[i]) > maxU) maxU = fabs(u_scores[i]);
    }
      
    if( (maxDelta < *tol) && (maxU < *tol)) {
       *converge = 1;
       break;
    }
  
    if( (*max_iter) == 0){
      break;
    }

    for(i=0; i<(*n_gamma); i++) {
      gamma[i] = gamma_new[i];
    }

  } /* end while  N-R step*/

  /* info_robust_mat */


   for(p1=0; p1<(*n_gamma); p1++) {  
      for(t=0; t<(*len_type); t++) {	
	for(k=0; k<(*n_pos); k++) {

	  var_p = mu_p = 0.0;
	  arp_func(type[t], upos[k], gamma, *n_gamma, *eps, &mu_p, &var_p, deriv, (t+1) );
	 
	  begin = start_pair[t];
	  end = stop_pair[t];
	  ratio = (deriv[p1]/var_p);

	  for(i=begin; i<end; i++) {
	    u_mat[pedIndx[i]][p1] +=  ratio * (Smat[i][k] - mu_p);
	  }
	}
      }
   }
	 

  for(i=0; i<(*nPed); i++){
    for(p1=0; p1<(*n_gamma); p1++){
      for(p2=0; p2<(*n_gamma); p2++){
        info_robust_mat[p1][p2] += u_mat[i][p1] * u_mat[i][p2];
      }
    }
  }


 
  /* copy new parameters to vector locations to be passed back */
  dMat_to_Vec(Smat, Svec, *n_pair, *n_pos);
  dMat_to_Vec(info_mat, info_vec, (*n_gamma), (*n_gamma));
  dMat_to_Vec(info_robust_mat, info_robust_vec, (*n_gamma), (*n_gamma));

 /* -----------Free C-allocated space---------------- */
 
 
  /* matrices */
  for(i=0; i<(*n_pair); i++) {
    Free(Smat[i]);
  }
  for(i=0;i<(*nPed);i++){
    Free(u_mat[i]);
  }
  for(i=0; i<((*n_gamma)); i++) {
    Free(info_mat_ch[i]);
    Free(info_mat[i]);
    Free(info_robust_mat[i]);
  }
  
  /* vectors */
  Free(Smat);
  Free(u_mat);
  Free(info_mat);
  Free(info_mat_ch);
  Free(info_robust_mat);

  Free(deriv);
  Free(deriv2);
  Free(start_pair);
  Free(stop_pair);
  Free(gamma_new);
  Free(u_scores_ch);  

}

/**************************************************************************************************************/
void gee_arp_lambda(
		    double  *upos,         /* unique positions, sorted vector        */
		    long    *n_pos,        /* number of unique positions      */
		    double  *Svec,         /* measured allele sharing  2*p(s=2) + p(s=1)   */
		    long    *n_pair,       /* number of relative pairs in Smat */
		    long    *pedIndx,      /* vec of integer indices for peds (ped id, with 0-offset) */
		    long    *nPed,         /*  number of unique pedigrees      */
		    long    *type,         /* unique Aff.Rel.Pair types 1:sibs 2:half-sibs, etc. */ 
		    long    *len_type,     /* length of type vectors */
		    long    *n_type,       /* numbers of pairs for each ARP type */
		    
		    /* control parameters for N-R methods */
		    long    *iter,         /* iterations used in optimization */
		    long    *max_iter,     /* max iterations for while loop   */
		    double  *tol,          /* N-R parameter change stopping rule */
		    double  *eps,      /* parameter from Liang for 'd' close to 0 */
		    
		    /* elements of N-R calculations */
		    double  *u_scores,      /* final score vector, ~=zero when complete */
		    double  *gamma,         /* tau and C coefs for different ARPs    */
		    double  *info_vec,      /* vector form of information matrix */
		    double  *info_robust_vec,   /* robust info mtx for var(gamma)  */
		    long    *rank,
		    long    *converge,
                    long    *alias,
                    long    *verbose      /* controls printing within iter looop  */   
		    ) {
  
  double pos_min, pos_max, ratio;
  
  double *gamma_new, *deriv, *deriv2, *u_scores_ch;
  long *stop_pair, *start_pair;
  
  double **Smat, **info_mat, **info_mat_ch, **info_robust_mat, **var_gamma_mat, **u_mat;
  
  double diff_sum, mu_p, var_p, maxDelta, maxU, mu1, mu2, var1, var2;
  long i, k, k1,k2, t, p1, p2, begin, end;  /* k:positions, t:types, p1,p2:parameters */
  
  /* Define limits for lambda based on these limites for C's
  ** double coef_low[5] = {-1.0, -0.5, -0.25, -0.5, -0.5}; */
  /* double coef_up[5]  = { 1.0,  0.5,  0.75,  0.5,  0.5}; */ 
  
  /* allocate vectors */
  gamma_new = (double *) Calloc(2, double);
  deriv = (double *) Calloc(2, double);
  deriv2 = (double *) Calloc(2, double);
  stop_pair = (long *) Calloc(*len_type, long);
  start_pair = (long *) Calloc(*len_type, long);
  u_scores_ch = (double *) Calloc(2, double);

  /* pos limits needed in N-R to keep tau in range */
  pos_min = upos[0];
  pos_max = upos[*n_pos-1];
  *converge = 0;

  /* fill vectors that keep track of types, and where Smat rows start and stop */
  start_pair[0] = 0;
  stop_pair[0] = n_type[0];

  for(i=1; i<(*len_type); i++) {
    stop_pair[i] = stop_pair[i-1] + n_type[i];
    start_pair[i] = stop_pair[i-1];
  }
  
  
  /* create matrices */
  Smat = dVec_to_Mat(Svec, *n_pair, *n_pos);
  info_mat = dVec_to_Mat(info_vec, 2, 2);
  info_mat_ch = dVec_to_Mat(info_vec, 2, 2);
  info_robust_mat = dVec_to_Mat(info_robust_vec, 2, 2);
  u_mat = dMatrix(*nPed, 2);
   

  for(p1=0; p1< 2; p1++){
    alias[p1] = 0;
  }


  *iter = 0;

  while(*iter <= *max_iter) {    /* all N-R steps within while */
    
    (*iter)++;
    
    for(p1=0; p1<2; p1++) {
      /* zero out summed terms, these will be re-computed during N-R step */
      /* make duplicates (_ch) of these that will be used in cholesky steps */
      u_scores[p1] = 0.0;
      u_scores_ch[p1] = 0.0;
      for(p2=0; p2<2; p2++) {
	info_mat[p1][p2] = 0.0;
        info_mat_ch[p1][p2] = 0.0;
      }
      
      for(t=0; t<(*len_type); t++) {
	
	for(k=0; k<(*n_pos); k++) {
	  var_p = mu_p = 0.0;
	  
	  arp_lambda(type[t], upos[k], gamma, *eps, &mu_p, &var_p, deriv);
	  
	  begin = start_pair[t];
	  end = stop_pair[t];
	  diff_sum = 0.0;
	  for(i=begin; i<end; i++) {
	    diff_sum += Smat[i][k] - mu_p;
	  }  /* end for i */

	  /* printf("deriv[%i] = %f\n",p1,deriv[p1]); */
	  /* printf("mu, var = %f, %f\n", mu_p, var_p); */


	  u_scores[p1] += (deriv[p1]/var_p) * diff_sum;

	  for(p2=0; p2<2; p2++){

            if(p1 <= p2)
              {
                info_mat[p1][p2] += n_type[t] * (deriv[p1]/var_p)*deriv[p2];
              }
            else
	      {
                info_mat[p1][p2] = info_mat[p2][p1];
	      }
	  }  /* end for p2 */

	} /* end for k in upos */
      } /* end for t in type */
    } /* end p1 in parameters */

  
    /* do N-R updates using info matrix and distance */

    for(p1=0; p1 < 2; p1++){
      u_scores_ch[p1] = u_scores[p1];
      for(p2=0; p2< 2; p2++){
        info_mat_ch[p1][p2] = info_mat[p1][p2];
      }
    }

    
    *rank = cholesky2(info_mat_ch, 2);
    
   if(*rank < 2 ){
      for(p1=0; p1< 2; p1++){
        if( fabs(info_mat_ch[p1][p1]) < EPSILON) alias[p1] = 1;
      }
    }
    
    chsolve2(info_mat_ch, 2, u_scores_ch);
    
    /* set new values to current, and do another step */
    for(i=0; i<2; i++) {
      gamma_new[i] = gamma[i] + u_scores_ch[i];
    }

    if((*verbose)==1){
      printf("\nIter = %i\n", *iter);
      printdVec(u_scores,2, "u_scores");
      printdVec(gamma_new,2,"gamma");
      printdMat(info_mat,2, 2, "info_mat");
    }
    
    
    /* checks on N-R steps, out of range, or meet converge criteria */
    
    if((gamma_new[0] < pos_min) || (gamma_new[0] > pos_max)) {
      *converge = -1;
      break;
    }

    if(gamma_new[1] < 0.0) {
      *converge = -2;
      break;
    }

  
    maxDelta = 0.0;
    maxU = 0.0;
    for(i=0; i<2; i++){
      if(fabs(u_scores_ch[i]) > maxDelta) maxDelta = fabs(u_scores_ch[i]) ;
      if(fabs(u_scores[i]) > maxU) maxU = fabs(u_scores[i]);
    }
    
    if( (maxDelta < *tol) && (maxU < *tol)) {
      *converge = 1;
      break;
    }

   if( (*max_iter) == 0){
      break;
    }

    
    for(i=0; i<2; i++) {
      gamma[i] = gamma_new[i];
    }
    
  } /* end while  N-R step*/
  

  /* fill info_robust_mat */
   
  for(p1=0; p1<(2); p1++) {  
    for(t=0; t<(*len_type); t++) {	
      for(k=0; k<(*n_pos); k++) {
	
	var_p = mu_p = 0.0;
	arp_lambda(type[t], upos[k], gamma, *eps, &mu_p, &var_p, deriv);
	
	begin = start_pair[t];
	end = stop_pair[t];
	ratio = (deriv[p1]/var_p);
	
	for(i=begin; i<end; i++) {
	  u_mat[pedIndx[i]][p1] +=  ratio * (Smat[i][k] - mu_p);
	}
      }
    }
  }
  

 
  for(i=0; i < (*nPed); i++){
    for(p1=0; p1 < 2; p1++){
      for(p2=0; p2<2; p2++){
        info_robust_mat[p1][p2] += u_mat[i][p1] * u_mat[i][p2];
      }
    }
  }

  /* copy new parameters to vector locations to be passed back */
  dMat_to_Vec(Smat, Svec, *n_pair, *n_pos);
  dMat_to_Vec(info_mat, info_vec, (2), (2));
  dMat_to_Vec(info_robust_mat, info_robust_vec, (2), (2));
  
  /* -----------Free C-allocated space---------------- */
  
  
  /* matrices */
  for(i=0; i<(*n_pair); i++) {
    Free(Smat[i]);
  }
  for(i=0;i<(*nPed);i++){
    Free(u_mat[i]);
  }
  for(i=0; i< 2; i++) {
    Free(info_mat_ch[i]);
    Free(info_mat[i]);
    Free(info_robust_mat[i]);
    
  }
  
  /* vectors */
  Free(Smat);
  Free(u_mat);
  Free(info_mat);
  Free(info_mat_ch);
  Free(info_robust_mat);
  
  Free(deriv);
  Free(deriv2);
  Free(start_pair);
  Free(stop_pair);
  Free(gamma_new);
  Free(u_scores_ch);
  
}

/*****************************************************************/
static void arp_func(long type,
		     double pos,
		     double *gamma, 
		     long n_gamma, 
		     double eps,
		     double *mu, 
		     double *var, 
		     double *deriv, 
		     long indx_c) {

  double tau, dist, delta;
  double a, b, c, f;
  long i;

  for(i=0; i< n_gamma; i++){
    deriv[i] = 0.0;
  }

  tau = gamma[0];

  dist = pos - tau;

  if(fabs(dist) <= eps) 
   {
    delta = (dist*dist)/(2.0*eps) + eps/2.0;
   } 
  else 
   { 
    delta = fabs(dist); 
   }
 

 
  if(dist > eps) 
   { 
     f = -1.0;
   }
  else if (-dist > eps)
   {
      f = 1.0;
   }
  else 
   { 
    f = -dist/eps;
   }
  

  switch(type) {
  case 1:    /* full sibs */
    a=1.0;
    b=exp(-.04*delta);
    c=gamma[indx_c];
    *mu = a + b*c;
    *var = 0.5 - exp(-.08*delta)*(c*c); 
    deriv[0] = c*b*f*(-.04);
    deriv[indx_c] = b;
    break;

  case 2:   /* half-sibs */
    a=0.5;
    b=exp(-.04*delta);
    c=gamma[indx_c];
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c*b*f*(-.04);
    deriv[indx_c] = b;
    break;

  case 3:    /* first cousins */

    a=0.25;
    b=0.5*exp(-.04*delta) + 0.33333*exp(-.06*delta) + 0.16666*exp(-.08*delta);
    c=gamma[indx_c];
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c * (0.5*exp(-.04*delta)*f*(-.04)+.33333*exp(-.06*delta)*f*(-.06)+.16666*exp(-.08*delta)*f*(-.08));
    deriv[indx_c] = b;
    break;

  case 4:    /* grandparent - grandchild */
    a = 0.5;
    b = exp(-.02*delta);
    c = gamma[indx_c];
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c*b*f*(-.02);
    deriv[indx_c] = b;
    break;

  case 5:    /* uncle-nephew */
    a = 0.5;
    b = 0.5*exp(-.04*delta) + 0.5*exp(-.06*delta);
    c = gamma[indx_c];
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c* (0.5*exp(-.04*delta)*f*(-.04)+(0.5)*exp(-.06*delta)*f*(-.06));
    deriv[indx_c] = b;
    break;
  }


}
/*****************************************************************/
static void arp_lambda(long type,
		       double pos,
		       double *gamma,
		       double eps,
		       double *mu,
		       double *var,
		       double *deriv) {

  double tau, lambda, dist, delta;
  double a, b, c, f;
  long i;


  tau = gamma[0];
  lambda = gamma[1];
  dist = pos - tau;

  if(fabs(dist) <= eps)
   {
    delta = (dist*dist)/(2.0*eps) + eps/2.0;
   } 
  else 
   { 
    delta = fabs(dist); 
   }

 
  if(dist > eps) 
   { 
     f = -1.0;
   }
  else if (-dist > eps)
   {
      f = 1.0;
   }
  else 
   { 
    f = -dist/eps;
   }
  

  switch(type) {
  case 1:    /* full sibs */
    a = 1.0;
    b = exp(-.04*delta);
    c = (lambda-1.0)/(2.0*lambda);
    *mu = a + b*c;
    *var = 0.5 - exp(-.08*delta)*(c*c);
    deriv[0] = c*b*f*(-.04);
    deriv[1] = b/(2.0*lambda*lambda);
    break;

  case 2:   /* half-sibs */
    a=0.5;
    b=exp(-.04*delta);
    c =  (lambda-1.0)/(2.0*(lambda+1));
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c*b*f*(-.04);
    deriv[1] = b/((lambda+1.0)*(lambda+1.0));
    break;

  case 3:    /* first cousins */

    a = 0.25;
    b = 0.5*exp(-.04*delta) + 0.33333*exp(-.06*delta) + 0.16666*exp(-.08*delta);
    c = 3.0*(lambda-1.0)/(4.0*(lambda+3.0));
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c * (0.5*exp(-.04*delta)*f*(-.04)+.33333*exp(-.06*delta)*f*(-.06)+.16666*exp(-.08*delta)*f*(-.08));
    deriv[1] = 3.0*b/((lambda+3.0)*(lambda+3.0));
    break;

  case 4:    /* grandparent - grandchild */
    a = 0.5;
    b = exp(-.02*delta);
    c = (lambda-1.0)/(2.0*(lambda+1.0));
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c*b*f*(-.02);
    deriv[1] = b/((lambda+1.0)*(lambda+1.0));
    break;

  case 5:    /* uncle-nephew */
    a = 0.5;
    b = 0.5*exp(-.04*delta) + 0.5*exp(-.06*delta);
    c = (lambda-1.0)/(2.0*(lambda+1.0));
    *mu = a + b*c;
    *var = (*mu)*(1.0 - (*mu));
    deriv[0] = c* (0.5*exp(-.04*delta)*f*(-.04)+(0.5)*exp(-.06*delta)*f*(-.06));
    deriv[1] = b/((lambda+1.0)*(lambda+1.0));
    break;
  }

}

/***********************************************************/
static void dMat_to_Vec(double **Ymat, 
			double *Yvec,
			long nr, 
			long nc) {
  long i, j, k;
  k=0;
  for(j=0; j<nc; j++) {
    for(i=0; i<nr; i++) {
      Yvec[k] = Ymat[i][j];
      k++;
    }
  }
}

/************************************************/
static double **dMatrix(long nr, 
			long nc) {
     long i;
     double **mat;

     mat = (double **) Calloc(nr, double *);
     if(!mat) printf("Error, memory allocation failed in dmatrix.\n");
     
     for(i=0; i<nr; i++) {
       mat[i] = (double *) Calloc(nc, double);
       if(!mat[i]) printf("Error, memory allocation failed in dmatrix.\n");
     }
     return mat;
}

/***********************************************/
static double **dVec_to_Mat(double *Yvec, 
			    long nr, 
			    long nc) {

  long i,j,k;
  double **Y;
  
  Y = dMatrix(nr,nc);
  k=0;
  for (j=0;j<nc;j++){
    for (i=0;i<nr;i++){
      Y[i][j]=Yvec[k];
      k++;
    }
  }
  return Y;
}


/*****************************************************************/
static void chsolve2(double **matrix, int n, double *y) 
     {
     register int i,j;
     register double temp;

     /*
     ** solve Fb =y
     */
     for (i=0; i<n; i++) {
          temp = y[i] ;
          for (j=0; j<i; j++)
               temp -= y[j] * matrix[i][j] ;
          y[i] = temp ;
          }
     /*
     ** solve DF'z =b
     */
     for (i=(n-1); i>=0; i--) {
          if (matrix[i][i]==0)  y[i] =0;
          else {
              temp = y[i]/matrix[i][i];
              for (j= i+1; j<n; j++)
                   temp -= y[j]*matrix[j][i];
              y[i] = temp;
              }
          }
     }

/*************************************************************/
/*#define EPSILON .000000001     /* <= EPS is considered a zero */  

static int cholesky2(double **matrix, int n)
    {
    register double temp;
    register int  i,j,k;
    double eps, pivot;
    int rank;

    eps =0;
    for (i=0; i<n; i++) {
        if (matrix[i][i] > eps)  eps = matrix[i][i];
        for (j=(i+1); j<n; j++)  matrix[j][i] = matrix[i][j];
        }
    eps *= EPSILON;

    rank =0;
    for (i=0; i<n; i++) {
        pivot = matrix[i][i];
        if (pivot < eps) matrix[i][i] =0;
        else  {
            rank++;
            for (j=(i+1); j<n; j++) {
                temp = matrix[j][i]/pivot;
                matrix[j][i] = temp;
                matrix[j][j] -= temp*temp*pivot;
                for (k=(j+1); k<n; k++) matrix[k][j] -= temp*matrix[k][i];
                }
            }
        }
    return(rank);
    }


/*********************************************************************************/
static void printdVec(double *x, long n, char *label){
  long i;

  printf("\n");
  for(i=0; i<n; i++){
    printf("%s[%i] = %f\n",label,i,x[i]);
  }

}
/*********************************************************************************/
static void printdMat(double **x, long nr, long nc, char *label){
  long i,j;

  printf("\n");
  for(i=0; i<nr; i++){
    printf("%s[%i][0,..,%i] = ",label,i,(nc-1));
    for(j=0; j<nc; j++){
      printf("%12.6f, ",x[i][j]);
    }
    printf("\n");

  }

}
