#$Author: sinnwell $
#$Date: 2007/10/16 21:35:45 $
#$Header: /people/biostat3/sinnwell/genet/rpack/Armitage/RCS/armitage.q,v 1.2 2007/10/16 21:35:45 sinnwell Exp $
#$Locker:  $
#$Log: armitage.q,v $
#Revision 1.2  2007/10/16 21:35:45  sinnwell
#put PACKAGE= in .C call
#
#Revision 1.1  2007/10/15 19:33:18  sinnwell
#Initial revision
#

armitage <- function(status, genoMatrix, snpLabels = NULL, dose=0:2, nsim=0, seed=NULL){


  # no missing values for status (coded as 0 for controls,
  # 1 for cases), but allow mising values for genoMatrix

  # genotypes must be coded 0,1,2, and NA for missing

  if(length(dose)!=3)
    {
      stop("length of dose !=3")
    }
  
  if(is.matrix(genoMatrix))
    {
      ngeno <- ncol(genoMatrix)
      nsubj <-  nrow(genoMatrix)
    } else
  {
    ngeno <- 1
    nsubj <- length(genoMatrix)
  }

  if(length(status) != nsubj){
    stop("Dim of status and genoMatrix inconsistent")
  }
  
  if(any(is.na(status))){
    stop("Missing values for status not allowed")
  }

  if( !all(status ==0 | status ==1)){
    stop("status should be 0 or 1")
  }
  
 
  gvec <- as.vector(genoMatrix)
  zed <- !is.na(gvec)
  gcheck <- gvec[zed]
  
  if( !all(gcheck==0 | gcheck ==1 | gcheck ==2) )
    {
      stop("genotype codes not 0, 1, or 2")
    }


  if(is.null(snpLabels))
    {
      if(is.null(dimnames(genoMatrix)[[2]]))
        {
          snpLabels <- paste("snp", 1:ncol(genoMatrix), sep="-")
        } else
      {
        snpLabels <- dimnames(genoMatrix)[[2]]
      }
    }

  if(length(snpLabels) != ncol(genoMatrix))
    {
      stop("Number of snpLabels != number of cols of genoMatrix")
    }
  

  genoNumeric <- as.numeric(gvec)

  genoNumeric <- ifelse(is.na(genoNumeric), 3, genoNumeric)
  
  # note: Now code SNP genotypes as 0,1,2 with missing coded as 3

 
  if(is.null(seed))
    {
      seed <- sample(2^20, size=1)
    }


  
  save <- .C("armitage",
             nsubj = as.integer(nsubj),
             ngeno = as.integer(ngeno),
             status = as.integer(status),
             genovec  = as.integer(genoNumeric),
             dose = as.integer(dose),
             seed = as.integer(seed),
             nsim = as.integer(nsim),
             statObs = as.double(numeric(ngeno)),
             statSim = as.double(numeric(nsim * ngeno)),
             PACKAGE="armitage")

  save$nsubj <- table(factor(status,labels=c('Controls','Cases')))
  save$snpLabels = snpLabels
  save$status <- NULL
  save$genovec <- NULL
  save$seed <- seed
  save$statSim <-  matrix(save$statSim, ncol=ngeno, byrow=TRUE)

  if(exists("is.R") && is.function(is.R) && is.R()) {
     class(save) <- "armitage"
   } else {
     oldClass(save) <- "armitage"
   }
  
  return(save)
}

print.armitage <- function(x, limit.snpLabel = 10, digits=max(options()$digits -2,5), ...) {

  if(!inherits(x, "armitage"))
    stop("Not legitimate armitage object")

  if(!is.matrix(x$nsubj)) {
    cat("No. Controls = ",x$nsubj[1],"\n")
    cat("No. Cases    = ",x$nsubj[2],"\n\n")
  } else {
    cat("Object Labels= ",dimnames(x$nsubj)[[1]],"\n")
    cat("No. Controls = ",x$nsubj[,1],"\n")
    cat("No. Cases    = ",x$nsubj[,2],"\n")
  }
    
  cat("No. Markers  = ",x$ngeno,"\n")
  cat("Marker Labels:\n")
  if(length(x$snpLabels) > limit.snpLabel) {
    print(c(x$snpLabels[1:limit.snpLabel],"..."))
  } else {
    print(x$snpLabels)
  }
  cat("\n")
  cat("No. Simulations = ",x$nsim,"\n")
  cat("Summary of Observed Statistics:\n")
  print(round(summary(x$statObs),6))
  
  invisible()
}

  
summary.armitage <- function(object){
  pval.asymp <- 1-pchisq(object$statObs, 1)
  df <- data.frame(snp=object$snpLabels,
                   chistat=object$statObs,
                   pval.asymp = pval.asymp)
  if(object$nsim==0)
    {
      return(df)
    }
                  

  # correct pvals for multiple testing by method described on page 75
  # of Westfall et a., "Multiple comparisons and multiple tests
  # using SAS"

  # corrected p-values are Pr(max sim stat >= obs stat) for
  # each obs stat

  simMax <- apply(object$statSim, 1, max)
  tmp <- outer(simMax, object$statObs, ">=")
  pval.corrected <- apply(tmp, 2, sum)/object$nsim
  
  pval.perm  <- apply(t(object$statSim) >= object$statObs, 1, sum)/object$nsim

  df <- cbind(df, pval.perm  = pval.perm,
              pval.corrected = pval.corrected)

  return(df)
}


armitage.concat <- function(x){

  if(!is.list(x)){
    stop("Input is not a list")
  }

  listLen <-  length(x)
  if(listLen < 2){
    stop("Fewer than 2 items in list")
  }

  if(any(sapply(x,class)!="armitage")) {
    step("All items must be armitage class")
  }
  #if( class(x[[1]]) != "armitage"){
  #  stop("First item not an armitage class")
  #}

  nsim <- x[[1]]$nsim
  for(i in 2:listLen){
    nsim <- c(nsim, x[[i]]$nsim)
  }

  if(any(nsim != nsim[1])){
    stop("Unequal number of simulations")
  }

  items.concat <- names(x)
  
  statObs <- x[[1]]$statObs
  snpLabels <- paste(items.concat[1], x[[1]]$snpLabels, sep="-")
  statSim <- x[[1]]$statSim
  nsubj <- x[[1]]$nsubj
  ngeno <- x[[1]]$ngeno
  
  for(i in 2:listLen){
    
    statObs <- c(statObs, x[[i]]$statObs)
    snpLabels <-  c(snpLabels,
                    paste(items.concat[i], x[[i]]$snpLabels, sep="-"))

  
    statSim <- cbind(statSim, x[[i]]$statSim)
    nsubj <- rbind(nsubj, x[[i]]$nsubj)
    ngeno <- c(ngeno, x[[i]]$ngeno)
   
  }
  
  dimnames(nsubj) <- list(items.concat, c("Controls","Cases"))
  names(ngeno) <- items.concat
  
  save <- list(items.concat=items.concat,
               statObs = statObs,
               statSim = statSim,
               nsim = nsim[1],
               snpLabels = snpLabels,
               nsubj = nsubj,
               ngeno = ngeno
               )
  
  if(exists("is.R") && is.function(is.R) && is.R()) {
    class(save) <- "armitage"
  } else {
    oldClass(save) <- "armitage"
  }

  return(save)
  
}







 
