#!/usr/bin/perl

=head1 NAME
   sample.input.pl

=head1 SYNOPSIS
    USAGE: sample.input.pl -r=run_info.txt

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	output directory

B<--help,-h>


=head1  DESCRIPTION
	Align a subset of fastq file to get mean and std-dev

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./sampling.pl -r=run_info.pl

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'stats|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
############################################
my $LONG_WAIT = 300;

## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/UpdateRunInfo.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Update RunInfo started");

#### check input
check_input($options{stats});

open ("FHD", "<", $options{stats})
	or $logger->logdie("Could now open stats file $!");
my @info = <FHD>;
close(FHD);

## remove end of line chr
chomp $info[1];
my @numbers = split(/\t/, $info[1]);

## update mean insert size and std-dev with values from inner_distance stats.
## file is expected to be of following format
##
#		Mean insert size	Median insert size	Standard deviation
#		-43	-53	52
#

$config->{ToolInfo}->{insert_size}->{value} = int($numbers[0]); ## update with mean insert size
$config->{ToolInfo}->{mate_sd}->{value} = int($numbers[2]); ## update with
$config->{ToolInfo}->export("$options{output_dir}/tool_info.txt");


#### Set sampling status complete in dashboard.
my $cmd = "$config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
$cmd .= " -r=$options{run_info} -t=Sampling -c=1";
execute_cmd($cmd);

$logger->info("Update RunInfo complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir stats);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $send_mail = 1;

	while (! -s $file) {
		if ($send_mail) {
			$send_mail = 0;

			$util->missingInput($config->{RunInfo}->{email},
								"UpdateRunInfo",
								"Sampling.InnerDistance",
								"$file",
								$config->{RunInfo}->{tool});
		}

		sleep $LONG_WAIT;
	}
}
