#!/usr/bin/perl

=head1 NAME
   transfer.pl

=head1 SYNOPSIS
    USAGE: transfer.pl -r=run_info.txt -o=output_dir

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Transfer relavent files from secondary analysis output to destination folder

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./transfer.pl -r=run_info.txt -o=output_dir

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});

my $this;
$this->{output_dir} = "$config->{RunInfo}->{delivery_folder}";

if ($this->{output_dir} !~ /$config->{RunInfo}->{output_folder}/) {
	$this->{output_dir} .= "/$config->{RunInfo}->{output_folder}";
}

#### exit if output folder for delivery already exists.
if (-d $this->{output_dir}) {
	print "Output dir $this->{output_dir} already exists.  Terminating transfer";
	exit();
}

my $cmd = "";

#### if delivery folder does not exists create one
if (! -e $this->{output_dir}) {
	$cmd = "mkdir -p $this->{output_dir}";
	execute_cmd($cmd);
}

#### move variant files.
$cmd = "mkdir $this->{output_dir}/variant";
execute_cmd($cmd);

$cmd = "mv $options{output_dir}/variant/*.vcf* $this->{output_dir}/variant/.";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/variant";
execute_cmd($cmd);


#### move reports
$cmd = "mkdir $this->{output_dir}/Reports";
execute_cmd($cmd);

$cmd = "mv $options{output_dir}/Reports/* $this->{output_dir}/Reports/.";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/Reports";
execute_cmd($cmd);


#### move fusion
$cmd = "mkdir $this->{output_dir}/fusion";
execute_cmd($cmd);

$cmd = "mv $options{output_dir}/fusion/* $this->{output_dir}/fusion/.";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/fusion";
execute_cmd($cmd);

#### move fancybox
$cmd = "cp -r $options{output_dir}/fancybox $this->{output_dir}/.";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/fancybox";
execute_cmd($cmd);


#### move fastqc
$cmd = "cp -r $options{output_dir}/fastqc $this->{output_dir}/.";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/fastqc";
execute_cmd($cmd);


foreach my $sample (keys %{$config->{SampleInfo}}) {
	#### move RSeQC per sample
	$cmd = "mkdir -p $this->{output_dir}/RSeQC/$sample";
	execute_cmd($cmd);

	$cmd = "mv $options{output_dir}/RSeQC/$sample/* $this->{output_dir}/RSeQC/$sample/.";
	execute_cmd($cmd);


	#### move alignment per sample
	$cmd = "mkdir -p $this->{output_dir}/alignment/tophat_$sample";
	execute_cmd($cmd);

	$cmd = "mv $options{output_dir}/alignment/tophat_$sample/* $this->{output_dir}/alignment/tophat_$sample/.";
	execute_cmd($cmd);
}

$cmd = "rm -rf $options{output_dir}/RSeQC";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/alignment";
execute_cmd($cmd);


#### move fusion
$cmd = "mkdir $this->{output_dir}/QC";
execute_cmd($cmd);

$cmd = "mv $options{output_dir}/QC/* $this->{output_dir}/QC/.";
execute_cmd($cmd);

$cmd = "rm -rf $options{output_dir}/QC";
execute_cmd($cmd);


#### move all files
my @toMove = qw(Appendix.html Main_Document.html StatisticsDescription.html
				igv_session.xml IGV_Setup.doc SampleStatistics.tsv
				mrnaseq_workflow.png);

foreach my $m (@toMove) {
	$cmd = "mv $options{output_dir}/$m $this->{output_dir}/";
	execute_cmd($cmd);
}


#### copy info files.
$cmd = "cp -r $options{output_dir}/sample_info.txt $this->{output_dir}/.";
execute_cmd($cmd);

$cmd = "cp -r $options{output_dir}/tool_info.txt $this->{output_dir}/.";
execute_cmd($cmd);

$cmd = "cp -r $options{output_dir}/run_info.txt $this->{output_dir}/.";
execute_cmd($cmd);

#### update dashboard for delivery status
#### setup dashboard status. to Delivery
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
$cmd .= " -r=$options{run_info} -t=Delivered -c=1";
execute_cmd($cmd);


exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'beauty'} = 0 unless ($options{'beauty'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	print $cmd."\n\n";
	system($cmd);

	if (( $? >> 8 ) != 0 ){
		print STDERR "Above command failed to execute: $!\n";
	}
}
