#!/usr/bin/perl

=head1 NAME
   sample.input.pl

=head1 SYNOPSIS
    USAGE: sampling.pl -r=run_info.txt -o=output_dir -s=sample

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--sample, =s>
	Sample name

B<--help,-h>


=head1  DESCRIPTION
	Align a subset of fastq file to get mean and std-dev

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


=head1 EXAMPLE
	./sampling.pl -r=run_info.pl

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
my $MAX_JOB_LIMIT = 3000;
my $LONG_WAIT = 300;
my $SHORT_WAIT = 30;
my $WAIT = 5;

## make sure everything passed was peachy
&check_parameters(\%options);

my $SAMPLING_SIZE = 100000;

## parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $this;
$this->{output_dir} = "$options{output_dir}/sampling";
$this->{run_on_sge} = 0;
$this->{job} = "";
$this->{q_args} = "";

#### if standalone is not defined (back compatibiliby)
#### and its set to anything other than "yes" then assume jobs are to be submmitted
#### to the grid.
#### if standalone is defined and set to "yes" then run locally.
if ((! defined $config->{ToolInfo}->{standalone}->{value}) || ($config->{ToolInfo}->{standalone}->{value} !~ /yes/i)){
	$this->{run_on_sge} = 1;
}

#### check if running on the grid
if ($this->{run_on_sge}) {
	$this->{q_args} = "-wd $config->{RunInfo}->{logs} -b y";
	$this->{q_args} .= " -q $config->{ToolInfo}->{queue}->{value}";
	$this->{q_args} .= " -m $config->{ToolInfo}->{notification_option}->{value}";
	$this->{q_args} .= " -M $config->{RunInfo}->{email} -l h_stack=10M";

	$this->{job} = $config->{RunInfo}->{tool} .".". $config->{RunInfo}->{version};
}



my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/Sampling.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Sampling started");

## check directory structure
create_dir_struct(\%options);

## write new run_info file in sampling dir
## with sample input dir pointing to local input file.
$config->{SampleInfo}->export_sample("$this->{output_dir}/sample_info.txt", $options{sample});
$config->{RunInfo}{input_dir} = "$this->{output_dir}/input";
$config->{RunInfo}{samplenames} = $options{sample};
$config->{RunInfo}{sample_info} = "$this->{output_dir}/sample_info.txt";
$config->{RunInfo}->export("$this->{output_dir}/run_info.txt");

my $cmd = "";
my $job_ids = -1;

my $read_count = `wc -l $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read1}`;
$read_count =~ s/^\s*(\d+)\s+.*/$1/;

$logger->info("Read count is: $read_count");

if ($read_count > 400000) {
	if ($config->{SampleInfo}{$options{sample}}[0]->{read1} =~ /\.gz$/) {
		$cmd = "gunzip -c $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read1}";
		$cmd .= " | head -n 400000 | gzip -c - > $this->{output_dir}/input/$config->{SampleInfo}{$options{sample}}[0]->{read1}";
		execute_cmd($cmd);

		$cmd = "gunzip -c $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read2}";
		$cmd .= " | head -n 400000 | gzip -c - > $this->{output_dir}/input/$config->{SampleInfo}{$options{sample}}[0]->{read2}";
		execute_cmd($cmd);
	} else {
		$cmd = "head -n 400000 $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read1}";
		$cmd .= " > $this->{output_dir}/input/$config->{SampleInfo}{$options{sample}}[0]->{read1}";
		execute_cmd($cmd);

		$cmd = "head -n 400000 $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read2}";
		$cmd .= " > $this->{output_dir}/input/$config->{SampleInfo}{$options{sample}}[0]->{read2}";
		execute_cmd($cmd);
	}
} else {
	$cmd = "cp $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read1}";
	$cmd .= " $this->{output_dir}/input/$config->{SampleInfo}{$options{sample}}[0]->{read1}";
	execute_cmd($cmd);

	$cmd = "cp $config->{SampleInfo}{$options{sample}}[0]->{filepath}/$config->{SampleInfo}{$options{sample}}[0]->{read2}";
	$cmd .= " $this->{output_dir}/input/$config->{SampleInfo}{$options{sample}}[0]->{read2}";
	execute_cmd($cmd);
}

## run alignment with default parameters.
$job_ids = run_alignment_module($options{sample});

## run rseqc inner_distance module
$job_ids = run_rseqc_module($options{sample}, $job_ids);

## update run info file.
$job_ids = run_update_runinfo($job_ids);

## write job it to be used by main alignment process and reports.
open(HID, ">", "$options{output_dir}/job_ids/sampling.ids")
	or $logger->logdie("Could not open file to write\n$!");
print HID $job_ids."\n";
close(HID);

$logger->info("Sampling complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	## check if analysis type folder exist else create one
	my $dir = "$options->{output_dir}/sampling/input";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	$dir = "$options->{output_dir}/sampling/RSeQC/$options{sample}";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub run_alignment_module {
	my $sample = shift;
	my $hold = "";

	my $dir = "$this->{output_dir}/alignment/tophat_$sample";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	my $job_name = "";
	my $name = "";
	my $cmd = "";

	my $reads_2 = 0;
	$reads_2 = 1 if ((exists $config->{SampleInfo}{$sample}[0]->{read2}) &&
					(length($config->{SampleInfo}{$sample}[0]->{read2})));


	##################################### run STEP-1 pre-processing
	$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".PreProcess";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/preProcess.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=0";

	execute_qsub_cmd($cmd);

	$hold = $name;


	##################################### run STEP-2 initial align
	### each process is threaded so keep that in mind while
	### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	## use left reads only
	$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".InitialAlignLeft";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{initalign_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/initialAlign.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=0 -e=left";

	execute_qsub_cmd($cmd);

	$job_name .= "$name,";

	#### right side
	if ($reads_2) {
		### each process is threaded so keep that in mind while
		### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
		$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".InitialAlignRight";

		$cmd = "";

		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{initalign_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		## use right reads only
		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/initialAlign.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=0 -e=right";

		execute_qsub_cmd($cmd);

		$job_name .= "$name,";
	}
	$job_name =~ s/,$//;
	$hold = $job_name;

	##################################### run STEP-3 align-segment
	### each process is threaded so keep that in mind while
	### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";
	my $num_seg = ($config->{RunInfo}->{readlength}/$config->{ToolInfo}->{segment_size}->{value});

	foreach my $seg (1..$num_seg){
		#### align left segments
		$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".LeftSeg.$seg";

		$cmd = "";

		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=0 -e=left -y=Segments -g=$seg";

		execute_qsub_cmd($cmd);

		$job_name .= "$name,";


		#### right side
		if ($reads_2) {
			### each process is threaded so keep that in mind while
			### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request

			#### align right segments
			$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
			$name .= "." . $config->{RunInfo}->{unique_id} . ".RightSeg.$seg";

			$cmd = "";

			#### check if running on the grid
			if ($this->{run_on_sge}) {
				$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
				$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
				$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

				if ( length($hold) ) {
					$cmd .= " -hold_jid $hold";
				}
			}

			$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
			$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
			$cmd .= " -f=0 -e=right -y=Segments -g=$seg";

			execute_qsub_cmd($cmd);

			$job_name .= "$name,";
		}
	}
	$job_name =~ s/,$//;
	$hold = $job_name;

	##################################### run STEP-4 align-gather
	### each process is threaded so keep that in mind while
	### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";
	chdir($dir);

	$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".AlignGather";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignGather.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=0";


	execute_qsub_cmd($cmd);

	$hold = $name;


	##################################### run STEP-5 align juncs
	### each process is threaded so keep that in mind while
	### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	foreach my $seg (1..$num_seg){
		#### align left segments
		chdir($dir);

		$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".LeftJunc.$seg";

		$cmd = "";

		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=0 -e=left -y=Juncs -g=$seg";

		execute_qsub_cmd($cmd);

		$job_name .= "$name,";

		#### right side
		if ($reads_2) {
			### each process is threaded so keep that in mind while
			### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
			#### align right segments
			chdir($dir);

			$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
			$name .= "." . $config->{RunInfo}->{unique_id} . ".RightJunc.$seg";

			$cmd = "";

			#### check if running on the grid
			if ($this->{run_on_sge}) {
				$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
				$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
				$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

				if ( length($hold) ) {
					$cmd .= " -hold_jid $hold";
				}
			}

			$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
			$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
			$cmd .= " -f=0 -e=right -y=Juncs -g=$seg";

			execute_qsub_cmd($cmd);

			$job_name .= "$name,";
		}
	}
	$job_name =~ s/,$//;
	$hold = $job_name;


	##################################### run STEP-6 post process
	### each process is threaded so keep that in mind while
	### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	#### left post process
	chdir($dir);

	$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".LeftPostProcess";

	$cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{postalign_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/postProcess.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=0 -e=left";

	execute_qsub_cmd($cmd);

	$job_name = $name;


	#### right post process
	if ($reads_2) {
		### each process is threaded so keep that in mind while
		### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
		chdir($dir);

		$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".RightPostProcess";

		$cmd = "";

		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{postalign_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/postProcess.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=0 -e=right";

		execute_qsub_cmd($cmd);

		$job_name .= ",$name";
	}
	$hold = $job_name;

	##################################### run STEP-7 generate tophat report
	### each process is threaded so keep that in mind while
	### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	$name = $this->{job} .".Sampling_Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".TophatReporting";

	$cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
		$cmd .= "  -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/reporting.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=0";

	execute_qsub_cmd($cmd);

	return $name;
}

#############################################################################
sub run_rseqc_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".Sampling_RSeQC.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id} . ".InnerDistance";

	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{innerdist_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/innerdistance.pl";
	$cmd .= " -i=$this->{output_dir}/alignment/tophat_$sample/accepted_hits.bam";
	$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_update_runinfo {
	my $hold = shift;

	my $job_name = $this->{job} .".Sampling_UpdateRunInfo.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{updateinfo_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/update_runinfo.pl";
	$cmd .= " -r=$options{run_info} -o=$options{output_dir}";
	$cmd .= " -s=$this->{output_dir}/RSeQC/$options{sample}/$options{sample}.inner_distance.stats";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub execute_qsub_cmd {
	my $cmd = shift;

	#### check if running on the grid
	if (! $this->{run_on_sge}) {
		execute_cmd($cmd);
		return;
	}

	my $count = 0;
	my $send_mail = 1;

	#### check if job limit is hit
	check_job_limit();

	sleep $WAIT;

	$logger->info("$cmd");
	system($cmd);

	while ((( $? >> 8 ) != 0 ) && ($count < 5)){
		sleep $SHORT_WAIT;
		$count ++;

		system($cmd);
	}

	if (( $? >> 8 ) != 0 ) {
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/Cmd.err";
		$util->createErrorFile($error_name, $cmd);

		while (-e $error_name) {

			#### pass Filename, cmd executed, email to, step running, sample name.
			if ($send_mail) {
				$util->reportError($error_name,
							   $cmd,
							   $config->{RunInfo}->{email},
							   "A command",
							   $options{sample},
							   "",
							   "");
				$send_mail = 0;
			}
			sleep $LONG_WAIT;
		}
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub check_job_limit {
	my $cmd = "$config->{ToolInfo}->{sge}->{value}/qstat";

	my @cmd_output = qx{$cmd};

	#### wait till number of jobs in queue are less than job limit to
	#### continue execution.
	while (scalar(@cmd_output) >= $MAX_JOB_LIMIT) {
		sleep $LONG_WAIT;

		@cmd_output = qx{$cmd};
	}
}
