#!/usr/bin/perl

=head1 NAME
   bam2wig.pl

=head1 SYNOPSIS
    USAGE: bam2wig.pl -r=run_info.txt -o=output -i=input -s=sample

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output, -o>
	Output prefix with dir path if needed

B<--input, -i>
	Input bam file

B<--sample, -s>
	Sample name

B<--help,-h>


=head1  DESCRIPTION
	Convert bam file to wiggle file.

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./bam2wig.pl -r=run_info.txt -o=output_prefix_with_dir -i=input_bam -s=sample

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output|o=s',
						  'input|i=s',
						  'sample|s=s',
						  'limit|k=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
############################################
my $LONG_WAIT = 300;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/BAM2WIG.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Bam2Wig started");

## export env var.
if ( defined $ENV{'PATH'} ) {
	$ENV{'PATH'}="$config->{ToolInfo}->{rseqcpath}->{value}:$config->{ToolInfo}->{r}->{value}:$ENV{'PATH'}";
} else {
	$ENV{'PATH'}="$config->{ToolInfo}->{rseqcpath}->{value}:$config->{ToolInfo}->{r}->{value}";
}
if ( defined $ENV{'PYTHON'} ) {
	$ENV{'PYTHONPATH'} = "$config->{ToolInfo}->{rseqcpath}->{value}:$config->{ToolInfo}->{pythonpath}->{value}:$ENV{'PYTHONPATH'}";
} else {
	$ENV{'PYTHONPATH'} = "$config->{ToolInfo}->{rseqcpath}->{value}:$config->{ToolInfo}->{pythonpath}->{value}";
}

#### check input file
check_input($options{input});

my $cmd = " $config->{ToolInfo}->{python}->{value}/python";
$cmd .= " $config->{ToolInfo}->{rseqc}->{value}/bam2wig.py";
$cmd .= " -i $options{input}";
$cmd .= " -o $options{output}";
$cmd .= " -s $config->{ToolInfo}->{chromsize}->{value}";

execute_cmd($cmd);

$logger->info("Bam2Wig completed");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output input sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
	$options{'fusion'} = 1 unless (defined $options{'fusion'});
	$options{'limit'} = 1000000 unless (defined $options{'limit'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);
}

#############################################################################
sub check_input {
	my $file = shift;

	my $send_mail = 1;

	while (! -s $file) {
		if ($send_mail) {
			$send_mail = 0;

			$util->missingInput($config->{RunInfo}->{email},
								"Bam2Wig",
								"Alignment",
								"$file",
								$config->{RunInfo}->{tool});
		}

		sleep $LONG_WAIT;
	}
}
