#!/usr/bin/perl

=head1 NAME
   mrna.pl

=head1 SYNOPSIS

    USAGE: mrna.pl -r=run_info.txt

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--auto_clean, -c>
	Optional. Remove all temp file and folders from execution space.
	This should always be set to 0 and run the script
	Indivicually after verifiying all outputs

B<--beauty, -b>
	Optional. Specify if this is a beauty run, (used by auto_clean only as of v1.0)

B<--subset, -s>
	Optional. Specify if running the workflow on a subset.
	A senario where, output folder, and direcotry structure already exists
	for a flowcell/run_id.  Add adding output for this run to existing folder.
		- a case when all but one sample failed and needs a re-run
		- a case when user forgot to add all samples in sample info file
		  and need to run another one for a given flowcell.

B<--fqc_override, -f>
	Optional. Force stop execution of FastQC module.  FastQC module will execute IF
		Use case 1:  user sets 'no' for FastQC analysis in the run_info file and
					 do not set the fastqc_folder location in run_info file in
					 this case the fastqc link will become invalid in the final
					 HTML document.

		Use case 2: user sets 'no' for FastQC analysis in the run_info file and
					set the fastqc_folder location in run_info file but the
					location is empty/invalid/does not contain reports in this
					case the fastqc link will become invalid in the final
					HTML document.

		default is 1, set it to 0 to stop execution of FastQC

B<--help,-h>


=head1 DESCRIPTION
	Execute mrna workflow

=head1 INPUT
	Run info file.

=head1 OUTPUT

=head1 VERSION
	0.9.1

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./mrna.pl -r=run_info.pl

=cut


BEGIN { 
	# set up the environment 
	use Cwd 'abs_path'; 
	use File::Basename;
	my $base_path = dirname(abs_path($0)); 
	$base_path =~ s/\/src$//; 
	print $base_path . "\n"; 
	push(@INC, $base_path . "/lib/" , $base_path . "/lib/perl5" , $base_path . "/lib/perl5/lib/perl5", $base_path . "/lib/perl5/x86_64-linux/auto");

	my @new_path = (
		$base_path . "/bin/samtools/0.1.19" , 
		$base_path . "/bin/bedtools/2.17.0" , 
		$base_path . "/bin/bowtie/0.12.9" , 
		$base_path . "/bin/bedtools/2.17.0" , 
		$base_path . "/bin/circos/0.64/bin" , 
		$base_path . "/bin/bowtie/0.12.9" , 
		$base_path . "/bin/fastqc/0.10.1" , 
		$base_path . "/bin/gatk/1.6/" , 
		$base_path . "/bin/java/bin" , 
		$base_path . "/bin/htseq/0.5.3p9/bin" , 
		$base_path . "/bin/picard/1.92" , 
		$base_path . "/bin/ucsc" , 
		$base_path . "/bin/rseqc/2.3.7/bin" , 
		$base_path . "/bin/tophat/2.0.6" , 
		);
	$ENV{'PATH'} = $ENV{'PATH'} . join(":", @new_path); 

	my @python_path = ( 
		$base_path . "/bin/htseq/0.5.3p9/lib/python2.7/site-packages/",
		$base_path . "/bin/rseqc/2.3.7/lib/python2.7/site-packages/"
		);	

	$ENV{'PYTHONPATH'} = $ENV{'PYTHONPATH'} . join(":", @python_path); 

	my @perl5lib_path = ( 
		"/usr/lib/perl5" , # ubuntu default
		"/usr/lib64/perl5" , # centos default
		$base_path . "/lib",
		$base_path . "/lib/perl5",
		$base_path . "/lib/perl5/lib/perl5",
		$base_path . "/lib/perl5/lib/perl5/x86_64-linux", 
		$base_path . "/lib/perl5/lib/perl5/x86_64-linux-thread-multi/",
		$base_path . "/lib/perl5/x86_64-linux/auto",
		);

	$ENV{'PERL5LIB'} = $ENV{'PERL5LIB'} . join(":", @perl5lib_path); 
}

use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use POSIX;
use ParseConfig;
use MyUtility;
use Workflow::Logger;



my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'auto_clean|c=s',
						  'subset|s=s',
						  'beauty|b=s',
						  'fqc_override|f=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
my $MAX_JOB_LIMIT = 3000;
my $LONG_WAIT = 300;
my $SHORT_WAIT = 30;
my $WAIT = 5;

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger = new Workflow::Logger('LOG_FILE'=>$options{'log'},
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Workflow started");

if (! $options{subset}){
	#### check directory structure
	create_dir_struct($config);
}

#### set local variables.
my $cmd = "";
my $this;

$this->{output_dir} = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}";
$this->{run_on_sge} = 0;
$this->{job} = "";
$this->{q_args} = "";

if (! $options{subset}) {

	#### create local copy of X_info files
	create_local_copy($options{run_info}, "run_info.txt");
	create_local_copy($config->{RunInfo}->{tool_info}, "tool_info.txt");
	create_local_copy($config->{RunInfo}->{sample_info}, "sample_info.txt");
	create_local_copy($config->{RunInfo}->{memory_info}, "memory_info.txt");

	#### prep output dir, copy workflow image, and basic docs.
	prep_output_folder($config);
}

#### if standalone is not defined (back compatibiliby)
#### and its set to anything other than "yes" then assume jobs are to be submmitted
#### to the grid.
#### if standalone is defined and set to "yes" then run locally.
if ((! defined $config->{ToolInfo}->{standalone}->{value}) || ($config->{ToolInfo}->{standalone}->{value} !~ /yes/i)){
	$this->{run_on_sge} = 1;
}

#### check if running on the grid
if ($this->{run_on_sge}) {
	$this->{q_args} = "-wd $config->{RunInfo}->{logs} -b y";
	$this->{q_args} .= " -q $config->{ToolInfo}->{queue}->{value}";
	$this->{q_args} .= " -m $config->{ToolInfo}->{notification_option}->{value}";
	$this->{q_args} .= " -M $config->{RunInfo}->{email} -l h_stack=10M";

	$this->{job} = $config->{RunInfo}->{tool} .".". $config->{RunInfo}->{version};
}


#### export env var.
if (defined $ENV{'PATH'}){
	$ENV{'PATH'}="$config->{ToolInfo}->{r}->{value}:$ENV{'PATH'}";
} else {
	$ENV{'PATH'}="$config->{ToolInfo}->{r}->{value}";
}

if (defined $ENV{'PYTHONPATH'}) {
	$ENV{'PYTHONPATH'} = "$config->{ToolInfo}->{pythonpath}->{value}:$config->{ToolInfo}->{rseqcpath}->{value}:$ENV{'PYTHONPATH'}";
} else {
	$ENV{'PYTHONPATH'} = "$config->{ToolInfo}->{pythonpath}->{value}:$config->{ToolInfo}->{rseqcpath}->{value}";
}


my $sampling_job_name = "";
my $pri_job_name = "";
my $sec_job_name = "";
my $counts_name = "";
my $umapreads_name = "";
my $fusion_hold = "";
my $expression_hold = "";
my @ret;
my $mod_gtf_job_name = "";

#### if its part of an existing workflow do not update dashboard.
if (! $options{subset}) {
	if ((defined $config->{ToolInfo}->{dashboard_update}->{value}) && ($config->{ToolInfo}->{dashboard_update}->{value} =~ /yes/i)) {
		#### setup dashboard status. to Beginning
		$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
		$cmd .= " -r=$options{run_info} -t=Beginning -c=1";
		execute_cmd($cmd);

		$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
		$cmd .= " -r=$options{run_info} -t=Start -c=1";
		execute_cmd($cmd);
	}

	######################################################################
	#### run modify gtf module
	$cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		#### run GTF modification module
		$mod_gtf_job_name = $this->{job} . ".ModGTF.MULTI." . $config->{RunInfo}->{output_folder};
		$mod_gtf_job_name .= "." . $config->{RunInfo}->{unique_id};

		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{gen_mem}";
		$cmd .= " -N $mod_gtf_job_name";
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/modify_gtf.pl";
	$cmd .= " -o=$this->{output_dir} -r=$this->{output_dir}/run_info.txt";

	execute_qsub_cmd($cmd);

	######################################################################
	#### check input file quality score if sanger convert to illumina 1.3
	#$cmd = "";

	#### check if running on the grid
	#if ($this->{run_on_sge}) {
		#### run GTF modification module
		#$mod_gtf_job_name = $this->{job} . ".FASTQCHECK.MULTI." . $config->{RunInfo}->{output_folder};
		#$mod_gtf_job_name .= "." . $config->{RunInfo}->{unique_id};

		#$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		#$cmd .= " $config->{MemoryInfo}->{fastqcheck_mem}";
		#$cmd .= " -N $mod_gtf_job_name";
	#}

	#$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/sangercheck.pl";
	#$cmd .= " -o=$this->{output_dir} -r=$this->{output_dir}/run_info.txt";

	#execute_qsub_cmd($cmd);
}

#### run sampling process.
if ((($config->{RunInfo}->{analysis} =~ /all/) || ($config->{RunInfo}->{analysis} =~ /sampling/) || ($config->{RunInfo}->{analysis} =~ /samplingplus/)) && (! $options{subset})) {
	$sampling_job_name = $this->{job} . ".Sampling.MULTI." . $config->{RunInfo}->{output_folder};
	$sampling_job_name .= "." . $config->{RunInfo}->{unique_id};

	my @samples = keys %{$config->{SampleInfo}};

	my $idx = int(rand(scalar(@samples)));

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{sample_mem}";
		$cmd .= " -hold_jid $mod_gtf_job_name";
		$cmd .= " -N $sampling_job_name";
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/sampling.pl";
	$cmd .= " -o=$this->{output_dir} -r=$this->{output_dir}/run_info.txt -s=$samples[$idx]";

	execute_qsub_cmd($cmd);
}


#### run main workflow step
foreach my $sample (keys %{$config->{SampleInfo}}) {

	#### run alignment module for a given sample
	if (($config->{RunInfo}->{analysis} =~ /all/) || ($config->{RunInfo}->{analysis} =~ /samplingplus/) ||
		($config->{RunInfo}->{analysis} =~ /align/) || ($config->{RunInfo}->{analysis} =~ /alignplus/)) {

		if (length($sampling_job_name) && $this->{run_on_sge}) {
			while (! -s "$this->{output_dir}/job_ids/sampling.ids") {
				print "Waiting for sampling jobs to be submitted\n";
				sleep $LONG_WAIT;
			}

			open (FHD, "<", "$this->{output_dir}/job_ids/sampling.ids")
				or $logger->logdie("Could not open file to read $this->{output_dir}/job_ids/sampling.ids\n$!");
			while(<FHD>) {
				chomp $_;

				$sampling_job_name .= ",". $_;
			}
			close(FHD);
		}

		if (length($mod_gtf_job_name)){
			$sampling_job_name .= ",$mod_gtf_job_name";
		}

		#### call alignment module on complete sample.
		$pri_job_name = run_alignment_module($sample, $sampling_job_name);

		#### fusion post process should only be ran after all the alignment
		#### jobs are completed as only one folder is created for all samples
		$fusion_hold .= "$pri_job_name,";

		if ($config->{RunInfo}->{fastqc} =~ /yes/i){
			#### run fastqc for each sample.
			$sec_job_name .= run_fastqc_module($sample) .",";
		} else {
			#### copy fastqc files
			my $fqc = copy_fastqc_reports($sample);

			##### if fastqc reports don't exists, and override flag is set
			##### run fastqc even thought run_info file states "NO"
			if ((! $fqc) && ($options{fqc_override})){
				$sec_job_name .= run_fastqc_module($sample) .",";
			}
		}

	} #### end of alignment module

	#### run various analysis on given sample.
	if (($config->{RunInfo}->{analysis} =~ /all/) || ($config->{RunInfo}->{analysis} =~ /alignplus/) ||
		($config->{RunInfo}->{analysis} =~ /analyze/)) {

			#### merge unmapped and accepted_hits bam file into one.
			$sec_job_name .= run_merge_bam_module($sample, $pri_job_name);

			#### re-arrange bam file
			$pri_job_name = run_sortbam_module($sample, $pri_job_name);

			#### run RSeQC
			$sec_job_name .= run_rseqc_module($sample, $pri_job_name) .",";

			#### run SNV calling
			$sec_job_name .= run_snv_module($sample, $pri_job_name) .",";

			#### run gene count
			$counts_name = run_genecount_module($sample, $pri_job_name) .",";

			#### run exon count
			$counts_name .= run_exoncount_module($sample, $pri_job_name) .",";

			#### run qc on each bam file.
			$umapreads_name .= run_umapreads_module($sample, $pri_job_name) .",";

			$expression_hold .= $counts_name .",";

			#### run numbers report
			$sec_job_name .= run_numbers_report($sample, $counts_name) .",";

	} #### analysis if condition
} #### end of each sample loop.

if (($config->{RunInfo}->{analysis} =~ /all/) || ($config->{RunInfo}->{analysis} =~ /alignplus/) ||
	($config->{RunInfo}->{analysis} =~ /analyze/)) {

	#### run Tophat fusion post process
	$fusion_hold =~ s/,$//;
	$sec_job_name .= run_tophat_post($fusion_hold). ",";

	#### run unique mapped reads plot
	$umapreads_name =~ s/,$//;
	$sec_job_name .= run_mreadplot_module($umapreads_name);

	#### run Expression counts
	$expression_hold =~ s/,,/,/g;
	$expression_hold =~ s/,$//;
	$sec_job_name .= run_mergecount_module($expression_hold);

	#### run Report
	$sec_job_name =~ s/,$//;
	$sec_job_name =~ s/,,/,/g;
	$pri_job_name = run_report_module($sec_job_name);

	#### run auto clean if option specified. this should be set to zero
	#### for easy of qc and backtracking just incase something goes wrong
	#### user should run cleanup script after double checking all data.
	if ($options{auto_clean}) {
		#### clean up unwanted files and move files
		run_cleanup_module($pri_job_name);
	}
}

#### complete status udpate for each process.
if ($this->{run_on_sge}) {
	my @process = qw(SortBam NumbersReport VQSRFiltering Alignment);
	foreach my $p (@process){
		my $dependent_job_name = "";

		my $name = $this->{job} .".$p.DASHBOARD.";
		$name .= $config->{RunInfo}->{output_folder}. ".";
		$name .= $config->{RunInfo}->{unique_id};

		foreach my $sample (keys %{$config->{SampleInfo}}) {
			#### create job dependencies.
			$dependent_job_name .= $this->{job} .".$p.$sample.";
			$dependent_job_name .= $config->{RunInfo}->{output_folder}. ".";
			$dependent_job_name .= $config->{RunInfo}->{unique_id};

			if ($p =~ /^Alignment$/) {
				$dependent_job_name .= ".TophatReporting";
			}

			$dependent_job_name .= ",";
		}

		$dependent_job_name =~ s/,$//;

		my $type = "Alignment";

		if ($p =~ /^NumbersReport$/){
			$type = "ExpressionCount";
		} elsif ($p =~ /^SortBam$/) {
			$type = "Statistics";
		}elsif ($p =~ /^VQSRFiltering$/){
			$type = "VariantCalling";
		}

		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{dashbrd_mem} -N $name";
		$cmd .= " -hold_jid $dependent_job_name";
		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
		$cmd .= " -r=$options{run_info} -t=$type -c=1";
		execute_qsub_cmd($cmd);
	}
}

$logger->info("Workflow complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	#### create a default log file.
	unless($options{'log'}){
		my $dir = getcwd;
		my ($second, $minute, $hour) = localtime();
		$options{'log'} = $dir."/MAPRSeq." . $config->{RunInfo}->{output_folder};
		$options{'log'} .= "." .$hour. "" .$minute. "" .$second. ".log";
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});

	##### init cleanup script at the end of run.  It is NOT recommended
	#### to set this option as you want to manually curate the output folder
	#### before deleting temp files.
	$options{'auto_clean'} = 0 unless (defined $options{'auto_clean'});

	#### this option is only used right now to be passed to cleanup script
	$options{'beauty'} = 0 unless (defined $options{'beauty'});

	#### if this run is a subset of previous run, then don't
	#### setup, create folder structure or copy required files.
	#### it is assumed that folder structure already exists.
	$options{'subset'} = 0 unless (defined $options{'subset'});

	#### override fastqc set to 1 by default
	#### if reports are not available in /data2/bsi/reports/RUN_ID/SAMPLE
	#### then run fastqc even though run_info file states "NO"
	$options{'fqc_override'} = 0 unless (defined $options{'fqc_override'});

	$options{'rerun'} = 0 unless (defined $options{'rerun'});  ####unused v1.0
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	#### check if PI folder exist else create one
	my $dir = "$options->{RunInfo}->{base_output_dir}/$options->{RunInfo}->{pi}";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	#### check if analysis type folder exist else create one
	$dir = "$options->{RunInfo}->{base_output_dir}/$options->{RunInfo}->{pi}/$options->{RunInfo}->{type}";
	if ( -d $dir ) {
		$logger->info("Analysis type $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	#### check if output folder exists else create one.
	$dir = "$options->{RunInfo}->{base_output_dir}/$options->{RunInfo}->{pi}/$options->{RunInfo}->{type}/$options->{RunInfo}->{output_folder}";
	if ( -d $dir ) {
		$logger->logdie("Output run folder $dir exists");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	my $outdir = $dir;

	my @dirs = qw(job_ids error RSeQC variant/logs variant/temp variant/plot fastqc fastq fusion);

	foreach my $d (@dirs) {
		#### check if folder exists else create one.
		$dir = "$outdir/$d";
		if ( -d $dir ) {
			$logger->logdie("Output folder $dir exists");
		} else {
			execute_cmd("mkdir -p $dir");
		}
	}

	#### create workflow logs dir.
	$dir = $options->{RunInfo}->{logs};
	execute_cmd("mkdir -p $dir");
}

#############################################################################
sub create_local_copy {
	my $filename = shift;
	my $local_name = shift;

	my $cmd = "cp $filename $this->{output_dir}/$local_name";
	execute_cmd("$cmd");
}

#############################################################################
sub prep_output_folder {
	my $options = shift;

	#### copy workflow diagram
	$cmd = "cp $options->{ToolInfo}->{workflow_path}->{value}/$options->{RunInfo}->{type}_workflow.png $this->{output_dir}/.";
	execute_cmd("$cmd");

	#### copy fancybox dir
	$cmd = "cp -r $options->{ToolInfo}->{workflow_path}->{value}/fancybox $this->{output_dir}/.";
	execute_cmd("$cmd");

	#### copy IGV doc.
	$cmd = "cp $options->{ToolInfo}->{workflow_path}->{value}/IGV_Setup.doc $this->{output_dir}/.";
	execute_cmd("$cmd");

	#### update run_info with local copy of sample, tool and memory info
	$config->{RunInfo}->{sample_info} = "$this->{output_dir}/sample_info.txt";
	$config->{RunInfo}->{tool_info} = "$this->{output_dir}/tool_info.txt";
	$config->{RunInfo}->{memory_info} = "$this->{output_dir}/memory_info.txt";

	#### export run_info updates
	$config->{RunInfo}->export("$this->{output_dir}/run_info.txt");
}

#############################################################################
sub run_fastqc_module {
	my $sample = shift;

	my $job_name = $this->{job} .".FastQC.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{fastqc_mem}";
		$cmd .= " -N $job_name";
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/fastqc.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub copy_fastqc_reports {
	my $sample = shift;

	foreach my $s (@{$config->{SampleInfo}{$sample}}) {
		my $read1 = "$s->{read1}";
		$read1 =~ s/\.fastq.*$/_fastqc/;

		if (-d "/data2/bsi/reports/$config->{RunInfo}->{output_folder}/fastqc/$read1") {
			$cmd = "cp /data2/bsi/reports/$config->{RunInfo}->{output_folder}/fastqc/$read1";
			$cmd .= " $this->{output_dir}/fastqc/. &";
			execute_cmd($cmd);
		} else {
			return 0;  #### return 0 to force fastqc module run
		}

		if ((exists $s->{read2}) && (length($s->{read2}))) {
			my $read2 = "$s->{read1}";
			$read2 =~ s/\.fastq.*$/_fastqc/;

			if (-d "/data2/bsi/reports/$config->{RunInfo}->{output_folder}/fastqc/$read2") {
				$cmd = "cp /data2/bsi/reports/$config->{RunInfo}->{output_folder}/fastqc/$read2";
				$cmd .= " $this->{output_dir}/fastqc/. &";
				execute_cmd($cmd);
			} else {
				return 0;  #### return 0 to force fastqc module run
			}
		}
	}

	return 1;
}

#############################################################################
sub run_alignment_module {
	my $sample = shift;
	my $hold = shift;

	my $dir = "$this->{output_dir}/alignment/tophat_$sample";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	my $job_name = "";
	my $name = "";
	my $cmd = "";

	my $reads_2 = 0;
	$reads_2 = 1 if ((exists $config->{SampleInfo}{$sample}[0]->{read2}) &&
					(length($config->{SampleInfo}{$sample}[0]->{read2})));

	##################################### run STEP-1 pre-processing
	$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".PreProcess";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/preProcess.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=1";

	execute_qsub_cmd($cmd);

	$hold = $name;


	##################################### run STEP-2 initial align
	#### each process is threaded so keep that in mind while
	#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	#### use left reads only
	$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".InitialAlignLeft";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{initalign_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/initialAlign.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=1 -e=left";

	execute_qsub_cmd($cmd);

	$job_name .= "$name,";

	#### right side
	if ($reads_2) {
		#### each process is threaded so keep that in mind while
		#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
		$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".InitialAlignRight";

		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{initalign_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		#### use right reads only
		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/initialAlign.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=1 -e=right";

		execute_qsub_cmd($cmd);

		$job_name .= "$name,";
	}
	$job_name =~ s/,$//;
	$hold = $job_name;


	##################################### run STEP-3 align-segment
	#### each process is threaded so keep that in mind while
	#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";
	my $num_seg = ($config->{RunInfo}->{readlength}/$config->{ToolInfo}->{segment_size}->{value});

	foreach my $seg (1..$num_seg){
		#### align left segments
		$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".LeftSeg.$seg";

		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=1 -e=left -y=Segments -g=$seg";

		execute_qsub_cmd($cmd);

		$job_name .= "$name,";


		#### right side
		if ($reads_2) {
			#### each process is threaded so keep that in mind while
			#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request

			#### align right segments
			$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
			$name .= "." . $config->{RunInfo}->{unique_id} . ".RightSeg.$seg";

			$cmd = "";
			#### check if running on the grid
			if ($this->{run_on_sge}) {
				$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
				$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
				$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

				if ( length($hold) ) {
					$cmd .= " -hold_jid $hold";
				}
			}

			$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
			$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
			$cmd .= " -f=1 -e=right -y=Segments -g=$seg";

			execute_qsub_cmd($cmd);

			$job_name .= "$name,";
		}
	}
	$job_name =~ s/,$//;
	$hold = $job_name;


	##################################### run STEP-4 align-gather
	#### each process is threaded so keep that in mind while
	#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".AlignGather";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignGather.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=1";

	execute_qsub_cmd($cmd);

	$hold = $name;


	##################################### run STEP-5 align juncs
	#### each process is threaded so keep that in mind while
	#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	foreach my $seg (1..$num_seg){
		#### align left segments

		$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".LeftJunc.$seg";

		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=1 -e=left -y=Juncs -g=$seg";

		execute_qsub_cmd($cmd);

		$job_name .= "$name,";

		#### right side
		if ($reads_2) {
			#### each process is threaded so keep that in mind while
			#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
			#### align right segments

			$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
			$name .= "." . $config->{RunInfo}->{unique_id} . ".RightJunc.$seg";

			$cmd = "";
			#### check if running on the grid
			if ($this->{run_on_sge}) {
				$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
				$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
				$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

				if ( length($hold) ) {
					$cmd .= " -hold_jid $hold";
				}
			}

			$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/alignSegJunc.pl";
			$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
			$cmd .= " -f=1 -e=right -y=Juncs -g=$seg";

			execute_qsub_cmd($cmd);

			$job_name .= "$name,";
		}
	}
	$job_name =~ s/,$//;
	$hold = $job_name;


	##################################### run STEP-6 post process
	#### each process is threaded so keep that in mind while
	#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".LeftPostProcess";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{postalign_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/postProcess.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=1 -e=left";

	execute_qsub_cmd($cmd);

	$job_name = $name;


	#### right post process
	if ($reads_2) {
		#### each process is threaded so keep that in mind while
		#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request

		$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} . ".RightPostProcess";

		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{postalign_mem} -N $name";
			$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/postProcess.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
		$cmd .= " -f=1 -e=right";

		execute_qsub_cmd($cmd);

		$job_name .= ",$name";
	}
	$hold = $job_name;

	##################################### run STEP-7 generate tophat report
	#### each process is threaded so keep that in mind while
	#### setting $config->{ToolInfo}->{sge}->{value}/qsub mem request
	$job_name = "";

	$name = $this->{job} .".Alignment.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".TophatReporting";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{align_mem} -N $name";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{align_threads}";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat/reporting.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$dir -s=$sample";
	$cmd .= " -f=1";

	execute_qsub_cmd($cmd);

	return $name;
}

#############################################################################
sub run_merge_bam_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".MergeBAMs.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{merge_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment to complete
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/merge_bams.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_sortbam_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".SortBam.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{sort_mem}";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{sort_threads} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/sortbam.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_rseqc_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = "";
	my $input_file = "$this->{output_dir}/alignment/tophat_$sample/accepted_hits.bam";

	my $dir = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/RSeQC/$sample";
	if ( -d $dir ) {
		$logger->info("Output run folder $dir exists");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	#### Junction Annotation
	my $name = $this->{job} .".RSeQC.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id}. ".JunctionAnnotation";

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{juncanno_mem}";
		$cmd .= " -N $name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/junctionannotation.pl";
	$cmd .= " -i=$input_file";
	$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";

	$job_name .= "$name,";

	execute_qsub_cmd($cmd);


	#### Junction Saturation
	$name = $this->{job} .".RSeQC.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} .".JunctionSaturation";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{juncsat_mem}";
		$cmd .= " -N $name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/junctionsaturation.pl";
	$cmd .= " -i=$input_file";
	$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";

	$job_name .= "$name,";

	execute_qsub_cmd($cmd);


	#### Inner distance
	$name = $this->{job} .".RSeQC.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} .".InnerDistance";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{innerdist_mem}";
		$cmd .= " -N $name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/innerdistance.pl";
	$cmd .= " -i=$input_file -k=1000000";
	$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";

	$job_name .= "$name,";

	execute_qsub_cmd($cmd);


	#### Gene Body coverage

		#### convert bam file to wig.
		$name = $this->{job} . ".RSeQC.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} .".BAM2Wig";

		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{bam2wig_mem}";
			$cmd .= " -N $name";

			if ( length($hold) ) {
				$cmd .= " -hold_jid $hold";
			}
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/bam2wig.pl";
		$cmd .= " -i=$input_file";
		$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";

		execute_qsub_cmd($cmd);

		#### convert wig to bigwig
		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{wig2bigwig_mem}";
			$cmd .= " -N $this->{job}.RSeQC.$sample." . $config->{RunInfo}->{output_folder};
			$cmd .= "." . $config->{RunInfo}->{unique_id} . ".Wig2BigWig";
			$cmd .= " -hold_jid $name";
		}

		$cmd .= " $config->{ToolInfo}->{ucsc_bin}->{value}/wigToBigWig -clip";
		$cmd .= " $this->{output_dir}/RSeQC/$sample/$sample.wig";
		$cmd .= " $config->{ToolInfo}->{chromsize}->{value}";
		$cmd .= " $this->{output_dir}/RSeQC/$sample/$sample.bw";

		execute_qsub_cmd($cmd);

		$name = $this->{job} .".RSeQC.$sample." . $config->{RunInfo}->{output_folder};
		$name .= "." . $config->{RunInfo}->{unique_id} .".GeneBodyCoverage";

		$cmd = "";
		#### check if running on the grid
		if ($this->{run_on_sge}) {
			$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
			$cmd .= " $config->{MemoryInfo}->{gbc_mem}";
			$cmd .= " -N $name";
			$cmd .= " -hold_jid $this->{job}.RSeQC.$sample." . $config->{RunInfo}->{output_folder};
			$cmd .= "." . $config->{RunInfo}->{unique_id} . ".Wig2BigWig";
		}

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/genebodycoverage.pl";
		$cmd .= " -i=$this->{output_dir}/RSeQC/$sample/$sample.bw";
		$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";

		$job_name .= "$name,";

		execute_qsub_cmd($cmd);


	#### Read duplication
	$name = $this->{job} .".RSeQC.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id} . ".ReadDuplication";

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{readdup_mem}";
		$cmd .= " -N $name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/rseqc/readduplication.pl";
	$cmd .= " -i=$input_file";
	$cmd .= " -o=$this->{output_dir}/RSeQC/$sample/$sample";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -s=$sample";


	$job_name .= "$name,";

	execute_qsub_cmd($cmd);

	$job_name =~ s/,$//;
	return $job_name;
}

#############################################################################
sub run_snv_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".UnifiedGenotyper.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $gatk_q = $this->{q_args};
	$gatk_q =~ s/$config->{ToolInfo}->{queue}->{value}/$config->{ToolInfo}->{gatk_queue}->{value}/;

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $gatk_q";
		$cmd .= " $config->{MemoryInfo}->{gatk_mem} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	#### run gatk unified genotyper
	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/unifiedGenotyper.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	#### blat verify
	# my $blat_name = $this->{job} .".BlatVerify.$sample." . $config->{RunInfo}->{output_folder};
	# $blat_name .= "." . $config->{RunInfo}->{unique_id};

	# $cmd = "";
	#### check if running on the grid
	# if ($this->{run_on_sge}) {
	#	$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
	#	$cmd .= " $config->{MemoryInfo}->{blat_mem} -N $blat_name";
	#	$cmd .= " -hold_jid $job_name";
	#}

	#$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/vcf_blat_verify.pl";
	#$cmd .= " -i=$this->{output_dir}/variant/$sample.gatk.vcf";
	#$cmd .= " -o=$this->{output_dir}/variant/$sample.gatk.new.vcf";
	#$cmd .= " -sam=$config->{ToolInfo}->{samtools}->{value}";
	#$cmd .= " -b=$config->{ToolInfo}->{blat}->{value} -br=$config->{ToolInfo}->{blat_ref}->{value}";
	#$cmd .= " -r=$config->{ToolInfo}->{ref_genome}->{value} -w=50 -s=70 -t=90";

	#execute_qsub_cmd($cmd);

	#### run vqsr filtering
	my $name = $this->{job}.".VQSRFiltering.$sample." . $config->{RunInfo}->{output_folder};
	$name .= "." . $config->{RunInfo}->{unique_id};

	$cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $gatk_q";
		$cmd .= " $config->{MemoryInfo}->{vqsr_mem} -N $name";
		$cmd .= " -hold_jid $job_name";
		#$cmd .= " -hold_jid $blat_name"; ## modified because blat verify is removed
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/vqsr.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir}/variant";
	$cmd .= " -s=$sample";

	execute_qsub_cmd($cmd);

	return $name;
}

#############################################################################
sub run_tophat_post {
	my $hold = shift;

	my $job_name = $this->{job} .".TophatFusionPost.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{fusionreport_mem}";
		#$cmd .= " -pe threaded $config->{MemoryInfo}->{threads} -N $job_name";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/tophat_post_process.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir}/alignment";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_genecount_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".GeneCount.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{gene_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/gene_count.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_exoncount_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".ExonCount.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{exon_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/exon_count.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_umapreads_module {
	my $sample = shift;
	my $hold = shift;

	my @chr = split(/:/, $config->{RunInfo}->{chrindex});

	my $job_name = $this->{job} .".QC.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{unmapread_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}

		$cmd .= " -t 1-".scalar(@chr);

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/unique_mapreads_count.pl";
		$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";
		execute_qsub_cmd($cmd);

	} else {
		foreach my $id (1..scalar(@chr)) {
			$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/unique_mapreads_count.pl";
			$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample -c=$id";
			execute_qsub_cmd($cmd);
		}
	}


	return $job_name;
}

#############################################################################
sub run_numbers_report {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".NumbersReport.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{gen_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		$hold =~ s/,$//;
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/numbers_report.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_mreadplot_module {
	my $hold = shift;

	my $job_name = $this->{job} .".MapReadsPlot.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{mread_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/mapped_reads_plot.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir}";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_mergecount_module {
	my $hold = shift;

	my $job_name = $this->{job} .".MergeCounts.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{mergecount_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/merge_counts.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir}";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_report_module {
	my $hold = shift;

	my $job_name = $this->{job} .".FinalReports.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " $config->{MemoryInfo}->{report_mem}";
		$cmd .= " -N $job_name";

		#### check all other job_ids before executing reports
		opendir(DIR, "$this->{output_dir}/job_ids")
			or $logger->logdie("Could not open dir $this->{output_dir}/job_ids");

		while(my $idFiles = readdir(DIR)) {
			next if ($idFiles !~ /\.ids$/);

			open(FHD, "<", "$this->{output_dir}/job_ids/$idFiles")
				or $logger->logdie("Could not open file to read $this->{output_dir}/job_ids/$idFiles");
			while(<FHD>) {
				chomp $_;

				$hold .= "," . $_;
			}
			close(FHD);

			$hold =~ s/,,/,/g;  ## remove empty lines from ids file.
			$hold =~ s/,$//;
		}


		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/reports.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir}";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_cleanup_module {
	my $hold = shift;

	my $job_name = $this->{job} .".Cleanup.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args} -l h_vmem=1G -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/cleanup.pl";
	$cmd .= " -r=$this->{output_dir}/run_info.txt -o=$this->{output_dir}";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub execute_qsub_cmd {
	my $cmd = shift;

	#### check if running on the grid
	if (! $this->{run_on_sge}) {
		execute_cmd($cmd);
		return;
	}

	my $count = 0;
	my $send_mail = 1;

	#### check if job limit is hit
	check_job_limit();

	sleep $WAIT;

	$logger->info("$cmd");
	system($cmd);

	while ((( $? >> 8 ) != 0 ) && ($count < 5)){
		sleep $SHORT_WAIT;
		$count ++;

		system($cmd);
	}

	if (( $? >> 8 ) != 0 ) {
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/Cmd.err";
		$util->createErrorFile($error_name, $cmd);

		while (-e $error_name) {

			#### pass Filename, cmd executed, email to, step running, sample name.
			if ($send_mail) {
				$util->reportError($error_name,
							   $cmd,
							   $config->{RunInfo}->{email},
							   "Executing command",
							   $options{sample},
							   "",
							   "");
				$send_mail = 0;
			}
			sleep $LONG_WAIT;
		}
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	print "Command \"$cmd\"\n";
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub check_job_limit {
	my $cmd = "$config->{ToolInfo}->{sge}->{value}/qstat";

	my @cmd_output = qx{$cmd};

	#### wait till number of jobs in queue are less than job limit to
	#### continue execution.
	while (scalar(@cmd_output) >= $MAX_JOB_LIMIT) {
		sleep $LONG_WAIT;

		@cmd_output = qx{$cmd};
	}
}
