#!/usr/bin/perl

=head1 NAME
   merge_counts.pl

=head1 SYNOPSIS
    USAGE: merge_counts.pl -r=run_info.txt -o=output_dir

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Merge all gene and exon counts in one reports file

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./merge_counts.pl -r=run_info.txt -o=output_dir

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### make sure everything passed was peachy
&check_parameters(\%options);

#### parse X_info files
my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/MergeCounts.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Merge counts started");

create_dir_struct(\%options);

my $this = {};
$this->{input} = "$options{output_dir}/counts";
$this->{output} = "$options{output_dir}/Reports";

#### create list of all individual sample gene and exon list
opendir(DIR, $this->{input}) or $logger->logdie("Could not open dir $this->{input}");
open(EX, ">", "$this->{input}/exon.file.list") or $logger->logdie("Could not create exon file list");
open(GE, ">", "$this->{input}/gene.file.list") or $logger->logdie("Could not create gene file list");

while(my $file = readdir(DIR)){
	if ($file =~ /\.gene\.count\.tsv$/) {
		print GE "$this->{input}/$file\n";
	} elsif ($file =~ /\.exon\.count\.tsv$/) {
		print EX "$this->{input}/$file\n";
	}
}

close(DIR);
close(EX);
close(GE);

my $cmd="";

#### merge all gene counts and calculate RPKM value
$cmd = "perl $config->{ToolInfo}->{workflow_path}->{value}/merge_gene.pl -i=$this->{input}/gene.file.list";
$cmd .= " -g=$config->{ToolInfo}->{gene_mapping}->{value} -o=$this->{output}/GeneCount.tsv";
execute_cmd($cmd);

if (! -s "$this->{output}/GeneCount.tsv") {
	$logger->logdie("Error creating merged gene count list");
}

#### merge all exon counts
$cmd = "perl $config->{ToolInfo}->{workflow_path}->{value}/merge_exon.pl -i=$this->{input}/exon.file.list";
$cmd .= " -o=$this->{output}/ExonCount.tsv";
execute_cmd($cmd);

if (! -s "$this->{output}/ExonCount.tsv") {
	$logger->logdie("Error creating merged exon count list");
}

$cmd = "rm $this->{input}/exon.file.list | rm $this->{input}/gene.file.list";
execute_cmd($cmd);

$logger->info("Merge counts completed");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/Reports";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}
