#!/usr/bin/perl

## Saurabh Baheti 09/22/2010
## Script to make a html report for mRNA seq runs

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/MainDocument.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Generate main html started");


my @sampleArray = split(/:/, $config->{RunInfo}->{samplenames});
my $num_samples=scalar(@sampleArray);
@sampleArray = sort(@sampleArray);

my $displayName = {};
for(my $i=0; $i<scalar(@sampleArray); $i++){
	#$displayName->{$sampleArray[$i]} = "Sample_".($i+1);
	$displayName->{$sampleArray[$i]} = $sampleArray[$i];
}

my $output = "$options{output_dir}/Main_Document.html";

open (OUT,">", $output)
	or $logger->logdie("Could not open file $output");

print OUT "<html>";
print OUT "<head>";
print OUT "<title>mRNAseq Main Document for $config->{RunInfo}->{output_folder}</title>";

print OUT getStyle();

print OUT qq{<link rel="stylesheet" href="fancybox/source/jquery.fancybox.css?v=2.0.6" type="text/css" media="screen" />
			<link rel="stylesheet" href="fancybox/source/helpers/jquery.fancybox-buttons.css?v=1.0.2" type="text/css" media="screen" />
			<script type="text/javascript" src="http://ajax.googleapis.com/ajax/libs/jquery/1.7/jquery.min.js"></script>
			<script type="text/javascript" src="fancybox/source/jquery.fancybox.pack.js?v=2.0.6"></script>
			<script type="text/javascript" src="fancybox/source/helpers/jquery.fancybox-buttons.js?v=1.0.2"></script>

			<script type="text/javascript">
				\$(document).ready(function() {
					\$(".fancybox").fancybox({
								openEffect	: 'none',
								closeEffect	: 'none'
							});
				});
			</script>};

print OUT "</head>";
print OUT "<body>";

print OUT "<div id='pageTitle'><a name=\"DocumentStart\">&#167;&#167;&#167; Mayo BIC PI Support &#167;&#167;&#167;</a></div>";
## making the index for the document
print OUT "<span id=\"top\"><a href=\"#DocumentStart\">[ TOP ]</a></span>";
print OUT qq{<div id="toc">
				<span class="heading">Contents</span>
				<div class="section">
					<ul>
						<li class="toclevel-1">
							<a href="#Project Title"><span class="tocnumber">1</span>
							<span class="toctext">Project Title</span></a>
						</li>
						<li class="toclevel-1">
							<a href="#Initial Meetings and Timeline"><span class="tocnumber">2</span>
							<span class="toctext">Initial Meetings and Timeline</span></a>
						</li>
						<li class="toclevel-1">
							<a href="#Project Description"><span class="tocnumber">3</span>
							<span class="toctext">Project Description</span></a>
						</li>
						<ul>
							<li class="toclevel-2">
								<a href="#Goals of the project"><span class="tocnumber">3.1</span>
								<span class="toctext">Goals of the project</span></a>
							</li>
						</ul>
						<li class="toclevel-1">
							<a href="#Analysis plan"><span class="tocnumber">4</span>
							<span class="toctext">Analysis plan</span></a>
						</li>
						<li class="toclevel-1">
							<a href="#Received Data"><span class="tocnumber">5</span>
							<span class="toctext">Received Data</span></a>
						</li>
						<ul>
							<li class="toclevel-2">
								<a href="#Sample Summary"><span class="tocnumber">5.1</span>
								<span class="toctext">Sample Summary</span></a>
							</li>
						</ul>
						<li class="toclevel-1">
							<a href="#Results Summary"><span class="tocnumber">6</span>
							<span class="toctext">Results Summary</span></a>
						</li>
						<ul>
							<li class="toclevel-2">
								<a href="#QC steps"><span class="tocnumber">6.1</span>
								<span class="toctext">QC steps</span></a>
							</li>
							<li class="toclevel-2">
								<a href="#Statistics based on per sample analysis"><span class="tocnumber">6.2</span>
								<span class="toctext">Statistics based on per sample analysis</span></a>
							</li>
						</ul>
						<li class="toclevel-1">
							<a href="#Appendix"><span class="tocnumber">7</span>
							<span class="toctext">Appendix</span></a>
						</li>
					</ul>
				</div>
			</div>\n};


## project title
print OUT qq{<div>
				<a name="Project Title" id="Project Title"></a>
				<span class="heading">I. Project Title : </span>
				<div class="section">NGS Bioinformatics for $config->{RunInfo}->{type} sequencing</div>
			</div><br/>\n};


## projet background
my $read_call;	if($config->{RunInfo}->{paired} == 1)	{	$read_call = 'PE';	}	else	{	$read_call = 'SR';	}
my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime;   ## to pull todays date
$year += 1900;$mon++;

my $delivery=$config->{RunInfo}->{delivery_folder};
my $tertiary=$config->{RunInfo}->{tertiary_folder};
$delivery =~ s/^\///;
$tertiary =~ s/^\///;
$delivery =~ s/\//\\/g;
$tertiary =~ s/\//\\/g;

print OUT qq{<div>
				<a name="Project Description" id="Project Description"></a>
				<span class="heading">II. Project Description</span>

				<div class="section">
					<a name="Background" id="Background"></a>1. Background
					<br/><br/>
					<table>
						<tr>
							<th class="border">item</th>
							<th class="border">description</th>
						</tr>
						<tr>
							<td class="border">Disease Type</td>
							<td class="border">$config->{RunInfo}->{disease}</td>
						</tr>
						<tr>
							<td class="border">Number of Samples</td>
							<td class="border">$num_samples</td>
						</tr>
						<tr>
							<td class="border">Read Length</td>
							<td class="border">$config->{RunInfo}->{readlength}</td>
						</tr>
						<tr>
							<td class="border">PairedEnd(PE)/SingleRead(SR)</td>
							<td class="border">$read_call</td>
						</tr>
						<tr>
							<td class="border">Genome Build (hg18/hg19)</td>
							<td class="border">$config->{RunInfo}->{genomebuild}</td>
						</tr>
						<tr>
							<td class="border">StartDate</td>
							<td class="border">$config->{RunInfo}->{date}</td>
						</tr>
						<tr>
							<td class="border">EndDate</td>
							<td class="border">$mon/$mday/$year</td>
						</tr>
					</table>
					<span class="note">
					Note: Further raw NGS data will be used for statistical analysis<br/>
					Further analysis space:: \\\\rcfcluster-cifs\\$tertiary <br/><br/>
					Data is available for 60 Days from the Delivered Date at<br/>
					Delivery space:: \\\\rcfcluster-cifs\\$delivery
					</span>
				</div>
			};


print OUT qq{	<div class="section">
					<a name="Study design" id="Study design"></a>2. Study design
					<ul>
						<li class="ques">What are the samples?</li>
						<li class="ans">$config->{RunInfo}->{sampleinformation}</li>

						<li class="ques">Goals of the project</li>
						<li class="ans">Identify how well $config->{RunInfo}->{type} sequencing worked on these samples,and obtain Gene and exon expression</li>
					</ul>
				</div>
			</div>
		};


print OUT qq{<div>
				<a name="Analysis Plan" id="Analysis Plan"></a>
				<span class="heading">III. Analysis Plan</span>

				<div class="section">
					<p align="center">
						<img src="$config->{RunInfo}->{type}_workflow.png" width="600">
					</p>
				</div>
			</div>
			};

print OUT qq{<div>
				<a name="Received Data" id="Received Data"></a>
				<span class="heading">IV. Received Data</span>

				<div class="section">
					<ul>
						<li class="ques">Run Name </li>
						<li class="ans">$config->{RunInfo}->{output_folder}</li>
			};

#### printing the samples information table
print OUT qq{<a name="Sample Summary" id="Sample Summary"></a><li class="ques">Sample Summary</li>
			<br><table>
				<tr>
					<th class="border">Lane</th>
					<th class="border">Index</th>
					<th class="border">Sample Names</th>
				</tr>
			};

foreach my $sample (sort keys %{$config->{SampleInfo}}) {
	print OUT qq{
			<tr>
				<td class="border">$config->{SampleInfo}->{$sample}[0]->{laneindex}</td>
				<td class="border">$config->{SampleInfo}->{$sample}[0]->{labindex}</td>
				<td class="border">$displayName->{$sample}</td>
			</tr>
		};
}
print OUT "</table></ul></div></div>";

my $fqc_file = 0;
if (-d "$options{output_dir}/fastqc") {
	opendir(IMD, "$options{output_dir}/fastqc");
	my @thefiles= readdir(IMD);
	close(IMD);

	if (scalar(@thefiles)) {
		$fqc_file = 1;
	}
}

print OUT qq{<br/><div>
				<a name="Results Summary" id="Results Summary"></a>
				<span class="heading">V.  Results Summary:</span>

				<div class="section">
					<ul>
						<li class="ques"><a name="QC steps" id="QC steps"></a> QC steps - FastQC-report</li>
						<li class="ans">
							FastQC aims to provide a simple way to do some quality control checks on raw sequence
							data coming from high throughput sequencing pipelines. It provides a modular set of
							analyses which you can use to give a quick impression of whether your data has any
							problems of which you should be aware before doing any further analysis.<br/>
			};

			if ($fqc_file) {
				print OUT qq {<a href="fastqc/" target="_blank">FastQC Reports</a>};
			} else {
				print OUT qq {<font color="#FF3300">FastQC Reports are not available</font>};
			}

			print OUT qq {
						</li>

						<li class="ques">
							<a name="Statistics based on per Sample Analysis" id="Statistics based on per Sample Analysis"></a>
							Statistics based on per Sample Analysis (<a href="StatisticsDescription.html" target="_blank">ColumnDescription</a>)
						</li>
						<li class="ans">
							Analysis is carried out using fastq sequence files as input and generates output tables.
							For paired-end runs, the tables contain counts for each sample combined from both reads.
						</li>
						};

my $sample_numbers;
my $uniq;

foreach my $sample (keys %{$config->{SampleInfo}}) {
	open(SAMPLE, "<", "$options{output_dir}/numbers/$sample.out")
		or $logger->logdie("Could not open $options{output_dir}/numbers/$sample.out\n$!");

	my @data;
	while(<SAMPLE>) {
		chomp $_;
		push @data, $_;
	}
	close(SAMPLE);

	my $hash;
	map { $hash->{$1} = $2 if( /([^=]+)\s*=\s*([^=]+)/ ) } @data;

	$sample_numbers->{$sample} = $hash;
}

print OUT qq{<table>
				<tr>
					<th class="border">sample(s)</th>
					<th class="border">total reads</th>
					<th class="border">used reads</th>
					<th class="border">mapped reads</th>
					<th class="border">mapped reads (genome)</th>
					<th class="border">mapped reads (junction)</th>
					<th class="border">gene count</th>
					<th class="border">exon count</th>
				</tr>
			};
foreach my $key (sort keys %$sample_numbers) {
	print OUT "<tr>";
	print OUT qq{<td class="border">$key</td>};
	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Total reads'}). "</td>";
	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Used reads'}) . "</td>";

	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Mapped reads'}) .
					" (" . sprintf("%.1f",($sample_numbers->{$key}{'Mapped reads'}/$sample_numbers->{$key}{'Total reads'})*100) .")</td>";

	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Mapped reads (Genome)'}) .
					" (" . sprintf("%.1f",($sample_numbers->{$key}{'Mapped reads (Genome)'}/$sample_numbers->{$key}{'Total reads'})*100) .")</td>";

	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Mapped reads (Junction)'}) .
					" (" . sprintf("%.1f",($sample_numbers->{$key}{'Mapped reads (Junction)'}/$sample_numbers->{$key}{'Total reads'})*100) .")</td>";

	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Gene count'}) .
					" (" . sprintf("%.1f",($sample_numbers->{$key}{'Gene count'}/$sample_numbers->{$key}{'Total reads'})*100) .")</td>";

	print OUT "<td class=\"border\">" .CommaFormatted($sample_numbers->{$key}{'Exon count'}) . "</td>";
	#				" (" . sprintf("%.1f",($sample_numbers->{$key}{'Exon count'}/$sample_numbers->{$key}{'Total reads'})*100) .")</td>";
	print OUT "</tr>";
}
print OUT "</table><br/></ul></div></div>";


print OUT qq{<div>
				<a name="Results Delivered" id="Results Delivered"></a>
				<span class="heading">VI. Results Delivered</span>

				<div class="section">
					<ul>
						<li class="ans">The following three sets of tables are delivered and column description is available in Appendix.</li>
						<li class="ques">
							<a href="Reports/ExonCount.tsv" target="_blank">Exon table</a>: contains counts for the number of times an exon has been detected
						</li>
						<li class="ans">count (raw) = sum of exon read counts</li>
						<li class="ans">count (RPKM)</li>

						<li class="ques">
							<a href="Reports/GeneCount.tsv" target="_blank">Gene table</a>: contains counts for the number of times a gene copy has been detected
						</li>
						<li class="ans">
							count (raw) = sum of exon read counts, with an exception that if reads start in different exons of the same gene twice, these are counted only once for the gene
						</li>
				};


print OUT qq{<li class="ques">RSeQC Plots:</li><br>
			<li class="ans">
			<table cellpadding="0" cellspacing="0" paddingBottom="20" width="100%">
			<tr>
				<th class="border">Sample Name</th>
				<th class="border">Junction Saturation</th>
				<th class="border">Splicing Junction</th>
				<th class="border">Splicing Event</th>
				<th class="border">Inner distance</th>
				<th class="border">Read duplication</th>
				<th class="border">Gene body coverage</th>
			</tr>};

foreach my $sample (sort keys %{$config->{SampleInfo}}) {
	print OUT "<tr>";
	#print OUT "<td class=\"border\">".uc($sample)."</td>";
	print OUT "<td class=\"border\">".uc($displayName->{$sample})."</td>";
	print OUT qq{<td class="border">
					<a class="fancybox" rel="$sample" href="RSeQC/$sample/$sample.junctionSaturation_plot.png" title="Junction saturation [sample id: $sample]">
						<img src="RSeQC/$sample/$sample.junctionSaturation_plot.png" width="200" />
					</a>
				</td>};
	print OUT qq{<td class="border">
					<a class="fancybox" rel=\"$sample\" href="RSeQC/$sample/$sample.splice_junction_plot.png" title="Junction annotation (splicing junction) [sample id: $sample]">
						<img src="RSeQC/$sample/$sample.splice_junction_plot.png" width="200" />
					</a>
				</td>};
	print OUT qq{<td class="border">
					<a class="fancybox" rel="$sample" href="RSeQC/$sample/$sample.splice_events_plot.png" title="Junction annotation (splicing event) [sample id: $sample]">
						<img src="RSeQC/$sample/$sample.splice_events_plot.png" width="200" />
					</a>
				</td>};
	print OUT qq{<td class="border">
					<a class="fancybox" rel="$sample" href="RSeQC/$sample/$sample.inner_distance_plot.png" title="Inner distance [sample id: $sample]">
						<img src="RSeQC/$sample/$sample.inner_distance_plot.png" width="200" />
					</a>
				</td>};
	print OUT qq{<td class="border">
					<a class="fancybox" rel="$sample" href="RSeQC/$sample/$sample.DupRate_plot.png" title="Read duplication [sample id: $sample]">
						<img src="RSeQC/$sample/$sample.DupRate_plot.png" width="200" />
					</a>
				</td>};
	print OUT qq{<td class="border">
					<a class="fancybox" rel="$sample" href="RSeQC/$sample/$sample.geneBodyCoverage.png" title="Gene body coverage [sample id: $sample]">
						<img src="RSeQC/$sample/$sample.geneBodyCoverage.png" width="200" />
					</a>
				</td>};
	print OUT "</tr>";
}
print OUT qq{</table> </li>};

if (! -s "$options{output_dir}/fusion/result.txt") {
	print OUT qq{<br/>
		<li class="ques">Tophat fusion (circos plot and raw files): </li>
		<li class="ans">There are NO fusion candidates.</li>
	};
} else {
	print OUT qq{<br/>
		<li class="ques">Tophat fusion (circos plot and raw files): </li>
		<li class="ans">Tab delimited fusion results are available <a href="fusion/tophat_fusion_report.txt" target="_blank">here</a></li>
		<li class="ans">Detail fusion report in HTML format is available <a href="fusion/result.html" target="_blank">here</a></li>
		<li class="ans">
			<table>
				<tr>
					<th class="border">Circos plot</th>
				<tr>
				<tr>
					<td class="border">
						<a class="fancybox" rel="Circos" href="fusion/circos_fusion_all.png" title="Fusion candidate(s) over all Chrs all samples">
							<img src="fusion/circos_fusion_all.png" width="500" />
						</a>
					</td>
				</tr>
			</table>
		</li>
	};
}

print OUT qq {
	<li class="ques">Statistics based on per Sample Analysis are recorded in the tab delimited file <a href="SampleStatistics.tsv" target="_blank">Details</a></li>
	<br/>
	<li class="ques">IGV Visualization</li>
	<li class="ans">
		The SNV annotation reports (both standard and filtered) include visualization links to IGV
		to enable a realistic view of the variants. Please follow steps in the following link to setup IGV
		(takes less than 5 minutes) and utilize this feature.
		<a href="IGV_Setup.doc" target="_blank">IGV setup for variant visualization</a>
	</li></ul></div></div>
};

print OUT qq{<div>
				<a name="Useful Links" id="Useful Links"></a>
				<span class="heading">VII. Useful Links</span>

				<div class="section">
					<ul>
						<li><a href="http://tophat.cbcb.umd.edu/" target="_blank">Tophat 2</a>
						<li><a href="http://tophat.cbcb.umd.edu/fusion_index.html" target="_blank">Tophat Fusion</a>
						<li><a href="http://www-huber.embl.de/users/anders/HTSeq/doc/overview.html" target="_blank">HtSeq</a>
						<li><a href="http://code.google.com/p/rseqc" target="_blank">RSeQC</a>
						<li><a href="http://www.broadinstitute.org/gatk" target="_blank">GATK</a>
						<li><a href="http://genome.ucsc.edu/" target="_blank">UCSC Genome Browser</a>
					</ul>
				</div>
			</div>

			<div>
				<a name="Appendix" id="Appendix"></a>
				<span class="heading">VIII. Appendix</span>

				<div class="section">
					Full Length cDNA Sequencing (mRNA- Seq) results delivery format (<a href="Appendix.html" target="_blank">Appendix</a>)
				</div>
			</div>
		};


print OUT qq{<div>
				<span class="heading">Authorship Consideration</span>
				<div class="section">
					Advancing scientific research is a primary motivation of all bioinformaticians and acknowledgment
					of contribution through authorship on manuscripts arising from this analysis is one way our work
					is assessed and attributed. We request to be considered for authorship on any manuscripts using
					the analysis results provided if you believe we have made substantive intellectual contributions to the study.
				</div>
			</div>

			<p align="center">&#167;&#167;&#167; <b>Powered by Mayo BIC PI Support</b> &#167;&#167;&#167;</p>
		};

print OUT qq{</body> </html>};
close(OUT);

desc();
apex();

$logger->info("Generate main html complete");

exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub parseInfoFile{
	my $file = $_[0];
	my $hash = ();

	open(FHD, "<", $file) or die "Could not open info file $file\n$!\n";

	while (<FHD>){
		next if ($_ =~ /^#/);

		chomp $_;
		my ($key, $value) = split(/=/,$_);
		$hash->{$key} = $value;
	}

	return $hash;
}

#############################################################################
sub CommaFormatted
{
	my $number = shift;

	#### if number is empty
	$number = 0 if ((!defined $number) || length($number) == 0);

	my $delimiter = ','; # replace comma if desired
	my($n,$d) = split(/\./, $number, 2);


	my @a = ();
	while($n =~ /\d\d\d\d/)
	{
		$n =~ s/(\d\d\d)$//;
		unshift @a,$1;
	}

	unshift @a, $n;
	$n = join($delimiter,@a);

	$n = "$n\.$d" if ((defined $d) && (length($d)) && ($d =~ /\d+/));

	return $n;
}


#############################################################################
sub getStyle {
	my $str = "";

	$str = qq {<style>
					*{
						outline:none;
					}
					body{
						font-family: Arial, "Times New Roman", sans-serif;
						font-size: 14px;
					}
					img{
						border:none;
					}
					p{
						font-family: Arial, "Times New Roman", sans-serif;
						font-size:14px;
						color:#5e5e5e;
						line-height:20px;
						margin-bottom:5px;
					}
					a {
						color: #0e94b6;
						text-decoration: none;
					}
					table{
						border-collapse: collapse;
						border-spacing: 0px;
						text-align : center;
					}
					table td{
						font-family: Arial, "Times New Roman", sans-serif;
						font-size: 14px;
						padding: 5px 5px 5px 5px;
						text-align : center;
					}
					table th{
						font-family: Arial, "Times New Roman", sans-serif;
						font-size: 16px;
						font-variant: small-caps;
						white-space:nowrap;
						background-color: #edf3fe;
						padding: 5px 5px 5px 5px;
						text-align: center;
					}
					th.border, td.border {
						border-bottom: 1px solid #666666;
						border-right: 1px solid #666666;
					}
					.left {
						text-align: left;
					}
					div {
						padding: 0px 0px 10px 0px;
					}
					li.ques {
						font-weight: bold;
					}
					li.ans {
						list-style-type: none;
						padding-bottom: 10px;
					}
					.heading{
						font-family: Arial, "Times New Roman", sans-serif;
						font-size: 16px;
						font-weight: bold;
						text-align: left;
					}
					.section {
						padding: 15px 0px 0px 25px;
					}
					#pageTitle {
						font-family: Arial, "Times New Roman", sans-serif;
						font-size: 20px;
						font-weight: bold;
						text-align: center;
					}
					#top {
						position: fixed;
						right: 5px;
						z-index: 999;
						top: 40%;
						color: #FF3300;
					}
					#top a {
						color: #FF3300;
					}
					.note {
						font-style: italic;
						font-size: 12px;
					}
				</style>};

	return $str;
}


#############################################################################
sub desc {
	my $desc = "$options{output_dir}/StatisticsDescription.html";

	open (DESC, ">", $desc)
		or $logger->logdie("Can not open file $desc : $!\n");

	print DESC "<html>";
	print DESC "<head>";
	print DESC "<title>$config->{RunInfo}->{type} Column Description for the Statistics per sample</title>";
	print DESC getStyle();
	print DESC "</head>";

	print DESC qq{<span class="heading">Row description for Results Summary Table:</span>\n};

	print DESC qq{
	<div class="section">
		<table>
			<tr>
				<th class="border">column</th>
				<th class="border">description</th>
			<tr>
			<tr>
				<th class="border left">total number of reads</th>
				<td class="border left">Total number of reads obtained</td>
			</tr>
			<tr>
				<th class="border left">total number of used reads</th>
				<td class="border left">Total number of reads used for the alignemnt</td>
			</tr>
			<tr>
				<th class="border left">mapped reads</th>
				<td class="border left">Number of reads that mapped to either the genome or junction reference database</td>
			</tr>
			<tr>
				<th class="border left">genome mapped reads</th>
				<td class="border left">Mapped Genome Reads and percentage (Mapped Genome Reads/Total number of reads)</td>
			</tr>
			<tr>
				<th class="border left">junction mapped reads</th>
				<td class="border left">Mapped Jucntion Reads and percentage (Mapped Junction Reads/ Total number of reads)</td>
			</tr>
			<tr>
				<th class="border left">gene counts </th>
				<td class="border left">Number of base-pairs of sequencing data mapping to the exon regions of UCSC annotated genes from the refFlat file.</td>
			</tr>
			<tr>
				<th class="border left">exon counts </th>
				<td class="border left">Number of base-pairs of sequencing data mapping to the exon regions of UCSC annotated genes from the refFlat file. May contain redundant counts due to overlapping exons.</td>
			</tr>
		</table>
	</div>
	</body>
	</html>
	};

	close DESC;
}


#############################################################################
sub apex {

	my $apen = "$options{output_dir}/Appendix.html";

	open(APEN, ">", $apen)
		or $logger->logdie("Can not open file $apen : $!\n");

	print APEN "<html>";
	print APEN "<head>";
	print APEN "<title>$config->{RunInfo}->{type} Column Description for the Expression Counts</title>";
	print APEN getStyle();
	print APEN "</head>";
	print APEN "<body>";

	print APEN qq{
	<span class="heading">The format of the tables delivered are as follows:</span>
	<div class="section">
		<ul>
			<li class="ques">Exon Table</li>
			<li class="ans">
				<table>
					<tr>
						<th class="border">chrid</th>
						<th class="border">start</th>
						<th class="border">stop</th>
						<th class="border">gene</th>
						<th class="border">sample-1_exoncount</th>
						<th class="border">sample-1_rpkm</th>
						<th class="border">...</th>
						<th class="border">sample-N_exoncount</th>
						<th class="border">sample-N_rpkm</th>
					</tr>
				</table>
				<br>

				<b>Column description for Exon Table:</b>

				<table>
					<tr>
						<th class="border">column</th>
						<th class="border">description</th>
					<tr>
					<tr>
						<th class="border left">chrid</th>
						<td class="border left">Chr Index for the Gene</td>
					</tr>
					<tr>
						<th class="border left">start</th>
						<td class="border left">Start Genomic position for exon</td>
					</tr>
					<tr>
						<th class="border left">stop</th>
						<td class="border left">Stop Genomic position fo exon</td>
					</tr>
					<tr>
						<th class="border left">gene</th>
						<td class="border left">Symbol of the gene</td>
					</tr>
					<tr>
						<th class="border left">sample-N_ExonCount</th>
						<td class="border left">Sum of exon read counts of Nth sample</td>
					</tr>
					<tr>
						<th class="border left">sample-N_rpkm</th>
						<td class="border left">RPKM value for exon read counts of Nth sample</td>
					<tr>
				</table>
			</li>

			<li class="ques">Gene Table</li>
			<li class="ans">
				<table>
					<tr>
						<th class="border">chrid</th>
						<th class="border">gene</th>
						<th class="border">sample-1_genecount</th>
						<th class="border">sample-1_rpkm</th>
						<th class="border">...</th>
						<th class="border">sample-N_genecount</th>
						<th class="border">sample-N_rpkm</th>
					</tr>
				</table>
				<br>

				<b>Column description for Gene Table:</b>

				<table>
					<tr>
						<th class="border">column</th>
						<th class="border">description</th>
					<tr>
					<tr>
						<th class="border left">chrid</th>
						<td class="border left">Chr Index for the Gene</td>
					</tr>
					<tr>
						<th class="border left">gene</th>
						<td class="border left">Symbol of the gene</td>
					</tr>
					<tr>
						<th class="border left">sample-N_genecount</th>
						<td class="border left">Sum of gene read counts of Nth sample</td>
					</tr>
					<tr>
						<th class="border left">sample-N_rpkm</th>
						<td class="border left">RPKM value for gene read counts of Nth sample</td>
					<tr>
				</table>
			</li>
		</ul>
	</div>
	</body>
	</html>};

	close(APEN);
}
