#!/usr/bin/perl

=head1 NAME
   tophat_preprocessing.pl

=head1 SYNOPSIS
    USAGE: tophat_preprocessing.pl -r=run_info.txt -o=output_dir -s=sample -t=tophat_args -f=fusion_args

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Run Tophat Step 1 PreProcessing

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./tophat_preprocessing.pl -r=run_info.txt -o=output_dir -s=sample -t=tophat_args -f=fusion_args

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
############################################
my $LONG_WAIT = 300;

## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/UnifiedGenotyper.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Unified Genotyper started");

my $count = 0;
my $cmd = "";

#### check input
check_input("$options{output_dir}/alignment/tophat_$options{sample}/$options{sample}-sorted.unique.nonF.bam");

while (($count < 5) && (! -s "$options{output_dir}/variant/$options{sample}.gatk.vcf.idx")){
	### run gatk unified genotyper
	$cmd = "$config->{ToolInfo}->{java}->{value}/java";
	$cmd .= " -Xmx$config->{MemoryInfo}->{gatk_jvmx} -Xms512m";
	$cmd .= " -Djava.io.tmpdir=$options{output_dir}/variant/temp";
	$cmd .= " -jar $config->{ToolInfo}->{gatk}->{value}/GenomeAnalysisTK.jar";
	$cmd .= " -R $config->{ToolInfo}->{ref_genome}->{value}";
	$cmd .= " " . $config->{ToolInfo}->{gatk_ug_param}->{value} ; 

	if ((defined $config->{ToolInfo}->{master_gene_file}->{value}) &&
		(length($config->{ToolInfo}->{master_gene_file}->{value}))) {
		$cmd .= " -L $config->{ToolInfo}->{master_gene_file}->{value}";
	}

	$cmd .= " -T UnifiedGenotyper";
	$cmd .= " -I $options{output_dir}/alignment/tophat_$options{sample}/$options{sample}-sorted.unique.nonF.bam";

	if ((defined $config->{ToolInfo}->{dbsnp_ref}->{value}) &&
		(length($config->{ToolInfo}->{dbsnp_ref}->{value}))) {
		$cmd .= " --dbsnp $config->{ToolInfo}->{dbsnp_ref}->{value}";
	}

	$cmd .= " --out $options{output_dir}/variant/$options{sample}.gatk.vcf";

	execute_cmd($cmd);

	$count++
}

if (! -s "$options{output_dir}/variant/$options{sample}.gatk.vcf.idx"){
	$logger->logdie("GATK unified genotyper executed unsuccessfully");
}

$logger->info("Unified Genotyper complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);
}

#############################################################################
sub check_input {
	my $file = shift;

	my $send_mail = 1;

	while (! -s $file) {
		if ($send_mail) {
			$send_mail = 0;

			$util->missingInput($config->{RunInfo}->{email},
								"UnifiedGenotyper",
								"SortBam",
								"$file",
								$config->{RunInfo}->{tool});
		}

		sleep $LONG_WAIT;
	}
}
