#!/usr/bin/perl

=head1 NAME
   qc_readdist.pl

=head1 SYNOPSIS
    USAGE: qc_gene_noread_map.pl -r=run_info.txt -o=output_dir

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Generate plot of reads not mapped to any gene

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./qc_gene_noread_map.pl -r=run_info.txt -o=output_dir

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/GeneNoReadMappedQC.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Generate reports started");

$ENV{PATH} = $config->{ToolInfo}->{r}->{value} .":". $ENV{PATH};
create_dir_struct(\%options);

my $this = {};
$this->{output} = "$options{output_dir}/QC";
$this->{gene} = "$options{output_dir}/Reports/GeneCount.tsv";

chdir $this->{output};

if (! -s $this->{gene}) {
	$logger->logdie("$this->{gene} not exist");
}

open(SMN, ">", "$this->{output}/samples")
	or $logger->logdie("Could not open file to write $this->{output}/samples");
open(CNT, ">", "$this->{output}/numbers")
	or $logger->logdie("Could not open file to write $this->{output}/numbers");

print CNT "Count\n";

foreach my $sample (keys %{$config->{SampleInfo}}) {
	my $col_num = `awk  -F '\\t' -v pat=\"${sample}_GeneCount\" '{ for(i=1;i<=NF;i++){ if (\$i ~ pat) {print i} } }' $this->{gene}`;
	chomp $col_num;

	my $num_genes = `cat $this->{gene} | awk 'NR>1' | awk '{if (\$$col_num == 0) print \$2}' | wc -l`;

	print SMN $sample."\n";
	print CNT $num_genes."\n";
}
close(SMN);
close(CNT);

my $cmd = "Rscript $config->{ToolInfo}->{workflow_path}->{value}/rscript/QC_mRNASeq_noreadsmapped_Rplot.r";
$cmd .= " $this->{output}/numbers $this->{output}/samples";
execute_cmd($cmd);

if (-s "$this->{output}/GeneCount_NoReadsMapped.png") {
	#$cmd = "rm $this->{output}/numbers $this->{output}/samples";
	#execute_cmd($cmd);
	;
} else {
	$logger->logdie("Problem generating $this->{output}/QC_AllSamples_GeneCount_NoReadsMapped.png");
}

$logger->info("Generate reports complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}
#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/QC";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}
