#!/usr/bin/perl

=head1 NAME
   circos-fusion.pl

=head1 SYNOPSIS
    USAGE: circos-fusion.pl -r=run_info.txt -o=output_dir -f=fusion.out

=head1 OPTIONS

B<--run_info, -r>
	Required. Run info file

B<--output_dir, -o>
	Required. Output directory

B<--fusion, -f>
	Required. Full path to fusion.out file

B<--help,-h>


=head1  DESCRIPTION
	Create circos plot of all fusions

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./circos-fusion.pl -r=run_info.pl -o=output_dir -f=fusion

=cut

use strict;
use warnings;
use Getopt::Long;
use Pod::Usage;
use IO::File;
use File::Temp qw(tempfile);
use File::Spec;
use File::Slurp;
use List::MoreUtils qw/ uniq /;
use HTML::TableExtract;
use CircosUtil;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'fusion|f=s',
                          'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
############################################
my $LONG_WAIT = 300;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/CircosFusion.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Circos fusion Process started");

my $fuse = "$options{output_dir}/fuse.dat";
my $gene = "$options{output_dir}/gene.dat";

#### check input file
check_input($options{fusion});

#### eventually pull the extraction code in here or in the utils file
#### and do magic. that would mean changing the options and stuff.
#### extract_fusion_info($results_file, $fusion, $gene_locations);

&extract_fusion_info($options{fusion}, $fuse, $gene);

if( -s $fuse == 0 )
{
    print STDERR "no fusions found; exiting...\n";
    exit 0;
}

#### generate a config file. eventually call the same
#### sub for this, just with different args
my $config1 = "$options{output_dir}/circos-fusion-all.conf";
mk_fusion_conf($config1, $fuse, "circos_fusion_all.png", $config->{ToolInfo}->{karyotype}->{value}, $gene);

my $lib = $config->{ToolInfo}->{std_perl5lib}->{value};
$lib =~ s/:/ -I /g;

my $cmd = "$config->{ToolInfo}->{perl}->{value} -I $lib";
$cmd .= " $config->{ToolInfo}->{circos}->{value}/circos";
$cmd .= " -conf $config1";
execute_cmd($cmd);


#### try to only use the chromosomes we have in the data
#my $config2 = "$options{output_dir}/circos-fusion-chr.conf";
#mk_fusion_conf2($config2, $fuse, "circos_fusion_chr.png", $config->{ToolInfo}->{karyotype}->{value}, $gene);

#$cmd = "$config->{ToolInfo}->{circos}->{value}/circos -conf $config2";
#execute_cmd($cmd);

$logger->info("Circos fusion process complete");
#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info fusion output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}


#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $send_mail = 1;

	while (! -s $file) {
		if ($send_mail) {
			$send_mail = 0;

			$util->missingInput($config->{RunInfo}->{email},
								"CircosFusion",
								"TophatFusionPost",
								"$file",
								$config->{RunInfo}->{tool});
		}

		sleep $LONG_WAIT;
	}
}
