#!/usr/bin/perl

=head1 NAME
   install.pl

=head1 SYNOPSIS
    USAGE: install.pl --prefix=/location/of/install/dir

=head1 OPTIONS

B<--prefix, -p>
	Required. Prefix location where package will be installed.

B<--perl_exec, -e>
	Optional.  If perl exec is other than /usr/bin/perl please specify location of perl install

B<--help,-h>


=head1  DESCRIPTION
	Install package

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./install.pl --prefix=/prefix

=cut

use strict;
use warnings;
use Cwd;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use Cwd 'abs_path';

my %options = ();
my $results = GetOptions (\%options,
                          'prefix|p=s',
						  'perl_exec|e=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### make sure everything passed was peachy
&check_parameters(\%options);


duration_warning(); 
print "Beginning MAP-RSeq Installation ...\n"; 
sub duration_warning { 
	print "\nWelcome to the MAP-RSeq install. This process will unpack and build the included bioinformatics tools required for the pipeline.  It will also download and index a Human reference (hg19).\n";
	print "\nThis process can take several hours depending upon your connection to the internet and server speed.  Interrupting the installation mid-way can leave the software in a broken state from which manual recovery may be difficult.\n"; 
	print "\n\nPlease press [Enter] to continue or Control-C to quit.\n"; 
	my $enter = <STDIN>;  
}

 
#### print time now.
timestamp();

my $this = {};
my $progress = {};
my $cmd = "";

#### get current working dir
$this->{source} = getcwd();

$progress = getProgress();

#### make logs dir
$cmd = "mkdir -p $options{prefix}/logs";
execute_cmd($cmd);

#### unpack binary dir containing all binary to be installed
#### which are required for successfull run
print STDERR "\n\nInstalling binaries...\n";

#### install each package in binary folder.
my @packages = qw(bedtools bowtie1 circos fastqc gatk htseq tophat picard rseqc subread samtools ucsc);

foreach my $tool (@packages) {
	if ((exists $progress->{$tool}) && ($progress->{$tool})){
		print STDERR "\t$tool already installed. Skipping...\n";
	} else {
		print STDERR "\tInstalling $tool...\n";

		#### unpack and install each tool
		eval("install_${tool}()");
	}
}

#### installling libraries required for successfull run
install_libraries();

#### copy source code and update paths for perl and libs
install_source();

#### unzip and install references
install_references();

#### get chromosome reference files
install_chr();

#### create bowtie-1 index files
build_bowtie1_index();

#### create bowtie-1 index files
#build_bowtie2_index();

#### get blast databases files
#install_blast_db();

#### create a test run_info file.
setup_runInfo();

#### create a test tool_info file based on install prefix
setup_toolInfo();

#### create a test sample_info file based on example samples
setup_sampleInfo();

#### copy memory info to config dir.
setup_memoryInfo();

#### move sample input files
setup_testInput();

#### completion message
print "\n\n\nMAPRSeq installation complete.  \n\n"; 

print "Use following command to test the installed tools\n";
print "\n\t$options{prefix}/check_install -t $options{prefix}/config/tool_info.txt\n\n";

print "Use following command to initiate a test run\n";
print "\n\t$options{prefix}/src/mrna.pl -r=$options{prefix}/config/run_info.txt\n\n";

#### print time now
timestamp();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(prefix);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{prefix} =  abs_path( $options{prefix} );

	$options{'perl_exec'} = "/usr/bin/perl" unless($options{'perl_exec'});
}

#############################################################################
sub getProgress {
	my $hash = {};
	my @sofar;

	#### if file exists get progress so far.
	if (-s "$options{prefix}/progress.txt") {
		open(FHD, "<", "$options{prefix}/progress.txt") or die "Could not open file to read $options{prefix}/progress.txt";
		while(<FHD>){
			chomp $_;
			push @sofar, $_;
		}
		close(FHD);

		map { $hash->{$1} = $2 if( /([^=]+)\s*=\s*([^=]+)/ ) } @sofar;
	}

	#### return hash
	return $hash;
}

#############################################################################
sub setProgress {
	my $hash = shift;

	open(OUT, ">", "$options{prefix}/progress.txt") or die "Could not open file to write $options{prefix}/progress.txt";

	foreach my $key (keys %{$hash}){
		print OUT $key."=".$hash->{$key}."\n";
	}

	close(OUT);
}

#############################################################################
sub install_bedtools {
	#### check and setup install dir
	my $dir = "$options{prefix}/bin/bedtools/2.17.0";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/BEDTools.v2.17.0.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/bedtools-2.17.0");
	$cmd = "make 1>$options{prefix}/logs/bedtools_log.out";
	$cmd .= " 2>$options{prefix}/logs/bedtools_log.err";
	execute_cmd($cmd);

	$cmd = "mv $this->{source}/binary/bedtools-2.17.0/bin/*";
	$cmd .= " $options{prefix}/bin/bedtools/2.17.0/.";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/bedtools-2.17.0";
	execute_cmd($cmd);

	$progress->{bedtools} = 1;
	setProgress($progress);
}

#############################################################################
sub install_blat {
	#### check and install dir
	my $dir = "$options{prefix}/bin/blat/34";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/blat.v34x64.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	$cmd = "mv $this->{source}/binary/blat $options{prefix}/bin/blat/34/.";
	execute_cmd($cmd);

	$progress->{blat} = 1;
	setProgress($progress);
}

#############################################################################
sub install_blast {
	#### check and install dir
	my $dir = "$options{prefix}/bin/blast/2.2.28";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/ncbi-blast-2.2.28+-src.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/ncbi-blast-2.2.28+-src/c++");
	$cmd = "$this->{source}/binary/ncbi-blast-2.2.28+-src/c++/configure";
	$cmd .= " --prefix=$options{prefix}/bin/blast/2.2.28";
	$cmd .= " && make && make install";
	$cmd .= " 1>$options{prefix}/logs/blast_log.out";
	$cmd .= " 2>$options{prefix}/logs/blast_log.err";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/ncbi-blast-2.2.28+-src";
	execute_cmd($cmd);

	$progress->{blast} = 1;
	setProgress($progress);
}

#############################################################################
sub install_bowtie1 {
	#### check and install dir
	my $dir = "$options{prefix}/bin/bowtie/0.12.9";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/bowtie-0.12.9.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/bowtie-0.12.9");
	$cmd = "make";
	$cmd .= " 1>$options{prefix}/logs/bowtie1_log.out";
	$cmd .= " 2>$options{prefix}/logs/bowtie1_log.err";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/bowtie-0.12.9/*";
	$cmd .= " $options{prefix}/bin/bowtie/0.12.9/.";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/bowtie-0.12.9";
	execute_cmd($cmd);

	$progress->{bowtie1} = 1;
	setProgress($progress);
}

#############################################################################
sub install_bowtie2 {
	#### check and install dir
	my $dir = "$options{prefix}/bin/bowtie/2.1.0";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/bowtie2-2.1.0.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/bowtie2-2.1.0");
	$cmd = "make";
	$cmd .= " 1>$options{prefix}/logs/bowtie2_log.out";
	$cmd .= " 2>$options{prefix}/logs/bowtie2_log.err";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/bowtie2-2.1.0/*";
	$cmd .= " $options{prefix}/bin/bowtie/2.1.0/.";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/bowtie2-2.1.0";
	execute_cmd($cmd);

	$progress->{bowtie2} = 1;
	setProgress($progress);
}

#############################################################################
sub install_circos {
	#### check and install dir
	my $dir = "$options{prefix}/bin/circos/0.64";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/circos-0.64.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/circos-0.64/*";
	$cmd .= " $options{prefix}/bin/circos/0.64/.";
	execute_cmd($cmd);

	$cmd = "rm -rf $this->{source}/binary/circos-0.64";
	execute_cmd($cmd);

	$progress->{circos} = 1;
	setProgress($progress);
}

#############################################################################
sub install_fastqc {
	#### check and install dir
	my $dir = "$options{prefix}/bin/fastqc/0.10.1";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/fastqc-0.10.1.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/FastQC/*";
	$cmd .= " $options{prefix}/bin/fastqc/0.10.1/.";
	execute_cmd($cmd);

	$cmd = "rm -rf $this->{source}/binary/FastQC";
	execute_cmd($cmd);

	$progress->{fastqc} = 1;
	setProgress($progress);
}

#############################################################################
sub install_gatk {
	#### check and install dir
	my $dir = "$options{prefix}/bin/gatk/1.6";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/GenomeAnalysisTK-1.6-9.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/GenomeAnalysisTK-1.6-9/*";
	$cmd .= " $options{prefix}/bin/gatk/1.6/.";
	execute_cmd($cmd);

	$cmd = "rm -rf $this->{source}/binary/GenomeAnalysisTK-1.6-9";
	execute_cmd($cmd);

	$progress->{gatk} = 1;
	setProgress($progress);
}

#############################################################################
sub install_htseq {
	#### check and install dir
	my $dir = "$options{prefix}/bin/htseq/0.5.3p9";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/HTSeq-0.5.3p9.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### create lib dir
	$cmd = "mkdir -p $dir/lib/python2.7/site-packages/";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/HTSeq-0.5.3p9/");

	if (defined $ENV{'PYTHONPATH'}) {
		$ENV{'PYTHONPATH'} = "$dir/lib/python2.7/site-packages/:$ENV{'PYTHONPATH'}";
	} else {
		$ENV{'PYTHONPATH'} = "$dir/lib/python2.7/site-packages/";
	}

	#### move install to bin dir
	$cmd = "python setup.py install";
	$cmd .= " --prefix=$options{prefix}/bin/htseq/0.5.3p9";
	$cmd .= " 1>$options{prefix}/logs/htseq_log.out";
	$cmd .= " 2>$options{prefix}/logs/htseq_log.err";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/HTSeq-0.5.3p9";
	execute_cmd($cmd);

	$progress->{htseq} = 1;
	setProgress($progress);
}

#############################################################################
sub install_tophat {
	#### check and install dir
	my $dir = "$options{prefix}/bin/tophat/2.0.6";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/tophat_2.0.6.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/2.0.6/*";
	$cmd .= " $options{prefix}/bin/tophat/2.0.6/.";
	execute_cmd($cmd);

	$cmd = "rm -rf $this->{source}/binary/2.0.6";
	execute_cmd($cmd);

	$progress->{tophat} = 1;
	setProgress($progress);
}

#############################################################################
sub install_picard {
	#### check and install dir
	my $dir = "$options{prefix}/bin/picard/1.92";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/picard-tools-1.92.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/picard-tools-1.92/*";
	$cmd .= " $options{prefix}/bin/picard/1.92/.";
	execute_cmd($cmd);

	$cmd = "rm -rf $this->{source}/binary/picard-tools-1.92";
	execute_cmd($cmd);

	$progress->{picard} = 1;
	setProgress($progress);
}

#############################################################################
sub install_rseqc {
	#### check and install dir
	my $dir = "$options{prefix}/bin/rseqc/2.3.7";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	#### untar source
	$cmd = "tar -zxvf $this->{source}/binary/RSeQC-2.3.7.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### setup dir and env before installing
	$cmd = "mkdir -p $dir/lib/python2.7/site-packages/";
	execute_cmd($cmd);

	if ((defined $ENV{PYTHONPATH}) && (length $ENV{PYTHONPATH})) {
		$ENV{PYTHONPATH} .= ":$dir/lib/python2.7/site-packages/";
	} else {
		$ENV{PYTHONPATH} = "$dir/lib/python2.7/site-packages/";
	}

	print "PYTHONPATH " . $ENV{PYTHONPATH} . "\n"; 

	print `which python` ; 

	#### install to bin dir
	chdir("$this->{source}/binary/RSeQC-2.3.7");
	$cmd = "python setup.py install --prefix=$dir";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/RSeQC-2.3.7";
	execute_cmd($cmd);

	$progress->{rseqc} = 1;
	setProgress($progress);
}

#############################################################################
sub install_samtools {
	#### check and install dir
	my $dir = "$options{prefix}/bin/samtools/0.1.19";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/samtools-0.1.19.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/samtools-0.1.19");
	$cmd = "make";
	$cmd .= " 1>$options{prefix}/logs/samtools_log.out";
	$cmd .= " 2>$options{prefix}/logs/samtools_log.err";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/samtools-0.1.19/*";
	$cmd .= " $options{prefix}/bin/samtools/0.1.19/.";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/samtools-0.1.19";
	execute_cmd($cmd);

	$progress->{samtools} = 1;
	setProgress($progress);
}


sub install_subread { 
	#subread-1.4.4-source.tar.gz

	my $dir = "$options{prefix}/bin/subread/1.4.4";
	my $cmd = ""; 

	if (-d $dir){ 
		$cmd = " rm -rf $dir"; 
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar xfz $this->{source}/binary/subread-1.4.4-source.tar.gz -C $this->{source}/binary";

	print $cmd . "\n"; 
	execute_cmd($cmd);

	chdir("$this->{source}/binary/subread-1.4.4-source/src");
	$cmd = "make -f Makefile.Linux"; 
	$cmd .= " 1>$options{prefix}/logs/subread_log.out";
	$cmd .= " 2>$options{prefix}/logs/subread_log.err";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "mv $this->{source}/binary/subread-1.4.4-source/bin/*";
	$cmd .= " $options{prefix}/bin/subread/1.4.4/."; 
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/subread-1.4.4-source";
	execute_cmd($cmd);

	$progress->{subread} = 1 ; 
	setProgress($progress);
}	

#############################################################################
sub install_ucsc {
	#### check and install dir
	my $dir = "$options{prefix}/bin/ucsc";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/ucsc_x86_64.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### move install to bin dir
	$cmd = "mv $this->{source}/binary/ucsc/*";
	$cmd .= " $options{prefix}/bin/ucsc/.";
	execute_cmd($cmd);

	$cmd = "rm -rf $this->{source}/binary/ucsc";
	execute_cmd($cmd);

	$progress->{ucsc} = 1;
	setProgress($progress);
}

#############################################################################
sub install_libraries {
	if ((exists $progress->{libraries}) && ($progress->{libraries})){
		print STDERR "\tLibraries already installed. Skipping...\n";
		return;
	}

	print STDERR "\n\nInstalling libraries...\n\n";
	chdir($this->{source});

	$cmd = "cp -r $this->{source}/library $options{prefix}/lib";
	execute_cmd($cmd);

	$progress->{libraries} = 1;
	setProgress($progress);
}

#############################################################################
sub install_references {
	if ((exists $progress->{references}) && ($progress->{references})){
		print STDERR "\tReferences already installed. Skipping...\n";
		return;
	}

	#### unzip and create reference dataset
	chdir("$this->{source}");

	$cmd = "cp -r $this->{source}/references $options{prefix}/references";
	execute_cmd($cmd);

	$progress->{references} = 1;
	setProgress($progress);
}

#############################################################################
sub install_chr {
	if ((exists $progress->{chr}) && ($progress->{chr})){
		print STDERR "\tCHR already installed. Skipping...\n";
		return;
	}

	#### setup base url
	my $url = "ftp://hgdownload.cse.ucsc.edu/goldenPath/hg19/chromosomes";
	my @chr = qw(1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 X Y M);

	my $cmd = "mkdir -p $options{prefix}/references/hg19/37.1";
	execute_cmd($cmd);

	chdir ("$options{prefix}/references/hg19/37.1");
	print "Downloading reference chromosomes for hg19 Genome build.  This make take some time depending on your internet connection\n";

	foreach my $c (@chr){
		print "Downloading chr${c}...\n";
		$cmd = "wget -q $url/chr${c}.fa.gz";
		execute_cmd($cmd);

		$cmd = "gunzip -c chr${c}.fa.gz >> $options{prefix}/references/hg19/37.1/allchr.fa";
		execute_cmd($cmd);

		$cmd = "rm chr${c}.fa.gz";
		execute_cmd($cmd);
	}

	$progress->{chr} = 1;
	setProgress($progress);
}

#############################################################################
sub build_bowtie1_index {
	if ((exists $progress->{bowtie1_index}) && ($progress->{bowtie1_index})){
		print STDERR "\tBowtie1 indexes already installed. Skipping...\n";
		return;
	}

	my $cmd = "mkdir -p $options{prefix}/references/hg19/37.1/indexed";
	execute_cmd($cmd);

	#### add allchr.fa softlink
	$cmd = "ln -s $options{prefix}/references/hg19/37.1/allchr.fa  $options{prefix}/references/hg19/37.1/indexed/allchr.fa";
	execute_cmd($cmd);

	print "Building bowtie-1 indexes.  This may take some time depending your hardware resources\n";

	$cmd = "$options{prefix}/bin/bowtie/0.12.9/bowtie-build";
	$cmd .= " $options{prefix}/references/hg19/37.1/allchr.fa";
	$cmd .= " $options{prefix}/references/hg19/37.1/indexed/allchr";
	execute_cmd($cmd);

	$progress->{bowtie1_index} = 1;
	setProgress($progress);
}

#############################################################################
sub build_bowtie2_index {
	if ((exists $progress->{bowtie2_index}) && ($progress->{bowtie2_index})){
		print STDERR "\tBowtie2 indexes already installed. Skipping...\n";
		return;
	}

	my $cmd = "mkdir -p $options{prefix}/references/hg19/37.1/indexed";
	execute_cmd($cmd);

	print "Building bowtie-2 indexes.  This may take some time depending your hardware resources\n";

	$cmd = "$options{prefix}/bin/bowtie/2.1.0/bowtie2-build";
	$cmd .= " $options{prefix}/references/hg19/37.1/allchr.fa";
	$cmd .= " $options{prefix}/references/hg19/37.1/indexed/allchr";
	execute_cmd($cmd);

	$progress->{bowtie2_index} = 1;
	setProgress($progress);
}

#############################################################################
sub install_blast_db {
	if ((exists $progress->{blast_db}) && ($progress->{blast_db})){
		print STDERR "\tBLAST DB already installed. Skipping...\n";
		return;
	}

	my $url = "ftp://ftp.ncbi.nlm.nih.gov/blast/db/";

	my @nt = qw(00 01 02 03 04 05 06 07 08 09 10 11 12 13 14);
	my @hg = qw(00 01 02 03 04 05 06 07 08 09 10 11 12);

	my $cmd = "mkdir -p $options{prefix}/references/blast_db/nt";
	execute_cmd($cmd);

	$cmd = "mkdir -p $options{prefix}/references/blast_db/human_genomic";
	execute_cmd($cmd);

	print "Downloading BLAST DBs from NCBI.  This make take some time depending on your internet connection\n";

	chdir("$options{prefix}/references/blast_db/nt");
	foreach my $db (@nt){
		print "Downloading nt.${db}...\n";
		$cmd = "wget -q $url/nt.${db}.tar.gz";
		execute_cmd($cmd);

		$cmd = "tar -zxvf nt.${db}.tar.gz";
		execute_cmd($cmd);

		$cmd = "rm nt.${db}.tar.gz";
		execute_cmd($cmd);
	}

	chdir("$options{prefix}/references/blast_db/human_genomic");
	foreach my $db (@hg){
		print "Downloading human_genomic.${db}...\n";
		$cmd = "wget -q $url/human_genomic.${db}.tar.gz";
		execute_cmd($cmd);

		$cmd = "tar -zxvf human_genomic.${db}.tar.gz";
		execute_cmd($cmd);

		$cmd = "rm human_genomic.${db}.tar.gz";
		execute_cmd($cmd);
	}

	$progress->{blast_db} = 1;
	setProgress($progress);
}

#############################################################################
sub install_source {
	if ((exists $progress->{source}) && ($progress->{source})){
		print STDERR "\tSource already installed. Skipping...\n";
		return;
	}

	print STDERR "\n\nInstalling source...\n\n";

	#### create dir to store source code
	$cmd = "mkdir -p $options{prefix}/config";
	execute_cmd($cmd);

	$cmd = "cp -r $this->{source}/script $options{prefix}/src";
	execute_cmd($cmd);

	#### make sure all scripts are executable
	$cmd = "chmod -R +x $options{prefix}/src";
	execute_cmd($cmd);

	#### link the check_config script to the root install dir
	$cmd = "ln -s " . $options{prefix} . "/src/check_install " . $options{prefix} . "/check_install";
	execute_cmd($cmd); 

	
	#### copy the setup.sh script to the root install dir
	#$cmd = "cp $this->{source}/script/setup.sh $options{prefix}/"; 
	#execute_cmd($cmd); 
	#$cmd = "chmod -R +x $options{prefix}/setup.sh";
	#execute_cmd($cmd);


	#### check if perl exec location is other than /usr/bin/perl
	if ($options{perl_exec} !~ /^\/usr\/bin\/perl$/) {
		#### update perl exec path on top of each script.
		$options{perl_exec} =~ s/\//\\\//g;

		$cmd = "find $options{prefix}/src -name \"*.pl\" -print";
		$cmd .= " -exec sed -i 's/#!\/usr\/bin\/perl/#!$options{perl_exec}/' {} \\;";
		execute_cmd($cmd);
	}

	#### replace library references to local install
	my $lib = "$options{prefix}/lib";
	$lib =~ s/\//\\\//g;

	$cmd = "find $options{prefix}/src -name \"*.pl\" -print";
	$cmd .= " -exec sed -i 's/\\/data2\\/bsi\\/reference\\/perl_workflow_ref\\/lib/$lib/' {} \\;";
	execute_cmd($cmd);

	#### update Logger.pm lib
	$cmd = "find $options{prefix}/lib -name \"Logger.pm\" -print";
	$cmd .= " -exec sed -i 's/\\/data2\\/bsi\\/reference\\/tophat\\/lib\\/perl5\\//$lib\\/perl5/' {} \\;";
	execute_cmd($cmd);

	#### copy config files
	$cmd = "cp $options{prefix}/src/config/* $options{prefix}/config";
	execute_cmd($cmd);

	#$cmd = "rm -rf $this->{source}/script";
	#execute_cmd($cmd);

	$progress->{source} = 1;
	setProgress($progress);
}

#############################################################################
sub setup_runInfo {
	if ((exists $progress->{runinfo}) && ($progress->{runinfo})){
		print STDERR "\tRun info config upto date. Skipping...\n";
		return;
	}

	my $username = (getpwuid($<))[0];

	#### add username in PI location
	$cmd = "sed -i 's/\@USER\@/$username/' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### update config location value
	my $config = "$options{prefix}/config";
	$config =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@CONFIG_DIR\@/$config/g' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### update input dir location for test run
	my $sample = "$options{prefix}/sample_data";
	$sample =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@INPUT_DIR\@/$sample/' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### update output dir location for test run
	my $output = "$options{prefix}/sample_output";
	$output =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@OUTPUT_DIR\@/$output/' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### remove double //
	$cmd = "sed -i 's/\\/\\//\\//g' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	$progress->{runinfo} = 1;
	setProgress($progress);
}

#############################################################################
sub setup_toolInfo {
	if ((exists $progress->{toolinfo}) && ($progress->{toolinfo})){
		print STDERR "\tTool info config upto date. Skipping...\n";
		return;
	}

	#### update reference file location
	my $reference = "$options{prefix}/references";
	$reference =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@REFERENCE\@/$reference/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update binary location
	my $binary = "$options{prefix}/bin/";
	$binary =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@BINARY\@/$binary/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update library location
	my $library = "$options{prefix}/lib/";
	$library =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@LIBRARY\@/$library/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update java location
	my $java = `which java`;
	chomp $java;

	#### stop execution and die if JAVA does not exists
	die "JAVA 1.6.0 or higher is required, could not find JAVA install via which java" unless (length($java));

	$java =~ s/\/java$//;
	$java =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@JAVA\@/$java/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update perl location
	my $perl = `which perl`;
	chomp $perl;

	#### stop execution and die if PERL does not exists
	die "Perl 5.10.0 or higher is required, could not find Perl install via which perl" unless (length($perl));

	$perl =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@PERL\@/$perl/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update R
	my $r = `which R`;

	#### stop execution and die if R does not exists
	die "R package is required, could not find R install via which R" unless (length($r));

	chomp $r;
	$r =~ s/\/R$//;
	$r =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@R\@/$r/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update python
	my $python = `which python`;
	chomp $python;

	#### stop execution and die if Python does not exists
	die "Python 2.6,5 or higher with nympy is required, could not find Python install via which Python" unless (length($python));

	$python =~ s/\/python$//;
	$python =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@PYTHON\@/$python/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update source path
	my $src = "$options{prefix}/src";
	$src =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@SOURCE\@/$src/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### check for PYTHONPATH env var
	my $pythonpath = `echo \$PYTHONPATH`;
	chomp $pythonpath;

	if (! length($pythonpath)){
		#### get default python library path.

		$python = `python --version 2>&1`;
		chomp $python;

		$python =~ s/Python (\d+\.\d+).*/$1/;

		$pythonpath = "/usr/lib/python$python";

		if (-d "$pythonpath/site-packages") {
			$pythonpath .= "/site-packages";
		} elsif (-d "$pythonpath/dist-packages") {
			$pythonpath .= "/dist-packages";
		} else {
			die "Can not file standard python lib dir $pythonpath/site-packages or $pythonpath/dist-packages\nUnable to set default PYTHONPATH. Check your python install and set PYTHONPATH varaible. Install failed";
		}
	}

	$pythonpath =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@PYTHONPATH\@/$pythonpath/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### remove double //
	$cmd = "sed -i 's/\\/\\//\\//g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	$progress->{toolinfo} = 1;
	setProgress($progress);
}

#############################################################################
sub setup_sampleInfo {
}

#############################################################################
sub setup_memoryInfo {
}

#############################################################################
sub setup_testInput {
	if ((exists $progress->{testinput}) && ($progress->{testinput})){
		print STDERR "\tTest input already installed. Skipping...\n";
		return;
	}

	#### check and install dir
	my $dir = "$options{prefix}/sample_data";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
	} else {
		$cmd = "mkdir -p $dir";
	}
	execute_cmd($cmd);

	chdir("$this->{source}");
	$cmd = "cp -r $this->{source}/input/* $dir/.";
	execute_cmd($cmd);

	$progress->{testinput} = 1;
	setProgress($progress);
}


#############################################################################
sub execute_cmd {
	my $cmd = shift;

	my $result = system($cmd);

	print $result . "\n"; 

	#while (( $? >> 8 ) != 0 ){
	#	print STDERR "ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd\n";
	#	exit(-1);
	#}
}

#############################################################################
sub timestamp {
	my @months = qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
    my @weekDays = qw(Sun Mon Tue Wed Thu Fri Sat Sun);
    my ($second, $minute, $hour, $dayOfMonth, $month, $yearOffset, $dayOfWeek, $dayOfYear, $daylightSavings) = localtime();
    my $year = 1900 + $yearOffset;
    my $theTime = "$hour:$minute:$second, $weekDays[$dayOfWeek] $months[$month] $dayOfMonth, $year";
    print "Time now: " . $theTime."\n";
}
