#!/usr/bin/perl

=head1 NAME
   unique_mapreads_count.pl

=head1 SYNOPSIS
    USAGE: unique_mapreads_count.pl -r=run_info.txt -o=output_dir -s=sample name

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Generate unique mapped reads count per chr

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./unique_mapreads_count.pl -r=run_info.txt -o=output_dir

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'chr|c=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
############################################
my $LONG_WAIT = 300;

## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $logger ;

if(!defined($ENV{'SGE_TASK_ID'})){
        $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/UniqueMapReadsCount.$options{sample}.NON_SGE.log",
                                                                   'LOG_LEVEL'=>$options{'debug'});
} else {
        $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/UniqueMapReadsCount.$options{sample}.$ENV{'SGE_TASK_ID'}.log",
                                                                   'LOG_LEVEL'=>$options{'debug'});
}

$logger = Workflow::Logger::get_logger();

$logger->info("Generate unique mapped reads per chr started");

create_dir_struct(\%options);

my $this = {};
$this->{input} = "$options{output_dir}/alignment/tophat_$options{sample}/$options{sample}-sorted.unique.bam";
$this->{output} = "$options{output_dir}/QC";

#### check input
check_input($this->{input});

my @chr = split(/:/, $config->{RunInfo}->{chrindex});

my $chr_idx = "";

if (defined $ENV{'SGE_TASK_ID'}){
	$chr_idx = $ENV{'SGE_TASK_ID'} - 1;
} else {
	$chr_idx = $options{chr} - 1;
}

my $cmd = " $config->{ToolInfo}->{samtools}->{value}/samtools view";
$cmd .= " $this->{input}";
#$cmd .= " | awk -v num=$chr[$ENV{'SGE_TASK_ID'}-1] '\$3 ~ \"chr\"num'";
$cmd .= " | awk -v num=$chr[$chr_idx] '\$3 ~ \"chr\"num'";
$cmd .= " | wc -l >> $this->{output}/$options{sample}.chr$chr[$chr_idx].unique.mappedreads";

print $cmd . "\n"; 
execute_cmd($cmd);

$logger->info("Generate unique mapped reads per chr complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
	$options{'chr'} = 1 unless (defined $options{'chr'});
}


#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}


#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/QC";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $send_mail = 1;

	while (! -s $file) {
		if ($send_mail) {
			$send_mail = 0;

			$util->missingInput($config->{RunInfo}->{email},
								"UniqueMapReadsCount",
								"SortBam",
								"$file",
								$config->{RunInfo}->{tool});
		}

		sleep $LONG_WAIT;
	}
}
