#!/usr/bin/perl

=head1 NAME
   tophat_post_process.pl

=head1 SYNOPSIS
    USAGE: tophat_post_process.pl -r=run_info.txt -o=output_dir -s=samplename

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	output directory

B<--sample_name, -s>
	sample name

B<--help,-h>


=head1  DESCRIPTION
	run vqsr filtering

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./tophat_post_process.pl -r=run_info.pl -o=output_dir -s=samplename

=cut

use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});

if (defined $ENV{'PATH'}){
	$ENV{'PATH'}="$config->{ToolInfo}->{samtools}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{bowtie1}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{bowtie}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{tophat}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{blastc}->{value}:$ENV{'PATH'}";
} else {
	$ENV{'PATH'}="$config->{ToolInfo}->{samtools}->{value}";
	$ENV{'PATH'}="$config->{ToolInfo}->{bowtie1}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{bowtie}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{tophat}->{value}:$ENV{'PATH'}";
	$ENV{'PATH'}="$config->{ToolInfo}->{blastc}->{value}:$ENV{'PATH'}";
}

my $this;
$this->{output_dir} = $options{output_dir};
$this->{fusion_dir} = $options{output_dir};
$this->{fusion_dir} =~ s/alignment/fusion/;

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/TophatPostProcess.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("TopHat Post Process started");

create_dir_struct($config);

chdir $this->{output_dir};
my $cmd = "";

#### run tophat report to create fusion results
$cmd = "$config->{ToolInfo}->{python}->{value}/python $config->{ToolInfo}->{tophat}->{value}/tophat-fusion-post --skip-blast";

if ((defined $config->{ToolInfo}->{fusion}->{value}) && (length($config->{ToolInfo}->{fusion}->{value}))){
	$cmd .= " $config->{ToolInfo}->{fusion}->{value}";
}

$cmd .= " -p 1 $config->{ToolInfo}->{ref_bowtie}->{value}";
$cmd .= " -o $this->{fusion_dir}";

execute_cmd($cmd);

#### move all data to fusion folder.
$cmd = "mv $this->{output_dir}/tophatfusion_out/* $this->{fusion_dir}/.";
execute_cmd($cmd);


#### check if file exists.
#### if file does not exists then there is an error in Tophat fusion call
if (-e "$this->{fusion_dir}/result.txt") {
	#### check if file is not emtpy.
	#### if file exists and its empyt this usually means that Tophat didn't find
	#### any fusion candidates.
	if (-s "$this->{fusion_dir}/result.txt") {
		#### summarize fusion results in a tab delimited file
		$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/tophat_fusion_report.pl";
		$cmd .= " -i=$this->{fusion_dir}/result.txt";
		$cmd .= " -p=$this->{fusion_dir}/potential_fusion.txt";
		$cmd .= " -r=$options{run_info}";
		$cmd .= " -o=$this->{fusion_dir}";
		execute_cmd($cmd);

		chdir $this->{fusion_dir};

		#### create circos conf
		$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/circos-fusion.pl";
		$cmd .= " -r=$options{run_info} -o=$this->{fusion_dir} -f=$this->{fusion_dir}/result.html";
		execute_cmd($cmd);
	} else {
		$logger->info("There are no fusion candidates available for this run");
	}
} else {
	$logger->info("Problem running tophat fusion post");
}



$logger->info("Tophat Post Process complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $cmd = "ln -s $config->{ToolInfo}->{refgene}->{value} $this->{output_dir}/refGene.txt";
	execute_cmd($cmd);

	$cmd = "ln -s $config->{ToolInfo}->{ensgene}->{value} $this->{output_dir}/ensGene.txt";
	execute_cmd($cmd);

	if ((defined $config->{ToolInfo}->{fusion}->{value}) && ($config->{ToolInfo}->{fusion}->{value} =~ /non-human/)){
		;
	} else {
		$cmd = "ln -s $config->{ToolInfo}->{mcl}->{value} $this->{output_dir}/mcl";
		execute_cmd($cmd);
	}

	#create dir called blast and ln all human and nt blast dbs
	$cmd = "mkdir -p $this->{output_dir}/blast";
	execute_cmd($cmd);

	#$cmd = "ln -s $config->{ToolInfo}->{hg_blast_db}->{value}/* $this->{output_dir}/blast/.";
	#execute_cmd($cmd);

	#$cmd = "ln -s $config->{ToolInfo}->{nt_blast_db}->{value}/* $this->{output_dir}/blast/.";
	#execute_cmd($cmd);
}


#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);

	#while (( $? >> 8 ) != 0 ){
	#	$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

	#	exit(-1);
	#}
}
