#!/usr/bin/perl

=head1 NAME
   merge_bams.pl

=head1 SYNOPSIS
    USAGE: merge_bams.pl -r=run_info.txt -o=output_dir -s=samplename

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	output directory

B<--sample_name, -s>
	sample name

B<--help,-h>


=head1  DESCRIPTION
	merge accepted_hits and unmapped bams into one

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./merge_bams.pl -r=run_info.pl -o=output_dir -s=samplename

=cut

use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
############################################
my $LONG_WAIT = 300;

## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

my $this;
$this->{output_dir} = "$options{output_dir}/alignment/tophat_$options{sample}";
$this->{input_file1} = "$this->{output_dir}/accepted_hits.bam";
$this->{input_file2} = "$this->{output_dir}/unmapped.bam";
$this->{max_read_mem} = 1000000;

my $cmd = "";

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/MergeBAMs.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Merge BAM for $options{sample} started");

#### check input file
check_input($this->{input_file1});
check_input($this->{input_file2});


$cmd = "$config->{ToolInfo}->{java}->{value}/java";
$cmd .= " -Xmx$config->{MemoryInfo}->{mergebam_jvmx} -Xms512m";
$cmd .= " -Djava.io.tmpdir=$options{output_dir}/tmp";
$cmd .= " -jar $config->{ToolInfo}->{picard}->{value}/MergeSamFiles.jar";
$cmd .= " INPUT=$this->{input_file1}";
$cmd .= " INPUT=$this->{input_file2}";
$cmd .= " OUTPUT=$this->{output_dir}/$options{sample}_sorted.bam";
$cmd .= " SO=coordinate";
#$cmd .= " CREATE_INDEX=true";
$cmd .= " TMP_DIR=$this->{output_dir}/tmp";
$cmd .= " MAX_RECORDS_IN_RAM=$this->{max_read_mem}";
$cmd .= " VALIDATION_STRINGENCY=SILENT";
execute_cmd($cmd);

#### create index for accepted_hits.bam file created will be accepted_hits.bam.bai
$cmd = "$config->{ToolInfo}->{samtools}->{value}/samtools index $this->{output_dir}/$options{sample}_sorted.bam";
execute_cmd($cmd);


if (! -s "$this->{output_dir}/$options{sample}_sorted.bam") {
	$logger->logdie("ERROR : Sorted merged BAM is not generated for $options{sample}");
}


$logger->info("Merge BAM for $options{sample} complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $send_mail = 1;

	while (! -s $file) {
		if ($send_mail) {
			$send_mail = 0;

			$util->missingInput($config->{RunInfo}->{email},
								"MergeBam",
								"Alignment",
								"$file",
								$config->{RunInfo}->{tool});
		}

		sleep $LONG_WAIT;
	}
}
