#!/usr/bin/perl

=head1 NAME
   reports.pl

=head1 SYNOPSIS
    USAGE: reports.pl -r=run_info.txt -o=output_dir

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Generate reports

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./reports.pl -r=run_info.txt -o=output_dir

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### make sure everything passed was peachy
&check_parameters(\%options);

#### parse X_info files
my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/Reports.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Generate reports started");

create_dir_struct(\%options);

my $this = {};
$this->{input} = "$options{output_dir}/counts";
$this->{output} = "$options{output_dir}/Reports";

$this->{q_args} = "-wd $config->{RunInfo}->{logs} -b y";
$this->{q_args} .= " -q $config->{ToolInfo}->{queue}->{value}";
$this->{q_args} .= " -m $config->{ToolInfo}->{notification_option}->{value}";
$this->{q_args} .= " -M $config->{RunInfo}->{email} -l h_stack=10M";

$this->{job} = $config->{RunInfo}->{tool} .".". $config->{RunInfo}->{version} .".". $config->{RunInfo}->{output_folder};

my $job_name = "";

#### get read distribution plot
my $cmd = "$config->{ToolInfo}->{workflow_path}->{value}/qc_readdist.pl";
$cmd .= " -r=$options{output_dir}/run_info.txt -o=$options{output_dir}";
execute_cmd($cmd);

#### get gene no read distribution plot
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/qc_gene_noread_map.pl";
$cmd .= " -r=$options{output_dir}/run_info.txt -o=$options{output_dir}";
execute_cmd($cmd);

#### generate IGV session
$cmd = "perl $config->{ToolInfo}->{workflow_path}->{value}/create_igv.pl";
$cmd .= " -o=$options{output_dir} -r=$options{run_info}";
execute_cmd($cmd);

#### genereate HTML
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/MainDocument.pl";
$cmd .= " -r=$options{run_info} -o=$options{output_dir}";
execute_cmd($cmd);

#### create global error and warning log file.
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/globalLog.pl";
$cmd .= " -r=$options{run_info} -o=$options{output_dir}";
execute_cmd($cmd);

#### dashboard update
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
$cmd .= " -r=$options{run_info} -t=Results -c=1";
execute_cmd($cmd);

$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/dashboard.pl";
$cmd .= " -r=$options{run_info} -t=Complete -c=1";
execute_cmd($cmd);

print STDOUT "\n\n$config->{RunInfo}->{type} workflow completed execution of all components.";
print STDOUT " Output is available in following folder.";
print STDOUT " $config->{RunInfo}->{output_folder}\n\n";

#### send final email.
send_completion_email();

$logger->info("Generate reports complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/QC";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub send_completion_email{

	my $sub="$config->{RunInfo}->{type} workflow completion for RunID $config->{RunInfo}->{output_folder}";

	my $msg="$config->{RunInfo}->{type} workflow completed execution of all components";
	$msg .= " for $config->{RunInfo}->{output_folder} on " .timer();

	#### check if there were any warnings
	if (-s "$options{output_dir}/warnings.log") {
		$msg = "There were some warning(s) reported during execution of workflow.  Please look at $options{output_dir}/warnings.log for more detail\n";
	}

	#### check if there were any errors
	if (-s "$options{output_dir}/errors.log") {
		$msg = "There were some error(s) reported during execution of workflow.  Please look at $options{output_dir}/errors.log for more detail\n";
	}

	## send the completion email
	system("echo -e \"$msg\" | mailx -v -s \"$sub\" \"$config->{RunInfo}->{email}\"");
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

###############################################################################
sub timer {
    my @months = qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
    my @weekDays = qw(Sun Mon Tue Wed Thu Fri Sat Sun);
    my ($second, $minute, $hour, $dayOfMonth, $month, $yearOffset, $dayOfWeek, $dayOfYear, $daylightSavings) = localtime();
    my $year = 1900 + $yearOffset;
    my $theTime = "$hour:$minute:$second, $weekDays[$dayOfWeek] $months[$month] $dayOfMonth, $year";

	return $theTime;

    #print "Time now: " . $theTime."\n";
}
