#!/usr/bin/perl

=head1 NAME
   mapped_reads_plot.pl

=head1 SYNOPSIS
    USAGE: mapped_reads_plot.pl -r=run_info.txt -o=output_dir

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--help,-h>


=head1  DESCRIPTION
	Generate unique mapped reads plot using RScript

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./mapped_reads_plot.pl -r=run_info.txt -o=output_dir

=cut

use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/MappedReadsPlot.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Mapped reads plot started");

$ENV{PATH} = $config->{ToolInfo}->{r}->{value} .":". $ENV{PATH};
create_dir_struct(\%options);

my $this = {};
$this->{output} = "$options{output_dir}/QC";

open(SMN, ">", "$this->{output}/sample.names")
	or $logger->logdie("Could not open file to write $this->{output}/sample.names");
open(CHR, ">", "$this->{output}/chrs")
	or $logger->logdie("Could not open file to write $this->{output}/chrs");
open(VAL, ">", "$this->{output}/persample.values")
	or $logger->logdie("Could not open file to write $this->{output}/persample.values");

## write all sorted sample names to tab delimited file
my $first = 1;

foreach my $sample ( sort keys %{$config->{SampleInfo}}) {
	if ($first) {
		print SMN "$sample";
		$first = 0;
	} else {
		print SMN "\t$sample";
	}
}
print SMN "\n";
close(SMN);

## get unique mapped reads per chr per sample
my @chr = split(/:/, $config->{RunInfo}->{chrindex});

foreach my $c (@chr) {
	print CHR "chr$c\n";
	my $n = "";
	foreach my $sample (sort keys %{$config->{SampleInfo}}) {
		my $tmp = `cat $this->{output}/$sample.chr$c.unique.mappedreads | head -1`;
		chomp $tmp;

		if (length($n)){
			$n .= "\t".$tmp;
		} else {
			$n = $tmp;
		}
	}
	print VAL "$n\n";
}
close(VAL);
close(CHR);

### Rscript to generate the plot
chdir($this->{output});
my $cmd = "Rscript $config->{ToolInfo}->{workflow_path}->{value}/rscript/QC_mRNASeq_unqreads_perchr_Rplot.r";
$cmd .= " $this->{output}/persample.values $this->{output}/chrs $this->{output}/sample.names";
execute_cmd($cmd);

if (-s "$this->{output}/UniqueMappedReads_PerChr.png") {
	#$cmd = "rm $this->{output}/persample.values $this->{output}/chrs $this->{output}/sample.names $this->{output}/*.unique.mappedreads";
	#execute_cmd($cmd);
	;
} else {
	$logger->logdie("$this->{output}/UniqueMappedReads_PerChr.png not generated");
}

$logger->info("Mapped reads plot complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/QC";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}
