#!/usr/bin/perl

=head1 NAME
   fastqc.pl

=head1 SYNOPSIS
    USAGE: fastqc.pl -r=run_info.txt -o=output_dir -s=samplename

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	output directory

B<--sample_name, -s>
	sample name

B<--help,-h>


=head1  DESCRIPTION
	run alignment

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./fastqc.pl -r=run_info.pl -o=output_dir -s=samplename

=cut

use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

## parse X_info files
my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/FastQC.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("FASTQC for $options{sample} started");

create_dir_struct(\%options);

my $cmd = "";

## run fastqc check on each sample.
foreach my $s (@{$config->{SampleInfo}{$options{sample}}}) {
	my $filename = $s->{read1};

	if ($filename =~ /\.gz$/) {
		$filename =~ s/\.gz$//;
		$cmd = "gunzip -c $s->{filepath}/$s->{read1} > $options{output_dir}/fastq/$filename";
		execute_cmd($cmd);
	} else {
		$cmd = "ln -s $s->{filepath}/$s->{read1} $options{output_dir}/fastq/$filename";
		execute_cmd($cmd);
	}

	$cmd = "$config->{ToolInfo}->{fastqc}->{value}/fastqc";
	$cmd .= " -Xmx$config->{MemoryInfo}->{fastqc_jvmx}";
	$cmd .= " -Dfastqc.output_dir=$options{output_dir}/fastqc $options{output_dir}/fastq/$filename";
	execute_cmd($cmd);

	#remove fastq file as its not needed
	$cmd = "rm $options{output_dir}/fastq/$filename";
	execute_cmd($cmd);

	if ((exists $s->{read2}) && (length($s->{read2}))) {
		$filename = $s->{read2};

		if ($filename =~ /\.gz$/) {
			$filename =~ s/\.gz$//;
			$cmd = "gunzip -c $s->{filepath}/$s->{read2} > $options{output_dir}/fastq/$filename";
			execute_cmd($cmd);
		} else {
			$cmd = "ln -s $s->{filepath}/$s->{read2} $options{output_dir}/fastq/$filename";
		}

		$cmd = "$config->{ToolInfo}->{fastqc}->{value}/fastqc";
		$cmd .= " -Xmx$config->{MemoryInfo}->{fastqc_jvmx}";
		$cmd .= " -Dfastqc.output_dir=$options{output_dir}/fastqc $options{output_dir}/fastq/$filename";
		execute_cmd($cmd);

		#remove fastq file as its not needed
		$cmd = "rm $options{output_dir}/fastq/$filename";
		execute_cmd($cmd);
	}
}

$logger->info("FASTQC for $options{sample} complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
	$options{'fusion'} = 1 unless ($options{'fusion'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options->{output_dir}/fastq";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	$dir = "$options->{output_dir}/fastqc";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(-1);
	}
}
