#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   pfam.pl

=head1 SYNOPSIS

    USAGE: pfam.pl -r=run_info.txt -s=sample_name

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--sample, -s>
	Required. Sample name for which pfam should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Run pfam module for a given sample

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./pfam.pl -r=run_info.pl -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use POSIX;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'prefix|p=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

if (defined $ENV{'SGE_TASK_ID'}) {
	$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/PFam.$options{sample}.$ENV{SGE_TASK_ID}.log",
								  'LOG_LEVEL'=>$options{'debug'});
} else {
	$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/PFam.$options{sample}.$options{prefix}.log",
								  'LOG_LEVEL'=>$options{'debug'});
}

$logger = Workflow::Logger::get_logger();

$logger->info("PFam for $options{sample} started");

#### set local variables.
my $cmd = "";
my $this;

if (defined $ENV{'SGE_TASK_ID'}){
	$options{prefix} = $ENV{'SGE_TASK_ID'};
}

$this->{output_dir} = "$options{output_dir}/lincRNA/pfam/$options{sample}/output";
$this->{input} = "$options{output_dir}/lincRNA/pfam/$options{sample}/$options{prefix}.fsa";

check_input($this->{input});

create_dir_struct(\%options);

#### run hmmscan
$cmd = "$config->{ToolInfo}->{hmmscan}->{value}/hmmscan";
$cmd .= " --domtblout $this->{output_dir}/$options{prefix}.Pfam.results.A";
$cmd .= " --noali --incdomE $config->{ToolInfo}->{domain_inclusion_thresholds}->{value}";
$cmd .= " --domE $config->{ToolInfo}->{domain_reporting_thresholds}->{value}";
$cmd .= " --cpu $config->{MemoryInfo}->{pfam_threads}";
$cmd .= " $config->{ToolInfo}->{pfam_db_a}->{value} $this->{input}";
$cmd .= " > $this->{output_dir}/$options{prefix}.A.pfamlog";
execute_cmd($cmd);

#### PFam output fileds are separated by spaces convert them to tabs
$cmd = "sed -i 's/ \\+/\\t/g' $this->{output_dir}/$options{prefix}.Pfam.results.A";
execute_cmd($cmd);

#### replace all single space to tab
#$cmd = "sed -i 's/ /\\t/g' $this->{output_dir}/$options{prefix}.Pfam.results.A";
#execute_cmd($cmd);

#### extract non protein reads from fasta file.
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/extractNonProteinReads.pl";
$cmd .= " -f=$this->{input}";
$cmd .= " -t=$this->{output_dir}/$options{prefix}.Pfam.results.A";
$cmd .= " -o=$this->{output_dir}/$options{prefix}.NonProtein.candidates.PFam.fa";
$cmd .= " -y=pfam";
execute_cmd($cmd);

$logger->info("PFam complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{debug} = 3 unless (defined $options{debug});
	$options{prefix} = 1 unless (defined $options{prefix});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$this->{output_dir}";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/PFAM.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING PFam STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "PFam",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
