#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   mergeGTF.pl

=head1 SYNOPSIS

    USAGE: mergeGTF.pl -r=run_info.txt -o=output_dir

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--help,-h>


=head1 DESCRIPTION
	Merge all GTF files across all samples after scripture and cufflinks run.

=head1 INPUT


=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./mergeGTF.pl -r=run_info.txt -o=output_dir

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
my $MAX_JOB_LIMIT = 3000;
my $LONG_WAIT = 300;
my $SHORT_WAIT = 30;
my $WAIT = 5;

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/MergeGTF.log",
							  'LOG_LEVEL'=>$options{'debug'});

$logger = Workflow::Logger::get_logger();

$logger->info("Merge GTF started");

#### set local variables.
my $cmd = "";
my $this;

$this->{dir} = "$options{output_dir}/lincRNA/cpat";

my @gtfs = split(/\n/, `find $this->{dir}/* -name \"*.final.candidates.gtf\" -print`);

$this->{all_gtfs} = join(" ", @gtfs);

#### create union GTF of Cufflinks and Scripture
$cmd = "$config->{ToolInfo}->{cufflinks}->{value}/cuffcompare -o $options{output_dir}/lincRNA/all";
$cmd .= " $this->{all_gtfs}";
execute_cmd($cmd);

#### check and update tool info file.
$config->{ToolInfo}->{features}->{value} = "$options{output_dir}/lincRNA/all.combined.gtf";
$config->{ToolInfo}->export("$config->{RunInfo}->{tool_info}");

$logger->info("Merge GTF complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/MergeGTF.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING MergeGTF STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "MergeGTF",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
