#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   mergeScriptureGTF.pl

=head1 SYNOPSIS

    USAGE: mergeScriptureGTF.pl -r=run_info.txt -o=output_dir -s=sample_name

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--sample, -s>
	Required. Sample name for which cufflinks should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Merge scripture [and cufflinks] per chr GTF files.  Then create a union of
	scripture and cufflinks GTF

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./mergeScriptureGTF.pl -r=run_info.txt -o=output_dir -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
my $MAX_JOB_LIMIT = 3000;
my $LONG_WAIT = 300;
my $SHORT_WAIT = 30;
my $WAIT = 5;

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/MergeScriptureGTF.$options{sample}.log",
							  'LOG_LEVEL'=>$options{'debug'});

$logger = Workflow::Logger::get_logger();

$logger->info("Merge Scripture GTF for $options{sample} started");

#### set local variables.
my $cmd = "";
my $this;

$this->{dir} = "$options{output_dir}/lincRNA/scripture/$options{sample}";
$this->{cufflinks_dir} = "$options{output_dir}/lincRNA/cufflinks/$options{sample}";

my @chr = split(/:/, $config->{RunInfo}->{chrindex});

foreach my $c (@chr){
	if (! -s "$this->{dir}/$options{sample}.chr$c.gtf"){
		$logger->warn("Cannot find $this->{dir}/$options{sample}.chr$c.gtf.");
	} else {
		$cmd = "cat $this->{dir}/$options{sample}.chr$c.gtf >> $this->{dir}/$options{sample}.gtf";
		execute_cmd($cmd);

		#### delete temp files not used.
		$cmd = "rm $this->{dir}/$options{sample}.chr$c.bed* $this->{dir}/$options{sample}.chr$c.gtf";
		execute_cmd($cmd);
	}

	#### if cufflinks was ran as a split job process then
	#### concat all per chr transcripts into one.
	if ($config->{ToolInfo}->{splitbam}->{value} == 1) {
		if (! -s "$this->{cufflinks_dir}/chr$c/transcripts.gtf"){
			$logger->warn("Cannot find $this->{cufflinks_dir}/chr$c/transcripts.gtf.");
		} else {
			$cmd = "cat $this->{cufflinks_dir}/chr$c/transcripts.gtf >> $this->{cufflinks_dir}/transcripts.gtf";
			execute_cmd($cmd);
		}
	}
}

#### if cufflinks was ran a split job process then
#### unification of cufflinks id hasn't been innitiated yet.
#### do so now
if ($config->{ToolInfo}->{splitbam}->{value} == 1) {
	$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/uniquefy_cufflinks_transcripts.pl";
	$cmd .= " -i=$this->{cufflinks_dir}/transcripts.gtf -o=$this->{cufflinks_dir}/transcripts.mod.gtf";
	execute_cmd($cmd);

	$cmd = "mv $this->{cufflinks_dir}/transcripts.mod.gtf $this->{cufflinks_dir}/transcripts.gtf";
	execute_cmd($cmd);
}

#### change any mRNA feature name to trancrips for scriputre gtf.
$cmd = "sed -i 's/mRNA/transcript/g' $this->{dir}/$options{sample}.gtf";
execute_cmd($cmd);

#### create union GTF of Cufflinks and Scripture
$cmd = "$config->{ToolInfo}->{cufflinks}->{value}/cuffcompare -o $this->{dir}/$options{sample}";
$cmd .= " -s $config->{ToolInfo}->{ref_genome}->{value}";
$cmd .= " $this->{cufflinks_dir}/transcripts.gtf";
$cmd .= " $this->{dir}/$options{sample}.gtf";
execute_cmd($cmd);

#### convert Exon level GTF to transcript level bed file
$cmd = "perl $config->{ToolInfo}->{workflow_path}->{value}/gtf2bed.pl";
$cmd .= " -g $this->{dir}/$options{sample}.combined.gtf";
$cmd .= " -o $this->{dir}/$options{sample}.transcripts.bed";
execute_cmd($cmd);

$logger->info("Merge Scripture GTF complete for $options{sample}");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
	$options{'chr'} = 1 unless (defined $options{'chr'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/MergeScriptureGTF.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING Merge Scripture GTF STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Merge Scripture GTF",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
