#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   filters.pl

=head1 SYNOPSIS

    USAGE: filters.pl -r=run_info.txt -s=sample_name

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--sample, -s>
	Required. Sample name for which cufflinks should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Run filters module for a given sample

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./filters.pl -r=run_info.pl -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use POSIX;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/Filters.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Filters for $options{sample} started");

if ( defined $ENV{'PATH'} ) {
    $ENV{'PATH'}="$config->{ToolInfo}->{r}->{value}:$ENV{'PATH'}";
} else {
    $ENV{'PATH'}="$config->{ToolInfo}->{r}->{value}";
}

if (defined $ENV{'PYTHONPATH'}) {
	$ENV{'PYTHONPATH'} = "$config->{ToolInfo}->{python_cpat}->{value}:$ENV{'PYTHONPATH'}";
} else {
	$ENV{'PYTHONPATH'} = "$config->{ToolInfo}->{pythonpath}->{value}:$config->{ToolInfo}->{python_cpat}->{value}";
}

#### set local variables.
my $cmd = "";
my $this;

$this->{output_dir} = "$options{output_dir}/lincRNA/cpat/$options{sample}";
$this->{cDNA} = "$options{output_dir}/lincRNA/cDNA/$options{sample}";
$this->{scripture} = "$options{output_dir}/lincRNA/scripture/$options{sample}";

$this->{input} = "$this->{scripture}/$options{sample}.transcripts.bed";
check_input($this->{input});

create_dir_struct(\%options);

#### STEP 1: FILTER OUT TRANSCRIPTS LESS THAN 200BP
$cmd = "cat $this->{input} | awk '{if ((\$3-\$2) > 200) print}'";
$cmd .= " > $this->{output_dir}/$options{sample}.transcripts.cov.200bp.bed";
execute_cmd($cmd);

##### NOTE:merge refseq and ensembl for the protein filter
$cmd = "$config->{ToolInfo}->{bedtools}->{value}/windowBed";
$cmd .= " -a $this->{output_dir}/$options{sample}.transcripts.cov.200bp.bed";
$cmd .= " -b $config->{ToolInfo}->{protein_coding}->{value}";
$cmd .= " -w $config->{ToolInfo}->{merge_window}->{value}";
$cmd .= " -v > $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.bed";
execute_cmd($cmd);


#### STEP 2: CPAT FILTER

$cmd = "$config->{ToolInfo}->{python}->{value}/python";
$cmd .= " $config->{ToolInfo}->{cpat}->{value}/cpat.py";
$cmd .= " -g $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.bed";
$cmd .= " -o $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.output";
$cmd .= " -d $config->{ToolInfo}->{cpat_rdata}->{value}";
$cmd .= " -x $config->{ToolInfo}->{cpat_hextab}->{value}";
$cmd .= " -r $config->{ToolInfo}->{ref_genome}->{value}";
execute_cmd($cmd);

#### Extracting non-coding candidates by applying cpat threshold
$cmd = "awk '{ if (\$6 < $config->{ToolInfo}->{cpat_threshold}->{value}) print}'";
$cmd .= " $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.output";
$cmd .= " > $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.filtered.output";
execute_cmd($cmd);

#### Converting cpat output to bed format
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/FormatCPATOutput.pl";
$cmd .= " $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.filtered.output";
$cmd .= " $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.bed";
$cmd .= " $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.filtered.bed";
execute_cmd($cmd);

#### split CPAT bed file with multiple exons per line into one per line
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/split_per_exon.pl";
$cmd .= " $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.filtered.bed";
$cmd .= " $this->{output_dir}/$options{sample}.getfasta.bed";
execute_cmd($cmd);

#### convert extracted bed file to fasta from ref
#### get per exon fasta file
$cmd = "$config->{ToolInfo}->{bedtools}->{value}/fastaFromBed";
$cmd .= " -name -fi $config->{ToolInfo}->{ref_genome}->{value}";
$cmd .= " -bed $this->{output_dir}/$options{sample}.getfasta.bed";
$cmd .= " -fo $this->{output_dir}/$options{sample}.nucleotideFasta.fa";
execute_cmd($cmd);

#### get a list of all unique transcripts
$cmd = "cut -f1,2,3,4 $this->{output_dir}/$options{sample}.cov.200bp.nonprotein.cpat.filtered.bed";
$cmd .= " | awk '{print \$1\"\#\"\$2\"\#\"\$3\"\#\"\$4}' | sort | uniq | tr \"\#\" \"\\t\"";
$cmd .= " > $this->{output_dir}/$options{sample}.transcripts";
execute_cmd($cmd);

#### stitch all exons for a given transcripts into one cDNA.
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/stitch_cdna.pl";
$cmd .= " -t=$this->{output_dir}/$options{sample}.transcripts";
$cmd .= " -f=$this->{output_dir}/$options{sample}.nucleotideFasta.fa";
$cmd .= " -o=$this->{cDNA}";
execute_cmd($cmd);


$logger->info("Filters complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/lincRNA/cpat/$options{sample}";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	$dir = "$options{output_dir}/lincRNA/cDNA/$options{sample}";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/Filters.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING FILTER STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Filter",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
