#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   filterCPAT.pl

=head1 SYNOPSIS

    USAGE: filterCPAT.pl -r=run_info.txt -o=output_dir -s=sample_name

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--sample, -s>
	Required. Sample name for which cufflinks should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Extract CPAT output that matches with BLAST results.

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./filterCPAT.pl -r=run_info.txt -o=output_dir -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Bio::SeqIO;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/FilterCPAT.$options{sample}.log",
							  'LOG_LEVEL'=>$options{'debug'});

$logger = Workflow::Logger::get_logger();

$logger->info("Filter CPAT results $options{sample} started");

#### set local variables.
my $cmd = "";
my $this;
my $seen = {};

$this->{cpat} = "$options{output_dir}/lincRNA/cpat/$options{sample}/$options{sample}.cov.200bp.nonprotein.cpat.filtered.bed";
$this->{blast} = "$options{output_dir}/lincRNA/blast/$options{sample}/$options{sample}.NonProtein.candidates.blast.fsa";
$this->{output} = "$options{output_dir}/lincRNA/cpat/$options{sample}/$options{sample}.final.candidates.bed";

check_input($this->{cpat});
check_input($this->{blast});

open(FHD, "<", $this->{blast}) or $logger->logdie("Could not open file to read $this->{blast}");
open (OUT, ">", $this->{output}) or $logger->logdie("Could not open file to write $this->{output}");

while(<FHD>){
	next if ($_ !~ /^>/);

	chomp $_;

	#### split fasta sequence header line
	#### e.g header: >TCONS_00224936_@_chr7_2_ORF300 Translation of TCONS_00224936_@_chr7 in frame 2, ORF 300, threshold 1, 17aa
	my @info = split(/ /, $_);

	#### get only the id from string
	#### e.g string: >TCONS_00224936_@_chr7_2_ORF300
	my $id = substr($info[0], 1);

	#### split id by _
	#### e.g id: TCONS_00224936_@_chr7_2_ORF300
	@info = split(/_/, $id);

	#### get only the frist 4 bits of header
	$id = join("_", @info[0..scalar(@info)-3]);

	#### fasta file could have multiple seq only need one.
	if (! exists($seen->{$id})) {
		#### extract all lines from input bed file
		$cmd = "grep \"$id\" $this->{cpat}";
		my $rslt = `$cmd`;

		if (length($rslt)) {
			print OUT $rslt;
		}

		$seen->{$id} = 1;
	}
}
close(FHD);
close(OUT);

$cmd = "perl $config->{ToolInfo}->{workflow_path}->{value}/convert_finalbed_to_finalgtf.pl";
$cmd .= " $this->{output} $options{output_dir}/lincRNA/cpat/$options{sample}/$options{sample}.final.candidates.gtf $options{sample}";
execute_cmd($cmd);

$logger->info("Filter CPAT results complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
	$options{'chr'} = 1 unless (defined $options{'chr'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/FilterCPAT.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING Filter CPAT STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Filter CPAT",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
