#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   emboss.pl

=head1 SYNOPSIS

    USAGE: emboss.pl -r=run_info.txt -s=sample_name -o=output_dir

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--sample, -s>
	Required. Sample name for which cufflinks should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Run emboss to find orfs.

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./emboss.pl -r=run_info.pl -s=sample_name -o=output_dir

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use Data::Dumper;
use Bio::SeqIO;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/EMBOSS.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("EMBOSS six frame translation for $options{sample} started");

#### set local variables.
my $cmd = "";
my $this;

$this->{output_dir} = "$options{output_dir}/lincRNA/emboss/$options{sample}";
$this->{input} = "$options{output_dir}/lincRNA/cDNA/$options{sample}";

check_input($this->{input});

create_dir_struct(\%options);

#### final output file
#open(OUT, ">", "$options{output_dir}/$options{sample}.proteinFasta.fa")
#	or die "Can't open output file to write final proteinFasta file\n";

#### run six frame translation for each cDNA fasta file.
opendir(DIR, "$this->{input}") or die "Could not open dir $this->{input}\n";
while (my $file=readdir(DIR)) {
	next if ($file =~ /^\.|^\.\./);

	$cmd = "$config->{ToolInfo}->{emboss}->{value}/sixpack";
	$cmd .= " -mstart Yes -reverse No -sequence $this->{input}/$file";
	$cmd .= " -outfile $this->{output_dir}/$options{sample}";
	$cmd .= " -outseq $this->{output_dir}/$file.proteinFasta.fa";
	$cmd .= " -orfminsize=$config->{ToolInfo}->{min_orf_length}->{value}";
	$cmd .= " 2>/dev/null";
	execute_cmd($cmd);

	#### even thought min orf size is set to 6 and we are requesting orf that
	#### start with M (start codon) there are orfs less than 6 amino acids or
	#### orfs that don't start with M.
	#### apply manual filter.

	#### open fasta file from emboss
	my $seq_in = Bio::SeqIO->new( -file   => "$this->{output_dir}/$file.proteinFasta.fa",
								  -format => 'fasta' );

	my $seq_out = Bio::SeqIO->new( -file   => ">$this->{output_dir}/$file.proteinFasta.filtered.fa",
								   -format => 'fasta' );

	while (my $seq = $seq_in->next_seq) {
		if (($seq->length => $config->{ToolInfo}->{min_orf_length}->{value})
			&& ($seq->seq() =~ /^m/i)){
			$seq_out->write_seq($seq);
		}
	}

	$cmd = "cat $this->{output_dir}/$file.proteinFasta.filtered.fa >> $this->{output_dir}/$options{sample}.proteinFasta.fa";
	execute_cmd($cmd);

	#### remove file to save space.
	#$cmd = "rm $file.proteinFasta.fa";
	#execute_cmd($cmd);
}
close(DIR);


$logger->info("EMBOSS six frame translation complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$this->{output_dir}";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	#while (( $? >> 8 ) != 0 ){
	#	$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
	#	exit(-1);
	#}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -d $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/EMBOSS.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED DIR WHILE RUNNING Emboss STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Emboss",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
