#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   count_.pl

=head1 SYNOPSIS
    USAGE: bam_stats.pl -r=run_info.txt -o=output_dir -s=sample_name

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--output_dir, -o>
	Output directory

B<--sample, -s>
	Sample name

B<--help,-h>


=head1  DESCRIPTION
	Generate report on input bam file.

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./bam_stats.pl -r=run_info.txt -o=output_dir -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
## make sure everything passed was peachy
&check_parameters(\%options);

#### parse X_info files
my $config = new ParseConfig($options{run_info});

my $logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/BAMStats.$options{sample}.log",
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Generate BAM Stats started");

create_dir_struct(\%options);


my $cmd;
my $this = {};
$this->{output} = "$options{output_dir}/numbers/$options{sample}.out";
$this->{alignment} = "$options{output_dir}/alignment/tophat_$options{sample}";
$this->{counts} = "$options{output_dir}/counts";

my $unmapped_reads = 0;
my $mapped_reads = 0;

open(OUT, ">", $this->{output})
	or $logger->logdie("Could not open file to write $this->{output}");

#### check number of mapped and unmapped reads in original input file
#### regardless of the flags used to process cufflinks
my $length = scalar(@{$config->{SampleInfo}->{$options{sample}}});

my $i=0;
for (; $i < $length; $i++) {
	my $original_input = "$config->{SampleInfo}{$options{sample}}[$i]->{filepath}/$config->{SampleInfo}{$options{sample}}[$i]->{read1}";

	#### count total number of unmmaped reads
	#$cmd = "$config->{ToolInfo}->{samtools}->{value}/samtools view -f 4 $original_input | wc -l";
	$cmd = "$config->{ToolInfo}->{samtools}->{value}/samtools view -f 4 -c $original_input";
	my $ur = execute_cmd($cmd);
	chomp $ur;

	$ur = 0 if (! length($ur));
	$unmapped_reads += $ur;

	#### count total number of unique mapped reads
	$cmd = "$config->{ToolInfo}->{samtools}->{value}/samtools view -F 4 $original_input";
	$cmd .= " | cut -f 1 | sort -T $options{output_dir}/numbers/tmp | uniq | wc -l";
	my $mr = execute_cmd($cmd);
	chomp $mapped_reads;

	$mr = 0 if (! length($mr));
	$mapped_reads += $mr;
}

print OUT "Total reads=".($mapped_reads+$unmapped_reads)."\n";
print OUT "Used reads=".($mapped_reads+$unmapped_reads)."\n";

$logger->info("Generate BAM stats complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info($cmd);

	return `$cmd`;
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/numbers/tmp";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}
