#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   lincRNA.pl

=head1 SYNOPSIS

    USAGE: lincRNA.pl -r=run_info.txt

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--auto_clean, -c>
	Optional. Remove all temp file and folders from execution space.
	This should always be set to 0 and run the script
	Indivicually after verifiying all outputs

B<--subset, -s>
	Optional. Specify if running the workflow on a subset.
	A senario where, output folder, and direcotry structure already exists
	for a flowcell/run_id.  Add adding output for this run to existing folder.
		- a case when all but one sample failed and needs a re-run
		- a case when user forgot to add all samples in sample info file
		  and need to run another one for a given flowcell.

B<--help,-h>


=head1 DESCRIPTION
	Execute lincRNA workflow

=head1 INPUT
	Run info file.

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./lincRNA.pl -r=run_info.pl

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use POSIX;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'auto_clean|c=s',
						  'subset|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
my $MAX_JOB_LIMIT = 3000;
my $LONG_WAIT = 300;
my $SHORT_WAIT = 30;
my $WAIT = 5;

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger = new Workflow::Logger('LOG_FILE'=>$options{'log'},
								  'LOG_LEVEL'=>$options{'debug'});
$logger = Workflow::Logger::get_logger();

$logger->info("Workflow started");

#### set local variables.
my $cmd = "";
my $this;

$this->{output_dir} = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}";
$this->{run_info} = "$this->{output_dir}/config/lincRNA/run_info.txt";
$this->{run_on_sge} = 0;
$this->{job} = "";
$this->{q_args} = "";

#### if this run is not a subset of previous run, i.e: MAPRSeq is not run yet
#### run alignment module for MAPRseq first.
#### since alignment module is being run all direcotry strucutre will be create
#### by MAPRSeq workflow.

if (! $options{subset}) {
	#### check directory structure
	create_dir_struct($config);

	#### create local copy of X_info files
	create_local_copy($options{run_info}, "run_info.txt");
	create_local_copy($config->{RunInfo}->{tool_info}, "tool_info.txt");
	create_local_copy($config->{RunInfo}->{sample_info}, "sample_info.txt");
	create_local_copy($config->{RunInfo}->{memory_info}, "memory_info.txt");

	#### prep output dir, copy workflow image, and basic docs.
	prep_output_folder($config);
}

#### if standalone is not defined (back compatibiliby)
#### and its set to anything other than "yes" then assume jobs are to be submmitted
#### to the grid.
#### if standalone is defined and set to "yes" then run locally.
if ((! defined $config->{ToolInfo}->{standalone}->{value}) || ($config->{ToolInfo}->{standalone}->{value} !~ /yes/i)){
	$this->{run_on_sge} = 1;
}

#### check if running on the grid
if ($this->{run_on_sge}) {
	$this->{q_args} = "-wd $config->{RunInfo}->{logs} -b y";
	$this->{q_args} .= " -q $config->{ToolInfo}->{queue}->{value}";
	$this->{q_args} .= " -m $config->{ToolInfo}->{notification_option}->{value}";
	$this->{q_args} .= " -M $config->{RunInfo}->{email} -l h_stack=10M";

	$this->{job} = $config->{RunInfo}->{tool} .".". $config->{RunInfo}->{version};
}

my $job_name = "";
my $bam_stats_name = "";
my $scrp_job_name = "";
my $cuff_job_name = "";
my $gene_job_name = "";
my $exon_job_name = "";
my $hold_job_names = "";

$this->{input_dir} = "$config->{RunInfo}->{input_dir}";

#### check for directory structure
my @sname = keys %{$config->{SampleInfo}};
if ((! -d "$this->{input_dir}/tophat_$sname[0]") ||
	(! -s "$this->{input_dir}/tophat_$sname[0]/$config->{SampleInfo}->{$sname[0]}[0]->{read1}")) {
	#### input directory not as expected create dir structure
	#### before continuing on.

	run_createInputDirStruct_module();
}

foreach my $sample (keys %{$config->{SampleInfo}}) {

	#### reset job name so next sample does not wait on previous sample.
	$job_name = "";

	#### get bam stats for final report
	$bam_stats_name .= run_bamstats_module($sample, $job_name);

	#### split input bam file into accepted_hits.bam and unaccepted_hits.bam.
	#### and sort it for exon and gene count.
	$job_name = run_extractNsort_module($sample, $job_name);

	#### subset input bam file such that all known protein coding regions
	#### are not in the input bam as we are not intersted in these regions.
	if ($config->{ToolInfo}->{extractnonprotein}->{value} == 1) {
		$job_name = run_nonProteinCoding_module($sample, $job_name);
	}

	#### split bam because scripture would be running on per chr bases.
	#### if splitbam flag is set to in tool_info then cufflinks will use this data
	#### as well
	$job_name = run_splitbam_module($sample, $job_name);

	#### run cufflinks module.
	$cuff_job_name = run_cufflinks_module($sample, $job_name);

	#### run scripture module.
	$job_name = run_scripture_module($sample, $job_name);

	#### merge per chrom scripture output
	$job_name .= ",".$cuff_job_name;
	$job_name = run_mergeScriptureGTF_module($sample, $job_name);

	#### run filter module per sample.
	$job_name = run_filters_module($sample, $job_name);

	#### run emboss module per sample.
	$job_name = run_emboss_module($sample, $job_name);

	#### split fasta file for hmmscan
	$job_name = run_splitFASTA_module($sample, $job_name, 0);

	#### run pfam module per sample
	$job_name = run_pfam_module($sample, $job_name);

	#### run BLAST module per sample
	$job_name = run_blast_module($sample, $job_name);

	#### merge BLAST results
	$job_name = run_mergeBLASTout_module($sample, $job_name);

	#### convert final non protein fasta to bed and gtf
	$job_name = run_filtercpat_module($sample, $job_name);
	$scrp_job_name .= "$job_name,";
}

#### neeed to merge all gtfs across samples together before continuing with
#### exon and gene counts.

#### merge all gtfs into one and create update tool info reference
$job_name = run_mergegtf_module($scrp_job_name);

#### modify gtf file in accordance with gene/exon count as used
#### by MAPRSeq workflow.
$job_name = run_modifygtf_module($job_name);

my $mod_gtf_job = $job_name;
foreach my $sample (keys %{$config->{SampleInfo}}) {

	#### run all gene and exon counts at once
	#### after modifygtf module is complete.
	$job_name = $mod_gtf_job;

    #### run gene count
    $gene_job_name = run_genecount_module($sample, $job_name);

    #### run exon count
    $exon_job_name = run_exoncount_module($sample, $job_name);

    #### run chromatin map module
    $scrp_job_name = run_chromatinmap_module($sample, $job_name);

	$job_name = $gene_job_name .",". $exon_job_name . ",". $bam_stats_name;
	$job_name =~ s/,,/,/;
	$job_name =~ s/,$//;
	$job_name =~ s/^,//;
	$job_name = run_numbersreport_module($sample, $job_name);

	#### compile job names for other holds
	$hold_job_names .= $job_name .",";
}

#### merge counts file across sample into one.
$job_name = $hold_job_names;
$job_name =~ s/,,/,/;
$job_name =~ s/,$//;
$job_name =~ s/^,//;
$job_name = run_mergecount_module($job_name);

#### run nearest gene module
$job_name = run_nearestgene_module($job_name);

#### run annotation module
$job_name = run_annotation_module($job_name);

#### run reports.
run_report_module($job_name);

$logger->info("Workflow complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	#### create a default log file.
	unless($options{'log'}){
		my $dir = getcwd;
		my ($second, $minute, $hour) = localtime();
		$options{'log'} = $dir."/lincRNA." . $config->{RunInfo}->{output_folder};
		$options{'log'} .= "." .$hour. "" .$minute. "" .$second. ".log";
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});

	#### if this run is a subset of previous run, then don't
	#### setup, create folder structure or copy required files.
	#### it is assumed that folder structure already exists.
	$options{'subset'} = 0 unless (defined $options{'subset'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	#### check if PI folder exist else create one
	my $dir = "$options->{RunInfo}->{base_output_dir}/$options->{RunInfo}->{pi}";
	if ( -d $dir) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	#### check if analysis type folder exist else create one
	$dir = "$options->{RunInfo}->{base_output_dir}/$options->{RunInfo}->{pi}/$options->{RunInfo}->{type}";
	if ( -d $dir ) {
		$logger->info("Analysis type $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	#### check if output folder exists else create one.
	$dir = "$options->{RunInfo}->{base_output_dir}/$options->{RunInfo}->{pi}/$options->{RunInfo}->{type}/$options->{RunInfo}->{output_folder}";
	if ( -d $dir ) {
		$logger->info("Output run folder $dir exists");
	} else {
		execute_cmd("mkdir -p $dir");
	}

	my $outdir = $dir;

	my @dirs = qw(error lincRNA 'config/lincRNA');

	foreach my $d (@dirs) {
		#### check if folder exists else create one.
		$dir = "$outdir/$d";
		if ( -d $dir ) {
			$logger->info("Output folder $dir exists");
		} else {
			execute_cmd("mkdir -p $dir");
		}
	}

	#### create workflow logs dir.
	$dir = $options->{RunInfo}->{logs};
	execute_cmd("mkdir -p $dir");
}

#############################################################################
sub create_local_copy {
	my $filename = shift;
	my $local_name = shift;

	my $cmd = "cp $filename $this->{output_dir}/config/lincRNA/$local_name";
	execute_cmd("$cmd");
}

#############################################################################
sub prep_output_folder {
	my $options = shift;

	#### update run_info with local copy of sample, tool and memory info
	$config->{RunInfo}->{sample_info} = "$this->{output_dir}/config/lincRNA/sample_info.txt";
	$config->{RunInfo}->{tool_info} = "$this->{output_dir}/config/lincRNA/tool_info.txt";
	$config->{RunInfo}->{memory_info} = "$this->{output_dir}/config/lincRNA/memory_info.txt";

	#### export run_info updates
	$config->{RunInfo}->export("$this->{output_dir}/config/lincRNA/run_info.txt");
}

#############################################################################
sub run_createInputDirStruct_module {
	my $cmd = "";

	foreach my $sample (keys %{$config->{SampleInfo}}) {
		my $dir = "$this->{output_dir}/alignment/tophat_$sample";
		$cmd = "mkdir -p $dir";

		execute_cmd($cmd);

		my $length = scalar(@{$config->{SampleInfo}->{$sample}});

		my $i=0;
		for (; $i < $length; $i++) {
			if (! -s "$dir/${sample}_sorted.bam") {
				$cmd = "ln -s $config->{SampleInfo}{$sample}[$i]->{filepath}/$config->{SampleInfo}{$sample}[$i]->{read1}";
				$cmd .= " $dir/${sample}_sorted.bam";

				execute_cmd($cmd);
			}
		}
	}

	#### override input dir with new location.
	$this->{input_dir} = "$this->{output_dir}/alignment";
}

#############################################################################
sub run_extractNsort_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".SortBam.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{sort_mem}";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{sort_threads} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/extract_n_sort.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt -o=$this->{output_dir}";
	$cmd .= " -i=$this->{input_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_nonProteinCoding_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".NonProteinCoding.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{gen_mem} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/extract_nonProteinCoding.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt -o=$this->{output_dir}";
	$cmd .= " -i=$this->{input_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_bamstats_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".BAMStats.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= "  -l h_vmem=$config->{MemoryInfo}->{bam_stat} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/bam_stats.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt -o=$this->{output_dir}";
	$cmd .= " -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_splitbam_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".SplitBAM.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my @chr = split(/:/, $config->{RunInfo}->{chrindex});
	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{splitbam_mem} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}

		$cmd .= " -t 1-".scalar(@chr);

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/split_bam.pl";
		$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt -o=$this->{output_dir}";
		$cmd .= " -i=$this->{input_dir} -s=$sample";

		execute_qsub_cmd($cmd);
	} else {
		foreach my $idx (1..scalar(@chr)) {
			$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/split_bam.pl";
			$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt -o=$this->{output_dir}";
			$cmd .= " -i=$this->{input_dir} -s=$sample -c=$idx";

			execute_qsub_cmd($cmd);
		}
	}

	return $job_name;
}

#############################################################################
sub run_cufflinks_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".CUFFLINKS.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my @chr = split(/:/, $config->{RunInfo}->{chrindex});
	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{cufflinks_mem}";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{cufflinks_threads} -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}

		#### only run an array job if splitting cufflinks job by chr.
		if ($config->{ToolInfo}->{splitbam}->{value} == 1) {
			$cmd .= " -t 1-".scalar(@chr);
		}

		#### change hstack value
		$cmd =~ s/$config->{ToolInfo}->{queue}->{value}/$config->{ToolInfo}->{cuff_queue}->{value}/;
		$cmd =~ s/-l h_stack=10M/$config->{MemoryInfo}->{cufflinks_stack}/;

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/cufflinks.pl";
		$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";

		if ($config->{ToolInfo}->{splitbam}->{value} == 1) {
			$cmd .= " -i=$this->{output_dir}/lincRNA/splitBAM/$sample";
		} else {
			$cmd .= " -i=$this->{input_dir}/tophat_$sample";
		}

		$cmd .= " -o=$this->{output_dir} -s=$sample";

		execute_qsub_cmd($cmd);
	} else {
		if ($config->{ToolInfo}->{splitbam}->{value} == 1) {
			foreach my $idx (1..scalar(@chr)) {
				$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/cufflinks.pl";
				$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
				$cmd .= " -i=$this->{output_dir}/lincRNA/splitBAM/$sample";
				$cmd .= " -o=$this->{output_dir} -s=$sample -c=$idx";

				execute_qsub_cmd($cmd);
			}
		} else {
			$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/cufflinks.pl";
			$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
			$cmd .= " -i=$this->{input_dir}/tophat_$sample";
			$cmd .= " -o=$this->{output_dir} -s=$sample";

			execute_qsub_cmd($cmd);
		}
	}

	return $job_name;
}

#############################################################################
sub run_scripture_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".SCRIPTURE.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my @chr = split(/:/, $config->{RunInfo}->{chrindex});
	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{scripture_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}

		$cmd .= " -t 1-".scalar(@chr);

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/scripture.pl";
		$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
		$cmd .= " -i=$this->{output_dir}/lincRNA/splitBAM/$sample";
		$cmd .= " -o=$this->{output_dir} -s=$sample";

		execute_qsub_cmd($cmd);
	} else {
		foreach my $id (1..scalar(@chr)) {
			$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/scripture.pl";
			$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
			$cmd .= " -i=$this->{output_dir}/lincRNA/splitBAM/$sample";
			$cmd .= " -o=$this->{output_dir} -s=$sample -c=$id";

			execute_qsub_cmd($cmd);
		}
	}

	return $job_name;
}

#############################################################################
sub run_mergeScriptureGTF_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".MERGESCRIPTUREGTF.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{scripturegtf_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/mergeScriptureGTF.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_filters_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".FILTERS.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{filters_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/filters.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_emboss_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".EMBOSS.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{emboss_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/emboss.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_splitFASTA_module {
	my $sample = shift;
	my $hold = shift;
	my $for_blast = shift;

	my $job_name = $this->{job} .".SPLITFASTA.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{splitfsa_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/split_multifasta.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample -b=$for_blast";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_pfam_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".PFAM.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{pfam_mem}";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{pfam_threads}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}

		$cmd .= " -t 1-".$config->{ToolInfo}->{max_fsa_split}->{value};

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/pfam.pl";
		$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
		$cmd .= " -o=$this->{output_dir} -s=$sample";

		execute_qsub_cmd($cmd);
	} else {
		foreach my $id (1..$config->{ToolInfo}->{max_fsa_split}->{value}) {
			$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/pfam.pl";
			$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
			$cmd .= " -o=$this->{output_dir} -s=$sample -p=$id";

			execute_qsub_cmd($cmd);
		}
	}

	return $job_name;
}

#############################################################################
sub run_blast_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".BLASTP.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{blast_mem}";
		$cmd .= " -pe threaded $config->{MemoryInfo}->{blast_threads}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}

		$cmd .= " -t 1-".$config->{ToolInfo}->{max_fsa_split}->{value};

		$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/blast.pl";
		$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
		$cmd .= " -o=$this->{output_dir} -s=$sample";

		execute_qsub_cmd($cmd);
	} else {
		foreach my $id (1..$config->{ToolInfo}->{max_fsa_split}->{value}) {
			$cmd = " $config->{ToolInfo}->{workflow_path}->{value}/blast.pl";
			$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
			$cmd .= " -o=$this->{output_dir} -s=$sample -p=$id";

			execute_qsub_cmd($cmd);
		}
	}

	return $job_name;
}

#############################################################################
sub run_mergeBLASTout_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".MERGEBLAST.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{mergeblast_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/mergeBLAST.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_filtercpat_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".FilterCPAT.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{filtercpat_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/filterCPAT.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_mergegtf_module {
	my $hold = shift;

	my $cmd = "";

	my $job_name = $this->{job} . ".MergeGTF.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		#### run GTF modification module
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= "  -l h_vmem=$config->{MemoryInfo}->{mergegtf_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/mergeGTF.pl";
	$cmd .= " -o=$this->{output_dir} -r=$this->{output_dir}/config/lincRNA/run_info.txt";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_modifygtf_module {
	my $hold = shift;

	my $cmd = "";

	my $job_name = $this->{job} . ".ModGTF.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	#### check if running on the grid
	if ($this->{run_on_sge}) {
		#### run GTF modification module
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{gen_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/modify_gtf.pl";
	$cmd .= " -o=$this->{output_dir}/lincRNA -r=$this->{output_dir}/config/lincRNA/run_info.txt";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_genecount_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".GeneCount.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{gene_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/gene_count.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -i=$this->{input_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_exoncount_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".ExonCount.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{exon_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/exon_count.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -i=$this->{input_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_numbersreport_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".NumbersReport.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{num_report}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/numbers_report.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_mergecount_module {
	my $hold = shift;

	my $job_name = $this->{job} .".MergeCounts.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{mergecount_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/merge_counts.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir}";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_chromatinmap_module {
	my $sample = shift;
	my $hold = shift;

	my $job_name = $this->{job} .".ChromatinMap.$sample." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{chromatin_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/chromatin_map.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -s=$sample";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_nearestgene_module {
	my $hold = shift;

	my $job_name = $this->{job} .".NearestGene.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{neargene_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/nearest_gene.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir}";

	execute_qsub_cmd($cmd);

	return $job_name;
}


#############################################################################
sub run_annotation_module {
	my $hold = shift;

	my $job_name = $this->{job} .".Annotation.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{anno_mem}";
		$cmd .= " -N $job_name";

		#### hold on alignment or sortbam
		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/annotation.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir} -i=$this->{output_dir}/lincRNA";

	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub run_report_module {
	my $hold = shift;

	my $job_name = $this->{job} .".FinalReports.MULTI." . $config->{RunInfo}->{output_folder};
	$job_name .= "." . $config->{RunInfo}->{unique_id};

	my $cmd = "";
	#### check if running on the grid
	if ($this->{run_on_sge}) {
		$cmd = "$config->{ToolInfo}->{sge}->{value}/qsub $this->{q_args}";
		$cmd .= " -l h_vmem=$config->{MemoryInfo}->{report_mem}";
		$cmd .= " -N $job_name";

		if ( length($hold) ) {
			$cmd .= " -hold_jid $hold";
		}
	}

	$cmd .= " $config->{ToolInfo}->{workflow_path}->{value}/reports.pl";
	$cmd .= " -r=$this->{output_dir}/config/lincRNA/run_info.txt";
	$cmd .= " -o=$this->{output_dir}";
	execute_qsub_cmd($cmd);

	return $job_name;
}

#############################################################################
sub execute_qsub_cmd {
	my $cmd = shift;

	#### check if running on the grid
	if (! $this->{run_on_sge}) {
		execute_cmd($cmd);
		return;
	}

	my $count = 0;
	my $send_mail = 1;

	#### check if job limit is hit
	check_job_limit();

	sleep $WAIT;

	$logger->info("$cmd");
	system($cmd);

	while ((( $? >> 8 ) != 0 ) && ($count < 5)){
		sleep $SHORT_WAIT;
		$count ++;

		system($cmd);
	}

	if (( $? >> 8 ) != 0 ) {
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/Cmd.err";
		$util->createErrorFile($error_name, $cmd);

		while (-e $error_name) {

			#### pass Filename, cmd executed, email to, step running, sample name.
			if ($send_mail) {
				$util->reportError($error_name,
							   $cmd,
							   $config->{RunInfo}->{email},
							   "Executing command",
							   $options{sample},
							   "",
							   "");
				$send_mail = 0;
			}
			sleep $LONG_WAIT;
		}
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");

		exit(-1);
	}
}

#############################################################################
sub check_job_limit {
	my $cmd = "$config->{ToolInfo}->{sge}->{value}/qstat";

	my @cmd_output = qx{$cmd};

	#### wait till number of jobs in queue are less than job limit to
	#### continue execution.
	while (scalar(@cmd_output) >= $MAX_JOB_LIMIT) {
		sleep $LONG_WAIT;

		@cmd_output = qx{$cmd};
	}
}
