#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   extract_nonProteinCoding.pl

=head1 SYNOPSIS

    USAGE: extract_nonProteinCoding.pl -r=run_info.txt -o=output_dir -s=sample_name

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--input_dir, -i>
	Required. Input dir location.

B<--sample, -s>
	Required. Sample name for which cufflinks should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Extract non protein coding reads from input BAM file

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./extract_nonProteinCoding.pl -r=run_info.pl -o=output_dir -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use POSIX;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'input_dir|i=s',
						  'sample|s=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/NonProteinCoding.$options{sample}.log",
						  'LOG_LEVEL'=>$options{'debug'});

$logger = Workflow::Logger::get_logger();

$logger->info("Extract non protein coding for $options{sample} started");

#### set local variables.
my $cmd = "";
my $this;

$this->{input} = "$options{input_dir}/tophat_$options{sample}/accepted_hits.bam";
$this->{output} = "$options{output_dir}/alignment/tophat_$options{sample}/accepted_hits.nonProtein.bam";

check_input($this->{input});

$cmd = "$config->{ToolInfo}->{bedtools}->{value}/intersectBed -abam $this->{input}";
$cmd .= " -b $config->{ToolInfo}->{protein_coding}->{value} -v";
$cmd .= " > $this->{output}";

execute_cmd($cmd);

#### move files around, so subsequent files don't need to be changed.
#### rename original input file
$cmd = "mv $this->{input} $this->{input}.all";
execute_cmd($cmd);

#### remove index file for original input file
$cmd = "rm $this->{input}.bai";
execute_cmd($cmd);

#### rename output to original input name
$cmd = "mv $this->{output} $options{output_dir}/alignment/tophat_$options{sample}/accepted_hits.bam";
execute_cmd($cmd);

#### create a new bam index file for only nonProtein coding file.
$cmd = "$config->{ToolInfo}->{samtools}->{value}/samtools index $options{output_dir}/alignment/tophat_$options{sample}/accepted_hits.bam";
execute_cmd($cmd);

$logger->info("Extract non protein coding for $options{sample} complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$this->{output_dir}";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	#while (( $? >> 8 ) != 0 ){
	#	$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
	#	exit(-1);
	#}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/ExtractNonProteinCoding.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING Extract non protein coding STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Extract Non Protein Coding",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
