#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   dashboard.pl

=head1 SYNOPSIS
    USAGE: dashboard.pl -r=run_info.txt -t=stage -s=sample_name -c=0/1

=head1 OPTIONS

B<--run_info, -r>
	Run info file

B<--stage, -t>
	Stage at which anlaysis step is currently at.

B<--sample, -s>
	sample name

B<--complete, -c>
	If stage is complete or not.

B<--help,-h>


=head1  DESCRIPTION
	Indicate at what stage currently process is for a given sample and if its complete or not.

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./dashboard.pl -r=run_info.pl -t=stage -s=sample_name -c=0/1

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'sample|s=s',
						  'stage|t=s',
						  'complete|c=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### make sure everything passed was peachy
&check_parameters(\%options);

#### parse X_info files
my $config = new ParseConfig($options{run_info});

#### do not procced if dashboard update is not avaialbe or set to no
if ((! defined $config->{ToolInfo}->{dashboard_update}->{value}) || ($config->{ToolInfo}->{dashboard_update}->{value} =~ /no/i)) {
    exit(0);
}

#### if the run_id is not in the expected format skip update.
#### run_id should be of the following format
#### YYMMDD_ABCD_ABDD_DABDERE
#### e.g format expected is
#### 		121112_SN7001166_0111_BD1KD4ACXX
if ($config->{RunInfo}->{output_folder} !~ /^\d+_\w+_\w+_\w+$/) {
	print STDERR "INFO: Skipping dashboard update since run id/output folder is not of expected format\n";
	exit(0);
}

my $cmd = "$config->{ToolInfo}->{java}->{value}/java -Xmx";
$cmd .= $config->{MemoryInfo}->{dashbrd_jvmx};
$cmd .= " -Xms512m";
$cmd .= " -jar $config->{ToolInfo}->{dashboard_jar}->{value}/AddSecondaryAnalysis.jar";
$cmd .= " -p $config->{ToolInfo}->{dashboard_jar}->{value}/AddSecondaryAnalysis.properties";
$cmd .= " -a $config->{RunInfo}->{tool}";
$cmd .= " -r $config->{RunInfo}->{output_folder}";

#### get group id
if ((! defined $config->{RunInfo}->{group_id}) || ($config->{RunInfo}->{group_id} == 0)) {
	get_group_id();

	#### add all samples to given group id.
	foreach my $sample (keys %{$config->{SampleInfo}}) {
		#### duplicate command and append lane, lab index
		my $l_cmd = $cmd;

		if ((defined $config->{SampleInfo}{$sample}[0]->{laneindex}) &&
			(length($config->{SampleInfo}{$sample}[0]->{laneindex})) &&
			($config->{SampleInfo}{$sample}[0]->{laneindex} !~ /-/)) {

			$l_cmd .= " -l $config->{SampleInfo}{$sample}[0]->{laneindex}";
		}

		if ((defined $config->{SampleInfo}{$sample}[0]->{labindex}) &&
			(length($config->{SampleInfo}{$sample}[0]->{labindex})) &&
			($config->{SampleInfo}{$sample}[0]->{labindex} !~ /-/)) {
			$l_cmd .= " -i $config->{SampleInfo}{$sample}[0]->{labindex}";
		}

		$l_cmd .= " -g $config->{RunInfo}->{group_id} --addToAnalysis";

		#### only execute if this is a mayo run and
		execute_cmd($l_cmd);
	}
}

#### at each step of completion update status.
if ($options{complete}){
	$cmd .= " -g $config->{RunInfo}->{group_id} -c";
	$cmd .= " -s $options{stage}";
	execute_cmd($cmd);
}

exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info stage);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless ($options{'debug'});
	$options{'complete'} = "No" unless (defined $options{'complete'});
}


#############################################################################
sub get_group_id {

	my @samples = sort keys %{$config->{SampleInfo}};

	my $cmd = "$config->{ToolInfo}->{java}->{value}/java -Xmx";
	$cmd .= $config->{MemoryInfo}->{dashbrd_jvmx};
	$cmd .= " -Xms512m";
	$cmd .= " -jar $config->{ToolInfo}->{dashboard_jar}->{value}/AddSecondaryAnalysis.jar";
	$cmd .= " -p $config->{ToolInfo}->{dashboard_jar}->{value}/AddSecondaryAnalysis.properties";
	$cmd .= " -r $config->{RunInfo}->{output_folder}";
	$cmd .= " -l $config->{SampleInfo}{$samples[0]}[0]->{laneindex}";

	if ($config->{SampleInfo}{$samples[0]}[0]->{labindex} !~ /^-$/){
		$cmd .= " -i $config->{SampleInfo}{$samples[0]}[0]->{labindex}";
	}

	#### get unique group id for the run.
	my $group_id = `$cmd`;
	chomp $group_id;

	$config->{RunInfo}->{group_id} = $group_id;

	#### export run_info updates
	$config->{RunInfo}->export($options{run_info});
}


#############################################################################
sub execute_cmd {
	my $cmd = shift;

	print $cmd."\n";
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		print STDERR "Above command failed to execute: $!\n";
		exit(-1);
	}
}
