#!/usr/bin/perl

=head1 NAME
   install.pl

=head1 SYNOPSIS
    USAGE: install.pl --prefix=/location/of/install/dir

=head1 OPTIONS

B<--prefix, -p>
	Required. Prefix location where package will be installed.

B<--perl_exec, -e>
	Optional.  If perl exec is other than /usr/bin/perl please specify location of perl install

B<--verbos, -v>
	Optional. Run verbos mode

B<--help,-h>


=head1  DESCRIPTION
	Install package

=head1  INPUT

=head1  OUTPUT


=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./install.pl --prefix=/prefix

=cut

use strict;
use warnings;
use Cwd;
use FindBin qw ($Bin);
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);

my %options = ();
my $results = GetOptions (\%options,
                          'prefix|p=s',
						  'perl_exec|e=s',
						  'help|h') || pod2usage();

## display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### make sure everything passed was peachy
&check_parameters(\%options);

#### print time now.
timestamp();

my $this = {};
my $progress = {};
my $cmd = "";

#### get current working dir
$this->{source} = $Bin;

$progress = getProgress();

#### make logs dir
$cmd = "mkdir -p $options{prefix}/logs";
execute_cmd($cmd);

#### unpack binary dir containing all binary to be installed
#### which are required for successfull run
print STDERR "\n\nCheck for required binaries...\n"  if ($options{verbos});

#### run env_check  by Matt Bockol
$cmd = "$this->{source}/env_check/env_check --toolinfo=$this->{source}/script/config/tool_info.txt";
execute_cmd($cmd);

#### installling libraries required for successfull run
install_libraries();

#### copy source code and update paths for perl and libs
install_source();

#### unzip and install references
install_references();

#### install pfam database
install_pfam();

#### get chromosome reference files
install_chr();

#### get blast databases files
install_blast_db();

#### create a test run_info file.
setup_runInfo();

#### create a test tool_info file based on install prefix
setup_toolInfo();

#### create a test sample_info file based on example samples
setup_sampleInfo();

#### copy memory info to config dir.
setup_memoryInfo();

#### move sample input files
setup_testInput();

#### completion message
print "\n\n\nlincRNA installation complete.  Use following command to initiate a test run\n";
print "\n\t$options{prefix}/src/lincRNA.pl -r=$options{prefix}/config/run_info.txt\n\n";

#### print time now
timestamp();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(prefix);

	foreach my $key (@required) {
		unless ($options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'perl_exec'} = "/usr/bin/perl" unless($options{'perl_exec'});
	$options{verbos} = 0 unless(defined $options{verbos})
}

#############################################################################
sub getProgress {
	my $hash = {};
	my @sofar;

	#### if file exists get progress so far.
	if (-s "$options{prefix}/progress.txt") {
		open(FHD, "<", "$options{prefix}/progress.txt") or die "Could not open file to read $options{prefix}/progress.txt";
		while(<FHD>){
			chomp $_;
			push @sofar, $_;
		}
		close(FHD);

		map { $hash->{$1} = $2 if( /([^=]+)\s*=\s*([^=]+)/ ) } @sofar;
	}

	#### return hash
	return $hash;
}

#############################################################################
sub setProgress {
	my $hash = shift;

	open(OUT, ">", "$options{prefix}/progress.txt") or die "Could not open file to write $options{prefix}/progress.txt";

	foreach my $key (keys %{$hash}){
		print OUT $key."=".$hash->{$key}."\n";
	}

	close(OUT);
}

#############################################################################
sub install_cufflinks {
	#### check and setup install dir
	my $dir = "$options{prefix}/bin/cufflinks/2.1.1";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/cufflinks-2.1.1.Linux_x86_64.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	$cmd = "mv $this->{source}/binary/cufflinks-2.1.1.Linux_x86_64/*";
	$cmd .= " $options{prefix}/bin/cufflinks/2.1.1/.";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/cufflinks-2.1.1.Linux_x86_64";
	execute_cmd($cmd);

	$progress->{cufflinks} = 1;
	setProgress($progress);
}

#############################################################################
sub install_scripture {
	#### check and install dir
	my $dir = "$options{prefix}/bin/scripture/beta2";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/scripture-beta2.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	$cmd = "mv $this->{source}/binary/scripture-beta2.jar $options{prefix}/bin/scripture/beta2/.";
	execute_cmd($cmd);

	$progress->{scripture} = 1;
	setProgress($progress);
}

#############################################################################
sub install_hmmer {
	#### check and install dir
	my $dir = "$options{prefix}/bin/hmmer/3.1b1";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/hmmer-3.1b1-linux-intel-x86_64.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/hmmer-3.1b1-linux-intel-x86_64");
	$cmd = "$this->{source}/binary/hmmer-3.1b1-linux-intel-x86_64/configure";
	$cmd .= " --prefix=$options{prefix}/bin/hmmer/3.1b1";
	$cmd .= " && make && make install";
	$cmd .= " 1>$options{prefix}/logs/hmmer_log.out";
	$cmd .= " 2>$options{prefix}/logs/hmmer_log.err";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/hmmer-3.1b1-linux-intel-x86_64";
	execute_cmd($cmd);

	$progress->{hmmer} = 1;
	setProgress($progress);
}

#############################################################################
sub install_emboss {
	#### check and install dir
	my $dir = "$options{prefix}/bin/emboss/6.5.7";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/EMBOSS-6.5.7.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/EMBOSS-6.5.7");
	$cmd = "$this->{source}/binary/EMBOSS-6.5.7/configure";
	$cmd .= " --prefix=$options{prefix}/bin/emboss/6.5.7";
	$cmd .= " && make && make install";
	$cmd .= " 1>$options{prefix}/logs/emboss_log.out";
	$cmd .= " 2>$options{prefix}/logs/emboss_log.err";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/EMBOSS-6.5.7";
	execute_cmd($cmd);

	$progress->{emboss} = 1;
	setProgress($progress);
}

#############################################################################
sub install_cpat {
	#### check and install dir
	my $dir = "$options{prefix}/bin/cpat/1.2.1";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
		execute_cmd($cmd);
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	$cmd = "tar -zxvf $this->{source}/binary/CPAT-1.2.1.tar.gz -C $this->{source}/binary";
	execute_cmd($cmd);

	#### create lib dir
	$cmd = "mkdir -p $dir/lib/python2.7/site-packages/";
	execute_cmd($cmd);

	chdir("$this->{source}/binary/CPAT-1.2.1");

	if (defined $ENV{'PYTHONPATH'}) {
		$ENV{'PYTHONPATH'} = "$dir/lib/python2.7/site-packages/:$ENV{'PYTHONPATH'}";
	} else {
		$ENV{'PYTHONPATH'} = "$dir/lib/python2.7/site-packages/";
	}

	#### move install to bin dir
	$cmd = "python setup.py install";
	$cmd .= " --prefix=$dir";
	$cmd .= " 1>$options{prefix}/logs/cpat_log.out";
	$cmd .= " 2>$options{prefix}/logs/cpat_log.err";
	execute_cmd($cmd);

	chdir("$this->{source}/binary");
	$cmd = "rm -rf $this->{source}/binary/CPAT-1.2.1";
	execute_cmd($cmd);

	$progress->{cpat} = 1;
	setProgress($progress);
}

#############################################################################
sub install_libraries {
	if ((exists $progress->{libraries}) && ($progress->{libraries})){
		if ($options{verbos}) {
			print STDERR "\tLibraries already installed. Skipping...\n";
		}
		return;
	}

	print STDERR "\n\nInstalling libraries...\n\n";
	chdir($this->{source});

	$cmd = "cp -r $this->{source}/library $options{prefix}/lib";
	execute_cmd($cmd);

	$progress->{libraries} = 1;
	setProgress($progress);
}

#############################################################################
sub install_references {
	if ((exists $progress->{references}) && ($progress->{references})){
		if ($options{verbos}) {
			print STDERR "\tReferences already installed. Skipping...\n";
		}
		return;
	}

	print STDERR "\n\nInstalling references...\n\n";

	#### unzip and create reference dataset
	chdir("$this->{source}");

	$cmd = "cp -r $this->{source}/references $options{prefix}/references";
	execute_cmd($cmd);

	$progress->{references} = 1;
	setProgress($progress);
}

#############################################################################
sub install_pfam {
	if ((exists $progress->{pfam}) && ($progress->{pfam})){
		if ($options{verbos}) {
			print STDERR "\tPFam already installed. Skipping...\n";
		}
		return;
	}

	#my @famDB = qw(Pfam-A.hmm  Pfam-A.hmm.h3f  Pfam-A.hmm.h3i  Pfam-A.hmm.h3m  Pfam-A.hmm.h3p  Pfam-B.hmm  Pfam-B.hmm.h3f  Pfam-B.hmm.h3i  Pfam-B.hmm.h3m  Pfam-B.hmm.h3p);
	my @famDB = qw(Pfam-A.hmm Pfam-A.hmm.h3f Pfam-A.hmm.h3i Pfam-A.hmm.h3m Pfam-A.hmm.h3p);

	print "If PFAM database already exists please enter complete path or press ENTER to start download: ";
	my $path = <STDIN>;

	chomp $path;

	if (length($path) && $path !~ /^$/) {
		#### check path and ensure files exists

		die "Path invalid.  Try again" if (! -d $path);

		foreach my $f (@famDB){
			if (! -s "$path/$f") {
				die "Cannot find $path/$f. Try again\n";
			}
		}

		#### update path for pfam location
		$path =~ s/\//\\\//g;
		my $src = "\@REFERENCE\@/pfam/human";
		$src =~ s/\//\\\//g;
		$cmd = "sed -i 's/$src/$path/g' $options{prefix}/config/tool_info.txt";
		execute_cmd($cmd);
	} else {
		#### setup base url
		my $url = "ftp://ftp.sanger.ac.uk/pub/databases/Pfam/releases/Pfam27.0/";

		my $cmd = "mkdir -p $options{prefix}/references/pfam/human";
		execute_cmd($cmd);

		chdir ("$options{prefix}/references/pfam/human");
		print "Downloading Pfam 27.0 database.  This make take some time depending on your internet connection\n";

		print "\tDownloading Pfam-A...\n";
		$cmd = "wget -q $url/Pfam-A.hmm.gz";
		execute_cmd($cmd);

		$cmd = "gunzip Pfam-A.hmm.gz";
		execute_cmd($cmd);

		#print "\tDownloading Pfam-B...\n";
		#$cmd = "wget -q $url/Pfam-B.hmm.gz";
		#execute_cmd($cmd);
		#
		#$cmd = "gunzip Pfam-B.hmm.gz";
		#execute_cmd($cmd);

		#### create hmm compressed index
		$cmd = "hmmpress Pfam-A.hmm";
		execute_cmd($cmd);

		#$cmd = "hmmpress Pfam-B.hmm";
		#execute_cmd($cmd);
	}

	$progress->{pfam} = 1;
	setProgress($progress);
}

#############################################################################
sub install_chr {
	if ((exists $progress->{chr}) && ($progress->{chr})){
		if ($options{verbos}) {
			print STDERR "\tCHR already installed. Skipping...\n";
		}
		return;
	}
	my @chr = qw(1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 X Y M);

	print "If reference chromoroses for human genome (hg19 build 37.1) already exists please enter complete path or press ENTER to start download: ";
	my $path = <STDIN>;

	chomp($path);

	if (length($path) && $path !~ /^$/) {
		#### check path and ensure files exists

		die "Path invalid.  Try again" if (! -d $path);

		foreach my $c (@chr){
			if (! -s "$path/chr${c}.fa") {
				die "Cannot find $path/chr${c}.fa. Try again\n";
			}
		}

		#### update path for chr location
		$path =~ s/\//\\\//g;
		my $src = "\@REFERENCE\@/hg19/37.1";
		$src =~ s/\//\\\//g;
		$cmd = "sed -i 's/$src/$path/g' $options{prefix}/config/tool_info.txt";
		execute_cmd($cmd);
	} else {
		#### setup base url
		my $url = "ftp://hgdownload.cse.ucsc.edu/goldenPath/hg19/chromosomes";

		my $cmd = "mkdir -p $options{prefix}/references/hg19/37.1";
		execute_cmd($cmd);

		chdir ("$options{prefix}/references/hg19/37.1");
		print "Downloading reference chromosomes for hg19 Genome build.  This make take some time depending on your internet connection\n";

		foreach my $c (@chr){
			print "Downloading chr${c}...\n";
			$cmd = "wget -q $url/chr${c}.fa.gz";
			execute_cmd($cmd);

			$cmd = "gunzip chr${c}.fa.gz";
			execute_cmd($cmd);

			$cmd = "cat chr${c}.fa >> $options{prefix}/references/hg19/37.1/allchr.fa";
			execute_cmd($cmd);
		}
	}

	$progress->{chr} = 1;
	setProgress($progress);
}

#############################################################################
sub install_blast_db {
	if ((exists $progress->{blast_db}) && ($progress->{blast_db})){
		if ($options{verbos}) {
			print STDERR "\tBLAST DB already installed. Skipping...\n";
		}
		return;
	}

	my $url = "ftp://ftp.ncbi.nlm.nih.gov/blast/db/";
	my @nr = qw(00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16);

	print "If BLAST NR database already exists please enter complete path or press ENTER to start download: ";
	my $path = <STDIN>;
	chomp $path;

	if (length($path) && $path !~ /^$/) {
		#### check path and ensure files exists

		die "Path invalid.  Try again" if (! -d $path);

		foreach my $db (@nr){
			if (! -s "$path/nr.${db}.phr") {
				die "Cannot find $path/nr.${db}.phr. Try again\n";
			}
		}

		if (! -s "$path/nr.pal") {
			die "Cannot find $path/nr.pal. Try again\n";
		}

		#### update path for chr location
		$path =~ s/\//\\\//g;
		my $src = "\@REFERENCE\@/blast_db/nr";
		$src =~ s/\//\\\//g;
		$cmd = "sed -i 's/$src/$path/g' $options{prefix}/config/tool_info.txt";
		execute_cmd($cmd);
	} else {
		my $cmd = "mkdir -p $options{prefix}/references/blast_db/nr";
		execute_cmd($cmd);

		print "Downloading BLAST DBs from NCBI.  This make take some time depending on your internet connection\n";

		chdir("$options{prefix}/references/blast_db/nr");
		foreach my $db (@nr){
			print "Downloading nr.${db}...\n";
			$cmd = "wget -q $url/nr.${db}.tar.gz";
			execute_cmd($cmd);

			$cmd = "tar -zxvf nr.${db}.tar.gz";
			execute_cmd($cmd);

			$cmd = "rm nr.${db}.tar.gz";
			execute_cmd($cmd);
		}
	}

	$progress->{blast_db} = 1;
	setProgress($progress);
}

#############################################################################
sub install_source {
	if ((exists $progress->{source}) && ($progress->{source})){
		if ($options{verbos}) {
			print STDERR "\tSource already installed. Skipping...\n";
		}
		return;
	}

	print STDERR "\n\nInstalling source...\n\n";

	#### create dir to store source code
	$cmd = "mkdir -p $options{prefix}/config";
	execute_cmd($cmd);

	$cmd = "cp -r $this->{source}/script $options{prefix}/src";
	execute_cmd($cmd);

	#### make sure all scripts are executable
	$cmd = "chmod -R +x $options{prefix}/src";
	execute_cmd($cmd);

	#### update perl exec path on top of each script.
	$options{perl_exec} =~ s/\//\\\//g;

	$cmd = "find $options{prefix}/src -name \"*.pl\"";
	$cmd .= " -exec sed -i '1 s/^#!.*perl\$/#!$options{perl_exec}/' {} \\;";
	execute_cmd($cmd);

	#### replace library references to local install
	my $lib = "$options{prefix}/lib";
	$lib =~ s/\//\\\//g;

	$cmd = "find $options{prefix}/src -name \"*.pl\"";
	$cmd .= " -exec sed -i 's/\\/data2\\/bsi\\/reference\\/perl_workflow_ref\\/lib/$lib/' {} \\;";
	execute_cmd($cmd);

	#### update Logger.pm lib
	$cmd = "find $options{prefix}/lib -name \"Logger.pm\"";
	$cmd .= " -exec sed -i 's/\\/data2\\/bsi\\/reference\\/tophat\\/lib\\/perl5\\//$lib\\/perl5/' {} \\;";
	execute_cmd($cmd);

	#### copy config files
	$cmd = "cp $options{prefix}/src/config/* $options{prefix}/config";
	execute_cmd($cmd);

	#$cmd = "rm -rf $this->{source}/script";
	#execute_cmd($cmd);

	$progress->{source} = 1;
	setProgress($progress);
}

#############################################################################
sub setup_runInfo {
	if ((exists $progress->{runinfo}) && ($progress->{runinfo})){
		if ($options{verbos}) {
			print STDERR "\tRun info config upto date. Skipping...\n";
		}
		return;
	}

	print STDERR "\n\nInstalling run time config file 1...\n\n";

	my $username = (getpwuid($<))[0];

	#### add username in PI location
	$cmd = "sed -i 's/\@USER\@/$username/' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### update config location value
	my $config = "$options{prefix}/config";
	$config =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@CONFIG_DIR\@/$config/g' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### update input dir location for test run
	my $sample = "$options{prefix}/sample_data";
	$sample =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@INPUT_DIR\@/$sample/' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### update output dir location for test run
	my $output = "$options{prefix}/sample_output";
	$output =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@OUTPUT_DIR\@/$output/' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	#### remove double //
	$cmd = "sed -i 's/\\/\\//\\//g' $options{prefix}/config/run_info.txt";
	execute_cmd($cmd);

	$progress->{runinfo} = 1;
	setProgress($progress);
}

#############################################################################
sub setup_toolInfo {
	if ((exists $progress->{toolinfo}) && ($progress->{toolinfo})){
		if ($options{verbos}) {
			print STDERR "\tTool info config upto date. Skipping...\n";
		}
		return;
	}

	print STDERR "\n\nInstalling run time config file 2...\n\n";

	#### update toolinfo settings for MAPRSeq
	my $maprseq_ref = "$options{prefix}/references";
	my $maprseq_bin = "$options{prefix}/bin";
	my $maprseq_lib = "$options{prefix}/lib";
	my $maprseq_src = "$options{prefix}/src";

	$maprseq_ref =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@REFERENCE\@/$maprseq_ref/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	$maprseq_bin =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@BINARY\@/$maprseq_bin/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	$maprseq_lib =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@LIBRARY\@/$maprseq_lib/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	$maprseq_src =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@SOURCE\@/$maprseq_src/g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update java location
	my $java = `which java`;
	chomp $java;

	#### stop execution and die if JAVA does not exists
	die "JAVA 1.6.0 or higher is required, could not find JAVA install via which java" unless (length($java));

	$java =~ s/\/java$//;
	$java =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@JAVA\@/$java/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update perl location
	my $perl = `which perl`;
	chomp $perl;

	#### stop execution and die if PERL does not exists
	die "Perl 5.10.0 or higher is required, could not find Perl install via which perl" unless (length($perl));

	$perl =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@PERL\@/$perl/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update R
	my $r = `which R`;

	#### stop execution and die if R does not exists
	die "R package is required, could not find R install via which R" unless (length($r));

	chomp $r;
	$r =~ s/\/R$//;
	$r =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@R\@/$r/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update python
	my $python = `which python`;
	chomp $python;

	#### stop execution and die if Python does not exists
	die "Python 2.6,5 or higher with nympy is required, could not find Python install via which Python" unless (length($python));

	$python =~ s/\/python$//;
	$python =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@PYTHON\@/$python/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### update source path
	my $src = "$options{prefix}/src";
	$src =~ s/\//\\\//g;
	$cmd = "sed -i 's/\@SOURCE\@/$src/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### check for PYTHONPATH env var
	my $pythonpath = `echo \$PYTHONPATH`;
	chomp $pythonpath;

	if (! length($pythonpath)){
		#### get default python library path.

		$python = `python --version 2>&1`;
		chomp $python;

		$python =~ s/Python (\d+\.\d+).*/$1/;

		$pythonpath = "/usr/lib/python$python";

		if (-d "$pythonpath/site-packages") {
			$pythonpath .= "/site-packages";
		} elsif (-d "$pythonpath/dist-packages") {
			$pythonpath .= "/dist-packages";
		} else {
			die "Can not file standard python lib dir $pythonpath/site-packages or $pythonpath/dist-packages\nUnable to set default PYTHONPATH. Check your python install and set PYTHONPATH varaible. Install failed";
		}
	}

	$pythonpath =~ s/\//\\\//g;

	$cmd = "sed -i 's/\@PYTHONPATH\@/$pythonpath/' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	#### remove double //
	$cmd = "sed -i 's/\\/\\//\\//g' $options{prefix}/config/tool_info.txt";
	execute_cmd($cmd);

	$progress->{toolinfo} = 1;
	setProgress($progress);
}

#############################################################################
sub setup_sampleInfo {
}

#############################################################################
sub setup_memoryInfo {
}

#############################################################################
sub setup_testInput {
	if ((exists $progress->{testinput}) && ($progress->{testinput})){
		if ($options{verbos}) {
			print STDERR "\tTest input already installed. Skipping...\n";
		}
		return;
	}

	#### check and install dir
	my $dir = "$options{prefix}/sample_data";
	my $cmd = "";

	if (-d $dir){
		$cmd = "rm -rf $dir";
	}

	$cmd = "mkdir -p $dir";
	execute_cmd($cmd);

	chdir("$this->{source}");
	$cmd = "cp -r $this->{source}/input/* $dir/.";
	execute_cmd($cmd);

	$progress->{testinput} = 1;
	setProgress($progress);
}


#############################################################################
sub execute_cmd {
	my $cmd = shift;

	system($cmd);

	while (( $? >> 8 ) != 0 ){
		print STDERR "ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd\n";
		exit(-1);
	}
}

#############################################################################
sub timestamp {
	my @months = qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
    my @weekDays = qw(Sun Mon Tue Wed Thu Fri Sat Sun);
    my ($second, $minute, $hour, $dayOfMonth, $month, $yearOffset, $dayOfWeek, $dayOfYear, $daylightSavings) = localtime();
    my $year = 1900 + $yearOffset;
    my $theTime = "$hour:$minute:$second, $weekDays[$dayOfWeek] $months[$month] $dayOfMonth, $year";
    print "Time now: " . $theTime."\n";
}
