#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   scripture.pl

=head1 SYNOPSIS

    USAGE: scripture.pl -r=run_info.txt -o=output_dir -s=sample_name [-c=chridx]

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--sample, -s>
	Required. Sample name for which cufflinks should be executed.

B<--help,-h>


=head1 DESCRIPTION
	Run scripture module for a given sample

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./scripture.pl -r=run_info.pl -o=output_dir -s=sample_name

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Cwd;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use POSIX;
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'input_dir|i=s',
						  'sample|s=s',
						  'chr|c=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### set global vars
my $MAX_JOB_LIMIT = 3000;
my $LONG_WAIT = 300;
my $SHORT_WAIT = 30;
my $WAIT = 5;

#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

if (defined $ENV{'SGE_TASK_ID'}){
	$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/Scripture.$options{sample}.$ENV{'SGE_TASK_ID'}.log",
								  'LOG_LEVEL'=>$options{'debug'});
} else {
	$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/Scripture.$options{sample}.$options{chr}.log",
							  'LOG_LEVEL'=>$options{'debug'});
}

$logger = Workflow::Logger::get_logger();

$logger->info("Scripture for $options{sample} started");

#### set local variables.
my @chr = split(/:/, $config->{RunInfo}->{chrindex});
my $chr_idx = "";
my $cmd = "";
my $this;

if (defined $ENV{'SGE_TASK_ID'}) {
	$chr_idx = $ENV{'SGE_TASK_ID'} - 1;
} else {
	$chr_idx = $options{chr} - 1;
}

$this->{output_dir} = "$options{output_dir}/lincRNA/scripture/$options{sample}";
$this->{input} = "$options{input_dir}/$options{sample}_sorted.chr$chr[$chr_idx].bam";

create_dir_struct(\%options);

check_input($this->{input});

$this->{bam} = $this->{input};

#### execute scripture
$cmd = "$config->{ToolInfo}->{java}->{value}/java -Xmx$config->{MemoryInfo}->{scripture_jvm} -Xms512m";
$cmd .= " -jar $config->{ToolInfo}->{scripture}->{value}";
$cmd .= " -alignment $this->{bam} -maskFileDir $config->{ToolInfo}->{mask_file}->{value}";
$cmd .= " -out $this->{output_dir}/$options{sample}.chr$chr[$chr_idx].bed -chr chr$chr[$chr_idx]";
$cmd .= " -sizeFile $config->{ToolInfo}->{chromsize}->{value}";
$cmd .= " -chrSequence $config->{ToolInfo}->{ref_genome_dir}->{value}/chr$chr[$chr_idx].fa";
execute_cmd($cmd);

if (-s "$this->{output_dir}/$options{sample}.chr$chr[$chr_idx].bed") {
	#### convert bed output to gtf.
	$cmd = "$config->{ToolInfo}->{java}->{value}/java -Xmx$config->{MemoryInfo}->{scripture_jvm} -Xms512m";
	$cmd .= " -jar $config->{ToolInfo}->{scripture}->{value} -task toGFF";
	$cmd .= " -cufflinks -in $this->{output_dir}/$options{sample}.chr$chr[$chr_idx].bed";
	$cmd .= " -source SCRIPTURE -out $this->{output_dir}/$options{sample}.chr$chr[$chr_idx].gtf";
	$cmd .= " -prefix chr$chr[$chr_idx]";
	execute_cmd($cmd);
} else {
	$logger->logdie("Scritpure failed to create BED file for $options{sample}.");
}


$logger->info("Scripture complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir input_dir sample);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
	$options{'chr'} = 1 unless (defined $options{'chr'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/lincRNA/scripture/$options{sample}";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/Scripture.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING SCRIPTURE STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Scripture",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
