#!/usr/local/biotools/perl/5.14.2/bin/perl

=head1 NAME
   annotation.pl

=head1 SYNOPSIS

    USAGE: annotation.pl -r=run_info.txt -o=output_dir -i=input_dir

=head1 OPTIONS


B<--run_info,-r>
	Required. Complete path to run info file

B<--output_dir, -o>
	Required.  Root output dir.  eg: /data2/bsi/secondary/PI/mrnaseq/RUN_ID

B<--input_dir, -i>
	Required. Input directory

B<--help,-h>


=head1 DESCRIPTION
	Run annotation module

=head1 INPUT
	Run info file and sample name

=head1 OUTPUT

=head1 VERSION
	1.0

=head1  CONTACT
  bjaysheel@gmail.com


==head1 EXAMPLE
	./annotation.pl -r=run_info.txt -o=output_dir -i=input_dir

=cut

use lib "/data2/bsi/reference/perl_workflow_ref/lib";
use lib "/data2/bsi/reference/perl_workflow_ref/lib/perl5/x86_64-linux/auto";
use strict;
use warnings;
use Data::Dumper;
use Pod::Usage;
use Getopt::Long qw(:config no_ignore_case no_auto_abbrev pass_through);
use ParseConfig;
use MyUtility;
use Workflow::Logger;

my %options = ();
my $results = GetOptions (\%options,
                          'run_info|r=s',
						  'output_dir|o=s',
						  'input_dir|i=s',
						  'log|l=s',
			              'debug=s',
						  'help|h') || pod2usage();

#### display documentation
if( $options{'help'} ){
    pod2usage( {-exitval => 0, -verbose => 2, -output => \*STDERR} );
}

#############################################################################
#### create hash of all config info.
my $config = new ParseConfig($options{run_info});
my $util = new MyUtility;

#### make sure everything passed was peachy
&check_parameters(\%options);

#### setup log object
my $logger;

$logger = new Workflow::Logger('LOG_FILE'=>"$config->{RunInfo}->{logs}/Annotation.log",
							  'LOG_LEVEL'=>$options{'debug'});

$logger = Workflow::Logger::get_logger();

$logger->info("Annotation started");

#### set local variables.
my $cmd = "";
my $this;

$this->{output_dir} = "$options{output_dir}/lincRNA/annotation";
$this->{input} = "$options{input_dir}/all.combined.gtf";

check_input($this->{input});

create_dir_struct();


#### Covert GTF to BED
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/gtf2bed.pl";
$cmd .= " -g $this->{input}";
$cmd .= " -o $this->{output_dir}/all.combined.transcripts.bed";
execute_cmd($cmd);


#### Find transcripts not present in MIT and Gencode
$cmd = "$config->{ToolInfo}->{bedtools}->{value}/intersectBed -a $this->{output_dir}/all.combined.transcripts.bed";
$cmd .= " -b $config->{ToolInfo}->{mit_transcripts}->{value} -v > $this->{output_dir}/all.NotInMIT.bed";
execute_cmd($cmd);

$cmd = "$config->{ToolInfo}->{bedtools}->{value}/intersectBed -a $this->{output_dir}/all.combined.transcripts.bed";
$cmd .= " -b $config->{ToolInfo}->{gencode_bed}->{value} -v > $this->{output_dir}/all.NotInGencode.bed";
execute_cmd($cmd);


#### Get the transcript IDs
$cmd = "cat $this->{output_dir}/all.combined.transcripts.bed | cut -f1,2,3,4,6 > $this->{output_dir}/all.transcripts";
$cmd .= " && cat $this->{output_dir}/all.NotInMIT.bed | cut -f4 > $this->{output_dir}/NotInMIT.transcripts";
$cmd .= " && cat $this->{output_dir}/all.NotInGencode.bed | cut -f4 > $this->{output_dir}/NotInGencode.transcripts";
execute_cmd($cmd);

#### Fill in with Annotation columns
$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/FillAnnotation.pl";
$cmd .= " $this->{output_dir}/all.transcripts $this->{output_dir}/NotInMIT.transcripts";
$cmd .= " $this->{output_dir}/MIT.Annotation.txt";
execute_cmd($cmd);

$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/FillAnnotation.pl";
$cmd .= " $this->{output_dir}/all.transcripts $this->{output_dir}/NotInGencode.transcripts";
$cmd .= " $this->{output_dir}/Gencode.Annotation.txt";
execute_cmd($cmd);

$cmd = "paste $this->{output_dir}/MIT.Annotation.txt $this->{output_dir}/Gencode.Annotation.txt";
$cmd .= " | cut -f1,2,3,4,5,6,12 > $this->{output_dir}/KnownAnnot.tmp";
execute_cmd($cmd);

#### Fill in with Locus info for respective Transcript IDs
$cmd = "cat $this->{input} | cut -f2,4 -d \" \" | sed -e 's/\"//g' -e 's/;//g' > $this->{output_dir}/Loci.Transcripts";
execute_cmd($cmd);

$cmd = "$config->{ToolInfo}->{workflow_path}->{value}/FillLocusInfo.pl";
$cmd .= " $this->{output_dir}/Loci.Transcripts $this->{output_dir}/KnownAnnot.tmp $this->{output_dir}/LocusAdded";
execute_cmd($cmd);

#### Final Output for Known Gene Annotation
`echo -e "Chr\tStart\tStop\tTranscriptID\tLocusID\tStrand\tMIT_Status\tGencode_Status" >> $this->{output_dir}/KnownGeneAnnotation.txt`;
$cmd = "cat $this->{output_dir}/LocusAdded | awk 'BEGIN{OFS=\"\\t\"}{print \$1,\$2,\$3,\$4,\$8,\$5,\$6,\$7}' >> $this->{output_dir}/KnownGeneAnnotation.txt";
execute_cmd($cmd);

#### Delete temp files
my @files = qw(all.NotInMIT.bed all.NotInGencode.bed all.transcripts NotInMIT.transcripts NotInGencode.transcripts MIT.Annotation.txt Gencode.Annotation.txt KnownAnnot.tmp Loci.Transcripts LocusAdded);
foreach my $f (@files){
	$cmd = "rm $this->{output_dir}/$f";
	execute_cmd($cmd);
}

$logger->info("Annotation complete");
exit();

#############################################################################
sub check_parameters {
    my $options = shift;

	my @required = qw(run_info output_dir input_dir);

	foreach my $key (@required) {
		unless (defined $options{$key}) {
			print STDERR "ARG: $key is required\n";
			pod2usage({-exitval => 2,  -message => "error message", -verbose => 1, -output => \*STDERR});
			exit(-1);
		}
	}

	$options{'debug'} = 3 unless (defined $options{'debug'});
}

#############################################################################
sub create_dir_struct {
	my $options = shift;

	my $dir = "$options{output_dir}/lincRNA/annotation";
	if ( -d $dir ) {
		$logger->info("Directory $dir exist");
	} else {
		execute_cmd("mkdir -p $dir");
	}
}

#############################################################################
sub execute_cmd {
	my $cmd = shift;

	$logger->info("$cmd");
	system($cmd);

	while (( $? >> 8 ) != 0 ){
		$logger->logdie("ERROR: Following command failed to execute. Exiting execution of workflow\n$cmd");
		exit(100);
	}
}

#############################################################################
sub check_input {
	my $file = shift;

	my $job_id = 0;
	my $sgeerr = "";
	my $sgeout = "";

	if (defined $ENV{JOB_ID}){
		$job_id = $ENV{JOB_ID};
	}

	if (defined $ENV{SGE_STDERR_PATH}){
		$sgeerr = $ENV{SGE_STDERR_PATH};
	}

	if (defined $ENV{SGE_STDOUT_PATH}){
		$sgeout = $ENV{SGE_STDOUT_PATH};
	}

	if (! -s $file){
		my $error_name = "$config->{RunInfo}->{base_output_dir}/$config->{RunInfo}->{pi}/$config->{RunInfo}->{type}/$config->{RunInfo}->{output_folder}/error/Annotation.$options{sample}.err";
		$util->createErrorFile($error_name, "EXPECTED FILE WHILE RUNNING Annotation STEP IS MISSING\n\n$file");

		$util->reportErrorSGE($config->{RunInfo}->{email},
							  $file,
							  "Annotation",
							  $error_name,
							  $job_id,
							  $sgeerr,
							  $sgeout);
		exit(100);
	}
}
