DROP DATABASE IF EXISTS NGSPortal;

CREATE DATABASE NGSPortal;

# Grant Permissions to super-users
GRANT ALL ON NGSPortal.* TO 'designer'@'localhost';
GRANT GRANT OPTION ON NGSPortal.* TO 'designer'@'localhost';
#GRANT ALL ON NGSPortal.* TO 'Asif'@'localhost';
#GRANT GRANT OPTION ON NGSPortal.* TO 'Asif'@'localhost';
# GRANT SELECT ON mysql.proc TO 'designer'@'localhost';

# Grant basic user permissions.	Allow WebUser to be on any system
#GRANT SELECT ON NGSPortal.* TO 'WebUser';
GRANT SELECT ON NGSPortal.* TO 'WebUser'@'localhost';

# Save the Privileges
FLUSH PRIVILEGES;

# Now build the database
USE NGSPortal;

# Stored Procedures dump their errors here, for debugging
DROP TABLE IF EXISTS Errors;
CREATE TABLE Errors (
	message VARCHAR(255)
) ENGINE = MyISAM;


#	To keep track of Application Types
DROP TABLE IF EXISTS ApplicationTypes;
CREATE TABLE ApplicationTypes (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(7) NOT NULL UNIQUE,						# Also indexed
	description VARCHAR(31) NOT NULL DEFAULT '', 
	PRIMARY KEY (id)
) ENGINE = MyISAM;

INSERT INTO ApplicationTypes (name, description) VALUES ('GENOM', 'Whole genome resequencing');
INSERT INTO ApplicationTypes (name, description) VALUES ('MRNAS', 'mRNA Sequencing');
INSERT INTO ApplicationTypes (name, description) VALUES ('SRNAD', 'Small RNA Discovery');
INSERT INTO ApplicationTypes (name, description) VALUES ('METHL', 'Methylation');
INSERT INTO ApplicationTypes (name, description) VALUES ('CHIPS', 'ChIP-SEQ');
INSERT INTO ApplicationTypes (name, description) VALUES ('DGESQ', 'Digital gene expression');
INSERT INTO ApplicationTypes (name, description) VALUES ('EXOME', 'Exome');
INSERT INTO ApplicationTypes (name, description) VALUES ('AMPLN', 'Amplicon');
# Main DB has a blank at 10, so we must, too
INSERT INTO ApplicationTypes (name, description) VALUES ('Foo', 'Bar');
DELETE FROM ApplicationTypes WHERE name = 'Foo';
INSERT INTO ApplicationTypes (name, description) VALUES ('CSCAP', 'Custom SureSelect Capture');
INSERT INTO ApplicationTypes (name, description) VALUES ('MEDIP-S', 'Methylated DNA IP Sequencing');
INSERT INTO ApplicationTypes (name, description) VALUES ('MATEPAI', 'Mate Pair');
INSERT INTO ApplicationTypes (name, description) VALUES ('RIP-SEQ', 'RNA IP Sequencing');
INSERT INTO ApplicationTypes (name, description) VALUES ('LNCRNA', 'Long non coding RNA');
INSERT INTO ApplicationTypes (name, description) VALUES ('TotalRN', 'Total RNA sequencing');
INSERT INTO ApplicationTypes (name, description) VALUES ('HALO', 'Haloplex Custom');
INSERT INTO ApplicationTypes (name, description) VALUES ('ChIRP', 'ChIRP Sequencing');
INSERT INTO ApplicationTypes (name, description) VALUES ('CLIP-Se', 'CLIP-Seq');


#	To keep track of types of reads
DROP TABLE IF EXISTS ReadTypes;
CREATE TABLE ReadTypes (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(31) NOT NULL UNIQUE,						# Also indexed
	description VARCHAR(31) NOT NULL DEFAULT '', 
	PRIMARY KEY (id)
) ENGINE = MyISAM;

INSERT INTO ReadTypes (name, description) VALUES ("SINGLE_READ", "Single read");
INSERT INTO ReadTypes (name, description) VALUES ("PAIRED_END", "Paired End");
INSERT INTO ReadTypes (name, description) VALUES ("MATED_PAIRED_END", "Mated Paired End");


#	To keep track of types of QC Results
DROP TABLE IF EXISTS QCResults;
CREATE TABLE QCResults (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	sequenceCount INT UNSIGNED NOT NULL, 
	numBases INT UNSIGNED NOT NULL, 
	numBadBases INT UNSIGNED NOT NULL, 
	gcContent DOUBLE NULL,  								# NULL means NaN
	duplicationLevel DOUBLE NULL,  							# NULL means NaN
	percentSeqLow DOUBLE NULL,  							# NULL means NaN
	percentSeqHigh DOUBLE NULL,  							# NULL means NaN
	qcValue VARCHAR(31) NOT NULL, 
	dataURL VARCHAR(10000) NOT NULL, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;



#	To keep track of types of roles
DROP TABLE IF EXISTS Role;
CREATE TABLE Role (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(31) NOT NULL UNIQUE,						# Also indexed
	PRIMARY KEY (id)
) ENGINE = MyISAM;

INSERT INTO Role (name) VALUES ('PI');
INSERT INTO Role (name) VALUES ('Co-PI');
INSERT INTO Role (name) VALUES ('Researcher');
INSERT INTO Role (name) VALUES ('Informatics PhD Lead');
INSERT INTO Role (name) VALUES ('Informatics Lead');
INSERT INTO Role (name) VALUES ('Informatics Specialist');
INSERT INTO Role (name) VALUES ('Statistician');
INSERT INTO Role (name) VALUES ('IT');
INSERT INTO Role (name) VALUES ('SPA');
INSERT INTO Role (name) VALUES ('Lab Manager');
INSERT INTO Role (name) VALUES ('Lab Tech');
INSERT INTO Role (name) VALUES ('Admin');
INSERT INTO Role (name) VALUES ('Project Informatics Lead');
INSERT INTO Role (name) VALUES ('Project Informatics Specialist');
INSERT INTO Role (name) VALUES ('Project PI');
INSERT INTO Role (name) VALUES ('Project Member');


#	To keep track of our Contacts
DROP TABLE IF EXISTS Contacts;
CREATE TABLE Contacts (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	lastName VARCHAR(31) NOT NULL DEFAULT '', 
	firstName VARCHAR(31) NOT NULL DEFAULT '', 
	lanID VARCHAR(31) NOT NULL DEFAULT '', 
	internal BOOLEAN NOT NULL, 
	email VARCHAR(255) NOT NULL DEFAULT '', 
	useCount INT UNSIGNED NOT NULL DEFAULT 0, 
	lastUsed TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP, 
	PRIMARY KEY (id), 
	INDEX name (lastName (8), firstName (8), id)
) ENGINE = MyISAM;


#	Connect Contacts to Projects, by way of the ContactLinker, so know the users Role with that project
DROP TABLE IF EXISTS ContactLinker;
CREATE TABLE ContactLinker (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	contactID INT UNSIGNED NOT NULL REFERENCES Contacts (id),	# The Person being described. Many:1
	userRole INT UNSIGNED NOT NULL REFERENCES Role (id),	# The Role this contact has. Many:1
	PRIMARY KEY (id), 
	INDEX roles (userRole, contactID), 
	INDEX contacts (contactID, userRole), 
	INDEX contactInfo (contactID, id)
) ENGINE = MyISAM;


#	To keep track of our Projects
DROP TABLE IF EXISTS ProjectInformation;
CREATE TABLE ProjectInformation (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(63) NOT NULL UNIQUE, 
	description VARCHAR(255) NOT NULL DEFAULT '', 
	progress INT NOT NULL DEFAULT 0, 
	IRC INT NULL, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;


#	To keep track of our Secondary Projects
DROP TABLE IF EXISTS SecondaryProjectInformation;
CREATE TABLE SecondaryProjectInformation (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	notes VARCHAR(255) NULL DEFAULT NULL, 
	assignedDate DATE NULL, 
	deliveryDate DATE NULL, 
	projectType VARCHAR(65) NULL DEFAULT NULL, 
	name VARCHAR(255) NULL DEFAULT NULL UNIQUE, 
	isBatched BOOLEAN NOT NULL DEFAULT FALSE, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;


#	Connect Contacts to Secondary Projects
DROP TABLE IF EXISTS ContactSecondaryProjectLinker;
CREATE TABLE ContactSecondaryProjectLinker (
	contactRoleID INT UNSIGNED NOT NULL REFERENCES ContactLinker (id),	# The Person and Role. Many:1
	secProjectID INT UNSIGNED NOT NULL REFERENCES SecondaryProjectInformation (id),	# The Project. Many:1
	INDEX contacts (secProjectID, contactRoleID), 
	INDEX projects (contactRoleID, secProjectID)
) ENGINE = MyISAM;


#	Connect Contacts to Projects
DROP TABLE IF EXISTS ContactProjectLinker;
CREATE TABLE ContactProjectLinker (
	contactRoleID INT UNSIGNED NOT NULL REFERENCES ContactLinker (id),	# The Person and Role. Many:1
	projectID INT UNSIGNED NOT NULL REFERENCES ProjectInformation (id),	# The Project. Many:1
	INDEX contacts (projectID, contactRoleID), 
	INDEX projects (contactRoleID, projectID)
) ENGINE = MyISAM;


#	To keep track of our Groups
DROP TABLE IF EXISTS Groups;
CREATE TABLE Groups (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	creator INT UNSIGNED NOT NULL REFERENCES Contacts (id),	# The Person who created the group. Many:1
	name VARCHAR(31) NOT NULL UNIQUE, 
	description VARCHAR(255) NOT NULL, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;


#	Connect Contacts to Groups
DROP TABLE IF EXISTS ContactGroupLinker;
CREATE TABLE ContactGroupLinker (
	contactRoleID INT UNSIGNED NOT NULL REFERENCES ContactLinker (id),	# The Person and Role. Many:1
	groupID INT UNSIGNED NOT NULL REFERENCES Groups (id),				# The Group. Many:1
	INDEX contacts (groupID, contactRoleID) 
) ENGINE = MyISAM;


#	Many to 1 table (this table provides the 1)
DROP TABLE IF EXISTS PrimaryAnalysisStatus;
CREATE TABLE PrimaryAnalysisStatus (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	status VARCHAR(31) NOT NULL, 
	task VARCHAR(31) NOT NULL, 
	PRIMARY KEY (id), 
	UNIQUE INDEX stats (task, status)
) ENGINE = MyISAM;


# Table for tracking families.  The starting parents will each have their own family.  Their children will 
# have two families, etc.
DROP TABLE IF EXISTS Family;
CREATE TABLE Family (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(255) NOT NULL UNIQUE, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;


# Table for creating Many : 1 (FamilyMember : Sample) and Many : Many (Family : Sample) connections.
# The geneologically oldest Sample will belong to generation 1, have 1 Family, and will have a NULL 
# parentID.  Children will be generation + 1 of that family.  
# Example: Let us have samples from the paternal Grandfather, the Father, the Mother, and the Child.
# PG is family 1, generation 1.  Father is family 1, generation 2.  Mother is family 2, generation 1.
# Child is family 1, generation 3, AND family 2, generation 2
DROP TABLE IF EXISTS FamilyMember;
CREATE TABLE FamilyMember (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	sampleID INT UNSIGNED NOT NULL REFERENCES Sample (id),	# The sample this applies to Many : 1
	familyID INT UNSIGNED NOT NULL REFERENCES Family (id),	# One of the families this sample belongs to
	parentID INT UNSIGNED NULL DEFAULT NULL REFERENCES FamilyMember (id),	# The parent from familyID family
	generation INT NOT NULL DEFAULT 0, 						# = parent.generation + 1
	isMale BOOLEAN NULL DEFAULT NULL, 						# Sex, if known, NULL if not known
	name VARCHAR(255) NULL DEFAULT NULL, 
	PRIMARY KEY (id), 
	INDEX Families (familyID, sampleID), 
	INDEX Samples (sampleID, parentID), 
	INDEX Offspring (parentID, sampleID), 
	INDEX Identifier (id, sampleID), 
	INDEX Generations (familyID, generation, sampleID)
) ENGINE = MyISAM;

DROP TABLE IF EXISTS FileLocation;
CREATE TABLE FileLocation (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(31) NOT NULL UNIQUE, 
	description VARCHAR(255) NOT NULL DEFAULT '', 
	location VARCHAR(255) NOT NULL, 
	access VARCHAR(255) NOT NULL, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;

INSERT INTO FileLocation (name, description, location, access) 
	VALUES ('NL', 'Delivery Storage', '/data3/delivery/<pi name>/<run name>/primary/<sample name>', 
			'ftp://rcfisinl1-212/delivery/');
INSERT INTO FileLocation (name, description, location, access) 
	VALUES ('RA', 'Research Archive', '/NGS/delivery/<pi name>/<run name>/primary/<sample name>', 
			'http://research-archive/?B=%2FNGS%2Fdelivery%2F');
INSERT INTO FileLocation (name, description, location, access) 
	VALUES ('X', 'Analysis storage', '/data2/delivery/<pi name>/<run name>/primary/<sample name>', 
			'file://///rcfcluster-cifs/data2/delivery/');


DROP TABLE IF EXISTS FlagstatResults;
CREATE TABLE FlagstatResults (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	readCount BIGINT UNSIGNED NOT NULL, 
	mappedCount BIGINT UNSIGNED NOT NULL, 
	qcFailure INT UNSIGNED NOT NULL, 
	duplicates INT UNSIGNED NOT NULL, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;


#	Master table.  All the runs we are tracking
DROP TABLE IF EXISTS RunInformation;
CREATE TABLE RunInformation (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	runID VARCHAR(63) NOT NULL UNIQUE,					 	# We will enforce UNIQUENESS
	flowCell VARCHAR(15) NOT NULL, 							# Unique ID
	instrumentCompleteDate DATE NULL, 
	primaryCompleteDate DATE NULL, 
	primaryQCCompleteDate DATE NULL, 
	primaryDeliveryDate DATE NULL, 
	rlimsBatchID VARCHAR(31) NOT NULL, 						# RLIMS Unique ID
	instrument VARCHAR(15) NOT NULL, 						# instrument name
	primaryAnalysisStatus INT UNSIGNED NULL REFERENCES PrimaryAnalysisStatus (id), 
	startDate DATE NOT NULL, 
	runValidated BOOLEAN NOT NULL DEFAULT FALSE, 
	PRIMARY KEY (id), 
	INDEX LifeTime (startDate, instrumentCompleteDate, id)
) ENGINE = MyISAM;


# Master table.	 All the samples we are tracking
DROP TABLE IF EXISTS Sample;
CREATE TABLE Sample (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(63) NOT NULL DEFAULT '', 
	sampleID VARCHAR(63) NOT NULL UNIQUE, 
	species VARCHAR(31) NOT NULL DEFAULT '', 
	diseaseType VARCHAR(31) NULL DEFAULT NULL, 
	tissueType VARCHAR(31) NULL DEFAULT NULL, 
	PRIMARY KEY (id), 
	INDEX Names (name, id), 
	INDEX Disease (diseaseType, id), 
	INDEX Tissue (tissueType, id)
) ENGINE = MyISAM;


#	To keep track of RLIMSProjects
DROP TABLE IF EXISTS RLIMSProject;
CREATE TABLE RLIMSProject (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(31) NOT NULL UNIQUE,						# Also indexed
	PRIMARY KEY (id)
) ENGINE = MyISAM;


#	To keep track of CaptureKits
DROP TABLE IF EXISTS CaptureKit;
CREATE TABLE CaptureKit (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(255) NOT NULL UNIQUE,						# Also indexed
	PRIMARY KEY (id)
) ENGINE = MyISAM;


# Master table.	 All the samples we are tracking
DROP TABLE IF EXISTS SequencedSample;
CREATE TABLE SequencedSample (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,		# PRIMARY KEY field
	sampleID INT UNSIGNED NOT NULL REFERENCES Sample (id),		# The Sample that was sequenced. Many:1
	runID INT UNSIGNED NOT NULL REFERENCES RunInformation (id),	# The Run doing the sequencing. Many:1
	appType INT UNSIGNED NOT NULL REFERENCES ApplicationTypes (id),	# What type of run was it
	readType INT UNSIGNED NOT NULL REFERENCES ReadTypes (id),	# What type of read was it
	qcResults INT UNSIGNED NULL DEFAULT NULL REFERENCES QCResults (id),	# What happened w/ QC
	pairQCResults INT UNSIGNED NULL DEFAULT NULL REFERENCES QCResults (id),	# What happened w/ QC
	lane INT UNSIGNED NOT NULL, 
	cycles INT UNSIGNED NOT NULL, 
	isControl BOOLEAN NOT NULL, 								# Is it a control run, or a normal one
	indexVendor BOOLEAN NOT NULL DEFAULT FALSE, 				# Is it a vendor supplied index
	indexSequence VARCHAR(31) NOT NULL DEFAULT '', 
	flowCell VARCHAR(15) NOT NULL, 								# Unique ID
	rlimsID VARCHAR(15) NOT NULL, 								# Unique ID
	requestName VARCHAR(63) NOT NULL, 							# User supplied
	resultLocation INT UNSIGNED NOT NULL DEFAULT 0,				# Bitfield of flags
	fileSize BIGINT UNSIGNED NULL DEFAULT NULL, 
	keepFileRequestor VARCHAR(255) NULL DEFAULT NULL, 
	keepFileDate DATE NULL DEFAULT NULL, 
	flagstatResults INT UNSIGNED NULL DEFAULT NULL REFERENCES FlagstatResults (id), 
	pairFlagstatResults INT UNSIGNED NULL DEFAULT NULL REFERENCES FlagstatResults (id), 
	bamFileName VARCHAR(255) NOT NULL NULL DEFAULT NULL, 
	isCIM BOOLEAN NOT NULL DEFAULT FALSE, 
	isBIC BOOLEAN NULL DEFAULT NULL, 
	needUpdate BOOLEAN NOT NULL DEFAULT FALSE, 
	captureKit INT UNSIGNED NOT NULL DEFAULT 0 REFERENCES CaptureKit (id), 
	rlimsProjectID INT UNSIGNED NULL DEFAULT NULL REFERENCES RLIMSProject (id), 
	PRIMARY KEY (id), 
	UNIQUE KEY (flowCell, indexSequence, lane), 
	INDEX Joiner (runID, sampleID, id), 
	INDEX Requesters (keepFileRequestor), 
	INDEX KeepDates (keepFileDate)
) ENGINE = MyISAM;


#	Connect Secondary Projects to Samples, and to the rest of the data
DROP TABLE IF EXISTS SecondaryProjectSampleLinker;
CREATE TABLE SecondaryProjectSampleLinker (
	sequencingID INT UNSIGNED NOT NULL REFERENCES SequencedSample (id),	# The Sequenced Sample. Many:1
	secProjectID INT UNSIGNED NOT NULL REFERENCES SecondaryProjectInformation (id),	# The Project. Many:1
	UNIQUE INDEX sequences (sequencingID, secProjectID), 
	UNIQUE INDEX projects (secProjectID, sequencingID)
) ENGINE = MyISAM;


#	Connect Projects to Samples, and to the rest of the data
DROP TABLE IF EXISTS ProjectSampleLinker;
CREATE TABLE ProjectSampleLinker (
	sequencingID INT UNSIGNED NOT NULL REFERENCES SequencedSample (id),	# The Sequenced Sample. Many:1
	projectID INT UNSIGNED NOT NULL REFERENCES ProjectInformation (id),	# The Project. Many:1
	UNIQUE INDEX sequences (sequencingID, projectID), 
	UNIQUE INDEX projects (projectID, sequencingID)
) ENGINE = MyISAM;


#	Connect Contacts to Samples, and to the rest of the data
DROP TABLE IF EXISTS ContactSequenceLinker;
CREATE TABLE ContactSequenceLinker (
	sequencingID INT UNSIGNED NOT NULL REFERENCES SequencedSample (id),	# The Sequenced Sample. Many:1
	contactRoleID INT UNSIGNED NOT NULL REFERENCES ContactLinker (id),	# The Contact. Many:1
	UNIQUE INDEX sequences (sequencingID, contactRoleID), 
	UNIQUE INDEX projects (contactRoleID, sequencingID)
) ENGINE = MyISAM;


DROP TABLE IF EXISTS AnalysisType;
CREATE TABLE AnalysisType (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(255) NOT NULL, 
	versionNumber DOUBLE NOT NULL, 
	PRIMARY KEY (id), 
	UNIQUE INDEX info (name, versionNumber)
) ENGINE = MyISAM;

INSERT INTO AnalysisType (name, versionNumber) VALUES ('Exome', 3.2);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('mRNA-SEQ', 3.6);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('WholeGenome', 1.0);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('Methylation', 1.0);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('MAPRSeq', 1.2);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('ChipSeq', 2.0);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('miRNA', 1.0);
# Main DB has a blank at 8, so we must, too
INSERT INTO AnalysisType (name, versionNumber) VALUES ('Foo1', 1.0);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('Microbiome', 1.0);
# Main DB has a blank at 10,11, so we must, too
INSERT INTO AnalysisType (name, versionNumber) VALUES ('Foo2', 1.0);
INSERT INTO AnalysisType (name, versionNumber) VALUES ('Foo3', 1.0);
DELETE FROM AnalysisType WHERE name LIKE 'Foo%';
INSERT INTO AnalysisType (name, versionNumber) VALUES ('MAPRSeq', 1.1);


DROP TABLE IF EXISTS Status;
CREATE TABLE Status (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(255) NOT NULL UNIQUE, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;

INSERT INTO Status (name) VALUES ('Beginning');
INSERT INTO Status (name) VALUES ('Start');
INSERT INTO Status (name) VALUES ('AdapterTrimming');
INSERT INTO Status (name) VALUES ('Sampling');
INSERT INTO Status (name) VALUES ('QC');
INSERT INTO Status (name) VALUES ('unused1');
INSERT INTO Status (name) VALUES ('Trimming');
INSERT INTO Status (name) VALUES ('unused2');
INSERT INTO Status (name) VALUES ('ChimeraCheck');
INSERT INTO Status (name) VALUES ('Alignment');
INSERT INTO Status (name) VALUES ('unused3');
INSERT INTO Status (name) VALUES ('unused4');
INSERT INTO Status (name) VALUES ('unused5');
INSERT INTO Status (name) VALUES ('Realignment');
INSERT INTO Status (name) VALUES ('OTUCall');
INSERT INTO Status (name) VALUES ('unused6');
INSERT INTO Status (name) VALUES ('unused7');
INSERT INTO Status (name) VALUES ('ExpressionCount');
INSERT INTO Status (name) VALUES ('unused8');
INSERT INTO Status (name) VALUES ('unused9');
INSERT INTO Status (name) VALUES ('PeakCalling');
INSERT INTO Status (name) VALUES ('VariantCalling');
INSERT INTO Status (name) VALUES ('unused10');
INSERT INTO Status (name) VALUES ('CpgCalling');
INSERT INTO Status (name) VALUES ('unused12');
INSERT INTO Status (name) VALUES ('Annotation');
INSERT INTO Status (name) VALUES ('unused13');
INSERT INTO Status (name) VALUES ('unused14');
INSERT INTO Status (name) VALUES ('unused15');
INSERT INTO Status (name) VALUES ('Statistics');
INSERT INTO Status (name) VALUES ('unused16');
INSERT INTO Status (name) VALUES ('unused17');
INSERT INTO Status (name) VALUES ('unused18');
INSERT INTO Status (name) VALUES ('Results');
INSERT INTO Status (name) VALUES ('unused19');
INSERT INTO Status (name) VALUES ('Complete');
INSERT INTO Status (name) VALUES ('Delivered');
INSERT INTO Status (name) VALUES ('unused20');
INSERT INTO Status (name) VALUES ('unused21');
INSERT INTO Status (name) VALUES ('unused22');
INSERT INTO Status (name) VALUES ('unusedHold1');
INSERT INTO Status (name) VALUES ('unusedHold2');
INSERT INTO Status (name) VALUES ('unusedHold3');
INSERT INTO Status (name) VALUES ('unusedHold4');
INSERT INTO Status (name) VALUES ('unusedHold5');


# deleteDate: Date when secondary analysis files will be deleted.  
# User can change this if need files to stay around longer.
# Should we have an uneditable 'original delete date' field, so we can do queries to find people 
# who are pushing back their delete dates (so we can ask them WHY they''re doing it)?
DROP TABLE IF EXISTS SecondaryAnalysis;
CREATE TABLE SecondaryAnalysis (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	analysisType INT UNSIGNED NOT NULL REFERENCES AnalysisType (id), 
	complete BOOLEAN NOT NULL, 
	deleteDate DATE NULL,								# Date when to delete the analysis
	keepFileRequestor VARCHAR(255) NULL DEFAULT NULL, 
	failed BOOLEAN NOT NULL DEFAULT FALSE, 
	notes VARCHAR(255) NULL DEFAULT NULL, 
	anGroupID INT UNSIGNED NULL DEFAULT NULL REFERENCES AnalysisGroup (id),	# The AnalysisGroup. Many:1
	statusID INT UNSIGNED NOT NULL DEFAULT 1 REFERENCES Status (id),	# The Status. Many:1
	dateAdded DATE NOT NULL, 
	PRIMARY KEY (id), 
	INDEX types (analysisType, id), 
	INDEX Status (statusID, anGroupID), 
	INDEX Analysis (anGroupID, statusID, analysisType, complete)
) ENGINE = MyISAM;


#	Group that connects Analyses with samples, and Projects
DROP TABLE IF EXISTS AnalysisGroup;
CREATE TABLE AnalysisGroup (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	projectID INT UNSIGNED NULL REFERENCES SecondaryProjectInformation (id),	# The Project. Many:1
	INDEX projects (projectID, id), 
	INDEX toProjects (id, projectID)
) ENGINE = MyISAM;


#	Connect Analysis Groups to Sequenced Samples
DROP TABLE IF EXISTS AnalysisGroupLinker;
CREATE TABLE AnalysisGroupLinker (
	sequencingID INT UNSIGNED NOT NULL REFERENCES SequencedSample (id),	# The Sequenced Sample. Many:1
	analysisID INT UNSIGNED NOT NULL REFERENCES AnalysisGroup (id),	# The AnalysisGroup. Many:1
	UNIQUE INDEX sequences (sequencingID, analysisID), 
	UNIQUE INDEX projects (analysisID, sequencingID)
) ENGINE = MyISAM;


# List of all the Tables that can have notes attached
DROP TABLE IF EXISTS Tables;
CREATE TABLE Tables (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(31) NOT NULL UNIQUE, 
	PRIMARY KEY (id)
) ENGINE = MyISAM;

INSERT INTO Tables (name) VALUES ('AccessChange');
INSERT INTO Tables (name) VALUES ('AccessChangeLog');
INSERT INTO Tables (name) VALUES ('AdHoc');
INSERT INTO Tables (name) VALUES ('AnalysisType');
INSERT INTO Tables (name) VALUES ('ApplicationTypes');
INSERT INTO Tables (name) VALUES ('BadData');
INSERT INTO Tables (name) VALUES ('ContactLinker');
INSERT INTO Tables (name) VALUES ('ContactProjectLinker');
INSERT INTO Tables (name) VALUES ('ContactSequenceLinker');
INSERT INTO Tables (name) VALUES ('Contacts');
INSERT INTO Tables (name) VALUES ('Family');
INSERT INTO Tables (name) VALUES ('FamilyMember');
INSERT INTO Tables (name) VALUES ('FileLocation');
INSERT INTO Tables (name) VALUES ('FlagstatResults');
INSERT INTO Tables (name) VALUES ('Notes');
INSERT INTO Tables (name) VALUES ('PrimaryAnalysisStatus');
INSERT INTO Tables (name) VALUES ('ProjectInformation');
INSERT INTO Tables (name) VALUES ('ProjectSampleLinker');
INSERT INTO Tables (name) VALUES ('QCResults');
INSERT INTO Tables (name) VALUES ('ReadTypes');
INSERT INTO Tables (name) VALUES ('Role');
INSERT INTO Tables (name) VALUES ('RunInformation');
INSERT INTO Tables (name) VALUES ('Sample');
INSERT INTO Tables (name) VALUES ('SecondaryAnalysis');
INSERT INTO Tables (name) VALUES ('SecondaryProjectInformation');
INSERT INTO Tables (name) VALUES ('SequencedSample');
INSERT INTO Tables (name) VALUES ('Status');
INSERT INTO Tables (name) VALUES ('Tables');


#	Table to track when someoen gets or loses access to samples, groups, or projects
DROP TABLE IF EXISTS AccessChangeLog;
CREATE TABLE AccessChangeLog (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	authorizer INT UNSIGNED NOT NULL REFERENCES ContactLinker (id),	# ID & Role of person who granted / denied the access
	authorized INT UNSIGNED NOT NULL REFERENCES ContactLinker (id),	# ID & Role of person who received / lost the access
	changeDate TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP, 
	granted BOOLEAN NOT NULL, 
	PRIMARY KEY (id), 
	INDEX Changes (changeDate), 
	INDEX AccessChanges (authorizer, granted, authorized)
) ENGINE = MyISAM;


#	Table to track the actual changes that someone authorized
DROP TABLE IF EXISTS AccessChange;
CREATE TABLE AccessChange (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	changeID INT UNSIGNED NOT NULL REFERENCES AccessChangeLog (id), # The Chagne we are tracking
	theTable INT UNSIGNED NOT NULL REFERENCES Tables (id),	# The Table targeted
	theRow INT UNSIGNED NOT NULL,							# The row / id in the table
	PRIMARY KEY (id), 
	INDEX TableAccessChanges (theTable, theRow, changeID)
) ENGINE = MyISAM;


#	Table to track Key : Value Pairs for items in tables
DROP TABLE IF EXISTS AdHoc;
CREATE TABLE AdHoc (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	theTable INT UNSIGNED NOT NULL REFERENCES Tables (id),	# The Table
	theRow INT UNSIGNED NOT NULL,							# The row / id in the table
	theKey VARCHAR (255) NOT NULL, 
	theValue VARCHAR (255) NOT NULL, 
	PRIMARY KEY (id), 
	INDEX TableKeyValuePairs (theTable, theRow, theKey, theValue), 
	INDEX KeyValuePairs (theKey, theValue, theTable, theRow)
) ENGINE = MyISAM;


# List of all the Runs and Samples that have RLIMS Problems
DROP TABLE IF EXISTS BadData;
CREATE TABLE BadData (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(63) NOT NULL, 
	isRun Boolean NOT NULL DEFAULT FALSE, 
	dateAdded TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP, 
	PRIMARY KEY (id), 
	UNIQUE INDEX items (isRun, name)
) ENGINE = MyISAM;


#	Table to track notes about items in tables
DROP TABLE IF EXISTS Notes;
CREATE TABLE Notes (
	contactID INT UNSIGNED NOT NULL REFERENCES Contacts (id),	# ID of person who wrote the note
	theTable INT UNSIGNED NOT NULL REFERENCES Tables (id),	# The Table
	theRow INT UNSIGNED NOT NULL,							# The row / id in the table
	noteType INT UNSIGNED NOT NULL, 
	theDate DATE NOT NULL, 
	note VARCHAR (8000), 
	deleted BOOLEAN NOT NULL DEFAULT FALSE,	 
	INDEX tableNotes (theTable, theRow)
) ENGINE = MyISAM;


# List of all the Runs and Samples that have RLIMS Problems
DROP TABLE IF EXISTS BadData;
CREATE TABLE BadData (
	id INT UNSIGNED NOT NULL AUTO_INCREMENT DEFAULT NULL,	# PRIMARY KEY field
	name VARCHAR(63) NOT NULL, 
	isRun boolean NOT NULL DEFAULT FALSE, 
	dateAdded DATE NOT NULL, 
	PRIMARY KEY (id), 
	UNIQUE INDEX items (isRun, name)
) ENGINE = MyISAM;

