#!/bin/bash
# wrap.sh / v1 Matt Bockol bockol.matthew@mayo.edu
#
# This script is used to capture and record exit codes, STDOUT, and STDERR for
# the commands it wraps.
#
# Several files are generated by this script:
#
# command.log - contains a line for each invocation of the wrapper. It takes the format:
#
# Fri Dec 10 09:36:35 CST 2010::2468.13579:0:PASS:/bin/true --help
#
# The fields recorded are:
# Timestamp::PID.Random:Exit Code:Status:Wrapped command with arguments
#
# {$PID}.{$RANDOM}.log - contains the command and stdout / stderr captured for each command.
# The {$PID}.{$RANDOM} value is the same as the value specified in the command.log, making
# it possible to connect the log message with the output.
#
# exit.log - contains an exit code from each wrap.sh invocation, one per line.
# This file is only created if the EXITLOG environmental variable is set
#
# flag - serves as marker that the script has completed
# This file is only created if the FLAGFILE environmental variable is set. It is created
# at the start of the wrapper invocation and moved aside at the end. It exists primarily 
# to serve as a check that the exit.log file is complete and ready to be parsed.


# Several environmental variables can be used to control the script's behavior:
#
# LOGDIR=/path/to/logging/directory/
# You can redirect the log file destination by setting the LOGDIR env variable
# If left unset, LOGDIR defaults to ./wraptmp . This directory is created if it
# doesn't already exist. If the directory can't be created or file permissions
# prevent log files from being written the script will fail.
#
# CMDLOG=/path/to/command.log
# You can specify the full path to the command log file. This will set the 
# LOGDIR to "/path/to/" automatically
#
# EXITLOG=/path/to/exit.log
# This defines where to write the exit codes of each wrapped command. If it's not
# specified the file will not be created
#
# FLAGFILE=/path/to/flag
# This defines where to create a flag file. These are empty files created at the start
# of the wrap.sh invocation and deleted when the script is completed. They serve as a 
# signal that the script execution has completed and it's safe to check the EXITLOG
# for errors.


# Examples:
#
# ./wrap.sh /bin/true --help
# will write to wraptmp/command.log:
#
# Fri Dec 10 09:36:35 CST 2010::00000.00000:0:PASS:/bin/true --help
#
# The fields recorded are:
# Timestamp::PID.Random:Exit Code:Status:Wrapped command with arguments
#
# 
# If the wrapped command produces output it will be captured along with the
# executed command in the file $LOGDIR/PID.Random.log
#
# ./wrap.sh echo "foo"
# will write to $LOGDIR/command.log
#
# Fri Dec 10 09:43:55 CST 2010::4401.11675:0:PASS:echo foo
#
# and create the file $LOGDIR/4401.11675.log containing
#
# echo "foo"
# foo
#
# To redirect the log destination
# 
# LOGDIR=/tmp/ wrap.sh echo "foo"
# 
# The command.log and saved stdout logs will be placed in /tmp/
#
# To specify a CMDLOG explicitly
#
# CMDLOG=/tmp/othercommand.log wrap.sh echo "foo"
#
# Will write /tmp/othercommand.log instead of /wraptmp/command.log
# The LOGDIR will automatically be set to /tmp/ based on CMDLOG's dirname value,
# so the associated stdout/stderr files will also be created in /tmp/ 
#
# EXITLOG=/tmp/exit.log wrap.sh echo "foo"
#
# Will append the exit code of each command wrap.sh has been called to invoke to 
# /tmp/exit.log . Subsequent runs of wrap.sh with the same EXITLOG specified will
# append to /tmp/exit.log, resulting in a file with an exit code on each line.
# This file can be read to look for failures in a given set of invocations by
# searching for non-zero lines.
#
# FLAGFILE=/tmp/flag wrap.sh echo "foo" 
# 
# Will create the file /tmp/flag when the script has completed, regardless of exit status


# capture wrapper script and the command it has been passed
WRAPPER=$0 
COMMAND=$*

LOGDIR_WAS_SET=0

if [ ! -z "$LOGDIR" ];
then
    
    LOGDIR_WAS_SET=1

    if [ ! -z "$CMDLOG" ];
    then
        if [ "$CMDLOG" == `basename "$CMDLOG"` ];
        then
            CMDLOG="${LOGDIR}/${CMDLOG}" 
        else
            echo "When LOGDIR is set (currently \"$LOGDIR\") the CMDLOG value must not include path information (currently \"$CMDLOG\"). $WRAPPER will not execute \"$COMMAND\". Exiting."
            exit 1
        fi
    else
        CMDLOG="${LOGDIR}/command.log"
    fi
else
    if [ ! -z "$CMDLOG" ];
    then
        LOGDIR=`dirname "$CMDLOG"`
    else
        echo "At minimum a LOGDIR or CMDLOG environmental variable must be defined. $WRAPPER will not execute \"$COMMAND\". Exiting."
        exit 1
    fi
fi


if [ ! -z "$EXITLOG" ];
then
    if [ "$LOGDIR_WAS_SET" -eq 1 ];
    then
        if [ "$EXITLOG" == `basename "$EXITLOG"` ];
        then
            EXITLOG="${LOGDIR}/${EXITLOG}" 
        else
            echo "When LOGDIR is set (currently \"$LOGDIR\") the EXITLOG value must not include path information (currently \"$EXITLOG\"). $WRAPPER will not execute \"$COMMAND\". Exiting."
            exit 1
        fi
    fi
else
    EXITLOG=""
fi


if [ ! -z "$FLAGFILE" ];
then
    if [ "$LOGDIR_WAS_SET" -eq 1 ];
    then
        if [ "$FLAGFILE" == `basename "$FLAGFILE"` ];
        then
            FLAGFILE="${LOGDIR}/${FLAGFILE}" 
        else
            echo "When LOGDIR is set (currently \"$LOGDIR\") the FLAGFILE value must not include path information (currently \"$FLAGFILE\"). $WRAPPER will not execute \"$COMMAND\". Exiting."
            exit 1
        fi
    fi
else
    FLAGFILE=""
fi


# fail when the script attempts to use an uninitialized variable or something other than the wrapped command returns an error code
set -u
set -e

function checkFile {

    FILE=$1
    LABEL=$2

    # check the the file's enclosing directory exists
    DIRNAME=`dirname "$FILE"`

    # if not, fail
    if [ ! -e "$DIRNAME" ];
    then
        echo "$LABEL file $FILE's cannot be written because $DIRNAME does not exists. $WRAPPER will not execute \"$COMMAND\". Exiting."
        exit 1
    fi

    # check whether the log file exists
    if [ ! -e "$FILE" ];
    then
        # if not, create an empty copy
        touch "$FILE"

        # confirm that it was successfully created. If not, fail.
        if [ ! -e "$FILE" ];
        then
            echo "$LABEL file $FILE's cannot be created. $WRAPPER will not execute \"$COMMAND\". Exiting."
            exit 1
        fi
    fi

    # check that the file is writable. If not, fail. 
    if [ ! -w "$FILE" ];
    then
            echo "$LABEL file $FILE's exists but is not writable. $WRAPPER will not execute \"$COMMAND\". Exiting."
            exit 1
    fi
}

# record the exit status and output of the wrapped command
function saveStatus {

    EXIT=$1
    CODE=$2
    COMMAND=$3
    RESULT=$4
    ERR=$5

    # create a uniq id (PID.RND) for saving command output
    UNIQ=$$.$((RANDOM%99999+10000))

    # if there's no output and no error to save then the uniq id serves no purpose
    if [ -z "$RESULT" ] && [ ! -e "$ERR" ];
    then
        UNIQ="00000.00000"
    fi

    # update the log file
    DATE=`date`
    echo "$DATE::$UNIQ:$EXIT:$CODE:$COMMAND" >> $CMDLOG

    # if we have output to STDOUT or STDERR, save the command line to the log file
    if [ -n "$RESULT" ] || [ -e "$ERR" ];
    then
        echo "CMD:" >> "$LOGDIR/$UNIQ.log"
        echo "$COMMAND" >> "$LOGDIR/$UNIQ.log"
    fi

    # if the command had output, save it to a file with the uniq id
    if [ -n "$RESULT" ];
    then
        echo "OUTPUT:" >> "$LOGDIR/$UNIQ.log"
        echo "$RESULT" >> "$LOGDIR/$UNIQ.log"
    fi

    # if the command had stderr output, save it to a file with a uniq id
    if [ -e "$ERR" ];
    then
        echo "ERROR:" >> "$LOGDIR/$UNIQ.log"
        cat "$ERR" >> "$LOGDIR/$UNIQ.log"
    
        # eliminate the temporary stderr file
        rm "$ERR"
    fi

    if [ "$EXITLOG" != "" ];
    then
        echo $EXIT >> "$EXITLOG"
    fi

    if [ "$FLAGFILE" != "" ];
    then
        rm "$FLAGFILE" 
    fi

    # the wrapper script return value reflects the success of failure of the wrapped command

    if [ "$CODE" == "PASS" ];
    then
        exit 0 
    else
        exit 1
    fi
}




# check if LOGDIR exists, if not, create it then check again.    
if [ ! -e $LOGDIR ];
then
   `mkdir -p "$LOGDIR"` 

    if [ ! -e $LOGDIR ];
    then
        echo "Unable to create $LOGDIR. Exiting."
        exit 1
    fi

fi

# make sure the command log exists
checkFile "$CMDLOG" "Command log"

# make sure the exit log can be used if it was specified
if [ "$EXITLOG" != "" ];
then
    checkFile "$EXITLOG" "Exit log"
fi

# make sure the flag file can be used and doesn't already exist if it was specified
if [ "$FLAGFILE" != "" ];
then
    if [ -e "$FLAGFILE" ];
    then
        echo "Flag file $FLAGFILE already exists.  $WRAPPER will not execute \"$COMMAND\". Exiting."
        exit 1
    fi
    checkFile "$FLAGFILE" "Flag"
fi


# catch terminations of the wrapper script
trap "saveStatus \"$?\" \"FAIL TRAP\" \"$WRAPPER '$COMMAND'\" \"\" \"/tmp/nostderr\" " INT TERM HUP


# append the current directory to the path
PATH=.:$PATH


# set aside a temp file for saving stderr
ERR=`mktemp`


# permit the script to proceed if the $COMMAND fails
set +e


# execute the wrapped command
RESULT="$($COMMAND 2> $ERR)"
EXIT=$?


# resume script failure should something return an non-zero exit code
set -e

# check for a zero filesize on the $ERR file used to capture STDERR.  If not, delete it.
ERROR_EXISTS=`stat --printf="%s" "$ERR"`
if [ $ERROR_EXISTS -eq 0 ];
then
    rm "$ERR"
fi 


# check exit status and report
if [ $EXIT -eq 0 ]; 
then
    CODE="PASS"
    saveStatus "$EXIT" "$CODE" "$COMMAND" "$RESULT" "$ERR" 
else
    CODE="FAIL"
    saveStatus "$EXIT" "$CODE" "$COMMAND" "$RESULT" "$ERR"
fi

# we should never reach this point
exit 1
