/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.interval;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import net.sf.picard.util.Interval;
import net.sf.picard.util.IntervalList;
import net.sf.samtools.SAMFileHeader;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.gatk.datasources.reference.ReferenceDataSource;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.GenomeLocParser;
import org.broadinstitute.sting.utils.GenomeLocSortedSet;
import org.broadinstitute.sting.utils.bed.BedParser;
import org.broadinstitute.sting.utils.collections.Pair;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.interval.IntervalMergingRule;
import org.broadinstitute.sting.utils.interval.IntervalSetRule;
import org.broadinstitute.sting.utils.text.XReadLines;

public class IntervalUtils {
    private static Logger logger = Logger.getLogger(IntervalUtils.class);

    public static List<GenomeLoc> parseIntervalArguments(GenomeLocParser parser, List<String> argList, boolean allowEmptyIntervalList) {
        ArrayList<GenomeLoc> rawIntervals = new ArrayList<GenomeLoc>();
        if (argList != null) {
            for (String argument : argList) {
                for (String fileOrInterval : argument.split(";")) {
                    if (fileOrInterval.trim().toLowerCase().equals("all")) {
                        if (argList.size() != 1) {
                            throw new UserException.CommandLineException(String.format("Conflicting arguments: Intervals given along with \"-L all\"", new Object[0]));
                        }
                        return null;
                    }
                    if (IntervalUtils.isUnmapped(fileOrInterval)) {
                        rawIntervals.add(GenomeLoc.UNMAPPED);
                        continue;
                    }
                    if (IntervalUtils.isIntervalFile(fileOrInterval)) {
                        try {
                            rawIntervals.addAll(IntervalUtils.intervalFileToList(parser, fileOrInterval, allowEmptyIntervalList));
                            continue;
                        }
                        catch (UserException.MalformedGenomeLoc e) {
                            throw e;
                        }
                        catch (Exception e) {
                            throw new UserException.MalformedFile(fileOrInterval, "Interval file could not be parsed in any supported format.", e);
                        }
                    }
                    rawIntervals.add(parser.parseGenomeLoc(fileOrInterval));
                }
            }
        }
        return rawIntervals;
    }

    public static List<GenomeLoc> intervalFileToList(GenomeLocParser glParser, String file_name, boolean allowEmptyIntervalList) {
        File inputFile = new File(file_name);
        ArrayList<GenomeLoc> ret = new ArrayList<GenomeLoc>();
        if (file_name.toUpperCase().endsWith(".BED")) {
            BedParser parser = new BedParser(glParser, inputFile);
            ret.addAll(parser.getLocations());
        } else {
            boolean isPicardInterval = false;
            try {
                IntervalList il = IntervalList.fromFile(inputFile);
                isPicardInterval = true;
                int nInvalidIntervals = 0;
                for (Interval interval : il.getIntervals()) {
                    if (glParser.isValidGenomeLoc(interval.getSequence(), interval.getStart(), interval.getEnd(), true)) {
                        ret.add(glParser.createGenomeLoc(interval.getSequence(), interval.getStart(), interval.getEnd(), true));
                        continue;
                    }
                    ++nInvalidIntervals;
                }
                if (nInvalidIntervals > 0) {
                    logger.warn("Ignoring " + nInvalidIntervals + " invalid intervals from " + inputFile);
                }
            }
            catch (Exception e) {
                if (isPicardInterval) {
                    throw new UserException.CouldNotReadInputFile(inputFile, e);
                }
                try {
                    XReadLines reader = new XReadLines(new File(file_name));
                    for (String line : reader) {
                        if (line.trim().length() <= 0) continue;
                        ret.add(glParser.parseGenomeLoc(line));
                    }
                    reader.close();
                }
                catch (IOException e2) {
                    throw new UserException.CouldNotReadInputFile(inputFile, (Exception)e2);
                }
            }
        }
        if (ret.isEmpty() && !allowEmptyIntervalList) {
            throw new UserException("The interval file " + inputFile.getAbsolutePath() + " contains no intervals " + "that could be parsed, and the unsafe operation ALLOW_EMPTY_INTERVAL_LIST has " + "not been enabled");
        }
        return ret;
    }

    public static boolean isUnmapped(String interval) {
        return interval != null && interval.trim().toLowerCase().equals("unmapped");
    }

    public static List<GenomeLoc> mergeListsBySetOperator(List<GenomeLoc> setOne, List<GenomeLoc> setTwo, IntervalSetRule rule) {
        if (setOne == null || setOne.size() == 0 || setTwo == null || setTwo.size() == 0) {
            return setOne == null || setOne.size() == 0 ? setTwo : setOne;
        }
        if (rule == IntervalSetRule.UNION) {
            setOne.addAll(setTwo);
            return setOne;
        }
        int iOne = 0;
        int iTwo = 0;
        LinkedList<GenomeLoc> retList = new LinkedList<GenomeLoc>();
        while (iTwo < setTwo.size() && iOne < setOne.size()) {
            if (setTwo.get(iTwo).isBefore(setOne.get(iOne))) {
                ++iTwo;
                continue;
            }
            if (setOne.get(iOne).isBefore(setTwo.get(iTwo))) {
                ++iOne;
                continue;
            }
            retList.add(setOne.get(iOne).intersect(setTwo.get(iTwo)));
            if (setOne.get(iOne).getStop() < setTwo.get(iTwo).getStop()) {
                ++iOne;
                continue;
            }
            ++iTwo;
        }
        if (retList.size() == 0) {
            throw new UserException.BadInput("The INTERSECTION of your -BTI and -L options produced no intervals.");
        }
        return retList;
    }

    public static GenomeLocSortedSet sortAndMergeIntervals(GenomeLocParser parser, List<GenomeLoc> intervals, IntervalMergingRule mergingRule) {
        Collections.sort(intervals);
        intervals = IntervalUtils.mergeIntervalLocations(intervals, mergingRule);
        return GenomeLocSortedSet.createSetFromList(parser, intervals);
    }

    public static boolean isIntervalFile(String str) {
        return IntervalUtils.isIntervalFile(str, true);
    }

    public static boolean isIntervalFile(String str, boolean checkExists) {
        File file = new File(str);
        if (str.toUpperCase().endsWith(".BED") || str.toUpperCase().endsWith(".LIST") || str.toUpperCase().endsWith(".PICARD") || str.toUpperCase().endsWith(".INTERVAL_LIST") || str.toUpperCase().endsWith(".INTERVALS")) {
            if (!checkExists) {
                return true;
            }
            if (file.exists()) {
                return true;
            }
            throw new UserException.CouldNotReadInputFile(file, "The interval file does not exist.");
        }
        if (file.exists()) {
            throw new UserException.CouldNotReadInputFile(file, String.format("The interval file %s does not have one of the supported extensions (.bed, .list, .picard, .interval_list, or .intervals). Please rename your file with the appropriate extension. If %s is NOT supposed to be a file, please move or rename the file at location %s", str, str, file.getAbsolutePath()));
        }
        return false;
    }

    public static Map<String, Long> getContigSizes(File reference) {
        ReferenceDataSource referenceSource = new ReferenceDataSource(reference);
        List<GenomeLoc> locs = GenomeLocSortedSet.createSetFromSequenceDictionary(referenceSource.getReference().getSequenceDictionary()).toList();
        LinkedHashMap<String, Long> lengths = new LinkedHashMap<String, Long>();
        for (GenomeLoc loc : locs) {
            lengths.put(loc.getContig(), loc.size());
        }
        return lengths;
    }

    public static int countContigIntervals(List<GenomeLoc> locs) {
        int maxFiles = 0;
        String contig = null;
        for (GenomeLoc loc : locs) {
            if (contig != null && contig.equals(loc.getContig())) continue;
            ++maxFiles;
            contig = loc.getContig();
        }
        return maxFiles;
    }

    public static void scatterContigIntervals(SAMFileHeader fileHeader, List<GenomeLoc> locs, List<File> scatterParts) {
        IntervalList intervalList = null;
        int fileIndex = -1;
        int locIndex = 0;
        String contig = null;
        for (GenomeLoc loc : locs) {
            if (!(fileIndex + 1 >= scatterParts.size() || contig != null && contig.equals(loc.getContig()))) {
                if (intervalList != null) {
                    intervalList.write(scatterParts.get(fileIndex));
                    intervalList = null;
                }
                ++fileIndex;
                contig = loc.getContig();
            }
            if (intervalList == null) {
                intervalList = new IntervalList(fileHeader);
            }
            intervalList.add(IntervalUtils.toInterval(loc, ++locIndex));
        }
        if (intervalList != null) {
            intervalList.write(scatterParts.get(fileIndex));
        }
        if (fileIndex + 1 != scatterParts.size()) {
            throw new UserException.BadArgumentValue("scatterParts", String.format("Only able to write contigs into %d of %d files.", fileIndex + 1, scatterParts.size()));
        }
    }

    public static void scatterFixedIntervals(SAMFileHeader fileHeader, List<GenomeLoc> locs, List<Integer> splits, List<File> scatterParts) {
        if (splits.size() != scatterParts.size()) {
            throw new UserException.BadArgumentValue("splits", String.format("Split points %d does not equal the number of scatter parts %d.", splits.size(), scatterParts.size()));
        }
        int fileIndex = 0;
        int locIndex = 1;
        int start = 0;
        for (Integer stop : splits) {
            IntervalList intervalList = new IntervalList(fileHeader);
            for (int i = start; i < stop; ++i) {
                intervalList.add(IntervalUtils.toInterval(locs.get(i), locIndex++));
            }
            intervalList.write(scatterParts.get(fileIndex++));
            start = stop;
        }
    }

    public static List<Integer> splitFixedIntervals(List<GenomeLoc> locs, int numParts) {
        if (locs.size() < numParts) {
            throw new UserException.BadArgumentValue("scatterParts", String.format("Cannot scatter %d locs into %d parts.", locs.size(), numParts));
        }
        long locsSize = 0L;
        for (GenomeLoc loc : locs) {
            locsSize += loc.size();
        }
        ArrayList<Integer> splitPoints = new ArrayList<Integer>();
        IntervalUtils.addFixedSplit(splitPoints, locs, locsSize, 0, locs.size(), numParts);
        Collections.sort(splitPoints);
        splitPoints.add(locs.size());
        return splitPoints;
    }

    private static void addFixedSplit(List<Integer> splitPoints, List<GenomeLoc> locs, long locsSize, int startIndex, int stopIndex, int numParts) {
        if (numParts < 2) {
            return;
        }
        int halfParts = (numParts + 1) / 2;
        Pair<Integer, Long> splitPoint = IntervalUtils.getFixedSplit(locs, locsSize, startIndex, stopIndex, halfParts, numParts - halfParts);
        int splitIndex = (Integer)splitPoint.first;
        long splitSize = (Long)splitPoint.second;
        splitPoints.add(splitIndex);
        IntervalUtils.addFixedSplit(splitPoints, locs, splitSize, startIndex, splitIndex, halfParts);
        IntervalUtils.addFixedSplit(splitPoints, locs, locsSize - splitSize, splitIndex, stopIndex, numParts - halfParts);
    }

    private static Pair<Integer, Long> getFixedSplit(List<GenomeLoc> locs, long locsSize, int startIndex, int stopIndex, int minLocs, int maxLocs) {
        int splitIndex = startIndex;
        long splitSize = 0L;
        for (int i = 0; i < minLocs; ++i) {
            splitSize += locs.get(splitIndex).size();
            ++splitIndex;
        }
        long halfSize = locsSize / 2L;
        while (splitIndex < stopIndex - maxLocs && splitSize < halfSize) {
            splitSize += locs.get(splitIndex).size();
            ++splitIndex;
        }
        return new Pair<Integer, Long>(splitIndex, splitSize);
    }

    private static Interval toInterval(GenomeLoc loc, int locIndex) {
        return new Interval(loc.getContig(), loc.getStart(), loc.getStop(), false, "interval_" + locIndex);
    }

    public static List<GenomeLoc> mergeIntervalLocations(List<GenomeLoc> raw, IntervalMergingRule rule) {
        if (raw.size() <= 1) {
            return raw;
        }
        ArrayList<GenomeLoc> merged = new ArrayList<GenomeLoc>();
        Iterator<GenomeLoc> it = raw.iterator();
        GenomeLoc prev = it.next();
        while (it.hasNext()) {
            GenomeLoc curr = it.next();
            if (prev.overlapsP(curr)) {
                prev = prev.merge(curr);
                continue;
            }
            if (prev.contiguousP(curr) && rule == IntervalMergingRule.ALL) {
                prev = prev.merge(curr);
                continue;
            }
            merged.add(prev);
            prev = curr;
        }
        merged.add(prev);
        return merged;
    }
}

