/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.gatk.datasources.sample;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMReadGroupRecord;
import net.sf.samtools.SAMRecord;
import org.broadinstitute.sting.gatk.datasources.sample.PropertyDefinition;
import org.broadinstitute.sting.gatk.datasources.sample.Sample;
import org.broadinstitute.sting.gatk.datasources.sample.SampleAlias;
import org.broadinstitute.sting.gatk.datasources.sample.SampleFileParser;
import org.broadinstitute.sting.gatk.datasources.sample.SampleParser;
import org.broadinstitute.sting.utils.SampleUtils;
import org.broadinstitute.sting.utils.exceptions.StingException;
import org.broadinstitute.sting.utils.variantcontext.Genotype;
import org.broadinstitute.sting.utils.variantcontext.VariantContext;
import org.yaml.snakeyaml.TypeDescription;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;

public class SampleDataSource {
    private SAMFileHeader header;
    private final HashMap<String, Sample> samples = new HashMap();
    private HashMap<String, String> sampleAliases = new HashMap();
    public static final String[] specialProperties = new String[]{"familyId", "population", "gender"};
    public static final String[] specialRelationships = new String[]{"mother", "father"};

    public SampleDataSource(SAMFileHeader header, List<File> sampleFiles) {
        this();
        this.header = header;
        for (String sampleName : SampleUtils.getSAMFileSamples(header)) {
            if (this.hasSample(sampleName)) continue;
            Sample newSample = new Sample(sampleName);
            newSample.setSAMFileEntry(true);
            this.samples.put(sampleName, newSample);
        }
        if (sampleFiles != null) {
            for (File file : sampleFiles) {
                this.addFile(file);
            }
        }
    }

    public SampleDataSource() {
        this.samples.put(null, new Sample(null));
    }

    public void addSamplesFromSAMHeader(SAMFileHeader header) {
        for (String sampleName : SampleUtils.getSAMFileSamples(header)) {
            if (this.hasSample(sampleName)) continue;
            Sample newSample = new Sample(sampleName);
            newSample.setSAMFileEntry(true);
            this.samples.put(sampleName, newSample);
        }
    }

    public void addFile(File sampleFile) {
        SampleFileParser parser;
        BufferedReader reader;
        try {
            reader = new BufferedReader(new FileReader(sampleFile));
        }
        catch (IOException e) {
            throw new StingException("Could not open sample file " + sampleFile.getAbsolutePath(), e);
        }
        Constructor con = new Constructor(SampleFileParser.class);
        TypeDescription desc = new TypeDescription(SampleFileParser.class);
        desc.putListPropertyType("propertyDefinitions", PropertyDefinition.class);
        desc.putListPropertyType("sampleAliases", SampleAlias.class);
        con.addTypeDescription(desc);
        Yaml yaml = new Yaml(con);
        try {
            parser = (SampleFileParser)yaml.load(reader);
        }
        catch (Exception e) {
            throw new StingException("There was a syntactic error with the YAML in sample file " + sampleFile.getAbsolutePath(), e);
        }
        boolean restrictProperties = parser.getAllowedProperties() != null;
        boolean restrictRelationships = parser.getAllowedRelationships() != null;
        boolean restrictPropertyValues = parser.getPropertyDefinitions() != null;
        HashMap<String, HashSet> propertyValues = null;
        if (restrictPropertyValues) {
            propertyValues = new HashMap<String, HashSet>();
            for (PropertyDefinition def : parser.getPropertyDefinitions()) {
                HashSet<String> set = new HashSet<String>();
                for (String value : def.getValues()) {
                    set.add(value);
                }
                propertyValues.put(def.getProperty(), set);
            }
        }
        this.validateAliases(parser);
        for (SampleParser sampleParser : parser.getSamples()) {
            try {
                Sample sample = this.getSampleById(sampleParser.getId());
                if (sample == null) {
                    sample = new Sample(sampleParser.getId());
                }
                this.addSample(sample);
                sample.setSampleFileEntry(true);
                if (sampleParser.getProperties() != null) {
                    for (String property : sampleParser.getProperties().keySet()) {
                        if (restrictProperties && !this.isPropertyValid(property, parser.getAllowedProperties())) {
                            throw new StingException(property + " is an invalid property. It is not included in the list " + "of allowed properties.");
                        }
                        if (restrictPropertyValues && !this.isValueAllowed(property, sampleParser.getProperties().get(property), propertyValues)) {
                            throw new StingException("The value of property '" + property + "' is invalid. " + "It is not included in the list of allowed values for this property.");
                        }
                        if (sample.getProperty(property) != null && sample.getProperty(property) != sampleParser.getProperties().get(property)) {
                            throw new StingException(property + " is a conflicting property!");
                        }
                        this.saveProperty(sample, property, sampleParser.getProperties().get(property));
                    }
                }
                if (sampleParser.getRelationships() == null) continue;
                for (String relationship : sampleParser.getRelationships().keySet()) {
                    String relativeId = sampleParser.getRelationships().get(relationship);
                    if (relativeId == null) {
                        throw new StingException("The relationship cannot be null");
                    }
                    if (restrictRelationships && !this.isRelationshipValid(relationship, parser.getAllowedRelationships())) {
                        throw new StingException(relationship + " is an invalid relationship");
                    }
                    if (sample.getRelationship(relationship) != null) {
                        if (sample.getRelationship(relationship).getId() == sampleParser.getProperties().get(relationship)) continue;
                        throw new StingException(relationship + " is a conflicting relationship!");
                    }
                    this.saveRelationship(sample, relationship, relativeId);
                }
            }
            catch (Exception e) {
                throw new StingException("An error occurred while loading this sample from the sample file: " + sampleParser.getId(), e);
            }
        }
    }

    private boolean isValueAllowed(String key, Object value, HashMap<String, HashSet> valuesList) {
        if (!valuesList.containsKey(key)) {
            return true;
        }
        if (value.getClass() != String.class) {
            return false;
        }
        return valuesList.get(key).contains(value);
    }

    private void validateAliases(SampleFileParser parser) {
        if (parser.getSampleAliases() == null) {
            return;
        }
        HashSet<String> mainIds = new HashSet<String>();
        HashSet<String> otherIds = new HashSet<String>();
        for (SampleAlias sampleAlias : parser.getSampleAliases()) {
            mainIds.add(sampleAlias.getMainId());
            for (String otherId : sampleAlias.getOtherIds()) {
                if (mainIds.contains(otherId)) {
                    throw new StingException(String.format("The aliases in your sample file are invalid - the alias %s cannot be both a main ID and an other ID", otherId));
                }
                if (otherIds.add(otherId)) continue;
                throw new StingException(String.format("The aliases in your sample file are invalid - %s is listed as an alias more than once.", otherId));
            }
        }
    }

    private boolean isPropertyValid(String property, String[] allowedProperties) {
        for (String allowedProperty : specialProperties) {
            if (!property.equals(allowedProperty)) continue;
            return true;
        }
        for (String allowedProperty : allowedProperties) {
            if (!property.equals(allowedProperty)) continue;
            return true;
        }
        return false;
    }

    private boolean isRelationshipValid(String relationship, String[] allowedRelationships) {
        for (String allowedRelationship : specialRelationships) {
            if (!relationship.equals(allowedRelationship)) continue;
            return true;
        }
        for (String allowedRelationship : allowedRelationships) {
            if (!relationship.equals(allowedRelationship)) continue;
            return true;
        }
        return false;
    }

    private void saveProperty(Sample sample, String key, Object value) {
        if (key.equals("gender")) {
            if (((String)value).toLowerCase().equals("male")) {
                value = Sample.Gender.MALE;
            } else if (((String)value).toLowerCase().equals("female")) {
                value = Sample.Gender.FEMALE;
            } else if (((String)value).toLowerCase().equals("unknown")) {
                value = Sample.Gender.UNKNOWN;
            } else if (value != null) {
                throw new StingException("'gender' property must be male, female, or unknown.");
            }
        }
        try {
            sample.setProperty(key, value);
        }
        catch (Exception e) {
            throw new StingException("Could not save property " + key, e);
        }
    }

    private void saveRelationship(Sample sample, String key, String relativeId) {
        Sample relative = this.getSampleById(relativeId);
        if (relative == null) {
            relative = new Sample(relativeId);
            this.addSample(relative);
        }
        sample.setRelationship(key, relative);
    }

    private String aliasFilter(String sampleId) {
        if (!this.sampleAliases.containsKey(sampleId)) {
            return sampleId;
        }
        return this.sampleAliases.get(sampleId);
    }

    private void addSample(Sample sample) {
        this.samples.put(sample.getId(), sample);
    }

    public boolean hasSample(String id) {
        return this.samples.get(this.aliasFilter(id)) != null;
    }

    public Sample getSampleById(String id) {
        return this.samples.get(this.aliasFilter(id));
    }

    public Sample getSampleByReadGroup(SAMReadGroupRecord readGroup) {
        String nameFromReadGroup = readGroup.getSample();
        return this.getSampleById(nameFromReadGroup);
    }

    public Sample getSampleByRead(SAMRecord read) {
        return this.getSampleByReadGroup(read.getReadGroup());
    }

    public int sampleCount() {
        return this.samples.size();
    }

    public Set<Sample> getFamily(String familyId) {
        HashSet<Sample> familyMembers = new HashSet<Sample>();
        for (Sample sample : this.samples.values()) {
            if (sample.getFamilyId() == null || !sample.getFamilyId().equals(familyId)) continue;
            familyMembers.add(sample);
        }
        return familyMembers;
    }

    public Set<Sample> getChildren(Sample sample) {
        HashSet<Sample> children = new HashSet<Sample>();
        for (Sample familyMember : this.getFamily(sample.getFamilyId())) {
            if (familyMember.getMother() != sample && familyMember.getFather() != sample) continue;
            children.add(familyMember);
        }
        return children;
    }

    public Set<Sample> getSamples() {
        HashSet<Sample> set = new HashSet<Sample>();
        set.addAll(this.samples.values());
        return set;
    }

    public Set<Sample> getSamples(Collection<String> sampleNameList) {
        HashSet<Sample> samples = new HashSet<Sample>();
        for (String name : sampleNameList) {
            try {
                samples.add(this.getSampleById(name));
            }
            catch (Exception e) {
                throw new StingException("Could not get sample with the following ID: " + name, e);
            }
        }
        return samples;
    }

    public Set<Sample> getSamplesWithProperty(String key) {
        HashSet<Sample> toReturn = new HashSet<Sample>();
        for (Sample s : this.samples.values()) {
            if (!s.hasProperty(key)) continue;
            toReturn.add(s);
        }
        return toReturn;
    }

    public Set<Sample> getSamplesWithProperty(String key, String value) {
        Set<Sample> toReturn = this.getSamplesWithProperty(key);
        for (Sample s : toReturn) {
            if (s.getProperty(key).equals(value)) continue;
            toReturn.remove(s);
        }
        return toReturn;
    }

    public Sample getOrCreateSample(String id) {
        Sample sample = this.getSampleById(id);
        if (sample == null) {
            sample = new Sample(id);
            this.addSample(sample);
        }
        return sample;
    }

    public Set<Sample> getSAMFileSamples() {
        HashSet<Sample> toReturn = new HashSet<Sample>();
        for (Sample sample : this.samples.values()) {
            if (!sample.hasSAMFileEntry()) continue;
            toReturn.add(sample);
        }
        return toReturn;
    }

    public Set<Sample> getSamplesByVariantContext(VariantContext context) {
        HashSet<Sample> samples = new HashSet<Sample>();
        for (String sampleName : context.getSampleNames()) {
            samples.add(this.getOrCreateSample(sampleName));
        }
        return samples;
    }

    public VariantContext subContextFromSampleProperty(VariantContext context, String key, String value) {
        HashSet<String> samplesWithProperty = new HashSet<String>();
        for (String sampleName : context.getSampleNames()) {
            Sample s = this.samples.get(sampleName);
            if (s == null || !s.hasProperty(key) || !s.getProperty(key).equals(value)) continue;
            samplesWithProperty.add(sampleName);
        }
        Map<String, Genotype> genotypes = context.getGenotypes(samplesWithProperty);
        return context.subContextFromGenotypes(genotypes.values());
    }

    public static SampleDataSource createEmptyDataSource() {
        SAMFileHeader header = new SAMFileHeader();
        return new SampleDataSource(header, null);
    }
}

