#!/bin/bash
#
#   Setup environment variables for the secondary analysis system
#   environment
#
#   This file should be sourced in your .profile, for example
#
#set -x

####

usage()
{
cat << EOF
######
##      eSNV-Detect v1.0 installation script
##      Script Options:
##      -d	<directory>	-	(REQUIRED) full/path/to/installation directory for eSNV-Detect
##      -h		-	Display this usage/help text (No arg)
##
#################################################################################################################
##
## Authors:             Saurabh Baheti
## Creation Date:       January 19 2014
## Last Modified:       January 19 2014
##
## For questions, comments, or concerns, contact Saurabh (baheti.saurabh@mayo.edu)
##
#################################################################################################################
EOF
}

while getopts "d:h" OPTION; do
  case $OPTION in
        h) usage
        exit ;;
        d) directory=$OPTARG ;;
   \?) echo "Invalid option: -$OPTARG. See output file for usage." >&2
       usage
       exit ;;
    :) echo "Option -$OPTARG requires an argument. See output file for usage." >&2
       usage
       exit ;;
  esac
done

if [[ -z "$directory"  && ! -d "$directory" ]]
then
        echo "Must provide at least required options. See output file for usage." >&2
        usage
        exit 1;
fi

echo -e "\n###############################################################################################"
echo -e "#"
echo -e "# This is the eSNV-Detect v1.0 installer. "
echo -e "# It will work under a bash shell"
echo -e "# It will download all necessary files and install them. "
echo -e "# Please restart your shell to make changes take effect"
echo -e "#"
echo -e "###############################################################################################"
START=$(date +%s)

####change this if user wants to run some other species or build on genome
genome="hg19"
####################

#### path to the location where package has been downloaded and unpacked.
path=$directory

###chnage this if uder wants to run some chromsome or some other specicies
chrs="1:2:3:4:5:6:7:8:9:10:11:12:13:14:15:16:17:18:19:20:21:22:X:Y:M"
####################

## make all the files executables
chmod -Rf 777 $path
echo -e "\n*** Checking for Executables in your system ***"

#### check if wget exists
#### used to download resources and ref files.
wget=`which wget`
wg=`wget -V 2>&1`
if [[ ! `echo $wg | grep Wget` ]]
then
	echo "No commandline download tool found on your system. Please install wget on your machine and restart the installer"
	exit 1;
else
	ver=`echo $wg | awk '{print $3}'`
	echo "wget version: $ver                                      already installed, nothing to do ..."
fi

#### check for gcc complier
gcc=`gcc --version 2>&1`
if [[ ! `echo $gcc | grep GCC` ]]
then
	echo "no gcc compiler installed. Please install a gcc compiler on your machine and restart the installer"
	exit 1;
else
	ver=`echo $gcc | awk '{print $3}'`
	echo "gcc version: $ver                                      already installed, nothing to do ..."	
fi

#### check and add Perl to env variable
perl=`which perl`
pp=`perl --version`
if [[ ! `echo $pp | grep Perl`  ]]
then
	echo "No perl found, please install perl on your machine and restart the installer"
	exit 1;
else
	ver=`echo $pp | awk -F ',' '{print $2}'`
	echo "perl version: $ver                                      already installed, nothing to do ..."	
fi
PERL_HOME=$perl_path
export PERL_HOME
export PERL5LIB=$perl:$PERL5LIB

### check for gunzip
gunzip=`which gunzip`
gu=`gunzip -V 2>&1`
if [[ ! `echo $gu | grep -E 'gunzip|gzip'`  ]]
then
	echo "No gunzip found, please install gunzip on your machine and restart the installer"
	exit 1;
else
	ver=`echo $gu | awk '{print $2}'`
	echo "gunzip version: $ver                                      already installed, nothing to do ..."
fi

### check for unzip
unzip=`which unzip`
uz=`unzip -V 2>&1`
if [[ ! `echo $uz | grep -E 'UnZip'`  ]]
then
	echo "No unzip found, please install unzip on your machine and restart the installer"
	exit 1;
else
	ver=`echo $uz | awk '{print $2}'`
	echo "unzip version: $ver                                      already installed, nothing to do ..."
fi

### check for unzip
make=`which make`
ma=`make -v 2>&1`
if [[ ! `echo $ma | grep -E 'Make'`  ]]
then
	echo "No make found, please install make on your machine and restart the installer"
	exit 1;
else
	ver=`echo $ma | awk '{print $3}'`
	echo "make version: $ver                                      already installed, nothing to do ..."
fi

### check for tar ball
tar=`which tar`
ut=`tar --version 2>&1`
if [[ ! `echo $ut | grep -E 'GNU tar'`  ]]
then
	echo "No GNU tar found, please install tar on your machine and restart the installer"
	exit 1;
else
	ver=`echo $ut | cut -d ' ' -f4`
	echo "tar version: $ver                                      already installed, nothing to do ..."
fi

#### check and add JAVA to env variable
java=`which java`
jj=`java -version 2>&1`
if [[ ! `echo $jj | grep Java` ]]
then
	echo "No JAVA found, please install java on your machine and restart the installer"
	exit 1;
else
	ver=`echo $jj | awk '{print $3}'`
	echo "java version: $ver                                      already installed, nothing to do ..."
fi
JAVA_HOME=$java_path
export JAVA_HOME

#### remove last /java from java_path
#### use later to add to tool_info file
java=${java%*/*}
perl=${perl%*/*}
path=${path%*/}
if [ ! -d $path/bin ] 
then
	mkdir $path/bin
else
	rm -Rf $path/bin
	mkdir $path/bin	
fi
#### install all the tools
echo -e "\n*** Installing all the required tools ***"
### PICARD
############
if [ ! -d $path/bin/picard/picard-tools-1.106 ]
then
	echo -e "\n PICARD 1.106 \n"
	mkdir $path/bin/picard
	cd $path/bin/picard
	$wget -q http://downloads.sourceforge.net/project/picard/picard-tools/1.106/picard-tools-1.106.zip
	echo -e " - unpacking picard\n";
	unzip_picard=`$unzip -d $path/bin/picard/ picard-tools-1.106.zip 2>&1`
else
	echo -e "PICARD version: 1.106                                      already installed, nothing to do ..."
fi
picard="$path/bin/picard/picard-tools-1.106"
### TABIX
############
if [ ! -d $path/bin/tabix/tabix-0.2.6 ]
then
	echo -e "\n TABIX 0.2.6 \n"
	mkdir $path/bin/tabix
	cd $path/bin/tabix
	$wget -q http://sourceforge.net/projects/samtools/files/tabix/tabix-0.2.6.tar.bz2/download
	echo -e " - unpacking tabix\n";
	untar_tabix=`$tar -C $path/bin/tabix -xvjf tabix-0.2.6.tar.bz2 2>&1`
        echo -e " - installing tabix\n"
        $make -i -k -C $path/bin/tabix/tabix-0.2.6/ > $path/bin/tabix/tabix-0.2.6/log.txt 2>&1       
else
	echo -e "TABIX version: 0.2.6                                      already installed, nothing to do ..."
fi
tabix="$path/bin/tabix/tabix-0.2.6"
### GATK
###########
if [ ! -d $path/bin/gatk/GenomeAnalysisTK-1.6-9-g47df7bb ]
then
	echo -e "\n GATK 1.6.9 \n";
	mkdir $path/bin/gatk
	cd $path/bin/gatk
	$wget -q ftp://ftp.broadinstitute.org/distribution/gsa/GenomeAnalysisTK/GenomeAnalysisTK-1.6-9-g47df7bb.tar.bz2
	echo -e " - unpacking GATK\n";
        untar_gatk=`$tar -C $path/bin/gatk -xvjf GenomeAnalysisTK-1.6-9-g47df7bb.tar.bz2 2>&1`
else
	echo -e "GATK version: 1.6.9                                      already installed, nothing to do ..."
fi
gatk="$path/bin/gatk/GenomeAnalysisTK-1.6-9-g47df7bb"
### SAMTOOLS
###########
if [ ! -d $path/bin/samtools/samtools-0.1.19 ]
then
	echo -e "\n SAMTOOLS 0.1.19 \n";
	mkdir $path/bin/samtools
	cd $path/bin/samtools
	$wget -q http://downloads.sourceforge.net/project/samtools/samtools/0.1.19/samtools-0.1.19.tar.bz2
	echo -e " - unpacking samtools\n";
	untar_samtools=`$tar -C $path/bin/samtools -xvjf samtools-0.1.19.tar.bz2 2>&1` 
	echo -e " - installing samtools\n";
	$make -i -k -C $path/bin/samtools/samtools-0.1.19/ > $path/bin/samtools/samtools-0.1.19/log.txt 2>&1
else
	echo -e "SAMTOOLS version: 0.1.19                                      already installed, nothing to do ..."
fi

### making all the tools executables to all
chmod -Rf 777 $path/bin
### creating local variables	
samtools=$path/bin/samtools/samtools-0.1.19

# change directory to resources folder
cd $path/resources
echo -e "\n Downloading all the required references\n"
### reference genoome fasta file
###############################
if [ -s $path/resources/allchr.$genome.fa ]; then rm $path/resources/allchr.$genome.fa; fi
for chr in `echo $chrs | tr ":" " "`
do
	echo -e  "\n Downloading chr$chr ...\n"
    if [ -s $path/resources/chr${chr}.fa.gz ];then rm $path/resources/chr${chr}.fa.gz;fi
    $wget -q ftp://hgdownload.cse.ucsc.edu/goldenPath/${genome}/chromosomes/chr${chr}.fa.gz
	zcat $path/resources/chr${chr}.fa.gz >> $path/resources/allchr.$genome.fa
	rm $path/resources/chr${chr}.fa.gz
done
echo -e "\n creating the index for $genome ...\n"
$samtools/samtools faidx $path/resources/allchr.$genome.fa

cd $path/resources
### dbSNP referece island file
#####################
echo -e "\nDownloading the dbSNP reference file for $genome\n"
if [ -s $path/resources/00-All.vcf.gz ];then rm $path/resources/00-All.vcf.gz;fi

if [ -s $path/resources/dbsnp.vcf.gz ];then rm $path/resources/dbsnp.vcf.gz;fi
$wget -q ftp://ftp.ncbi.nih.gov/snp/organisms/human_9606/VCF/00-All.vcf.gz
zcat $path/resources/00-All.vcf.gz | awk '{if($0 ~ /^#/) {print} else {print "chr"$0};}' | sed -e '/chrMT/s//chrM/g' | $tabix/bgzip > $path/resources/dbsnp.vcf.gz
$tabix/tabix -f -p vcf $path/resources/dbsnp.vcf.gz

### create the config too info files
config=$( cat $path/resources/config.txt | tr "\n" "||")

####replace @PATH@ with actual path in tool info file.
config=${config//@PATH@/$path}

#### replace @JAVA@ with actual path to java
config=${config//@JAVA@/$java}

#### replace @JAVA@ with actual path to java
config=${config//@PERL@/$perl}

echo $config | tr "||" "\n" > $path/config.txt

echo -e "\n configuration file is generated here : $path/config.txt ...\n"
echo -e "\n USER needs to download annovar and install it in $path/bin using the annovar.sh utility script provided with the package"
chmod -Rf 777 $path
cd $path
END=$(date +%s)
DIFF=$(( $END - $START ))
echo -e "\n ****setup script for took $DIFF seconds**** \n"
	

